# -*- coding: utf-8 -*-
'''  
Created on 3 oct. 2004
Ported in Python in May 2020
'''
import sys
import os 
import logging
import json
import copy
from collections import namedtuple

import numpy as np
from scipy import spatial

from langutil import C_OK, C_FAIL, C_ERROR
from pypos3d.wftk.WFBasic import Vector3d, Point3d, BoundingBox, BoundingSphere, calcMLS
from pypos3d.wftk.Repere import Repere
from pypos3d.wftk.PoserFileParser import PoserFileParser, ParsingErrorException
from pypos3d.pftk.PoserBasic import PoserConst, PoserObject, PToken, PoserToken, Lang, TAS, index, create, WBoolLine, RemoveQuotes, nodeNameNo
from pypos3d.pftk.SimpleAttribut import KSA, AddChildSA, ValueOpDelta, OffSA

 
PtMapping = namedtuple('PtMapping', ['srcNo', 'refNo', 'refGeom', 'dist' ])
  
#   
class StructuredAttribut(PoserObject):
  ''' Structured attribute of a Poser file : NAME { list of PoserObject } '''
 
  def __init__(self, n=''):
    super(StructuredAttribut, self).__init__()
    self.setName(n)
    self._lstAttr = [ ]

  def getLstAttr(self): return self._lstAttr

  # 
  # Read a structured attribute data from the file. This class name, the name
  # and the opening bracket are supposed to be consumed by the caller.
  # Because it is the mean it uses to recognize a structured attribute.
  #    
  def read(self, st):
    nextLine = True
    st.getToken()
    while st.ttype != PoserFileParser.TT_EOF:
      if st.ttype == PoserFileParser.TT_EOL:
        st.getToken()
      if st.ttype == PoserFileParser.TT_WORD:
        if st.isRightBracket():
          #  End of current structured attribute
          break

        cn = st.sval
        try:
          vc = Lang[st.sval]
          #  Known word
          if vc.isStructured:
            nom = st.getToNextLine()
            if st.ttype == PoserFileParser.TT_EOL:
              nom = ""

            sta = create(vc, nom)

            self.addAttribut(sta)

            #  Read the opening bracket
            st.getToken()
            if st.isLeftBracket():
              sta.read(st)
            else:
              if sta.isAmbi():
                #  Skip to next line
                st.pushBack()
                nextLine = False
              else:
                logging.warning( "Line[%s]: '{' is missing for %s", st.lineno, cn)
          else:
            if vc.isDirect:
              st.getToNextLine()
              self.setDirect(vc, st.sval)
            else:
              sa = create(vc, st.sval)
              #  Try : 17AOU2007 : Read before add
              sa.read(st)
              self.addAttribut(sa)
            nextLine = False

        except KeyError: #  Mot inconnu
          logging.info("Line[%s] - Unknown word:%s", st.lineno(), cn)

      if nextLine:
        st.skipToNextLine()
      else:
        nextLine = True
      #  Get next token
      st.getToken()
      # End of While

  def isAmbi(self): return self.ambiguious

  # @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    for po in self._lstAttr:
      po.write(fw, nextPfx)

    if len(self._lstAttr) == 0:
      fw.write('\n')

    fw.write(nextPfx + "}\n")

  def getAttribut(self, attrTypeOrName):
    try:
      if isinstance(attrTypeOrName, PToken):
        po = next(x for x in self._lstAttr if x.getPoserType() == attrTypeOrName)
      else:
        po = next(x for x in self._lstAttr if x.getName() == attrTypeOrName)

      return po
    except StopIteration:
      #  20080417 : Fix - Shall return null if not found
      return None


  def replaceAttribut(self, attrTypeName, nattr):
    for i,po in enumerate(self._lstAttr):
      if po.getPoserType()==attrTypeName:
        self._lstAttr[i] = nattr
        return True

    return False

  def deleteAttribut(self, attrName):
    for po in self._lstAttr:
      if po.getName() == attrName:
        self._lstAttr.remove(po)
        return True

    return False

  def addAttribut(self, po):
    self._lstAttr.append(po)

  def setDirect(self, tokenID, val):
    #  Dummy (debug) implementation
    logging.warning("Unexpected Direct:" + tokenID.token + " (" + (val if val else 'null') + ") in " + str(self.__class__) + " for " + self.getPoserType().token)

  def findAttribut(self, attrTypeName:PoserToken, name=None) -> PoserObject :
    if name:
      try:
        po = next(x for x in self._lstAttr if (x.getPoserType() == attrTypeName) and (x.getName() == name))
        return po
      except:
        #  20080417 : Fix - Shall return null if not found
        return None
    else:
      return self.getAttribut(attrTypeName)

#TODO 
#  * This class represents a set of vertice indexes used by 
#  * Poser to adapt dynamic clothes (I think). 
#  * Moreover it can contain embedded attribute like 'stitchVertsGroupProperties'
#  
class VertsGroup(StructuredAttribut):
  def __init__(self):
    super(VertsGroup, self).__init__()
    self.vertList = None
    self.vertTab = []

  #    * @see deyme.v3d.poser.PoserObject#read(deyme.v3d.poser.PoserFileParser)
  #    
  def read(self, st):
    self.vertTab = [ ]

    fin = False
    while not fin:
      st.getToken()
      if st.ttype == PoserFileParser.TT_EOF:
        fin = True
        continue 

      if st.sval == None:
        continue 

      if st.ttype == PoserFileParser.TT_WORD:
        if st.sval == "v":
          st.getNumber()
          v = int(st.nval)
          st.skipToNextLine()
          self.vertTab.append(v)
          continue 

        if st.isRightBracket():
          fin = True
        else:
          #  Other keyword found
          cn = st.sval
          try:
            vc = Lang[st.sval]
            #  Known word
            if vc.isStructured:
              st.getToNextLine()
              if st.ttype == PoserFileParser.TT_EOL:
                nom = ""

              # sta = vc.create(nom)
              sta = create(vc, nom)
              self.addAttribut(sta)
              #  Read the opening bracket
              st.getToken()
              if st.isLeftBracket():
                sta.read(st)
              else:
                if sta.isAmbi():
                  #  Skip to next line
                  st.pushBack()
                else:
                  logging.warning("Line[%s]:  '{' is missing for %s", st.lineno(), cn)
            else:
              #sa = vc.createSimple(st.sval)
              sa = create(vc, st.sval)
              #  Try : 17AOU2007 : Read before add
              sa.read(st)
              self.addAttribut(sa)

          except:
            #  Mot inconnu
            logging.info("Line[%s] - Unknown word:%s",st.lineno() ,  cn)
      else:
        logging.warning("Line[%s] - Not Accepted :%s", st.lineno() , st.sval)
        raise ParsingErrorException()

    # End of read

  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    #  FIXME: Wrong output - Other attributes (like stitchVertsGroupProperties are forgotten)
    fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    nextPfx = pfx + "\tv "

    for vi in self.vertTab:
      fw.write(nextPfx + str(vi) + '\n')
 
    fw.write(pfx + "\t}\n")

# 
# This class represents the "textureFile" element of Poser files.
#  
class TextureFile(StructuredAttribut):

  def __init__(self):
    super(TextureFile, self).__init__()
    self._file = '' #  "NO_MAP" 
    self._warped = False

  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_file:
      self._file = None if val == PoserConst.C_NO_MAPG or val == PoserConst.C_NO_MAP else val
    elif tokenID==PoserToken.E_warped:
      self._warped = val.startswith("1")
    else:
      logging.warning("Unexpected Direct:%s (%s)",tokenID.token, val)

  #  (non-Javadoc)
  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  #    
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    fw.write(nextPfx + "file " + (PoserConst.C_NO_MAPG if (self._file == None) else self._file) + '\n') #  "NO_MAP" 
    WBoolLine(fw, nextPfx, "warped", self._warped )
    for po in self._lstAttr:
      po.write(fw, nextPfx)
    if len(self._lstAttr) == 0:
      fw.write('\n')
    fw.write(nextPfx + "}\n")

# 
# This class represents the "nodeInput" element of Poser files.
#  
class NodeInput(StructuredAttribut):
  def __init__(self, nodeName='', colorStr='1 1 1'):
    super(NodeInput, self).__init__()
    self._poserType = PoserToken.E_nodeInput
    self._nodeName = nodeName
    if nodeName:
      self._name = nodeName
      
    self._value    = colorStr #  22 0 32768
    self._parmR    = "" #  NO_PARM
    self._parmG    = "" #  NO_PARM
    self._parmB    = "" #  NO_PARM
    self._node     = "" #  NO_NODE
    self._file     = "" #  "NO_MAP" 
    
  # Set the name.
  def setName(self, s):
    self._name = RemoveQuotes(s)
    self._nodeName = self._name

  def read(self, st):
       
    while True:
      st.getToken()
      if st.ttype==PoserFileParser.TT_EOF or st.isRightBracket():
        break

      if not st.sval:
        continue

      if st.ttype == PoserFileParser.TT_WORD:
        try:
          tokenID = Lang[st.sval]        
          val = st.getToNextLine()
        
          if tokenID==PoserToken.E_name:
            self._nodeName = RemoveQuotes(val)
          elif tokenID==PoserToken.E_value:
            self._value = val
          elif tokenID==PoserToken.E_parmR:
            self._parmR = None if val==PoserConst.C_NO_PARM else val
          elif tokenID==PoserToken.E_parmG:
            self._parmG = None if val==PoserConst.C_NO_PARM else val
          elif tokenID==PoserToken.E_parmB:
            self._parmB = None if val==PoserConst.C_NO_PARM else val
          elif tokenID==PoserToken.E_node:
            self._node = None if val==PoserConst.C_NO_NODE else RemoveQuotes(val)
          elif tokenID==PoserToken.E_file:
            val = val.strip()
            self._file = None if val==PoserConst.C_NO_MAPG or val==PoserConst.C_NO_MAP else val
          else:
            logging.warning("Unexpected Direct:%s (%s)",tokenID.token, val)
        
        except KeyError: #  Mot inconnu
          logging.info("Line[%s] - Unknown word:%s", st.lineno(), st.sval)

        if st.isRightBracket():
          break
      else:
        logging.warning("L[%d] - Not Accepted :%s", st.lineno(), st.sval)
        raise ParsingErrorException()




  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  #    
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + ' "' + self.getName() +'"\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + '{\n')
    fw.write(nextPfx + 'name "' + self._nodeName +'"\n')
    fw.write(nextPfx + "value " + self._value +'\n')
    fw.write(nextPfx + "parmR " + (self._parmR if self._parmR else PoserConst.C_NO_PARM ) +'\n')
    fw.write(nextPfx + "parmG " + (self._parmG if self._parmG else PoserConst.C_NO_PARM ) +'\n')
    fw.write(nextPfx + "parmB " + (self._parmB if self._parmB else PoserConst.C_NO_PARM ) +'\n')
    fw.write(nextPfx + 'node "' + (self._node if self._node else PoserConst.C_NO_NODE )+'"\n')
    fw.write(nextPfx + "file " + (self._file if self._file else PoserConst.C_NO_MAPG ) +'\n')
    fw.write(nextPfx + '}\n')

  def getFile(self): return self._file

  #
  # Mimic Poser Python interface
  # Set the _file with a filename value
  #
  def setString(self, fn):
    self._file = fn

  #
  # Mimic Poser Python interface
  # Set the _parmR with a float value
  #
  def setFloat(self, f):
    t = self._value.split(' ')
    self._value = str(f) + ' ' + t[1] + ' ' + t[2]

  def getInNode(self, sht):
    if self._node and self._node!=PoserConst.C_NO_NODE:
      return sht.getNodeByInternalName(self._node)
    return None

#
#  Shader Tree Node
#
class Node(StructuredAttribut):
  def __init__(self):
    super(Node, self).__init__()
    self._poserType = PoserToken.E_node
    self.ambiguious = True
    self._nodeName  = ''
    self._typeName  = ''
    self._pos       = '0 0'

  def setDirect(self, tokenID, val):
    if tokenID == PoserToken.E_name:
      self._nodeName = RemoveQuotes(val)
    else:
      logging.warning("Unexpected Direct:%s (%s)",tokenID.token, val)
  
  def addAttribut(self, po):
    if po.getPoserType()==PoserToken.E_pos:
      self._pos = po.getValue()
    else:
      super(Node, self).addAttribut(po)

  # Set the name.
  def setName(self, s):
    if s:
      t = s.split(' ')
      self._name = RemoveQuotes(t[1])
      self._typeName = RemoveQuotes(t[0])
    else:
      self._name = s
      
      
  # @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + ' "' + self._typeName + '" "' + self.getName() + '"\n')
    if self._lstAttr:
      nextPfx = pfx + "\t"
      fw.write(nextPfx + "{\n")
      fw.write(nextPfx + 'name "' + self._nodeName + '"\n')
      fw.write(nextPfx + "pos " + self._pos + '\n')
      for po in self._lstAttr:
        po.write(fw, nextPfx)
      if len(self._lstAttr)==0:
        fw.write('\n')
      fw.write(nextPfx + "}\n")

  def getInputByInternalName(self, inpName='Blending'):
    for no in self.getLstAttr():
      if isinstance(no, NodeInput) and no._nodeName==inpName:
        return no
    return None

  def setLocation(self, x, y):
    self._pos = str(x)+ ' ' + str(y)

  def getNodeType(self):
    return PoserConst.kNodeTypeCodeBLENDER if self._typeName=='blender' else PoserConst.kNodeTypeCodeIMAGEMAP if self._typeName=='image_map' else C_ERROR

#
# Shader Tree
# 
class ShaderTree(StructuredAttribut):
  def __init__(self):
    super(ShaderTree, self).__init__()

  def getNodeByInternalName(self, intName):
    for po in self.getLstAttr():
      if isinstance(po, Node) and po._nodeName==intName:
        return po
    return None

  # Create a Node in a shader tree
  # Partial Implementation for 'blender' and 'image_map'
  #
  # Pos as (x,y)
  # Default Blender:
  #			node "blender" "Blender"
  #				{
  #				name "Blender"
  #				pos 396 54
  #				nodeInput "Input_1"
  #					{
  #					name "Input_1"
  #					value 1 1 1
  #					parmR NO_PARM
  #					parmG NO_PARM
  #					parmB NO_PARM
  #					node NO_NODE
  #					file NO_MAP
  #					}
  #				nodeInput "Input_2"
  #					{
  #					name "Input_2"
  #					value 1 1 1
  #					parmR NO_PARM
  #					parmG NO_PARM
  #					parmB NO_PARM
  #					node NO_NODE
  #					file NO_MAP
  #					}
  #				nodeInput "Blending"
  #					{
  #					name "Blending"
  #					value 0.5 0 1
  #					parmR NO_PARM
  #					parmG NO_PARM
  #					parmB NO_PARM
  #					node NO_NODE
  #					file NO_MAP
  #					}  
  def CreateNode(self, poserkNodeTypeCode, pos=None): # kNodeTypeCodeBLENDER or kNodeTypeCodeIMAGEMAP
    n = Node()
    # Generated a name according to the type
    baseName = 'Blender_' if poserkNodeTypeCode==PoserConst.kNodeTypeCodeBLENDER else 'Image_Map_'

    lstno = [ nodeNameNo(no._nodeName) for no in self.getLstAttr() \
      if no.getNodeType()==poserkNodeTypeCode ]

    no = max(lstno) + 1 if lstno else 0

    n._name = baseName+str(no)
    n._nodeName = n._name
    n._typeName = 'blender' if poserkNodeTypeCode==PoserConst.kNodeTypeCodeBLENDER else 'image_map'

    # Add the relevant NodeInput to the new Node
    if poserkNodeTypeCode==PoserConst.kNodeTypeCodeBLENDER:
      lstni = (("Input_1", "1 1 1"), ("Input_2", "1 1 1"), ("Blending", "0 0 1"))
    else:
      lstni = (("Image_Source", "-1 0 0"), ("Auto_Fit", "0 0 0"), ("U_Scale", "1 -1 1"),\
                  ("V_Scale", "1 -1 1"), ( "U_Offset", "0 0 1"), ("V_Offset", "0 0 1"), \
                  ("Texture_Coords", "1 0 0"), ("Image_Mapped", "4 0 0"), ("Background", "1 1 1"),\
		  ("Global_Coordinates", "0 0 1"), ("Mirror_U", "0 0 0"), ("Mirror_V", "0 0 0"), \
                  ("Texture_Strength", "1 0 1"), ("Filtering", "3 0 0"))

    n._lstAttr += [ NodeInput(tni[0], colorStr=tni[1]) for tni in lstni ]
    self._lstAttr.append(n)

    if pos:
      n.setLocation(pos[0], pos[1])
    return n
 
  def AttachTreeNodes(self, nodeDest, inputName, nodeRef):
    ni = nodeDest.getInputByInternalName(inpName=inputName)
    ni._node = nodeRef.getName()

#  
# material class.  
#  
class PoserMaterial(StructuredAttribut):
  # 
  # Create the default "skin" material.
  #    
  def __init__(self):
    super(PoserMaterial, self).__init__()

  def getTextureMap(self): return (self.findAttribut(PoserToken.E_textureMap)).getPath()

  def getBumpMap(self): return (self.findAttribut(PoserToken.E_bumpMap)).getPath()

  def getReflectionMap(self): return (self.findAttribut(PoserToken.E_reflectionMap)).getPath()

  def getTransparencyMap(self): return (self.findAttribut(PoserToken.E_transparencyMap)).getPath()

  def getLstNodes(self):
    st = self.findAttribut(PoserToken.E_shaderTree)
    return [ po for po in st.getLstAttr() if isinstance(po, Node) ] if st else [ ]




# 
# Representation of an Inverse Kynematic Chain
#  
class InkyChain(StructuredAttribut):

  # on
  def __init__(self):
    super(InkyChain, self).__init__()
    self._pmName = None
    self._active = False

  # 
  #    * Return true if the actor or the prop is ON
  #    * @return a boolean
  #    
  def isActive(self): return self._active

  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_name:
      self._pmName = val
    elif tokenID==PoserToken.E_on:
      self._active = True
    elif tokenID==PoserToken.E_off:
      self._active = False
    else:
      logging.warning("Unexpected Direct:%s (%s)", tokenID.token, val)

  # 
  #    * @see pftk.poser.kern.StructuredAttribut#write(java.io.PrintWriter,
  #    
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName()+'\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    fw.write(nextPfx + "name " + self._pmName + '\n')
    fw.write(nextPfx + ("on\n" if self._active else "off\n"))
    for po in self._lstAttr:
      po.write(fw, nextPfx)
    if len(self._lstAttr) == 0:
      fw.write('\n')
    fw.write(nextPfx + "}\n")

# 
#  * This class represents the "keys" entry of every channel definition.
#  * 
#  * keys
#  *  {
#  *  static <0|1>
#  *  k <long> <real>
#  *  [sl <0|1>
#  *  spl|lin|con
#  *  sm|br]
#  *  }
#  * 
#  * static determines whether this channel is animating (=0) or static (=1).
#  * k denotes the key information. The first values is the frame, starting at 0. The second value is the channel value at this frame.
#  * sl determines whether or not to loop animation.
#  * spl|lin|con denote the section interpolation type: spline, linear, constant.
#  * sm|br represent either a smooth transition or break in the transition.
#  
class Keys(StructuredAttribut):

  def __init__(self, src=None):
    super(Keys, self).__init__()
    self.setPoserType(PoserToken.E_keys)
    self._static = src._static if src else False
    self._hshKeys = { frameNo:KSA(src=ksa) for frameNo,ksa in src._hshKeys.items() } if src else { } # Replaced by a dict frameNo:KSA

  def read(self, st):
    fin = False
    curK = None
    while not fin:
      st.getToken()
      if st.ttype == PoserFileParser.TT_EOF:
        fin = True
        continue 

      if st.sval == None:
        continue 

      if st.ttype == PoserFileParser.TT_WORD:
        try:
          vc = Lang[st.sval]

          if vc==PoserToken.E_static:
            self._static = st.getBoolToNextLine()
            
          elif vc==PoserToken.E_k:
            curK = KSA(int(st.getNumber()), st.getNumber())
            self._hshKeys[curK._noFrame] = curK

          elif vc==PoserToken.E_sl:
            curK.setSl(st.getBoolToNextLine())
            
          elif (vc==PoserToken.E_spl) or (vc==PoserToken.E_lin) or (vc==PoserToken.E_con):
            curK.setCurveType(vc)

          elif (vc==PoserToken.E_sm) or (vc==PoserToken.E_br):
            curK.setCurveCnx(vc)
          else:
            logging.warning("L[%d] - Not Accepted :%s", st.lineno(), st.sval)
            
          continue 
        except KeyError:
          pass

        if st.isRightBracket():
          fin = True

      else:
        logging.warning("L[%d] - Not Accepted :%s", st.lineno(), st.sval)
        raise ParsingErrorException()


  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token+'\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    WBoolLine(fw, nextPfx, "static", self._static )
    for ks in sorted(self._hshKeys.values(), key=lambda k:k._noFrame):
      ks.write(fw, nextPfx)
    fw.write(nextPfx + "}\n")

  def setStatic(self, s):
    self._static = s

  def addKey(self, noFrame, factor):
    self._hshKeys[noFrame] = KSA(noFrame, factor)

  # 
  # Return the K factor for a given frameNo.
  # Intermediate values are not yet calculated.
  #  
  # @param frameNo 0 based frame index
  # @return the found key factor. 0.0 if not found.
  #    
  def getKeyFactor(self, frameNo):
    try:
      return self._hshKeys[frameNo]._factor
    except KeyError:
      return 0.0

  # 
  #    * Set the K factor for a given frameNo.
  #    * Intermediate values are not yet calculated.
  #    * 
  #    * @param frameNo 0 based frame index
  #    * @return C_OK when the frame no exists and C_FAIL if not found.
  #    
  def setKeyFactor(self, frameNo, k):
    try:
      self._hshKeys[frameNo].setFactor(k)
      return C_OK
    except KeyError:
      return C_FAIL

  # 
  #  Return the list of declared "k" frames.
  #    
  # def getLstKey(self): return self._lstKeys

  # 
  # Convert a GenericTransform to a static one (i.e. no keys)
  # @return    The number of deleted keys
  #    
  def toStatic(self, staticValue):
    l = len(self._hshKeys)
    self._static = True
    #  Remove all keys
    self._hshKeys.clear()
    cle = KSA(0, staticValue)
    cle.setSl(True)
    cle.setCurveType(PoserToken.E_spl)
    cle.setCurveCnx(PoserToken.E_sm)
    self._hshKeys[0] = cle
    return l - 1

class HairGrowthGroup(StructuredAttribut):
  pass

class DocDescription(StructuredAttribut):
  def __init__(self):
    super(DocDescription, self).__init__()
    self._dimensions = None
    self._screenPlace = None
    self._displayMode = None

  #  USEPARENT
  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_dimensions:
      self._dimensions = val
    elif tokenID==PoserToken.E_screenPlace:
      self._screenPlace = val
    elif tokenID==PoserToken.E_displayMode:
      self._displayMode = val
    else:
      logging.warning("Unexpected Direct:%s (%s)", tokenID.token, val)

  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token+ '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    if self._dimensions:
      fw.write(nextPfx + "dimensions " + self._dimensions + '\n')

    if self._screenPlace:
      fw.write(nextPfx + "screenPlace " + self._screenPlace + '\n')

    if self._displayMode:
      fw.write(nextPfx + "displayMode " + self._displayMode + '\n')

    for po in self._lstAttr:
      po.write(fw, nextPfx)

    fw.write(nextPfx + "}\n")







# 
# Description part of a figure.
#  
class FigureDescription(StructuredAttribut):

  def __init__(self):
    super(FigureDescription, self).__init__()
    self._displayMode = None
    self._locked = False
    self._root = None # Root Actor Name

  # 
  #    * Change the name of referenced part (actor, prop, hairProp, controlProp)  
  #    * @param oldPartName
  #    * @param newPartName
  #    
  def changeReference(self, oldPartName, newPartName):
    #  Change Attributes : addChild and weld
    for po in self.getLstAttr():
      if (po.getPoserType() == PoserToken.E_root) or (po.getPoserType() == PoserToken.E_defaultPick):
        po.TAS(oldPartName, newPartName)
      else:
        if isinstance(po, AddChildSA):
          po.changeReference(oldPartName, newPartName)
        else:
          if po.getPoserType() == PoserToken.E_inkyChain:
            for po2 in po.getLstAttr():
              if (po2.getPoserType() == PoserToken.E_goal) or (po2.getPoserType() == PoserToken.E_addLink):
                po2.TAS(oldPartName, newPartName)

  # 
  #  Add the list of actors as starting from pa, following pa's figure child and weld trees
  #  * @param parent
  #  * @param pa
  #    
  def addActor(self, parent, ldesc, weldActors=True):
    seen = False
    last = -1

    for i,po in enumerate(self.getLstAttr()):
      if po.getPoserType() == PoserToken.E_addChild:
        seen = True
      elif seen: #  Add before "i"
        last = i
        for pfils in ldesc:
          ac = AddChildSA(PoserToken.E_addChild, pfils.getName(), pfils.getParent())
          self.getLstAttr().insert(last, ac)
          last += 1
        break

    if weldActors:
      seen = False
      for i,po in enumerate(self.getLstAttr()):
        #po = self.getLstAttr[i]
        if po.getPoserType() == PoserToken.E_weld:
          seen = True
        elif seen: #  Add before "i"
          last = i
          for pfils in ldesc:
            gwp = pfils.getWeldParent()
            if gwp:
              ac = AddChildSA(PoserToken.E_weld, pfils.getName(), gwp)
              self.getLstAttr().insert(last, ac)
              last += 1
          break
        
  def getRoot(self): return self._root

  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_name:
      self.setName(val)
    elif tokenID==PoserToken.E_locked:
      self._locked = (val[0] == "1")
    elif tokenID==PoserToken.E_displayMode:
      self._displayMode = val
    elif tokenID==PoserToken.E_root:
      self._root = val
    else:
      logging.warning("Unexpected Direct:%s (%s)", tokenID.token, val)

  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token+'\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    fw.write(nextPfx + "name " + self.getName() + '\n')
    fw.write(nextPfx + "root " + self.getRoot() + '\n')
    
    for po in self._lstAttr:
      po.write(fw, nextPfx)
    if self._displayMode:
      fw.write(nextPfx + "displayMode " + self._displayMode + '\n')

    WBoolLine(fw, nextPfx, "locked", self._locked )

    fw.write(nextPfx + "}\n")

class AlternateGeom(StructuredAttribut):
  def __init__(self):
    super(AlternateGeom, self).__init__()
    #  Index of the Alternate geometry in the list (not a real attribut) 
    self._no = 0
    self._printName = None

    #  Geom filename formated like Poser with ':' or ""  
    self._geomFileName = None

  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_name:
      self._printName = val
    elif tokenID==PoserToken.E_objFile:
      tmps1 = val.strip()
      bci1 = tmps1.rfind(' ')
      self._geomFileName = tmps1[(bci1 + 1):]
    else:
      logging.warning("Unexpected Direct:%s (%s)", tokenID.token, val)

  #  @see pftk.poser.kern.StructuredAttribut#write(java.io.PrintWriter,
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    fw.write(nextPfx + "name " + self._printName +'\n')
    fw.write(nextPfx + "objFile 20 " + self._geomFileName +'\n')
    fw.write(nextPfx + "}\n")

  def setNo(self, n):
    self._no = n

  def getNo(self): return self._no

  # Set the print name 
  #  @param gfn
  def setPrintName(self, gfn):
    self._printName = gfn

  # Set the filename as represented on the "objFile" line.
  #  @param gfn
  def setGeomFileName(self, gfn):
    self._geomFileName = gfn

  def getGeomFileName(self): return self._geomFileName


# Represents a delta for the vertex identified by noPt.
class DeltaPoint(Vector3d):

  def __init__(self, no=0, dx=0.0, dy=0.0, dz=0.0):
    super(DeltaPoint, self).__init__(dx,dy,dz)
    self.noPt = no

  def getPointNo(self): return self.noPt

  def setPointNo(self, n):
    self.noPt = n

  def __str__(self):
    return "d {0:d} {1: 11.8f} {2: 11.8f} {3: 11.8f}".format(self.noPt, self.x, self.y, self.z)

  # Round the value to C_MIN_DELTA
  #@classmethod --> Replaced by the buildin round(f,ndigit)
  #def round(cls, v):

  def getVector(self): return self

  def setVector(self, v):
    self.x = v.x
    self.y = v.y
    self.z = v.z

  def toV3d(self): return Vector3d(self)

# 
# This class represents a Deltas attribute used to morph a meshed object. 
#  
class Deltas(StructuredAttribut):

  def __init__(self, src=None):
    super(Deltas, self).__init__()
    self.setPoserType(PoserToken.E_deltas)

    #In python deltaTab replaced by a dict { noPt:DeltaPoint() }
    self.deltaSet = { dp.noPt:DeltaPoint(dp.noPt, dp.x, dp.y, dp.z) for dp in src.deltaSet.items() } if src else { }

  #    * @see deyme.v3d.poser.PoserObject#read(deyme.v3d.poser.PoserFileParser)
  def read(self, st):
    self.deltaSet = { }
    fin = False
    while not fin:
      st.getToken()
      if st.ttype == PoserFileParser.TT_EOF:
        break

      if not st.sval:
        continue

      if st.ttype == PoserFileParser.TT_WORD:
        if st.sval == "d":
          p = DeltaPoint(int(st.getNumber()), st.getNumber(), st.getNumber(), st.getNumber())
          st.skipToNextLine()
          self.deltaSet[p.noPt] = p
          continue 
        if st.isRightBracket():
          fin = True
      else:
        logging.warning("L[%d] - Not Accepted :%s", st.lineno(), st.sval)
        raise ParsingErrorException()

  # @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    for d in sorted(self.deltaSet.values(), key=lambda p:p.noPt):
      fw.write(nextPfx + str(d) + '\n')
    fw.write(nextPfx + "}\n")

  def clear(self):
    self.deltaSet.clear() # = { }

  @classmethod
  def enhancement(cls, srcWG, setFoundDeltas, setNewDeltas, enhance, boundingType, useVicinityLoop, alpha):
    res = C_OK
    mlsFailed = False

    # Populate result with extact matches
    setNewDeltas.update(setFoundDeltas)

    if enhance != PoserConst.C_NO_ENHANCEMENT:
      # 03FEV2008 : Check For missed/not complete faces 
      for grp in srcWG.getGroups():

        # Create an empty sphere (i.e. negative volume)
        # Or (default C_NO_BOUNDING or BOX) Create an empty box (i.e. negative volume)
        bnd = BoundingSphere(Point3d(), -1.0) if boundingType==PoserConst.C_SPHERE_BOUNDING else \
              BoundingBox(Point3d(1.0, 1.0, 1.0), Point3d(-1.0, -1.0, -1.0))

        for faceno in range(0, grp.getNbFace()):
          startIdx = grp.stripCount[faceno]
          lastIdx = grp.stripCount[faceno + 1]
          
          nbEdgeMoved = 0
          faceEdge = [ ]
          faceEdgeIdx = [ ]
          
          for vi in grp.vertIdx[startIdx:lastIdx]: 
            try:
              dp = setFoundDeltas[vi]
              nbEdgeMoved += 1
              # Keep in bounding box, points that have a move
              bnd.combine(grp.coordList[vi])
            except KeyError: #if not found:
              # Record a copy of the point that has no delta              
              faceEdge.append( Point3d(grp.coordList[vi]) )
              faceEdgeIdx.append(vi)
          
          nbmissing = lastIdx - startIdx - nbEdgeMoved
          if (nbEdgeMoved > 0) and (nbmissing > 0):
            # Some vertex are not moved!
            # Get the vertex that compose the "hole face"
            edgePtIdx = grp.vertIdx[startIdx:lastIdx]

            # Calculate '1' loop Vicinity
            edgeLoopPtIdx =  copy.copy(edgePtIdx)

            # First Loop Vicinity
            if useVicinityLoop:
              grp.extendLoopVicinity(edgePtIdx, edgeLoopPtIdx)

            if enhance==PoserConst.C_MLS_ENHANCEMENT:
              # Keep trace of math. error to apply other type of enhancement
              mlsFailed = False

              # Apply deltas to points designed by edgeLoopPtIdx[]
              nuageVoisin = [ ] 
              nuageIdx = [ ] 
              nbPtNuage = 0

              for vi in edgeLoopPtIdx:
                # Add to the cloud (nuage) only vertex that have a delta
                try:
                  faceEdgeIdx.index(vi)
                except ValueError:  #if not found:
                  nuageVoisin.append( Point3d(srcWG.coordList[vi]) )
                  nuageIdx.append(nbPtNuage)
                  try:
                    dp = setFoundDeltas[vi]
                    nuageVoisin[nbPtNuage].add(dp)
                  except KeyError:
                    pass
                  
                  nbPtNuage += 1
              # End for vi

              try:
                # Compute projection Repere with SVD matrix decomposition
                repUVN = Repere(nbPtNuage, nuageIdx, nuageVoisin)

                tabproj = repUVN.project(0, 0, nbPtNuage, nuageIdx, nuageVoisin, None)
                tabdest = repUVN.project(0, 0, nbmissing, nuageIdx, faceEdge, None)

                # Calculate Point Altitudes in the UVN repere
                calcMLS(tabdest, 0, nbmissing, tabproj, alpha)

                ncl = repUVN.reserveProject(0, 0, nbmissing, tabdest, None)

                # Create some new deltas
                for i,vi in enumerate(faceEdgeIdx):
                  if (boundingType==PoserConst.C_NO_BOUNDING) or bnd.intersect(grp.coordList[vi]):
                    dx = round(ncl[i].x - srcWG.coordList[vi].x, 6)
                    dy = round(ncl[i].y - srcWG.coordList[vi].y, 6)
                    dz = round(ncl[i].z - srcWG.coordList[vi].z, 6)
                    if (dx != 0.0) or (dy != 0.0) or (dz != 0.0):
                      dp = DeltaPoint(vi, dx, dy, dz)
                      setNewDeltas[vi]= dp
                    
              except Exception as e:
                logging.warning("Math. Error in %s[%s face=%d] : %s", srcWG.getName(), grp.getName(), faceno, e)
                mlsFailed = True

            if (enhance==PoserConst.C_AVG_ENHANCEMENT) or ((enhance==PoserConst.C_MLS_ENHANCEMENT) and mlsFailed):
              # Find deltas of points designed by edgeLoopPtIdx[]
              dx = 0.0
              dy = 0.0
              dz = 0.0
              nbdelta = 0
              for vi in edgeLoopPtIdx:
                try:
                  dp = setFoundDeltas[vi]
                  dx += dp.x
                  dy += dp.y
                  dz += dp.z
                  nbdelta += 1
                except KeyError:
                  pass

              # Calculage average displacment
              # Value less than 1e-6 are set to 0 by Poser + Null Delta shall not be kept
              if nbdelta>0:
                dx = round(dx / nbdelta, 6)
                dy = round(dy / nbdelta, 6)
                dz = round(dz / nbdelta, 6)
                if (dx != 0.0) or (dy != 0.0) or (dz != 0.0):
                  setNewDeltas.update( { vi:DeltaPoint(vi, dx, dy, dz) for vi in faceEdgeIdx if (boundingType==PoserConst.C_NO_BOUNDING) or bnd.intersect(grp.coordList[vi]) } )

      # Sort the list according to point index (I'm not sure of Poser behavior if not sorted)
      #lstNewDeltas.sort(key=lambda dp: dp.noPt)

    return res



# 
#  * Generic Transformation dedicated to represent : targetGeom <name> valueParm
#  * <name> geomChan <name> xOffsetA|yOffsetA|zOffsetA <name>
#  * xOffset|yOffsetB|zOffsetB <name> taperX|Y|Z <name> scale|X|Y|Z <name>
#  * propagatingScale|X|Y|Z <name> translateX|Y|Z <name> rotateX|Y|Z <name>
#  * smoothScaleX|Y|Z <name> twistX|Y|Z <name> jointX|Y|Z <name> curveX|Y|Z <name>
#  * handGrasp|thumbGrasp|handSpread <name> pointAtParm <name> hairDynamicsParm 
#  * curve  <name>
#  * <name> ... And others ...
#  
#  This method should actually take a ValueOpDelta[] because a single
#  transform can have many valueOpDelta fields!!!
class GenericTransform(StructuredAttribut):
  def __init__(self, poserType=None, channelName=None, vodFigure=None, vodActor=None, vodChannel=None):
    super(GenericTransform, self).__init__()

    #  PoserMeshedObject that contains the channel list 
    self._pmo = None

    #  Optimized attributs
    self._transfName = None
    self._initValue = 0.0
    self._hidden = False
    self._enabled = True

    #  V9 attribut (unknown effect)
    self._forceLimits = 4.0
    self._min = -100000
    self._max = 100000
    self._trackingScale = 0.001
    self._interpStyleLocked = -sys.maxsize
    self._keys = None
    # self._indexes = -sys.maxsize - Replaced by a computed value
    self._numbDeltas = -sys.maxsize
    self._dlt = None
    self._lstVOD = [ ]

    self._hasStaticValue = False
    self._staticValue = 0.0

    if poserType:
      self.setPoserType(poserType)
      self.setName(channelName)

      #  20080821: Do not add a prefix to printable name (old P4 Stuff)
      self._transfName = channelName[3:] if (channelName.startswith("PBM") or channelName.startswith("FBM")) else channelName
      self.addKeyFrame(0, 0.0)
      #  Default keys are static=0
      self._interpStyleLocked = 0

      #  Create the deltas table
      if poserType != PoserToken.E_valueParm:
        self._numbDeltas = 0
        self._dlt = Deltas()

        if vodActor or vodChannel:
          if vodFigure:
            #  HACK HACK HACK Hardwiring valueOpDeltaAdd for now. This will let the
            #  typical case of
            #  valueOpDeltaAdd work for now but all others (valueOpDeltaPlus,
            #  valueOpDeltaMinus, etc)
            #  will be broken! I'll fix this after Olivier adds his hair stuff.
            vod = ValueOpDelta(PoserToken.E_valueOpDeltaAdd, vodFigure, vodActor, vodChannel, 1.0)
            self.addVOD(vod)

  #
  # Copy all data from src, except the _pmo if parentPO is set
  #
  def copy(self, src, parentPO=None):
    self.setPoserType(self.getPoserType())
    self.setName(src.getName())

    self._pmo = parentPO if parentPO else src._pmo
    self._transfName = src._transfName
    self._initValue = src._initValue
    self._hidden = src._hidden
    self._enabled = src._enabled
    self._forceLimits = src._forceLimits
    self._min = src._min
    self._max = src._max
    self._trackingScale = src._trackingScale
    self._interpStyleLocked = src._interpStyleLocked
    self._hasStaticValue = src._hasStaticValue
    self._staticValue = src._staticValue
    self._numbDeltas = src._numbDeltas
    self._keys = Keys(src=src._keys)
    self._dlt = Deltas(src=src._dlt)
    self._lstVOD = [ ValueOpDelta(src=v) for v in src.getVOD() ]
    self._transfName = src._transfName



  def addKeyFrame(self, frameno, v):
    if not self._keys:
      self._keys = Keys() #  Default keys are static=0

    self._keys.addKey(frameno, v)


  def getKeysFactor(self, frameNo):
    return self._keys.getKeyFactor(frameNo) if self._keys else 0.0

  def getDeltas(self): return self._dlt

  def ishasDeltas(self): return self._dlt and (len(self._dlt.deltaSet)>0)

  # 
  #    * Remove the deltas of tha current channel
  #    
  def removeDeltas(self):
    #  20090711 : Not sure that should be REALLY cleared?
    #  As of 20100226 do not delete anymore the calculation dependencies : _lstVOD.clear();
    #self._indexes = 0
    self._numbDeltas = 0
    #  20190105 : Fix for robusness
    if self.ishasDeltas():
      self.getDeltas().deltaSet.clear()

  # 
  # Optimize deltas : Delete delta with a too small norme
  #    
  def optimizeDeltas(self, refNorm):
    dl = self.getDeltas()
    #t = dl.getDeltaTab()
    oldLength = len(dl.deltaSet)
    optSet = { t.noPt:t for t in dl.deltaSet if t.norm() > refNorm }

    # optTabFinal = Arrays.copyOf(optTab, nbpt)
    dl.setDeltaTab(optSet)
    indexes = len(optSet)
    logging.info("Channel[%s] reduced from %d to %d", self.getName(), oldLength, indexes )
    return indexes 

  def getVOD(self): return self._lstVOD

  # Add an operator to the GenericChannel
  def addVOD(self, vop):
    self._lstVOD.append(vop)

  def findValueOp(self, vodType:'ValueOpDelta', targetName:'str'=None):
    for vop in self._lstVOD:
      if (vop.getPoserType()==vodType) and ((targetName is None) or (vop.getChannelName()==targetName)):
        return vop
       
    return None

  # 
  #    * Delete any reference to the given GenericTransform
  #    * @param gt
  #    
  def deleteChannelRef(self, gt):
    chName = gt.getName()
    grpName = gt.getPoserMeshedObject().getName()
    self._lstVOD[:] = [ vod for vod in self._lstVOD if (grpName!=vod.getGroupName()) or (chName!=vod.getChannelName()) ]
    #self._vodTab = None


  # 
  #    * Delete any reference to the given GenericTransform
  #    * @param gt
  #    
  def deleteFigureRef(self, fig):
    fi = fig.getBodyIndex()
    self._lstVOD[:] = [ vod for vod in self._lstVOD if index(vod.getGroupName())!=fi ]
    #self._vodTab = None


  # 
  #    * Change the name of referenced part (actor, prop, hairProp, controlProp)  
  #    * @param oldPartName
  #    * @param newPartName
  #    
  def changeReference(self, oldPartName, newPartName, changeRefToBODY=True):
    for vod in self._lstVOD:
      if not vod.getGroupName().startswith(PoserConst.C_BODY) or changeRefToBODY:
        vod.setGroupName(TAS(vod.getGroupName(), oldPartName, newPartName))
      
    for po in self.getLstAttr():
      if po.getPoserType() == PoserToken.E_otherActor:
        po.TAS(oldPartName, newPartName)

  OPS = (PoserToken.E_valueOpDeltaAdd, PoserToken.E_valueOpPlus, PoserToken.E_valueOpMinus, PoserToken.E_valueOpTimes, PoserToken.E_valueOpDivideBy, PoserToken.E_valueOpDivideInto, PoserToken.E_valueOpKey) 

  def addAttribut(self, po):
    if po.getPoserType() in GenericTransform.OPS:
      self._lstVOD.append(po)
    elif po.getPoserType()==PoserToken.E_keys:
      self._keys = po
    elif po.getPoserType()==PoserToken.E_deltas:
      self._dlt = po
    else:
      super(GenericTransform, self).addAttribut(po)

  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_name:
      self._transfName = val
    elif tokenID==PoserToken.E_initValue:
      self._initValue = float(val)
    elif tokenID==PoserToken.E_hidden:
      self._hidden = val.startswith("1")
    elif tokenID==PoserToken.E_trackingScale:
      self._trackingScale = float(val)
    elif tokenID==PoserToken.E_min:
      self._min = float(val)
    elif tokenID==PoserToken.E_max:
      self._max = float(val)
    elif tokenID==PoserToken.E_forceLimits:
      self._forceLimits = float(val)
    elif tokenID==PoserToken.E_flipped:
      #  Bad fix ... but for the moment
      self.addAttribut(OffSA("flipped"))
    elif tokenID==PoserToken.E_indexes:
      #self._indexes = int(val)
      pass
    elif tokenID==PoserToken.E_numbDeltas:
      self._numbDeltas = int(val)
    elif tokenID==PoserToken.E_enabled:
      self._enabled = val.startswith("1")
    elif tokenID==PoserToken.E_interpStyleLocked:
      self._interpStyleLocked = int(val)
    elif tokenID==PoserToken.E_staticValue:
      self.setStaticValue(float(val))
    else:
      logging.warning("Unexpected Direct:%s (%s)", tokenID.token, val)

  #    * @see pftk.poser.kern.StructuredAttribut#write(java.io.PrintWriter,
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName() +'\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + '{\n')
    fw.write(nextPfx + "name " + self._transfName +'\n')
    fw.write(nextPfx + "initValue " + str(self._initValue) +'\n')
    #fw.write(nextPfx + "hidden " + (1 if self._hidden else 0) +'\n')
    WBoolLine(fw, nextPfx, "hidden", self._hidden )
    
    #  Test Poser language version before writing
    if self._pmo and (self._pmo.getPoserFile().isVersion(PoserConst.POSER_V9)):
      WBoolLine(fw, nextPfx, "enabled", self._enabled )
    fw.write(nextPfx + "forceLimits " + str(self._forceLimits) +'\n')
    fw.write(nextPfx + "min " + str(self._min) +'\n')
    fw.write(nextPfx + "max " + str(self._max) +'\n')
    fw.write(nextPfx + "trackingScale " + str(self._trackingScale) +'\n')
    #  Print Keys
    if self._keys != None:
      self._keys.write(fw, nextPfx)
    if self._interpStyleLocked != -sys.maxsize:
      fw.write(nextPfx + "interpStyleLocked " + str(self._interpStyleLocked) +'\n')
    if self._hasStaticValue:
      fw.write(nextPfx + "staticValue " + str(self._staticValue) +'\n')
    for vod in self._lstVOD:
      vod.write(fw, nextPfx)
    for po in self._lstAttr:
      po.write(fw, nextPfx)
    if self._dlt and (len(self._dlt.deltaSet)>0):
      fw.write(nextPfx + "indexes " + str(len(self._dlt.deltaSet)) + '\n')
      fw.write(nextPfx + "numbDeltas " + str(self._numbDeltas) +'\n')
      self._dlt.write(fw, nextPfx)
    fw.write(nextPfx + "}\n")

  def setPoserMeshedObject(self, pmo):
    self._pmo = pmo

  def getPoserMeshedObject(self): return self._pmo

  def getPrintName(self): return self._transfName

  def setPrintName(self, n):
    self._transfName = n

  def isHidden(self): return self._hidden

  def setHidden(self, h):
    self._hidden = h

  def getMin(self): return self._min

  def getMax(self): return self._max

  def setMin(self, m):
    self._min = m

  def setMax(self, m):
    self._max = m

  def setInterpStyleLocked(self, isl):
    self._interpStyleLocked = isl

  def getPrintableDependencies(self):
    return [ vod.getPoserType().token + " " +vod.getGroupName() + " " + vod.getChannelName() for vod in self.getVOD() ]

  def getTrackingScale(self): return self._trackingScale

  def setTrackingScale(self, m):
    self._trackingScale = m

  def setNumbDeltas(self, n):
    self._numbDeltas = n

  def getNumbDeltas(self): return self._numbDeltas

  def getInitValue(self): return self._initValue

  def setInitValue(self, m):
    self._initValue = m

  def getStaticValue(self): return self._staticValue

  def setStaticValue(self, m):
    self._hasStaticValue = True
    self._staticValue = m

  def getForceLimits(self): return self._forceLimits

  def setForceLimits(self, m):
    self._forceLimits = m

  def setDeltaTab(self, tdp):
    self._dlt.deltaSet = tdp
    

  def getKeys(self): return self._keys

  # Convert a GenericTransform to a static one (i.e. no keys)
  def toStatic(self):
    ret = C_OK
    if self._hasStaticValue:
      logging.info("Channel[%s]: keep static=%d", self.getName(), self.getStaticValue())
    else:
      self.setStaticValue(self.getInitValue())
      logging.info("Channel[%s]: set static=%d", self.getName(), self.getStaticValue())

    if self._keys:
      ret = self._keys.toStatic(self.getStaticValue())
    return ret

  # Determine if a target morph shall be taken into account. 
  # @param attrName        Name of the targetGeom or valueParm
  # @param setTargetMorph  Set of names
  # @return    true when the set of target morph is empty AND the attrName starts with "PBM" OR when the set of target contains the given attrName
  @classmethod
  def concerned(cls, attrName, setTargetMorph):
    if setTargetMorph == None:
      return attrName.startswith("PBM")
    else:
      return attrName in setTargetMorph


class Scale(GenericTransform):

  def __init__(self):
    super(Scale, self).__init__()
    self.ambiguious = True
    self._isSimple = True

  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  #    
  def write(self, fw, pfx):
    if self._isSimple:
      fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    else:
      super(Scale, self).write(fw, pfx)

  def addAttribut(self, po):
    super(Scale, self).addAttribut(po)
    self._isSimple = False

# 
# DepthMapSize attribut.
#  
class DepthMapSize(GenericTransform):
  def __init__(self):
    super(DepthMapSize, self).__init__()
    self.ambiguious = True
    self._isSimple = True

  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    if self._isSimple:
      fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    else:
      super(DepthMapSize, self).write(fw, pfx)

  def addAttribut(self, po):
    super(DepthMapSize, self).addAttribut(po)
    self._isSimple = False

# 
# Represents the list of channels of a PoserMeshedObject.
#  
class Channels(StructuredAttribut):

  def __init__(self):
    super(Channels, self).__init__()
    self.setPoserType(PoserToken.E_channels)
    self.setName("")

    #  PoserMeshedObject that contains the channel list 
    self._pmo = None

    #  Store the group presentation of channels 
    self._groups = None

  def setPoserMeshedObject(self, pmo):
    self._pmo = pmo

  def getPoserMeshedObject(self): return self._pmo

  # 
  #    * Delete the given channel.
  #    * @param gt
  #    
  def deleteChannel(self, gt):
    self.deleteAttribut(gt.getName())
    self.deleteGTReference(self._groups, gt)

  def deleteGTReference(self, grp, gt):
    #  FIX 2019-05-05 : Some (old) character may have not group
    if grp != None:
      # grp._lstAttr[:] = [ po for po in grp.getLstAttr() if (po.getPoserType()!=PoserToken.E_parmNode) or (gt.getName()==po.getValue()) ]
      # grp.getLstAttr().remove(po)

      for po in grp.getLstAttr():
        if po.getPoserType()==PoserToken.E_parmNode:
          if gt.getName()==po.getValue():
            grp.getLstAttr().remove(po)
            
        elif po.getPoserType()==PoserToken.E_groupNode:
          #  Recursive, because a group may contain another group
          self.deleteGTReference(po, gt)

  # 
  #    * Add the given channel.
  #    * @param gt
  #    
  def addChannel(self, gt):
    res = C_OK
    po = self.findAttribut(gt.getPoserType(), gt.getName())
    if not po:
      self.getLstAttr().append(gt)
    else:
      logging.info("Channel[%s:%s] already exists", gt.getPoserType(), gt.getName()  )
      res = C_FAIL
    return res

  # 
  # Update or create a targetGeom and create a dependency to the body homonyme valueParm (if bodyLinked=true)
  #
  # @param channelName
  #    *          TargetGeom name
  # @param targetFigureName
  #    *          Referenced Figure Name
  # @param srcWG
  #    *          Concerned WaveGeom
  # @param lstNewDeltas
  #    *          List of deltas
  # @param bodyIdx
  #    *          Body index of the referenced figure (silly)
  #    
  #     updateOrCreate(String channelName, String targetFigureName, WaveGeom3d srcWG, lstNewDeltas, bodyIdx, bodyLinked)
  # def updateOrCreate_0(self, channelName, targetFigureName, srcWG, lstNewDeltas, bodyIdx):
  def updateOrCreate(self, channelName, targetFigureName, srcWG, setNewDeltas, bodyIdx, bodyLinked=True):
    gt = None
    #  trouve -> "found"
    trouve = False
    for gt in self.getLstAttr():
      trouve = (gt.getPoserType() == PoserToken.E_targetGeom) and (gt.getName() == channelName)
      if trouve: break

    if not trouve: #  Create the missing targetGeom
      if bodyLinked:
        gt = GenericTransform(PoserToken.E_targetGeom, channelName, targetFigureName, \
                              None if not targetFigureName else "BODY:" + str(bodyIdx),\
                              None if not targetFigureName else channelName)
      else:
        gt = GenericTransform(PoserToken.E_targetGeom, channelName)
        #  Do not add a prefix to printable name (old P4 Stuff)
        gt.setPrintName(channelName[3:] if (channelName.startswith("PBM") or channelName.startswith("FBM")) else channelName)
        gt.addKeyFrame(0, 0.0)
        #  Default keys are static=0
        gt.setInterpStyleLocked(0)

      self.getLstAttr().insert(0, gt)

    gt.getDeltas().deltaSet = setNewDeltas # [ delta for delta in lstNewDeltas ] )
    gt.setNumbDeltas(len(srcWG.coordList))
    return gt

  # 
  # Update or create a valueParm and initializes it
  # @param channelName ValueParm name
  def updateOrCreateVP(self, channelName, minVal=0.0, maxVal=1.0, applyLimits=True, isHidden=False):
    vp = None
    #  trouve -> "found"
    trouve = False
    for vp in self.getLstAttr():
      trouve = (vp.getPoserType()==PoserToken.E_valueParm) and (vp.getName()==channelName)
      if trouve: break

    if not trouve:
      #  Create the missing valueParm
      vp = GenericTransform(PoserToken.E_valueParm, channelName)
      #  Do not add a prefix to printable name (old P4 Stuff)
      vp.setPrintName(channelName[3:] if (channelName.startswith("PBM") or channelName.startswith("FBM")) else channelName)
      vp.addKeyFrame(0, 0.0)
      #  Default keys are static=0
      vp.setInterpStyleLocked(0)

      self.getLstAttr().insert(0, vp)

    vp.setMin(minVal)
    vp.setMax(maxVal)
    vp.setForceLimits(1 if applyLimits else 0)
    vp.setHidden(isHidden)
    return vp

  def addAttribut(self, po):
    if po.getPoserType()==PoserToken.E_groups:
      self._groups = po
    elif po.getPoserType()==PoserToken.E_readScript:
      #  TODO : Read Channel Definition Script
      logging.error("No Management rule for %s [%s]", po.getName(), po.getPoserType().token)
    else:
      po.setPoserMeshedObject(self._pmo)
      super(Channels, self).addAttribut(po)

  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token+'\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    if self._groups:
      self._groups.write(fw, nextPfx)
    for po in self._lstAttr:
      po.write(fw, nextPfx)
    if len(self._lstAttr) == 0:
      fw.write('\n')
    fw.write(nextPfx + "}\n")

# 
#  * Used to create morph. 
#  
class ChannelMorphStatus(object):
  #    
  PRESENCE_IMG = ("Unused", "in Reference", "in Destination", "Both")
  ATTR = ("Channel", "Presence", "RefFactor", "RefNonNullDesc", "RefHasDelta", "SrcFactor", "SrcNonNullDesc", "SrcHasDelta", "Optimized")

  def __init__(self, cnOrgt, InitSetTargetMorph=None):

    # 
    # * 0x0 = Unused
    # * 0x1 = Used only in Reference figure
    # * 0x2 = Used only in Destination figure
    # * 0x3 = Used by both
    #    
    self.presence = 0
    self.peer = None
    self.hasDelta = False
    
    if isinstance(cnOrgt, GenericTransform):
      self.channelName = cnOrgt.getName()
      self.selected = GenericTransform.concerned(cnOrgt.getName(), InitSetTargetMorph)
      self.finalFactor = cnOrgt.getKeysFactor(0)
      self.updateDelta(cnOrgt)

    else: # cnOrgt is a String
      self.channelName = cnOrgt
      self.finalFactor = 0.0
      self.hasNonNullDesc = False     
      self.selected = False


  def updateDelta(self, attrDesc):
    dlt = attrDesc.getDeltas()
    self.hasDelta = self.hasDelta or dlt

  def getChannelName(self): return self.channelName

  def setChannelName(self, channelName):
    self.channelName = channelName

  def getFinalFactor(self): return self.finalFactor

  def setFinalFactor(self, finalFactor):
    self.finalFactor = finalFactor

  def isHasNonNullDesc(self): return self.hasNonNullDesc

  def setHasNonNullDesc(self, hasNonNullDesc):
    self.hasNonNullDesc = hasNonNullDesc

  def isHasDelta(self): return self.hasDelta

  def setHasDelta(self, hasDelta):
    self.hasDelta = hasDelta

  def isSelected(self): return self.selected

  def setSelected(self, selected):
    self.selected = selected

  def getPeer(self): return self.peer

  def setPeer(self, peer):
    self.peer = peer

  def getPresence(self): return self.presence

  def setPresence(self, presence):
    self.presence = presence


# 
# A list of ChannelMorphStatus
#  
class ChannelMorphStatusList(list):
  CMSL_FILE_VERSION = 0xf000000016112007
  FILTER_NONE = 0x0000
  FILTER_NONNULLREFDELTA = 0x0001
  FILTER_REFPRESENCE = 0x0002
  FILTER_NONNULLREFFACTOR = 0x0004


  # 
  # Create an empty list.
  # Create an filtered copy of list lstsrc.
  #    * @param filter
  #    
  # def __init___0(self, filterCode)
  # def __init__(self):
  def __init__(self, filterCode=None, lstsrc=None):
    super(ChannelMorphStatusList, self).__init__()
    self._filter = filterCode if filterCode else ChannelMorphStatusList.FILTER_NONE
    if lstsrc:
      for refCS in lstsrc:
        self.addFiltered(refCS, None)

  def __str__(self):
    return str(len(self))

  def find(self, channelName):
    for c in self:
      if channelName == c.channelName:
        return c
    return None

  def extract(self, channelName):
    for c in self:
      if channelName == c.channelName:
        self.remove(c)
        return c

    return None

  def addFiltered(self, refCS, srcCS):
    refCS.peer = srcCS
    result = True
    if self._filter != self.FILTER_NONE:
      if (self._filter & self.FILTER_NONNULLREFDELTA) == self.FILTER_NONNULLREFDELTA:
        result = result and refCS.hasDelta
      if (self._filter & self.FILTER_REFPRESENCE) == self.FILTER_REFPRESENCE:
        result = result and ((refCS.presence & 0x1) == 0x1)
      if (self._filter & self.FILTER_NONNULLREFFACTOR) == self.FILTER_NONNULLREFFACTOR:
        result = result and (refCS.finalFactor != 0.0)
    if result:
      self.append(refCS)

  def getFilter(self): return self._filter

  # 
  # Return a Set with the selected channels.
  #    
  def getChannelSet(self):
    return { cs.channelName for cs in self if cs.selected }

  # 
  #  Save the list of selected channels.
  #  @param fn  Full path name of the file to create.
  #    
  def saveSelected(self, fn):
    try:
      #  Write File Version
      ois = os.open(fn, "w")

      jsonD = { cs.channelName : cs for cs in self if cs.selected }

      json.dump(jsonD, ois)

      ois.close()
      logging.info("List saved in %s (JSON)", fn)
    except IOError as ioex:
      ioex.printStackTrace()

  @classmethod
  def loadList(cls, fn):
    l = ChannelMorphStatusList()
    try:
      ois = os.open(fn, "r")
      data = json.load(ois)
      for cs in data.values():
        l.append(cs)

      ois.close()
    except IOError as ioex:
      # ioex.printStackTrace()
      print(str(ioex))
      l = None
    return l


# 
# This class represents the "frontImage" & "sideImage" elements of Poser files.
#  
class FrontImage(StructuredAttribut):
  def __init__(self):
    super(FrontImage, self).__init__()
    self._loaded = False
    self._file = None #  "NO_MAP" 
    self._flipped = False

  def setDirect(self, tokenID, val):
    if tokenID==PoserToken.E_loaded:
      self._loaded = val.startswith("1")
    elif tokenID==PoserToken.E_file:
      self._file = None if val == PoserConst.C_NO_MAPG or val == PoserConst.C_NO_MAP else val
    elif tokenID==PoserToken.E_flipped:
      self._flipped = val.startswith("1")
    else:
      logging.warning("Unexpected Direct:%s (%s)", tokenID.token, val)

  #    * @see deyme.v3d.poser.PoserObject#write(java.io.FileWriter)
  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName()+'\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    fw.write(nextPfx + "loaded " + ("1" if self._loaded else "0") + '\n')
    fw.write(nextPfx + "file " + (self._file if self._file else PoserConst.C_NO_MAPG ) + '\n')
    fw.write(nextPfx + "flipped " + ("1" if self._flipped else "0")+ '\n')
    for po in self._lstAttr:
      po.write(fw, nextPfx)
    if len(self._lstAttr) == 0:
      fw.write('\n')
    fw.write(nextPfx + "}\n")

# package: pftk.poser.kern
class SideImage(FrontImage):
  def __init__(self):
    super(SideImage, self).__init__()



class WeightMap(StructuredAttribut):
  def __init__(self, Name=None):
    super(WeightMap, self).__init__(Name)
    self._numbVerts = 0
    self._vertTab = []

  # Overloaded reader    
  def read(self, st):
    self._vertTab = [ ]

    fin = False
    while not fin:
      st.getToken()
      if st.ttype == PoserFileParser.TT_EOF:
        fin = True
        continue 

      if st.sval == None:
        continue 

      if st.ttype == PoserFileParser.TT_WORD:
        if st.sval == 'numbVerts':          
          self._numbVerts = int(st.getNumber())
          st.skipToNextLine()
          continue
        
        if st.sval == "v":
          self._vertTab.append( ( int(st.getNumber()), st.getNumber() ) )
          st.skipToNextLine()
          continue 

        if st.isRightBracket():
          fin = True
        else:
          #  Mot illegal
          logging.info("Line[%s] - Not accepted word:%s",st.lineno() ,  st.sval)
      else:
        logging.warning("Line[%s] - Not Accepted :%s", st.lineno() , st.sval)
        raise ParsingErrorException()

    # End of read

  def write(self, fw, pfx):
    fw.write(pfx + self.getPoserType().token + " " + self.getName() + '\n')
    nextPfx = pfx + "\t"
    fw.write(nextPfx + "{\n")
    fw.write(nextPfx + "numbVerts " + str(self._numbVerts)+'\n')

    nextPfx = pfx + "\tv {0:d} {1:9.6f}\n"

    for vi in self._vertTab:
      fw.write(nextPfx.format(vi[0], vi[1]))
 
    fw.write(pfx + "\t}\n")

class Zones(StructuredAttribut):
  def __init__(self, Name=None):
    super(Zones, self).__init__(Name)
   
class WeightMapZone(StructuredAttribut):
  def __init__(self, Name=None):
    super(WeightMapZone, self).__init__()
   
class SphereZone(StructuredAttribut):
  def __init__(self, Name=None):
    super(SphereZone, self).__init__()



# 
# Calculate point mapping table :<br>
# For each vertex of srcWG, find the closest vertex in the list of reference WaveGeom 
# If maxDist is not null, some vertex could have <b>no</b> mapping. 
# Should this occur, refGeom attribut of PtMapping is set to null. 
# 
# @param srcWG           Source WaveGeom
# @param lstRefGeom      List of reference WaveGeom
# @param translation     Translation between original MeshedObject and its localisation in this.
# @param maxDist         Maximal distance to take a point into account. If null, no maxDist are applied.
# @return                A Table of non null PtMapping
def calcMapping_KDTree(srcWG, lstRefGeom, translation, maxDist):

    vt = translation if translation else Vector3d()
    tabGeom = [ ]
    tabStart = [ ]

    # Create a numpy table Nx3
    nbsrc=0
    for refGC in lstRefGeom:
      refWG = refGC.getWaveGeom()
      nbsrc += len(refWG.coordList)


    refNo = 0
    npTab = np.zeros( (nbsrc, 3) )
    for refGC in lstRefGeom:
      refWG = refGC.getWaveGeom()
      startNo = refNo
      for p in refWG.coordList:
        npTab[refNo] = [ p.x, p.y, p.z ]
        refNo+=1
        tabGeom.append(refGC)
        tabStart.append(startNo)

    # Create an KDtree with the numpy table
    tree = spatial.KDTree(npTab, leafsize=10 if nbsrc<10000 else 100)

    svect = np.zeros((len(srcWG.coordList),3))
    for i,psrc in enumerate(srcWG.coordList):
      svect[i] = [ psrc.x - vt.x , psrc.y - vt.y , psrc.z - vt.z ]

    rest, resIdx = tree.query( svect, distance_upper_bound=maxDist )

    tabMapping = [ PtMapping(i, int(ri-tabStart[ri]), tabGeom[ri], float(rest[i])) \
                  for i,ri in enumerate(resIdx) if rest[i]<maxDist ]
    
    return tabMapping

# 
# Filter the result list to remove too short vectors.
# Does not modify the source set
#    
def filterLength(setDeltas, minVectLen):
  return { dp.noPt:dp for dp in setDeltas.values() if minVectLen <= dp.norme() }

# 
# @param tabMapping
# @param refFigure
# @param channelName
# @param minVectLen
# @return a set of { noPt:DeltaPoint }
#    
# findNewDelta(PtMapping[] tabMapping, Figure refFigure, String channelName)
# findNewDelta(PtMapping[] tabMapping, PoserMeshedObject refMeshedObj, String channelName)
def findNewDelta(tabMapping, refObj, channelName):
    setNewDeltas = { }
    emptySet = { }

    #if isinstance(refObj, "Figure"):
    if refObj.__class__.__name__ == "Figure":
      lastGeom = None
      for pm in tabMapping:
        if pm.refGeom:
          if pm.refGeom!=lastGeom:
            lastGeom = pm.refGeom
            gt = refObj.getPoserFile().findActor(pm.refGeom.getName()).getTargetGeom(channelName)            
            cachDelta = gt.getDeltas().deltaSet if gt else emptySet
              
          #srcDp = refObj.getPoserFile().findActor(pm.refGeom.getName()).findDelta(channelName, pm.refNo)
          try:
            srcDp = cachDelta[pm.refNo]
            dp = DeltaPoint(pm.srcNo, srcDp.x, srcDp.y, srcDp.z)
            setNewDeltas[dp.noPt] = dp
          except KeyError:
            pass
    else: # refObj is supposed to be a PoserMeshedObject
      gt = refObj.getTargetGeom(channelName)
  
      cachDelta = gt.getDeltas().deltaSet if gt.getDeltas() else emptySet

      for pm in tabMapping:
        if pm.refGeom:
          try:
            srcDp = cachDelta[pm.refNo]
            dp = DeltaPoint(pm.srcNo, srcDp.x, srcDp.y, srcDp.z)
            setNewDeltas[dp.noPt] = dp
          except KeyError:
            pass

    return setNewDeltas


# Increase recursion limit for very big data sets - For KDTree
sys.setrecursionlimit(20000)
