# -*- encoding: windows-1252 -*-
#
# This file is part of Dragonfly.
# (c) Copyright 2007, 2008 by Christo Butcher
# Licensed under the LGPL.
#
#   Dragonfly is free software: you can redistribute it and/or modify it
#   under the terms of the GNU Lesser General Public License as published
#   by the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   Dragonfly is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   Lesser General Public License for more details.
#
#   You should have received a copy of the GNU Lesser General Public
#   License along with Dragonfly.  If not, see
#   <http://www.gnu.org/licenses/>.
#

import copy
import locale
import unittest

from six import string_types, text_type, binary_type

from dragonfly.engines.base.dictation   import DictationContainerBase
from dragonfly.grammar.elements         import Compound, Dictation
from dragonfly.test.element_testcase    import ElementTestCase
from dragonfly.test.element_tester      import ElementTester


#===========================================================================
class DictationElementTestCase(ElementTestCase):

    input_output = []

    def test_element(self):
        # Translate all inputs and outputs to the windows-1252 encoding for
        # recognition.
        encoding = locale.getpreferredencoding()

        def translate(string):
            # Any binary strings in this file will be encoded with
            # windows-1252 as per the specified source file encoding.
            if isinstance(string, binary_type):
                string = string.decode("windows-1252").encode(encoding)

            return string

        for i, (input, output) in enumerate(self.input_output):
            new_input = translate(input)
            new_output = translate(output)
            self.input_output[i] = (new_input, new_output)

        ElementTestCase.test_element(self)


class TestNonAsciiDictation(unittest.TestCase):

    def test_dictation_non_ascii(self):
        """ Test handling of non-ASCII characters in dictation. """

        def value_func(node, extras):
            return extras["text"]
        element = Compound("test <text>",
                           extras=[Dictation(name="text")],
                           value_func=value_func)
        tester = ElementTester(element)
        words = [u"test", u"touch"]
        dictation = tester.recognize(words)

        # Verify recognition returned dictation result.
        if not isinstance(dictation, DictationContainerBase):
            encoding = locale.getpreferredencoding()
            message = (u"Expected recognition result to be a dictation"
                       u" container, but received %r"
                       % (repr(dictation),))
            self.fail(message.encode(encoding))

        # Verifying dictation converts/encode successfully.
        string = "touch"
        if isinstance(string, binary_type):
            encoding = locale.getpreferredencoding()
            string = string.decode("windows-1252").encode(encoding)
        self.assertEqual(str(dictation), string)
        self.assertEqual(text_type(dictation), u"touch")
        self.assertTrue(isinstance(repr(dictation), string_types))


class NonAsciiUnicodeDictationTestCase(DictationElementTestCase):
    """ Verify handling of non-ASCII characters in unicode dictation. """

    def _build_element(self):
        def value_func(node, extras):
            return text_type(extras["text"])
        return Compound("test <text>",
                        extras=[Dictation(name="text")],
                        value_func=value_func)

    input_output = [
                    (u"test dictation",     u"dictation"),
                    (u"test touch",        u"touch"),
                    (u"test jalapeo",      u"jalapeo"),
                   ]


class NonAsciiStrDictationTestCase(DictationElementTestCase):
    """ Verify handling of non-ASCII characters in str dictation. """

    def _build_element(self):
        def value_func(node, extras):
            return str(extras["text"])
        return Compound("test <text>",
                        extras=[Dictation(name="text")],
                        value_func=value_func)

    input_output = [
                    ("test dictation",     "dictation"),
                    ("test touch",        "touch"),
                    ("test jalapeo",      "jalapeo"),
                   ]


class FormattedDictationTestCase(DictationElementTestCase):
    """ Verify handling of string methods applied to Dictation objects """

    def _build_element(self):
        def value_func(node, extras):
            return str(extras["text"])
        return Compound("test <text>",
                        extras=[Dictation(name="text").upper().replace(" ", "/")],
                        value_func=value_func)

    input_output = [
                    ("test some random dictation", "SOME/RANDOM/DICTATION"),
                    ("test touch jalapeo",       "TOUCH/JALAPEO"),
                   ]


class CamelDictationTestCase(DictationElementTestCase):
    """ Verify handling of camelCase formatting applied to Dictation objects """

    def _build_element(self):
        def value_func(node, extras):
            return str(extras["text"])
        return Compound("test <text>",
                        extras=[Dictation(name="text").camel()],
                        value_func=value_func)

    input_output = [
                    ("test some random dictation", "someRandomDictation"),
                    ("test touch jalapeo",       "touchJalapeo"),
                   ]


class ApplyDictationTestCase(DictationElementTestCase):
    """ Verify handling of arbitrary formatting applied to Dictation objects using apply() """

    f = lambda self, s: "".join(
        "_".join(s.split(" ")[:-1] + [s.split(" ")[-1].upper()])
    )

    def _build_element(self):
        def value_func(node, extras):
            return str(extras["text"])
        return Compound("test <text>",
                        extras=[Dictation(name="text").apply(self.f)],
                        value_func=value_func)

    input_output = [
                    ("test some random dictation", "some_random_DICTATION"),
                    ("test touch jalapeo",       "touch_JALAPEO"),
                   ]


class DictationCopyTestCase(unittest.TestCase):

    def test_copy(self):
        """ Test that Dictation elements can be copied. """
        element = Dictation("text")
        self.assertIsNot(element, copy.copy(element))
        element = Dictation("text").camel()
        self.assertIsNot(element, copy.copy(element))

#===========================================================================

if __name__ == "__main__":
    unittest.main()
