"""
Main functionality for the chat agent.
"""

import sys
import socket
import requests
import importlib
from typing import Optional, List, Dict, Tuple

from rich.console import Console
from rich.markdown import Markdown
from rich.prompt import Prompt, Confirm

from .config import config
from .storage import ChatSession

console = Console()


def check_internet_connectivity() -> Tuple[bool, str]:
    """Check if internet is accessible.

    Returns:
        Tuple[bool, str]: (is_connected, error_message)
    """
    try:
        # Try to connect to Google's DNS server
        socket.create_connection(("8.8.8.8", 53), timeout=3)
        return True, ""
    except OSError:
        # Try an alternative method using requests
        try:
            requests.get("https://www.google.com", timeout=3)
            return True, ""
        except requests.RequestException:
            msg = "Cannot connect to the internet. Please check your connection."
            return False, msg


class ChatAgent:
    """Chat agent that interfaces with AI providers via AirTrain."""

    def __init__(self, session_id: Optional[str] = None):
        """Initialize the chat agent with an optional session ID."""
        # Load or create a chat session
        if session_id:
            self.session = ChatSession.load(session_id)
            if not self.session:
                msg = (
                    f"[red]Session {session_id} not found. "
                    f"Creating a new session.[/red]"
                )
                console.print(msg)
                self.session = ChatSession()
        else:
            self.session = ChatSession()

        # Load the active provider and model
        self.provider = config.get_active_provider()
        self.model = config.get_active_model()

        # Check for API key and handle setup if missing
        if not config.is_api_key_set:
            self._handle_missing_credentials()

        # Check if we have credentials now
        if not config.is_api_key_set:
            console.print("[red]Error: Could not set up API credentials.[/red]")
            console.print(f"Please try again or set {self.provider.upper()}_API_KEY manually.")
            sys.exit(1)

        # Initialize the provider-specific client and skill
        self._initialize_provider()

    def _initialize_provider(self):
        """Initialize the provider-specific client and skill."""
        provider_module_path = config.get_provider_module_path(self.provider)
        if not provider_module_path:
            console.print(f"[red]Error: Provider '{self.provider}' is not supported.[/red]")
            sys.exit(1)

        try:
            # Import the provider's skill and credentials modules
            skill_module = importlib.import_module(f"{provider_module_path}.skills")
            creds_module = importlib.import_module(f"{provider_module_path}.credentials")

            # Determine class names based on provider
            if self.provider == "together":
                creds_class_name = "TogetherAICredentials"
                chat_skill_class_name = "TogetherAIChatSkill"
                input_class_name = "TogetherAIInput"
            else:
                creds_class_name = f"{self.provider.capitalize()}Credentials"
                chat_skill_class_name = f"{self.provider.capitalize()}ChatSkill"
                input_class_name = f"{self.provider.capitalize()}Input"

            # Get the classes
            creds_class = getattr(creds_module, creds_class_name)
            chat_skill_class = getattr(skill_module, chat_skill_class_name)
            self.input_class = getattr(skill_module, input_class_name)

            # Get the API key for this provider
            api_key = config.get_provider_api_key(self.provider)

            # Initialize credentials
            credentials_kwargs = {f"{self.provider}_api_key": api_key}
            self.credentials = creds_class(**credentials_kwargs)

            # Initialize chat skill
            self.chat_skill = chat_skill_class(credentials=self.credentials)

        except (ImportError, AttributeError) as e:
            console.print(f"[red]Error initializing provider '{self.provider}': {str(e)}[/red]")
            sys.exit(1)

    def _handle_missing_credentials(self) -> None:
        """Handle missing credentials by prompting the user."""
        console.print(f"[yellow]Warning: {self.provider.capitalize()} API key not found![/yellow]")
        msg = "The API key was not found in environment variables or credentials file."
        console.print(msg)
        creds_path = config.credentials_dir / f"{self.provider}.json"
        console.print(f"Credentials should be in: {creds_path}")
        console.print()

        set_key = Confirm.ask(f"Would you like to set your {self.provider.capitalize()} API key now?")
        if set_key:
            api_key = Prompt.ask(f"Enter your {self.provider.capitalize()} API key", password=True)
            if api_key:
                if config.save_provider_credentials(self.provider, api_key):
                    console.print(f"[green]API key saved successfully.[/green]")
                else:
                    console.print(f"[red]Failed to save API key.[/red]")
            else:
                console.print("[yellow]No API key provided.[/yellow]")

    def chat(self) -> None:
        """Start an interactive chat session."""
        # Remove connectivity checks from here - we'll check only when making API calls

        console.print("[bold blue]Chat Session Started[/bold blue]")
        console.print(f"Session ID: {self.session.session_id}")
        console.print(f"Using: [cyan]{self.provider.capitalize()}[/cyan] with model [cyan]{self.model}[/cyan]")
        console.print("Type 'exit', 'quit', or 'q' to end the session.")
        console.print("For multi-line input, use one of these triggers:")
        console.print("  - /m, /multiline, /multi, /p, /paste (end with /end)")
        console.print("  - \"\"\" (end with \"\"\")")
        console.print("  - ''' (end with ''')")
        console.print()

        # Display existing messages if any
        if self.session.messages:
            console.print("[bold]Chat History:[/bold]")
            for message in self.session.messages:
                if message["role"] == "user":
                    console.print(f"[bold green]You:[/bold green] {message['content']}")
                else:
                    console.print("[bold purple]AI:[/bold purple]")
                    console.print(Markdown(message["content"]))
            console.print("\n[bold]Continuing conversation...[/bold]\n")

        # Interactive chat loop
        while True:
            try:
                # Get user input (potentially multi-line)
                user_input = self._get_user_input()
                
                if user_input is None:  # User canceled input
                    continue

                # Check for exit command
                if user_input.lower() in ["exit", "quit", "q"]:
                    console.print("[bold blue]Ending chat session...[/bold blue]")
                    break

                # Add user message to session
                self.session.add_message("user", user_input)

                # Format messages for the API
                messages = self._format_messages_for_api()
                system_prompt = "You are a helpful assistant."

                # Check internet connectivity only when about to make an API call
                internet_connected, internet_error = check_internet_connectivity()
                if not internet_connected:
                    console.print(f"[bold red]Error: {internet_error}[/bold red]")
                    continue  # Continue the loop to allow user to try again or quit

                # Prepare the input for the provider's skill
                input_kwargs = {
                    "user_input": user_input,
                    "system_prompt": system_prompt,
                    "conversation_history": messages[:-1],  # Exclude the last user message
                    "model": self.model,
                    "max_tokens": 1024,
                    "temperature": 0.7,
                }
                
                input_data = self.input_class(**input_kwargs)

                # Call the API
                console.print("[bold purple]AI:[/bold purple]")
                with console.status("[bold]Thinking...[/bold]"):
                    result = self.chat_skill.process(input_data)

                # Get AI response
                ai_response = result.response
                console.print(Markdown(ai_response))

                # Add AI message to session
                self.session.add_message("assistant", ai_response)

            except KeyboardInterrupt:
                interrupt_msg = (
                    "\n[bold blue]Chat session interrupted. "
                    "Saving and exiting...[/bold blue]"
                )
                console.print(interrupt_msg)
                break
            except Exception as e:
                console.print(f"[bold red]Error: {str(e)}[/bold red]")
                continue

    def _get_user_input(self) -> Optional[str]:
        """Get user input, supporting multi-line input with various triggers."""
        # Get the first line of input
        first_line = console.input("[bold green]You:[/bold green] ")
        
        # Check for multi-line input triggers
        multiline_triggers = ["/m", "/multiline", "/multi", "/p", "/paste"]
        quote_triggers = ["\"\"\"", "'''"]
        
        # For command-based triggers
        if first_line in multiline_triggers:
            return self._collect_multiline_input("/end")
        
        # For quote-based triggers
        for trigger in quote_triggers:
            if first_line == trigger:
                return self._collect_multiline_input(trigger)
        
        # Regular single-line input
        return first_line
    
    def _collect_multiline_input(self, terminator: str) -> Optional[str]:
        """Collect multi-line input until the terminator is encountered."""
        console.print(
            f"[bold cyan]Multi-line input mode.[/bold cyan]"
            f" [bold cyan]End with '{terminator}' on a new line.[/bold cyan]"
        )
        lines = []
        
        try:
            while True:
                line = console.input()
                if line == terminator:
                    break
                lines.append(line)
                
            # Join the lines with newlines
            if not lines:  # Empty input
                console.print("[yellow]Empty input. Cancelled.[/yellow]")
                return None
                
            return "\n".join(lines)
            
        except KeyboardInterrupt:
            console.print("\n[yellow]Multi-line input cancelled.[/yellow]")
            return None

    def _format_messages_for_api(self) -> List[Dict[str, str]]:
        """Format the messages for the Together AI API."""
        # Filter out timestamp and other metadata
        return [
            {"role": msg["role"], "content": msg["content"]}
            for msg in self.session.messages
        ]
