"""Setup command for initializing Frappe/ERPNext development environment."""

import click
import sys
import shutil
from pathlib import Path

from ..utils import (
    Colors, print_info, print_success, print_warning, print_error,
    run_command
)


@click.command()
@click.option('--project-name', 
              help='Name of the project (default: current directory name)')
@click.option('--frappe-version', 
              default='v15.78.1',
              help='Frappe version to install (default: v15.78.1)')
@click.option('--python-path',
              default='python3',
              help='Path to Python executable (default: python3)')
@click.option('--force',
              is_flag=True,
              help='Force setup even if virtual environment exists')
@click.option('--install-docs',
              is_flag=True,
              default=True,
              help='Install TECHNICAL_DOCS.md template (default: True)')
@click.option('--init-git',
              is_flag=True,
              help='Initialize Git repository')
def setup(project_name, frappe_version, python_path, force, install_docs, init_git):
    """Setup Frappe/ERPNext development environment.
    
    This command creates a virtual environment, installs frappe-bench,
    initializes a new bench with the specified Frappe version, and optionally
    installs documentation templates and initializes a Git repository.
    """
    # Get the directory where this command is being run
    script_dir = Path.cwd()
    
    # Get project name from current directory if not provided
    if not project_name:
        project_name = script_dir.name
    
    print()
    print(f"{Colors.BLUE}{'=' * 63}{Colors.NC}")
    print(f"{Colors.GREEN}  Frappe/ERPNext Environment Setup{Colors.NC}")
    print(f"{Colors.BLUE}{'=' * 63}{Colors.NC}")
    print()
    print_info(f"Project Name: {project_name}")
    print_info(f"Script Directory: {script_dir}")
    print_info(f"Frappe Version: {frappe_version}")
    print()
    
    # Check if virtual environment already exists
    env_dir = script_dir / "env"
    if env_dir.exists() and not force:
        print_warning("Virtual environment already exists!")
        if not click.confirm("Do you want to continue anyway?"):
            print_info("Setup cancelled.")
            return
    
    # ═══════════════════════════════════════════════════════════
    # CREATE VIRTUAL ENVIRONMENT
    # ═══════════════════════════════════════════════════════════
    print_info("Setting up virtual environment...")
    
    if not run_command(f'{python_path} -m venv "{env_dir}"', cwd=script_dir):
        print_error("Failed to create virtual environment!")
        sys.exit(1)
    
    print_success("Virtual environment created")
    print()
    
    # Paths to executables in virtual environment
    python_exe = env_dir / "bin" / "python"
    pip_exe = env_dir / "bin" / "pip"
    
    # ═══════════════════════════════════════════════════════════
    # UPGRADE PIP
    # ═══════════════════════════════════════════════════════════
    print_info("Upgrading pip...")
    if not run_command(f'"{python_exe}" -m pip install --quiet --upgrade pip', 
                      cwd=script_dir, quiet=True):
        print_error("Failed to upgrade pip!")
        sys.exit(1)
    
    print_success("pip upgraded")
    print()
    
    # ═══════════════════════════════════════════════════════════
    # INSTALL FRAPPE-BENCH
    # ═══════════════════════════════════════════════════════════
    print_info("Installing frappe-bench...")
    if not run_command(f'"{python_exe}" -m pip install --quiet frappe-bench', 
                      cwd=script_dir, quiet=True):
        print_error("Failed to install frappe-bench!")
        sys.exit(1)
    
    print_success("frappe-bench installed")
    print()
    
    # ═══════════════════════════════════════════════════════════
    # INSTALL WHEEL
    # ═══════════════════════════════════════════════════════════
    print_info("Installing wheel...")
    if not run_command(f'"{python_exe}" -m pip install --quiet wheel', 
                      cwd=script_dir, quiet=True):
        print_error("Failed to install wheel!")
        sys.exit(1)
    
    print_success("wheel installed")
    print()
    
    # ═══════════════════════════════════════════════════════════
    # INITIALIZE BENCH
    # ═══════════════════════════════════════════════════════════
    print_info("Getting frappe and initializing bench...")
    print_warning("This may take a while...")
    print()
    
    # Change to parent directory to run bench init
    parent_dir = script_dir.parent
    
    # Build bench init command
    bench_cmd = (f'source "{env_dir}/bin/activate" && '
                f'bench init {project_name} --frappe-branch {frappe_version} --ignore-exist')
    
    if not run_command(bench_cmd, cwd=parent_dir, shell=True, check=True, quiet=False):
        print_error("Failed to initialize bench!")
        sys.exit(1)
    
    print()
    print_success("Bench initialized successfully!")
    print()
    
    # ═══════════════════════════════════════════════════════════
    # INSTALL TECHNICAL DOCS
    # ═══════════════════════════════════════════════════════════
    if install_docs:
        print_info("Installing TECHNICAL_DOCS.md template...")
        try:
            # Try to get the template file from the package
            import importlib.resources
            try:
                # Python 3.9+
                with importlib.resources.files("ws_frappe_cli.templates.docs").joinpath("TECHNICAL_DOCS.md").open("r") as f:
                    docs_content = f.read()
            except AttributeError:
                # Python 3.8 fallback
                with importlib.resources.open_text("ws_frappe_cli.templates.docs", "TECHNICAL_DOCS.md") as f:
                    docs_content = f.read()
            
            # Write to the root directory
            docs_file = script_dir / "TECHNICAL_DOCS.md"
            with open(docs_file, "w") as f:
                f.write(docs_content)
            
            print_success("TECHNICAL_DOCS.md installed in root directory")
        except Exception as e:
            print_warning(f"Could not install TECHNICAL_DOCS.md: {e}")
        print()
    
    # ═══════════════════════════════════════════════════════════
    # INITIALIZE GIT REPOSITORY
    # ═══════════════════════════════════════════════════════════
    if init_git:
        print_info("Initializing Git repository...")
        
        # Check if Git is available
        if not run_command("git --version", quiet=True):
            print_error("Git is not installed or not available in PATH!")
            print_warning("Skipping Git initialization.")
        else:
            git_dir = script_dir / ".git"
            if git_dir.exists():
                print_warning("Git repository already exists!")
            else:
                if run_command("git init", cwd=script_dir, quiet=True):
                    print_success("Git repository initialized")
                    
                    # Create a basic .gitignore for Frappe projects
                    gitignore_content = """# Frappe/ERPNext specific
env/
sites/*/private/
sites/*/public/
sites/*/locks/
*.pyc
__pycache__/
.DS_Store
*.log
logs/
node_modules/
.vscode/
.idea/

# Bench specific
sites/apps.txt
sites/common_site_config.json
sites/currentsite.txt

# Python
*.egg-info/
build/
dist/
.coverage
.pytest_cache/
"""
                    gitignore_file = script_dir / ".gitignore"
                    with open(gitignore_file, "w") as f:
                        f.write(gitignore_content)
                    print_success(".gitignore created")
                    
                    # Make initial commit if docs were installed
                    if install_docs and (script_dir / "TECHNICAL_DOCS.md").exists():
                        run_command("git add TECHNICAL_DOCS.md .gitignore", cwd=script_dir, quiet=True)
                        run_command('git commit -m "Initial commit: Add technical docs and gitignore"', cwd=script_dir, quiet=True)
                        print_success("Initial commit created")
                else:
                    print_error("Failed to initialize Git repository")
        print()
    
    # ═══════════════════════════════════════════════════════════
    # FINAL MESSAGE
    # ═══════════════════════════════════════════════════════════
    print()
    print(f"{Colors.BLUE}{'=' * 63}{Colors.NC}")
    print(f"{Colors.GREEN}  Setup Completed Successfully!{Colors.NC}")
    print(f"{Colors.BLUE}{'=' * 63}{Colors.NC}")
    print()
    print_success("Your Frappe development environment is ready!")
    
    if install_docs:
        print_success("TECHNICAL_DOCS.md template is available in the root directory")
    
    if init_git:
        print_success("Git repository has been initialized")
    
    print()
    print_info("Next steps:")
    print(f"  1. Activate the virtual environment:")
    print(f"     {Colors.YELLOW}source env/bin/activate{Colors.NC}")
    print(f"  2. Create a new site:")
    print(f"     {Colors.YELLOW}ws-frappe-cli create-site{Colors.NC}")
    print(f"  3. Install apps:")
    print(f"     {Colors.YELLOW}ws-frappe-cli fetch-app{Colors.NC}")
    print(f"  4. Start the development server:")
    print(f"     {Colors.YELLOW}bench start{Colors.NC}")
    
    if install_docs:
        print()
        print_info("Documentation:")
        print(f"  • Review and customize TECHNICAL_DOCS.md for your project")
        print(f"  • Update app versions and customizations as needed")
    
    if init_git:
        print()
        print_info("Git workflow:")
        print(f"  • Add your remote repository: {Colors.YELLOW}git remote add origin <your-repo-url>{Colors.NC}")
        print(f"  • Make your first commit with project files")
    
    print()


if __name__ == "__main__":
    setup()