# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aries_staticagent', 'aries_staticagent.dispatcher']

package_data = \
{'': ['*']}

install_requires = \
['PyNaCl>=1.4.0,<2.0.0',
 'aiohttp>=3.7.4,<4.0.0',
 'base58>=2.1.0,<3.0.0',
 'msgpack>=1.0.2,<2.0.0',
 'pydantic>=1.8.2,<2.0.0',
 'semver>=2.13.0,<3.0.0',
 'sortedcontainers>=2.4.0,<3.0.0']

setup_kwargs = {
    'name': 'aries-staticagent',
    'version': '0.9.0rc4',
    'description': 'Aries Static Agent library for Python',
    'long_description': '# Static Agent Library\n\n[![pypi release](https://img.shields.io/pypi/v/aries-staticagent)](https://pypi.org/project/aries-staticagent/)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n[![Tests](https://github.com/hyperledger/aries-staticagent-python/actions/workflows/tests.yml/badge.svg)](https://github.com/hyperledger/aries-staticagent-python/actions/workflows/tests.yml)\n\n\nThis repo contains an example Aries Static Agent Library in Python.\n\nA static agent is a form of agent that can speak DIDComm at a basic level but its keys and\nconnections are configured statically. Static Agents have a direct relationship with a single full\nagent. Static Agents do not have a wallet.\n\nExamples of static agents may include:\n- Remote administration interface for an agent\n- IoT devices\n- [Relays][1]\n- OpenAPI to DIDComm translator\n\nA static agent\'s configuration minimally consists of:\n- Its own public and private key\n- The public key of its full agent counterpart\n- The endpoint of its full agent counterpart\n\n_**It is up to you to secure your static agent\'s configuration.**_ The examples included in this\nrepository use command line arguments or environment variables to configure the agent for simplicity\nand demonstration purposes only. _**This is not recommended for production environments.**_\n\nThis library makes as few assumptions about it\'s running environment as possible. This includes\nfew dependencies, assumptions about web frameworks, etc.\n\n[1]: https://github.com/hyperledger/aries-rfcs/tree/master/concepts/0046-mediators-and-relays#summary\n\nQuick Start Guide\n-----------------\n\n#### Requirements\n\n- Python 3.6 or higher\n\n#### Running the included examples\n\nYou will need to pair the static agent with a full agent capable of basic DIDComm to complete the\nexamples. The [Indy Python Reference Agent][3] can be used as the full agent for these examples.\n\nCreate and activate python virtual environment:\n```sh\n$ python3 -m venv env\n$ source env/bin/activate\n```\n\nInstall dependencies and the library into the virtual environment:\n```sh\n$ pip install -e .\n```\n\n> If you want to run the included tests, install the `test` feature with pip:\n> `pip install -e .[test]`\n\nExecute `keygen()`:\n```sh\n$ python -c "import aries_staticagent; aries_staticagent.keygen()"\n\nFor full agent:\n        DID: <Base58 encoded DID>\n        VK: <Base58 encoded verkey>\n\nFor static agent:\n        VK: <the same Base58 encoded verkey>\n        SK: <Base58 encoded sigkey>\n```\n\nAs the output implies, the first section is intended to be entered in on the full agent to configure\na static connection. The second section is used to configure the static agent. The `verkey` (VK) in\nthe first and second section are the _same_ key, representing the key the static agent will use\nfor the connection. The `keygen` script does _not_ generate the keys that the full agent will use.\n\nIf using the [Indy Python Reference Agent][3], open the web interface and\nclick `Add Static Connection`. Enter the information output by `keygen.py` and a label of your\nchoice. The endpoint of the static agent is optional and must match the hostname and port you\nconfigure for the static agent if running the web server example. After clicking `Add`, a new\ndialogue window will open with the information needed to now start up the static agent.\n\nIf you are using another agent that supports configuring a static connection, follow the\ninstructions provided by that agent.\n\nStart the static agent (in this case, `exapmles/cron.py`):\n```sh\n$ python examples/cron.py --endpoint <the endpoint of the full agent> \\\n$ --endpointkey <the verkey output by the full agent> \\\n$ --mypublickey <the verkey output by keygen.py> \\\n$ --myprivatekey <the sigkey output by keygen.py>\n```\n\nIn the full agent\'s BasicMessages, you should now see a message sent from the static agent script.\n\n> TODO: Include screencast of running the example with the Indy Python Reference Agent\n\n[2]: https://download.libsodium.org/doc/installation\n[3]: https://github.com/hyperledger/indy-agent/tree/master/python\n\nUsing the library\n-----------------\n\nRefer to the `examples` directory for complete working examples of using this library.\n\n### Setting up a Static Agent Connection\n\n```python\nfrom aries_staticagent import Connection\n\n# endpoint, endpointkey, mypublickey, myprivatekey key are obtained through some form of static\n# configuration\n\nconn = Connection((mypublickey, myprivatekey), their_vk=endpointkey, endpoint=endpoint)\n```\n\nThis will open a static connection with the full agent reachable at `endpoint` and messages packed\nfor `endpointkey`.\n\n### Sending a message to the Full Agent\n\nWith the static agent connection `a`, to send messages to the full agent:\n\n```python\nconn.send({\n    "@type": "https://didcomm.org/basicmessage/1.0/message",\n    "~l10n": {"locale": "en"},\n    "sent_time": utils.timestamp(),\n    "content": "The Cron Script has been executed."\n})\n```\n\nAn asynchronous method is also provided:\n```python\nawait conn.send_async({\n    "@type": "https://didcomm.org/basicmessage/1.0/message",\n    "~l10n": {"locale": "en"},\n    "sent_time": utils.timestamp(),\n    "content": "The Cron Script has been executed."\n})\n```\n\n### Receiving messages from the Full Agent\n\nTransport mechanisms are completely decoupled from the Static Agent Library. This is intended to\nallow library consumers to choose which transport is appropriate for their use case. The\n`examples/webserver_aiohttp.py` example shows how one might use the `aiohttp` library as an inbound\ntransport mechanism for the static agent:\n\n```python\nfrom aiohttp import web\nfrom aries_staticagent import Connection, utils\n\n# ... Configuration omitted\n\n# Create static agent connection\nconn = Connection((args.mypublickey, args.myprivatekey), their_vk=args.endpointkey, endpoint=args.endpoint)\n\n# Register a handler for the basicmessage/1.0/message message type\n@conn.route("https://didcomm.org/basicmessage/1.0/message")\nasync def basic_message(msg, conn):\n    # Respond to the received basic message by sending another basic message back\n    await conn.send_async({\n        "@type": "https://didcomm.org/basicmessage/1.0/message",\n        "~l10n": {"locale": "en"},\n        "sent_time": utils.timestamp(),\n        "content": "You said: {}".format(msg[\'content\'])\n    })\n\n\n# aiohttp request handler\nasync def handle(request):\n    # Read request body and pass to Connection.handle\n    await conn.handle(await request.read())\n    raise web.HTTPAccepted()\n\n# Register aiohttp request handler\napp = web.Application()\napp.add_routes([web.post(\'/\', handle)])\n\n# Start the web server\nweb.run_app(app, port=args.port)\n```\n\nAs seen in this example, registering a handler for a DIDComm message is done using the\n`@conn.route(\'<message_type>\')` decorator. Passing raw, unpackaged messages to the static agent\nconnection over the decoupled transport mechanism is done by calling `conn.handle(<raw message>)`.\n\nStatic agents can only unpack messages sent by the full agent.\n\n### Unresolved Questions\n* Are we allowing Agent routing between a static agent and it\'s full agent?\n  * We\'re starting with no and will revisit in the future.\n\n## License\n\n[Apache License Version 2.0](https://github.com/hyperledger/aries-staticagent-python/blob/main/LICENSE)\n',
    'author': 'Daniel Bluhm',
    'author_email': 'dbluhm@pm.me',
    'maintainer': 'Daniel Bluhm',
    'maintainer_email': 'dbluhm@pm.me',
    'url': 'https://github.com/hyperledger/aries-staticagent-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
