# TypeScript Umbrel Linter - Complete Feature Analysis

This document provides a comprehensive analysis of all features and validations implemented in the original TypeScript Umbrel linter.

## Overview

The TypeScript linter is a comprehensive validation tool for Umbrel applications that checks:
- **umbrel-app.yml** manifest files
- **docker-compose.yml** files  
- **Directory structure** and file organization
- **Security vulnerabilities** and best practices
- **GitHub integration** and pull request validation

## Core Architecture

### Main Components
- **Lint Module** (`src/modules/lint.ts`): Core validation logic
- **Mock Module** (`src/modules/mock.ts`): Variable mocking for Docker Compose
- **Registry Module** (`src/modules/registry.ts`): Docker registry API integration
- **Image Module** (`src/modules/image.ts`): Docker image parsing and validation
- **YAML Utils** (`src/utils/yaml.ts`): YAML parsing and source mapping

### Error Types
The linter defines 20+ specific error types:
- `invalid_yaml_syntax`
- `invalid_docker_image_name`
- `invalid_yaml_boolean_value`
- `invalid_app_data_dir_volume_mount`
- `invalid_submission_field`
- `missing_file_or_directory`
- `docker_socket_mount`
- `empty_app_data_directory`
- `external_port_mapping`
- `invalid_image_architectures`
- `invalid_container_user`
- `filled_out_release_notes_on_first_submission`
- `filled_out_icon_or_gallery_on_first_submission`
- `container_network_mode_host`
- `invalid_app_proxy_configuration`
- `invalid_restart_policy`
- `duplicate_ui_port`

## umbrel-app.yml Validation Features

### 1. YAML Syntax Validation
- **Feature**: Validates YAML syntax and structure
- **Error**: `invalid_yaml_syntax`
- **Severity**: ERROR
- **Message**: "umbrel-app.yml is not a valid YAML file"

### 2. Schema Validation (Zod)
- **Feature**: Comprehensive schema validation using Zod
- **Checks**: All required fields, data types, format validation
- **Error**: Various Zod error codes
- **Severity**: ERROR

### 3. New App Submission Validation
- **Feature**: Validates fields for new app submissions
- **Checks**:
  - `submission` field must match pull request URL
  - `releaseNotes` must be empty for new submissions
  - `icon` and `gallery` must be empty for new submissions
- **Errors**: 
  - `invalid_submission_field`
  - `filled_out_release_notes_on_first_submission`
  - `filled_out_icon_or_gallery_on_first_submission`
- **Severity**: ERROR/WARNING

### 4. Port Uniqueness Validation
- **Feature**: Ensures each app uses a unique port
- **Error**: `duplicate_ui_port`
- **Severity**: ERROR
- **Message**: "Each app must use a unique port"

## docker-compose.yml Validation Features

### 1. YAML Syntax & Schema Validation
- **Feature**: Validates Docker Compose YAML syntax and schema
- **Uses**: AJV with Docker Compose JSON Schema
- **Error**: `invalid_yaml_syntax`
- **Severity**: ERROR

### 2. Variable Mocking
- **Feature**: Mocks environment variables for proper validation
- **Variables**: `${APP_DATA_DIR}`, `${APP_HOST}`, `${APP_PORT}`, etc.
- **Purpose**: Enables schema validation with dynamic values

### 3. Docker Image Validation

#### 3.1 Image Name Format
- **Feature**: Validates image naming convention
- **Format**: `<name>:<version-tag>@<sha256>`
- **Error**: `invalid_docker_image_name`
- **Severity**: ERROR
- **Message**: "Images should be named like '<name>:<version-tag>@<sha256>'"

#### 3.2 Latest Tag Warning
- **Feature**: Warns against using 'latest' tag
- **Error**: `invalid_docker_image_name`
- **Severity**: WARNING
- **Message**: "Images should not use the 'latest' tag"

#### 3.3 Architecture Validation
- **Feature**: Checks if images support arm64 and amd64 architectures
- **API**: Makes HTTP requests to Docker registries
- **Error**: `invalid_image_architectures`
- **Severity**: ERROR
- **Message**: "The image does not support the architectures 'arm64' and 'amd64'"

### 4. YAML Boolean Value Validation
- **Feature**: Ensures boolean values are strings (Docker Compose V1 compatibility)
- **Checks**: `environment`, `labels`, `extra_hosts` fields
- **Error**: `invalid_yaml_boolean_value`
- **Severity**: ERROR
- **Message**: "Boolean values should be strings like 'true' instead of true"

### 5. Volume Mount Validation

#### 5.1 APP_DATA_DIR Direct Mount Warning
- **Feature**: Warns against mounting directly to `${APP_DATA_DIR}`
- **Error**: `invalid_app_data_dir_volume_mount`
- **Severity**: WARNING
- **Message**: "Volumes should not be mounted directly into the '${APP_DATA_DIR}' directory!"

#### 5.2 Missing File/Directory Check
- **Feature**: Checks if mounted files/directories exist
- **Error**: `missing_file_or_directory`
- **Severity**: INFO
- **Message**: "The volume tries to mount the file/directory, but it is not present"

### 6. Security Validations

#### 6.1 Docker Socket Mount Warning
- **Feature**: Warns about Docker socket mounting
- **Error**: `docker_socket_mount`
- **Severity**: WARNING
- **Message**: "The volume mounts the Docker socket, which can be a security risk"

#### 6.2 Container User Validation
- **Feature**: Checks for unsafe user configurations
- **Checks**: Root user usage, UID/PUID environment variables
- **Error**: `invalid_container_user`
- **Severity**: INFO
- **Message**: "The default container user 'root' can lead to security vulnerabilities"
- **Exception**: Skips `app_proxy` service (needs root privileges)

#### 6.3 Host Network Mode Warning
- **Feature**: Warns about host network mode usage
- **Error**: `container_network_mode_host`
- **Severity**: INFO
- **Message**: "The host network mode can lead to security vulnerabilities"

### 7. Port Mapping Validation
- **Feature**: Informs about potentially unnecessary port mappings
- **Error**: `external_port_mapping`
- **Severity**: INFO
- **Message**: "Port mappings may be unnecessary for the app to function correctly"

### 8. App Proxy Configuration Validation

#### 8.1 APP_HOST Environment Variable
- **Feature**: Validates APP_HOST configuration
- **Checks**: Required, format validation
- **Error**: `invalid_app_proxy_configuration`
- **Severity**: ERROR/WARNING
- **Message**: "The app_proxy container needs to have the APP_HOST environment variable set"

#### 8.2 APP_PORT Environment Variable
- **Feature**: Validates APP_PORT configuration
- **Checks**: Required, numeric validation
- **Error**: `invalid_app_proxy_configuration`
- **Severity**: ERROR/WARNING
- **Message**: "The app_proxy container needs to have the APP_PORT environment variable set"

### 9. Restart Policy Validation
- **Feature**: Ensures restart policy is set to 'on-failure'
- **Error**: `invalid_restart_policy`
- **Severity**: WARNING
- **Message**: "The restart policy should be set to 'on-failure'"
- **Exception**: Skips `app_proxy` service

## Directory Structure Validation

### 1. Empty Directory Check
- **Feature**: Ensures empty directories have `.gitkeep` files
- **Error**: `empty_app_data_directory`
- **Severity**: ERROR
- **Message**: "Please add a '.gitkeep' file to the directory"

## Advanced Features

### 1. Source Map Integration
- **Feature**: Provides precise line/column information for errors
- **Implementation**: YAML source mapping with `getSourceMapForKey`
- **Benefit**: Enables precise error location in files

### 2. Registry API Integration
- **Feature**: Makes HTTP requests to Docker registries
- **Purpose**: Validates image existence and architectures
- **Rate Limiting**: Optional flag to prevent registry rate limiting
- **Timeout**: Configurable timeout for API requests

### 3. Variable Mocking System
- **Feature**: Comprehensive variable replacement system
- **Variables**: IP addresses, ports, passwords, directories, etc.
- **Purpose**: Enables proper schema validation with dynamic values

### 4. Multi-App Validation
- **Feature**: Validates port uniqueness across all apps
- **Implementation**: Cross-app port conflict detection
- **Benefit**: Prevents port conflicts in app stores

## Error Severity Levels

### ERROR (3)
- YAML syntax errors
- Schema validation failures
- Missing required files
- Invalid image formats
- Architecture validation failures
- Port conflicts
- Missing environment variables

### WARNING (2)
- Docker socket mounting
- Invalid restart policies
- APP_DATA_DIR direct mounting
- Invalid app proxy configuration
- Latest tag usage

### INFO (1)
- Port mapping notifications
- User security warnings
- Host network mode warnings
- Missing file/directory notifications

## Configuration Options

### Command Line Options
- `--checkImageArchitectures`: Enable architecture validation
- `--loglevel`: Set minimum log level (error/warning/info)
- `--app`: Lint specific app
- `--new-submission`: New app submission mode
- `--pr-url`: Pull request URL for validation

### Library Mode Options
- `isNewAppSubmission`: New submission validation
- `pullRequestUrl`: PR URL validation
- `allUmbrelAppYmlContents`: Cross-app validation
- `checkImageArchitectures`: Architecture checking

## Performance Considerations

### 1. Registry API Rate Limiting
- **Issue**: Multiple apps can trigger rate limits
- **Solution**: Optional `--checkImageArchitectures` flag
- **Default**: Disabled for bulk operations

### 2. Variable Mocking Performance
- **Feature**: Efficient variable replacement
- **Implementation**: Regex-based pattern matching
- **Benefit**: Fast processing of large files

### 3. Caching
- **Feature**: Registry response caching
- **Purpose**: Avoid duplicate API calls
- **Implementation**: In-memory cache for registry responses

## Integration Points

### 1. GitHub Actions
- **Feature**: Library mode for CI/CD integration
- **Options**: New submission validation, PR URL checking
- **Benefit**: Automated validation in pull requests

### 2. Umbrel App Store
- **Feature**: Bulk validation of app stores
- **Capabilities**: Cross-app validation, port conflict detection
- **Benefit**: Ensures app store integrity

### 3. Development Workflow
- **Feature**: Individual app validation
- **Options**: Specific app linting, detailed error reporting
- **Benefit**: Developer-friendly validation

## Security Focus

The linter prioritizes security with extensive checks for:
- **Container Security**: User privileges, network modes
- **Volume Security**: Docker socket mounting, data directory access
- **Network Security**: Port exposure, host network usage
- **Image Security**: Architecture validation, tag verification

## Best Practices Enforcement

The linter enforces Umbrel development best practices:
- **Naming Conventions**: Image tags, file formats
- **Directory Structure**: Proper organization, .gitkeep files
- **Configuration**: App proxy setup, restart policies
- **Documentation**: Release notes, submission URLs

This comprehensive validation system ensures that Umbrel applications meet high standards for security, functionality, and maintainability.
