import errno
import os
from dataclasses import dataclass
from typing import Optional

from frogml_storage.authentication._authentication_utils import get_credentials
from frogml_storage.cli._login_cli import login as frogml_login
from frogml_storage.constants import CONFIG_FILE_PATH

from frogml_core.exceptions import FrogmlLoginException
from frogml_core.inner.tool.auth import FrogMLAuthClient


@dataclass
class UserAccount:
    """
    User Account Configuration
    """

    # Assigned Token
    token: Optional[str] = None

    # Assigned username
    username: Optional[str] = None

    # Assigned password
    password: Optional[str] = None

    # Assigned URL
    url: Optional[str] = None

    # Anonymous login
    anonymous: bool = False

    # Interactive login
    is_interactive: bool = False


class UserAccountConfiguration:
    def __init__(
        self,
        config_file=CONFIG_FILE_PATH,
    ):
        self._config_file = config_file
        self._auth_client = FrogMLAuthClient()

    def configure_user(self, user_account: UserAccount):
        """
        Configure user authentication based on the authentication client type
        """

        # Use FrogML login flow
        success = frogml_login(
            url=user_account.url,
            username=user_account.username,
            password=user_account.password,
            token=user_account.token,
            anonymous=user_account.anonymous,
            is_interactive=user_account.is_interactive,
        )

        if not success:
            raise FrogmlLoginException("Failed to authenticate with JFrog")

        # Validate token
        token: Optional[str] = self._auth_client.get_token()

        if not token or len(token) <= 64:
            raise FrogmlLoginException(
                "Authentication with JFrog failed: Only Access Tokens are supported. "
                "Please ensure you are using a valid Access Token."
            )

    @staticmethod
    def _mkdir_p(path):
        try:
            os.makedirs(path)
        except OSError as exc:  # Python >2.5
            if not (exc.errno == errno.EEXIST and os.path.isdir(path)):
                raise

    @staticmethod
    def _safe_open(path):
        UserAccountConfiguration._mkdir_p(os.path.dirname(path))
        return open(path, "w")

    @staticmethod
    def get_user_config() -> UserAccount:
        """
        Get persisted user account from config file
        :return:
        """
        return UserAccount()

    @staticmethod
    def get_user_token() -> str:
        """
        Get persisted user account from config file
        :return:
        """
        _, auth = get_credentials()

        try:
            return getattr(auth, "token")
        except AttributeError as e:
            raise FrogmlLoginException("Token is not configured") from e
