r'''
# AWS Generative AI CDK Constructs

![Stability: Experimental](https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge)

> All classes are under active development and subject to non-backward compatible changes or removal in any
> future version. These are not subject to the [Semantic Versioning](https://semver.org/) model.
> This means that while you may use them, you may need to update your source code when upgrading to a newer version of this package.

---


[![View on Construct Hub](https://constructs.dev/badge?package=generative-ai-cdk-constructs)](https://constructs.dev/packages/@cdklabs/generative-ai-cdk-constructs)

[![PyPI version](https://img.shields.io/pypi/v/cdklabs.generative-ai-cdk-constructs)](https://pypi.org/project/cdklabs.generative-ai-cdk-constructs/)
[![npm version](https://img.shields.io/npm/v/@cdklabs/generative-ai-cdk-constructs)](https://www.npmjs.com/package/@cdklabs/generative-ai-cdk-constructs)

## Table of contents

* [Introduction](#introduction)
* [CDK Versions](#cdk-versions)
* [Contributing](#contributing)
* [Design guidelines and Development guide](#design-guidelines-and-development-guide)
* [Getting Started](#getting-started)
* [Catalog](#catalog)
* [Sample Use Cases](#sample-use-cases)
* [Additional Resources](#additional-resources)
* [Contributors](#contributors)
* [Operational Metrics Collection](#operational-metrics-collection)
* [Roadmap](#roadmap)
* [Legal Disclaimer](#legal-disclaimer)

# Introduction

The AWS Generative AI Constructs Library is an open-source extension of the [AWS Cloud Development Kit (AWS CDK)](https://docs.aws.amazon.com/cdk/v2/guide/home.html) that provides multi-service, well-architected patterns for quickly defining solutions in code to create predictable and repeatable infrastructure, called [constructs](https://docs.aws.amazon.com/cdk/v2/guide/constructs.html). The goal of AWS Generative AI CDK Constructs is to help developers build generative AI solutions using pattern-based definitions for their architecture.

The patterns defined in AWS Generative AI CDK Constructs are high level, multi-service abstractions of AWS CDK constructs that have default configurations based on well-architected best practices. The library is organized into logical modules using object-oriented techniques to create each architectural pattern model.

# CDK Versions

AWS Generative AI CDK Constructs and the AWS CDK are independent teams and have different release schedules. Each release of AWS Generative AI CDK Constructs is built against a specific version of the AWS CDK. The [CHANGELOG.md](./CHANGELOG.md) file lists the CDK version associated with each AWS Generative AI Constructs release. For instance, AWS Generative AI CDK Constructs v0.0.0 was built against AWS CDK v2.96.2. This means that to use AWS Generative AI CDK Constructs v0.0.0, your application must include AWS CDK v2.96.2 or later. You can continue to use the latest AWS CDK versions and upgrade the your AWS Generative AI CDK Constructs version when new releases become available.

# Contributing

Contributions of all kinds are welcome! Check out our [contributor guide](./CONTRIBUTING.md)

# Design guidelines and Development guide

If you want to add a new construct to the library, check out our [design guidelines](./DESIGN_GUIDELINES.md), then follow the [development guide](./DEVELOPER_GUIDE.md)

# Getting Started

#### For TypeScript:

* Create or use an existing CDK application in TypeScript.

  * `cdk init app --language typescript`
* Run `npm install @cdklabs/generative-ai-cdk-constructs`
* The package should be added to your package.json.
* Import the library:

  * `import * as genai from '@cdklabs/generative-ai-cdk-constructs';`

#### For Python:

* Create or use an existing CDK application in Python

  * `cdk init app --language python`
* Install the package:

  * `pip install cdklabs.generative-ai-cdk-constructs`
* Import the library:

  * `import cdklabs.generative_ai_cdk_constructs`

Refer to the documentation for additional guidance on a particular construct: [Catalog](#catalog)

# Catalog

The following constructs are available in the library:

**L3 constructs**

| **Construct** |Description| AWS Services used |
|:-------------|:-------------|:-------------|
| [Data ingestion pipeline - OpenSearch](./src/patterns/gen-ai/aws-rag-appsync-stepfn-opensearch/) | Ingestion pipeline providing a RAG (retrieval augmented generation) source for storing documents in a knowledge base. | Amazon OpenSearch, AWS Step Functions, Amazon Bedrock, AWS AppSync, AWS Lambda |
| [Data ingestion pipeline - Kendra](./src/patterns/gen-ai/aws-rag-appsync-stepfn-kendra/) | Ingestion pipeline providing a RAG (retrieval augmented generation) source for storing documents in a knowledge base. | Amazon Kendra, AWS Step Functions, AWS AppSync, AWS Lambda |
| [Question answering](./src/patterns/gen-ai/aws-qa-appsync-opensearch/) | Utilizing Large Language Models (Anthropic Claude V2.1.) for Question Answering on PDF documents with  RAG (retrieval augmented generation) source and/or long context. Additionally, leveraging Anthropic Claude 3 for visual question answering on images.| Amazon OpenSearch, AWS Lambda, Amazon Bedrock, AWS AppSync |
| [Summarization](./src/patterns/gen-ai/aws-summarization-appsync-stepfn/) | Document summarization with a large language model (Anthropic Claude V2.1). | AWS Lambda, Amazon Bedrock, AWS AppSync and Amazon ElastiCache for Redis.  |
| [SageMaker model deployment (JumpStart)](./src/patterns/gen-ai/aws-model-deployment-sagemaker/README_jumpstart.md) | Deploy a foundation model from Amazon SageMaker JumpStart to an Amazon SageMaker endpoint. | Amazon SageMaker |
| [SageMaker model deployment (Hugging Face)](./src/patterns/gen-ai/aws-model-deployment-sagemaker/README_hugging_face.md) | Deploy a foundation model from Hugging Face to an Amazon SageMaker endpoint. | Amazon SageMaker |
| [SageMaker model deployment (Custom)](./src/patterns/gen-ai/aws-model-deployment-sagemaker/README_custom_sagemaker_endpoint.md) | Deploy a foundation model from an S3 location to an Amazon SageMaker endpoint. | Amazon SageMaker |
| [Content Generation](./src/patterns/gen-ai/aws-contentgen-appsync-lambda/) | Generate images from text using Amazon titan-image-generator-v1 or stability.stable-diffusion-xl model.  | AWS Lambda, Amazon Bedrock, AWS AppSync |
| [Web crawler](./src/patterns/gen-ai/aws-web-crawler/) | Crawl websites and RSS feeds on a schedule and store changeset data in an Amazon Simple Storage Service bucket.  | AWS Lambda, AWS Batch, AWS Fargate, Amazon DynamoDB  |

**L2 Constructs**

| **Construct** |Description| AWS Services used |
|:-------------|:-------------|:-------------|
| [Lambda layer](./src/patterns/gen-ai/aws-langchain-common-layer/) | Python Lambda layer providing dependencies and utilities to develop generative AI applications on AWS. | AWS Lambda, Amazon Bedrock, Amazon SageMaker |
| [Amazon Bedrock](./src/cdk-lib/bedrock/README.md) | CDK L2 Constructs for Amazon Bedrock. | Amazon Bedrock, Amazon OpenSearch Serverless, AWS Lambda |
| [Amazon OpenSearch Serverless Vector Collection](./src/cdk-lib/opensearchserverless/README.md) | CDK L2 Constructs to create a vector collection. | Amazon OpenSearch Vector Index |
| [Amazon OpenSearch Vector Index](./src/cdk-lib/opensearch-vectorindex/README.md) | CDK L1 Custom Resource to create a vector index. | Amazon OpenSearch Serverless, AWS Lambda |

# Sample Use Cases

The official samples repository https://github.com/aws-samples/generative-ai-cdk-constructs-samples includes a collection of functional use case implementations to demonstrate the usage of AWS Generative AI CDK Constructs. These can be used in the same way as architectural patterns, and can be conceptualized as an additional "higher-level" abstraction of those patterns. Those patterns (constructs) are composed together into [stacks](https://docs.aws.amazon.com/cdk/latest/guide/stacks.html), forming a "CDK app".

# Additional Resources

| Resource |Description|
|:-------------|:-------------|
| [AWS re:Invent 2023 - Keynote with Dr. Werner Vogels](https://youtu.be/UTRBVPvzt9w?t=6252) | Dr. Werner Vogels, Amazon.com's VP and CTO, announces the AWS Generative AI CDK Constructs during his AWS re:Invent 2023 keynote. |
| [Workshop - Building Generative AI Apps on AWS with CDK](https://catalog.workshops.aws/building-genai-apps) | In this workshop, you will explore how to build a sample generative AI app on AWS using CDK and Generative AI CDK Constructs. |
| [Build generative AI applications with Amazon Titan Text Premier, Amazon Bedrock, and AWS CDK](https://aws.amazon.com/blogs/machine-learning/build-generative-ai-applications-with-amazon-titan-text-premier-amazon-bedrock-and-aws-cdk/) | Blog post exploring building and deploying two sample applications powered by Amazon Titan Text Premier using the Generative AI CDK constructs. |
| [aws-cdk-stack-builder-tool](https://github.com/aws-samples/aws-cdk-stack-builder-tool) | AWS CDK Builder is a browser-based tool designed to streamline bootstrapping of Infrastructure as Code (IaC) projects using the AWS Cloud Development Kit (CDK). |
| [CDK Live! Building generative AI applications and architectures leveraging AWS CDK Constructs!](https://www.youtube.com/watch?v=NI1F4Xxqyr8) | CDK Live! episode focused on building and deploying generative AI applications and architectures on AWS using the AWS Cloud Development Kit (CDK) and the AWS Generative AI CDK Constructs. |
| [Announcing AWS Generative AI CDK Constructs!](https://aws.amazon.com/blogs/devops/announcing-generative-ai-cdk-constructs/) | Blog post announcing the release of the AWS Generative AI CDK Constructs. |
| [aws-genai-llm-chatbot](https://github.com/aws-samples/aws-genai-llm-chatbot/tree/main) | Multi-Model and Multi-RAG Powered Chatbot Using AWS CDK on AWS allowing you to experiment with a variety of Large Language Models and Multimodal Language Models, settings and prompts in your own AWS account. |
| [amazon-bedrock-rag](https://github.com/aws-samples/amazon-bedrock-rag) | Fully managed RAG solution using Knowledge Bases for Amazon Bedrock. |
| [Amazon Bedrock Multimodal Search](https://github.com/aws-samples/amazon-bedrock-titan-multimodal-search) | Multimodal product search app built using Amazon Titan Multimodal Embeddings model. |
| [Amazon Bedrock Knowledge Bases with Private Data](https://blog.serverlessadvocate.com/amazon-bedrock-knowledge-bases-with-private-data-7685d04ef396) | Blog post and associated code sample demonstrating how to integrate Knowledge Bases into Amazon Bedrock to provide foundational models with contextual data from private data sources. |
| [Automating tasks using Amazon Bedrock Agents and AI](https://blog.serverlessadvocate.com/automating-tasks-using-amazon-bedrock-agents-and-ai-4b6fb8856589) | Blog post and associated code sample demonstrating how to deploy an Amazon Bedrock Agent and a Knowledge Base through a hotel and spa use case. |
| [Agents for Amazon Bedrock - Powertools for AWS Lambda (Python)](https://docs.powertools.aws.dev/lambda/python/latest/core/event_handler/bedrock_agents/#using-aws-cloud-developer-kit-cdk) | Create Agents for Amazon Bedrock using event handlers and auto generation of OpenAPI schemas. |
| [Text to SQL Bedrock Agent](https://github.com/aws-samples/amazon-bedrock-samples/tree/main/agents-for-bedrock/use-case-examples/text-2-sql-agent-cdk-enhanced) | Harnessing the power of natural language processing, the "Text to SQL Bedrock Agent" facilitates the automatic transformation of natural language questions into executable SQL queries. |

# Contributors

[![contributors](https://contrib.rocks/image?repo=awslabs/generative-ai-cdk-constructs&max=2000)](https://github.com/awslabs/generative-ai-cdk-constructs/graphs/contributors)

# Operational Metrics Collection

Generative AI CDK Constructs may collect anonymous operational metrics, including: the region a construct is deployed, the name and version of the construct deployed, and related information. We may use the metrics to maintain, provide, develop, and improve the constructs and AWS services.

# Roadmap

Roadmap is available through the [GitHub Project](https://github.com/orgs/awslabs/projects/136)

# Legal Disclaimer

You should consider doing your own independent assessment before using the content in this library for production purposes. This may include (amongst other things) testing, securing, and optimizing the CDK constructs and other content, provided in this library, based on your specific quality control practices and standards.

---


© Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk as _aws_cdk_ceddda9d
import aws_cdk.aws_appsync as _aws_cdk_aws_appsync_ceddda9d
import aws_cdk.aws_batch as _aws_cdk_aws_batch_ceddda9d
import aws_cdk.aws_codeguruprofiler as _aws_cdk_aws_codeguruprofiler_ceddda9d
import aws_cdk.aws_cognito as _aws_cdk_aws_cognito_ceddda9d
import aws_cdk.aws_dynamodb as _aws_cdk_aws_dynamodb_ceddda9d
import aws_cdk.aws_ec2 as _aws_cdk_aws_ec2_ceddda9d
import aws_cdk.aws_ecr as _aws_cdk_aws_ecr_ceddda9d
import aws_cdk.aws_ecr_assets as _aws_cdk_aws_ecr_assets_ceddda9d
import aws_cdk.aws_events as _aws_cdk_aws_events_ceddda9d
import aws_cdk.aws_iam as _aws_cdk_aws_iam_ceddda9d
import aws_cdk.aws_kms as _aws_cdk_aws_kms_ceddda9d
import aws_cdk.aws_lambda as _aws_cdk_aws_lambda_ceddda9d
import aws_cdk.aws_logs as _aws_cdk_aws_logs_ceddda9d
import aws_cdk.aws_opensearchserverless as _aws_cdk_aws_opensearchserverless_ceddda9d
import aws_cdk.aws_opensearchservice as _aws_cdk_aws_opensearchservice_ceddda9d
import aws_cdk.aws_s3 as _aws_cdk_aws_s3_ceddda9d
import aws_cdk.aws_sagemaker as _aws_cdk_aws_sagemaker_ceddda9d
import aws_cdk.aws_secretsmanager as _aws_cdk_aws_secretsmanager_ceddda9d
import aws_cdk.aws_sns as _aws_cdk_aws_sns_ceddda9d
import aws_cdk.aws_sqs as _aws_cdk_aws_sqs_ceddda9d
import aws_cdk.aws_stepfunctions as _aws_cdk_aws_stepfunctions_ceddda9d
import constructs as _constructs_77d1e7e8


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.AdapterProps",
    jsii_struct_bases=[],
    name_mapping={
        "compatible_architectures": "compatibleArchitectures",
        "compatible_runtimes": "compatibleRuntimes",
        "description": "description",
        "layer_version_name": "layerVersionName",
        "license": "license",
        "removal_policy": "removalPolicy",
    },
)
class AdapterProps:
    def __init__(
        self,
        *,
        compatible_architectures: typing.Optional[typing.Sequence[_aws_cdk_aws_lambda_ceddda9d.Architecture]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence[_aws_cdk_aws_lambda_ceddda9d.Runtime]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy] = None,
    ) -> None:
        '''(experimental) AdapterProps.

        :param compatible_architectures: The system architectures compatible with this layer. Default: [Architecture.X86_64]
        :param compatible_runtimes: The runtimes compatible with this Layer. Default: - All runtimes are supported.
        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14940da981f24c52deaeac165bd230136ed11f0570a4f7e7b9cb9ce527771c1e)
            check_type(argname="argument compatible_architectures", value=compatible_architectures, expected_type=type_hints["compatible_architectures"])
            check_type(argname="argument compatible_runtimes", value=compatible_runtimes, expected_type=type_hints["compatible_runtimes"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument layer_version_name", value=layer_version_name, expected_type=type_hints["layer_version_name"])
            check_type(argname="argument license", value=license, expected_type=type_hints["license"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if compatible_architectures is not None:
            self._values["compatible_architectures"] = compatible_architectures
        if compatible_runtimes is not None:
            self._values["compatible_runtimes"] = compatible_runtimes
        if description is not None:
            self._values["description"] = description
        if layer_version_name is not None:
            self._values["layer_version_name"] = layer_version_name
        if license is not None:
            self._values["license"] = license
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy

    @builtins.property
    def compatible_architectures(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_lambda_ceddda9d.Architecture]]:
        '''The system architectures compatible with this layer.

        :default: [Architecture.X86_64]
        '''
        result = self._values.get("compatible_architectures")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_lambda_ceddda9d.Architecture]], result)

    @builtins.property
    def compatible_runtimes(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_lambda_ceddda9d.Runtime]]:
        '''The runtimes compatible with this Layer.

        :default: - All runtimes are supported.
        '''
        result = self._values.get("compatible_runtimes")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_lambda_ceddda9d.Runtime]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description the this Lambda Layer.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layer_version_name(self) -> typing.Optional[builtins.str]:
        '''The name of the layer.

        :default: - A name will be generated.
        '''
        result = self._values.get("layer_version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license(self) -> typing.Optional[builtins.str]:
        '''The SPDX licence identifier or URL to the license file for this layer.

        :default: - No license information will be recorded.
        '''
        result = self._values.get("license")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy]:
        '''Whether to retain this version of the layer when a new version is added or when the stack is deleted.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AdapterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BaseClass(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.BaseClass",
):
    '''
    :stability: experimental
    '''

    def __init__(self, scope: _constructs_77d1e7e8.Construct, id: builtins.str) -> None:
        '''
        :param scope: -
        :param id: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba4d0ee9487405bc5711a0e2897cccdfb48b382a83933a20ede0b6969d0b52e2)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="addObservabilityToConstruct")
    def _add_observability_to_construct(
        self,
        *,
        construct_id: builtins.str,
        construct_name: "ConstructName",
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param construct_id: (experimental) construct id.
        :param construct_name: (experimental) name of the construct.
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev

        :stability: experimental
        '''
        props = BaseClassProps(
            construct_id=construct_id,
            construct_name=construct_name,
            observability=observability,
            stage=stage,
        )

        return typing.cast(None, jsii.invoke(self, "addObservabilityToConstruct", [props]))

    @jsii.member(jsii_name="updateConstructUsageMetricCode")
    def _update_construct_usage_metric_code(
        self,
        props: typing.Union["BaseClassProps", typing.Dict[builtins.str, typing.Any]],
        scope: _constructs_77d1e7e8.Construct,
        lambda_functions: typing.Sequence[_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction],
    ) -> None:
        '''
        :param props: -
        :param scope: -
        :param lambda_functions: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b76d1a279818be0e9eb34c96c33e37d05feefb17eb730e2f8f353003cfd48b55)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument lambda_functions", value=lambda_functions, expected_type=type_hints["lambda_functions"])
        return typing.cast(None, jsii.invoke(self, "updateConstructUsageMetricCode", [props, scope, lambda_functions]))

    @jsii.member(jsii_name="updateEnvSuffix")
    def _update_env_suffix(
        self,
        *,
        construct_id: builtins.str,
        construct_name: "ConstructName",
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param construct_id: (experimental) construct id.
        :param construct_name: (experimental) name of the construct.
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev

        :stability: experimental
        '''
        props = BaseClassProps(
            construct_id=construct_id,
            construct_name=construct_name,
            observability=observability,
            stage=stage,
        )

        return typing.cast(None, jsii.invoke(self, "updateEnvSuffix", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="usageMetricMap")
    def usage_metric_map(cls) -> typing.Mapping[builtins.str, jsii.Number]:  # pyright: ignore [reportGeneralTypeIssues]
        '''(experimental) Record<string, number> , maps construct name with number of deployments.

        :stability: experimental
        '''
        return typing.cast(typing.Mapping[builtins.str, jsii.Number], jsii.sget(cls, "usageMetricMap"))

    @usage_metric_map.setter # type: ignore[no-redef]
    def usage_metric_map(cls, value: typing.Mapping[builtins.str, jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4127ecc357816bf3302cf133e3437ce15aa8c1e68c9230808384225266154cdb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.sset(cls, "usageMetricMap", value)

    @builtins.property
    @jsii.member(jsii_name="constructUsageMetric")
    def construct_usage_metric(self) -> builtins.str:
        '''(experimental) construct usage metric , added in template description.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "constructUsageMetric"))

    @builtins.property
    @jsii.member(jsii_name="enablexray")
    def enablexray(self) -> builtins.bool:
        '''(experimental) enable disable xray tracing.

        :default: - True

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "enablexray"))

    @enablexray.setter
    def enablexray(self, value: builtins.bool) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d94cf2e70af4448135781ede6cdf1dface1c02c2f925449cf86b5acb2f9fdcf7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enablexray", value)

    @builtins.property
    @jsii.member(jsii_name="fieldLogLevel")
    def field_log_level(self) -> _aws_cdk_aws_appsync_ceddda9d.FieldLogLevel:
        '''(experimental) Default  log config for all constructs.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_appsync_ceddda9d.FieldLogLevel, jsii.get(self, "fieldLogLevel"))

    @field_log_level.setter
    def field_log_level(
        self,
        value: _aws_cdk_aws_appsync_ceddda9d.FieldLogLevel,
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__18c9089605f15ff3a723a07573bc4513d0f1919e19f30fccbcf6e3ffb13fe723)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "fieldLogLevel", value)

    @builtins.property
    @jsii.member(jsii_name="lambdaTracing")
    def lambda_tracing(self) -> _aws_cdk_aws_lambda_ceddda9d.Tracing:
        '''(experimental) enable disable lambda tracing.

        :default: - Active

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.Tracing, jsii.get(self, "lambdaTracing"))

    @lambda_tracing.setter
    def lambda_tracing(self, value: _aws_cdk_aws_lambda_ceddda9d.Tracing) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e94708ec2c46362d816de918945a5b4f20113230fb8b8405d9b9824b1be83ab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "lambdaTracing", value)

    @builtins.property
    @jsii.member(jsii_name="retention")
    def retention(self) -> _aws_cdk_aws_logs_ceddda9d.RetentionDays:
        '''(experimental) Default  log retention config for all constructs.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_logs_ceddda9d.RetentionDays, jsii.get(self, "retention"))

    @retention.setter
    def retention(self, value: _aws_cdk_aws_logs_ceddda9d.RetentionDays) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f82b6ba01833bfcbf363e315e04043c72f89cc18ee7ce7a822d0c59d6e694b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "retention", value)

    @builtins.property
    @jsii.member(jsii_name="stage")
    def stage(self) -> builtins.str:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "stage"))

    @stage.setter
    def stage(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb6137258e16f21128dc31a4b333e6760949adbb069429aa01b8994be58f6181)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stage", value)


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.BaseClassProps",
    jsii_struct_bases=[],
    name_mapping={
        "construct_id": "constructId",
        "construct_name": "constructName",
        "observability": "observability",
        "stage": "stage",
    },
)
class BaseClassProps:
    def __init__(
        self,
        *,
        construct_id: builtins.str,
        construct_name: "ConstructName",
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param construct_id: (experimental) construct id.
        :param construct_name: (experimental) name of the construct.
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ada02997f2d10f9b98eb5cee9804150834cb4d55315e4d8e81b855f959dcbbaf)
            check_type(argname="argument construct_id", value=construct_id, expected_type=type_hints["construct_id"])
            check_type(argname="argument construct_name", value=construct_name, expected_type=type_hints["construct_name"])
            check_type(argname="argument observability", value=observability, expected_type=type_hints["observability"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "construct_id": construct_id,
            "construct_name": construct_name,
        }
        if observability is not None:
            self._values["observability"] = observability
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def construct_id(self) -> builtins.str:
        '''(experimental) construct id.

        :stability: experimental
        '''
        result = self._values.get("construct_id")
        assert result is not None, "Required property 'construct_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def construct_name(self) -> "ConstructName":
        '''(experimental) name of the construct.

        :stability: experimental
        '''
        result = self._values.get("construct_name")
        assert result is not None, "Required property 'construct_name' is missing"
        return typing.cast("ConstructName", result)

    @builtins.property
    def observability(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable observability.

        Warning: associated cost with the services
        used. Best practice to enable by default.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("observability")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseClassProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/generative-ai-cdk-constructs.ConstructName")
class ConstructName(enum.Enum):
    '''(experimental) Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

    Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
    with the License. A copy of the License is located at Example::

       http://www.apache.org/licenses/LICENSE-2.0

    or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
    OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
    and limitations under the License.

    :stability: experimental
    '''

    AWSRAGAPPSYNCSTEPFNOPENSEARCH = "AWSRAGAPPSYNCSTEPFNOPENSEARCH"
    '''
    :stability: experimental
    '''
    AWSQAAPPSYNCOPENSEARCH = "AWSQAAPPSYNCOPENSEARCH"
    '''
    :stability: experimental
    '''
    AWSSUMMARIZATIONAPPSYNCSTEPFN = "AWSSUMMARIZATIONAPPSYNCSTEPFN"
    '''
    :stability: experimental
    '''
    AWSMODELDEPLOYMENTSAGEMAKER = "AWSMODELDEPLOYMENTSAGEMAKER"
    '''
    :stability: experimental
    '''
    CUSTOMSAGEMAKERENDPOINT = "CUSTOMSAGEMAKERENDPOINT"
    '''
    :stability: experimental
    '''
    HUGGINGFACESAGEMAKERENDPOINT = "HUGGINGFACESAGEMAKERENDPOINT"
    '''
    :stability: experimental
    '''
    JUMPSTARTSAGEMAKERENDPOINT = "JUMPSTARTSAGEMAKERENDPOINT"
    '''
    :stability: experimental
    '''
    AWSCONTENTGENAPPSYNCLAMBDA = "AWSCONTENTGENAPPSYNCLAMBDA"
    '''
    :stability: experimental
    '''
    AWSRAGAPPSYNCSTEPFNKENDRA = "AWSRAGAPPSYNCSTEPFNKENDRA"
    '''
    :stability: experimental
    '''
    AWSWEBCRAWLER = "AWSWEBCRAWLER"
    '''
    :stability: experimental
    '''


class ContainerImage(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.ContainerImage",
):
    '''(experimental) https://github.com/aws/deep-learning-containers/blob/master/available_images.md.

    :stability: experimental
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromAsset")
    @builtins.classmethod
    def from_asset(
        cls,
        directory: builtins.str,
        *,
        asset_name: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_ssh: typing.Optional[builtins.str] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_aws_ecr_assets_ceddda9d.DockerCacheOption, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_aws_cdk_aws_ecr_assets_ceddda9d.DockerCacheOption, typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_aws_cdk_aws_ecr_assets_ceddda9d.DockerImageAssetInvalidationOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_aws_cdk_aws_ecr_assets_ceddda9d.NetworkMode] = None,
        outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        platform: typing.Optional[_aws_cdk_aws_ecr_assets_ceddda9d.Platform] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_aws_cdk_ceddda9d.SymlinkFollowMode] = None,
        ignore_mode: typing.Optional[_aws_cdk_ceddda9d.IgnoreMode] = None,
    ) -> "ContainerImage":
        '''
        :param directory: -
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param build_secrets: Build secrets. Docker BuildKit must be enabled to use build secrets. Default: - no build secrets
        :param build_ssh: SSH agent socket or keys to pass to the ``docker build`` command. Docker BuildKit must be enabled to use the ssh flag Default: - no --ssh flag
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from options are passed to the build command
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to options are passed to the build command
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param outputs: Outputs to pass to the ``docker build`` command. Default: - no outputs are passed to the build command (default outputs are used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param target: Docker target to build to. Default: - no target
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__25f8364678c63568f662d1c08defb90957a4a2e7d3066311040cba2e3735b2fd)
            check_type(argname="argument directory", value=directory, expected_type=type_hints["directory"])
        options = _aws_cdk_aws_ecr_assets_ceddda9d.DockerImageAssetOptions(
            asset_name=asset_name,
            build_args=build_args,
            build_secrets=build_secrets,
            build_ssh=build_ssh,
            cache_disabled=cache_disabled,
            cache_from=cache_from,
            cache_to=cache_to,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            outputs=outputs,
            platform=platform,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("ContainerImage", jsii.sinvoke(cls, "fromAsset", [directory, options]))

    @jsii.member(jsii_name="fromEcrRepository")
    @builtins.classmethod
    def from_ecr_repository(
        cls,
        repository: _aws_cdk_aws_ecr_ceddda9d.IRepository,
        tag: typing.Optional[builtins.str] = None,
    ) -> "ContainerImage":
        '''
        :param repository: -
        :param tag: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d56d3018c15bc3c527de6fcfbab1fce78287d4b4a491912aa644b5edbd8864e)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument tag", value=tag, expected_type=type_hints["tag"])
        return typing.cast("ContainerImage", jsii.sinvoke(cls, "fromEcrRepository", [repository, tag]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        grantable: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> "ContainerImageConfig":
        '''
        :param scope: -
        :param grantable: -

        :stability: experimental
        '''
        ...


class _ContainerImageProxy(ContainerImage):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        grantable: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> "ContainerImageConfig":
        '''
        :param scope: -
        :param grantable: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98ebe648e608c12c055bbce76a5856ab8c5494b3267f8d897e31136265e41508)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument grantable", value=grantable, expected_type=type_hints["grantable"])
        return typing.cast("ContainerImageConfig", jsii.invoke(self, "bind", [scope, grantable]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, ContainerImage).__jsii_proxy_class__ = lambda : _ContainerImageProxy


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.ContainerImageConfig",
    jsii_struct_bases=[],
    name_mapping={"image_name": "imageName"},
)
class ContainerImageConfig:
    def __init__(self, *, image_name: builtins.str) -> None:
        '''
        :param image_name: (experimental) The image name. Images in Amazon ECR repositories can be specified by either using the full registry/repository:tag or registry/repository@digest. For example, ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest`` or ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE``.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__93de8836b2699c5ff29ea91bd61ddf2d833937fcf645c73a50bfb8c03c3b01e4)
            check_type(argname="argument image_name", value=image_name, expected_type=type_hints["image_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image_name": image_name,
        }

    @builtins.property
    def image_name(self) -> builtins.str:
        '''(experimental) The image name. Images in Amazon ECR repositories can be specified by either using the full registry/repository:tag or registry/repository@digest.

        For example, ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest`` or
        ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE``.

        :stability: experimental
        '''
        result = self._values.get("image_name")
        assert result is not None, "Required property 'image_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerImageConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ContentGenerationAppSyncLambda(
    BaseClass,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.ContentGenerationAppSyncLambda",
):
    '''
    :stability: experimental
    :summary: The ContentGenerationAppSyncLambda class.
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        custom_docker_lambda_props: typing.Optional[typing.Union["DockerLambdaCustomProps", typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_generated_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        generated_assets_bucket_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        lambda_provisioned_concurrency: typing.Optional[jsii.Number] = None,
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a a scope-unique id.
        :param cognito_user_pool: (experimental) Cognito user pool used for authentication. Default: - None
        :param custom_docker_lambda_props: (experimental) Optional. Allows to provide custom lambda code and settings instead of the existing
        :param existing_bus_interface: (experimental) Optional Existing instance of an EventBridge bus. If not provided, the construct will create one. Default: - None
        :param existing_generated_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``generatedAssetsBucketProps`` will cause an error. Default: - None
        :param existing_merged_api: (experimental) Existing merged Appsync GraphQL api. Default: - None
        :param existing_security_group: (experimental) Optional existing security group allowing access to opensearch. Used by the lambda functions built by this construct. If not provided, the construct will create one. Default: - none
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param generated_assets_bucket_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingGeneratedAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param lambda_provisioned_concurrency: (experimental) Optional. Allows a user to configure Lambda provisioned concurrency for consistent performance
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practive to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the Lambda functions the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: Constructs a new instance of the ContentGenerationAppSyncLambda class.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c875d95af0c83c518410a0ced49e645849cd6e1a0a7e51d09b941014c4f1064)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ContentGenerationAppSyncLambdaProps(
            cognito_user_pool=cognito_user_pool,
            custom_docker_lambda_props=custom_docker_lambda_props,
            existing_bus_interface=existing_bus_interface,
            existing_generated_assets_bucket_obj=existing_generated_assets_bucket_obj,
            existing_merged_api=existing_merged_api,
            existing_security_group=existing_security_group,
            existing_vpc=existing_vpc,
            generated_assets_bucket_props=generated_assets_bucket_props,
            lambda_provisioned_concurrency=lambda_provisioned_concurrency,
            observability=observability,
            stage=stage,
            vpc_props=vpc_props,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="cgLambdaFunction")
    def cg_lambda_function(self) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of appsync.GraphqlApi created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "cgLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="generatedImageBus")
    def generated_image_bus(self) -> _aws_cdk_aws_events_ceddda9d.IEventBus:
        '''(experimental) Returns the instance of events.IEventBus used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_events_ceddda9d.IEventBus, jsii.get(self, "generatedImageBus"))

    @builtins.property
    @jsii.member(jsii_name="graphqlApi")
    def graphql_api(self) -> _aws_cdk_aws_appsync_ceddda9d.GraphqlApi:
        '''(experimental) Returns an instance of appsync.GraphqlApi created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_appsync_ceddda9d.GraphqlApi, jsii.get(self, "graphqlApi"))

    @builtins.property
    @jsii.member(jsii_name="s3GenerateAssetsBucketInterface")
    def s3_generate_assets_bucket_interface(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns an instance of s3.IBucket created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "s3GenerateAssetsBucketInterface"))

    @builtins.property
    @jsii.member(jsii_name="securityGroup")
    def security_group(self) -> _aws_cdk_aws_ec2_ceddda9d.ISecurityGroup:
        '''(experimental) Returns the instance of ec2.ISecurityGroup used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup, jsii.get(self, "securityGroup"))

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _aws_cdk_aws_ec2_ceddda9d.IVpc:
        '''(experimental) Returns the instance of ec2.IVpc used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.IVpc, jsii.get(self, "vpc"))

    @builtins.property
    @jsii.member(jsii_name="s3GenerateAssetsBucket")
    def s3_generate_assets_bucket(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket]:
        '''(experimental) Returns an instance of s3.Bucket created by the construct. IMPORTANT: If existingGeneratedAssetsBucketObj was provided in Pattern Construct Props, this property will be undefined.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket], jsii.get(self, "s3GenerateAssetsBucket"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.ContentGenerationAppSyncLambdaProps",
    jsii_struct_bases=[],
    name_mapping={
        "cognito_user_pool": "cognitoUserPool",
        "custom_docker_lambda_props": "customDockerLambdaProps",
        "existing_bus_interface": "existingBusInterface",
        "existing_generated_assets_bucket_obj": "existingGeneratedAssetsBucketObj",
        "existing_merged_api": "existingMergedApi",
        "existing_security_group": "existingSecurityGroup",
        "existing_vpc": "existingVpc",
        "generated_assets_bucket_props": "generatedAssetsBucketProps",
        "lambda_provisioned_concurrency": "lambdaProvisionedConcurrency",
        "observability": "observability",
        "stage": "stage",
        "vpc_props": "vpcProps",
    },
)
class ContentGenerationAppSyncLambdaProps:
    def __init__(
        self,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        custom_docker_lambda_props: typing.Optional[typing.Union["DockerLambdaCustomProps", typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_generated_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        generated_assets_bucket_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        lambda_provisioned_concurrency: typing.Optional[jsii.Number] = None,
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) The properties for the ContentGenerationAppSyncLambdaProps class.

        :param cognito_user_pool: (experimental) Cognito user pool used for authentication. Default: - None
        :param custom_docker_lambda_props: (experimental) Optional. Allows to provide custom lambda code and settings instead of the existing
        :param existing_bus_interface: (experimental) Optional Existing instance of an EventBridge bus. If not provided, the construct will create one. Default: - None
        :param existing_generated_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``generatedAssetsBucketProps`` will cause an error. Default: - None
        :param existing_merged_api: (experimental) Existing merged Appsync GraphQL api. Default: - None
        :param existing_security_group: (experimental) Optional existing security group allowing access to opensearch. Used by the lambda functions built by this construct. If not provided, the construct will create one. Default: - none
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param generated_assets_bucket_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingGeneratedAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param lambda_provisioned_concurrency: (experimental) Optional. Allows a user to configure Lambda provisioned concurrency for consistent performance
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practive to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the Lambda functions the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        '''
        if isinstance(custom_docker_lambda_props, dict):
            custom_docker_lambda_props = DockerLambdaCustomProps(**custom_docker_lambda_props)
        if isinstance(generated_assets_bucket_props, dict):
            generated_assets_bucket_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**generated_assets_bucket_props)
        if isinstance(vpc_props, dict):
            vpc_props = _aws_cdk_aws_ec2_ceddda9d.VpcProps(**vpc_props)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c78dc6e99bf9b7ba9cc34a9d191b26cbd359a0a7388cb25ada4a9b125c36bff0)
            check_type(argname="argument cognito_user_pool", value=cognito_user_pool, expected_type=type_hints["cognito_user_pool"])
            check_type(argname="argument custom_docker_lambda_props", value=custom_docker_lambda_props, expected_type=type_hints["custom_docker_lambda_props"])
            check_type(argname="argument existing_bus_interface", value=existing_bus_interface, expected_type=type_hints["existing_bus_interface"])
            check_type(argname="argument existing_generated_assets_bucket_obj", value=existing_generated_assets_bucket_obj, expected_type=type_hints["existing_generated_assets_bucket_obj"])
            check_type(argname="argument existing_merged_api", value=existing_merged_api, expected_type=type_hints["existing_merged_api"])
            check_type(argname="argument existing_security_group", value=existing_security_group, expected_type=type_hints["existing_security_group"])
            check_type(argname="argument existing_vpc", value=existing_vpc, expected_type=type_hints["existing_vpc"])
            check_type(argname="argument generated_assets_bucket_props", value=generated_assets_bucket_props, expected_type=type_hints["generated_assets_bucket_props"])
            check_type(argname="argument lambda_provisioned_concurrency", value=lambda_provisioned_concurrency, expected_type=type_hints["lambda_provisioned_concurrency"])
            check_type(argname="argument observability", value=observability, expected_type=type_hints["observability"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument vpc_props", value=vpc_props, expected_type=type_hints["vpc_props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cognito_user_pool": cognito_user_pool,
        }
        if custom_docker_lambda_props is not None:
            self._values["custom_docker_lambda_props"] = custom_docker_lambda_props
        if existing_bus_interface is not None:
            self._values["existing_bus_interface"] = existing_bus_interface
        if existing_generated_assets_bucket_obj is not None:
            self._values["existing_generated_assets_bucket_obj"] = existing_generated_assets_bucket_obj
        if existing_merged_api is not None:
            self._values["existing_merged_api"] = existing_merged_api
        if existing_security_group is not None:
            self._values["existing_security_group"] = existing_security_group
        if existing_vpc is not None:
            self._values["existing_vpc"] = existing_vpc
        if generated_assets_bucket_props is not None:
            self._values["generated_assets_bucket_props"] = generated_assets_bucket_props
        if lambda_provisioned_concurrency is not None:
            self._values["lambda_provisioned_concurrency"] = lambda_provisioned_concurrency
        if observability is not None:
            self._values["observability"] = observability
        if stage is not None:
            self._values["stage"] = stage
        if vpc_props is not None:
            self._values["vpc_props"] = vpc_props

    @builtins.property
    def cognito_user_pool(self) -> _aws_cdk_aws_cognito_ceddda9d.IUserPool:
        '''(experimental) Cognito user pool used for authentication.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("cognito_user_pool")
        assert result is not None, "Required property 'cognito_user_pool' is missing"
        return typing.cast(_aws_cdk_aws_cognito_ceddda9d.IUserPool, result)

    @builtins.property
    def custom_docker_lambda_props(self) -> typing.Optional["DockerLambdaCustomProps"]:
        '''(experimental) Optional.

        Allows to provide custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_docker_lambda_props")
        return typing.cast(typing.Optional["DockerLambdaCustomProps"], result)

    @builtins.property
    def existing_bus_interface(
        self,
    ) -> typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus]:
        '''(experimental) Optional Existing instance of an EventBridge bus.

        If not provided, the construct will create one.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_bus_interface")
        return typing.cast(typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus], result)

    @builtins.property
    def existing_generated_assets_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Existing instance of S3 Bucket object, providing both this and ``generatedAssetsBucketProps`` will cause an error.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_generated_assets_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_merged_api(
        self,
    ) -> typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi]:
        '''(experimental) Existing merged Appsync GraphQL api.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_merged_api")
        return typing.cast(typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi], result)

    @builtins.property
    def existing_security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup]:
        '''(experimental) Optional existing security group allowing access to opensearch.

        Used by the lambda functions
        built by this construct. If not provided, the construct will create one.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup], result)

    @builtins.property
    def existing_vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc]:
        '''(experimental) Optional An existing VPC in which to deploy the construct.

        Providing both this and
        vpcProps is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc], result)

    @builtins.property
    def generated_assets_bucket_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional user provided props to override the default props for the S3 Bucket.

        Providing both this and ``existingGeneratedAssetsBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("generated_assets_bucket_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def lambda_provisioned_concurrency(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Optional.

        Allows a user to configure
        Lambda provisioned concurrency for consistent performance

        :stability: experimental
        '''
        result = self._values.get("lambda_provisioned_concurrency")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def observability(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable observability.

        Warning: associated cost with the services
        used. Best practive to enable by default.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("observability")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_props(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps]:
        '''(experimental) Optional custom properties for a VPC the construct will create.

        This VPC will
        be used by the Lambda functions the construct creates. Providing
        both this and existingVpc is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("vpc_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContentGenerationAppSyncLambdaProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.CrawlerTarget",
    jsii_struct_bases=[],
    name_mapping={
        "target_type": "targetType",
        "url": "url",
        "crawl_interval_hours": "crawlIntervalHours",
        "download_files": "downloadFiles",
        "file_types": "fileTypes",
        "max_files": "maxFiles",
        "max_requests": "maxRequests",
    },
)
class CrawlerTarget:
    def __init__(
        self,
        *,
        target_type: "CrawlerTargetType",
        url: builtins.str,
        crawl_interval_hours: typing.Optional[jsii.Number] = None,
        download_files: typing.Optional[builtins.bool] = None,
        file_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_files: typing.Optional[jsii.Number] = None,
        max_requests: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param target_type: (experimental) Type of URL to be crawled.
        :param url: (experimental) Target URL to be crawled.
        :param crawl_interval_hours: (experimental) Schedule the crawler to run every N hours following the completion of the previous job. Default: - not scheduled
        :param download_files: (experimental) Download files from the web site. Default: - true
        :param file_types: (experimental) File types (extensions) to be downloaded. Default: - all file types
        :param max_files: (experimental) Maximum number of files to be downloaded. Default: - crawler limit
        :param max_requests: (experimental) Maximum number of requests to be made by crawler. Default: - crawler limit

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c0c564265805d82c2a307dfbf5106b4a567067cc9f1394de3a20df7af914d7c4)
            check_type(argname="argument target_type", value=target_type, expected_type=type_hints["target_type"])
            check_type(argname="argument url", value=url, expected_type=type_hints["url"])
            check_type(argname="argument crawl_interval_hours", value=crawl_interval_hours, expected_type=type_hints["crawl_interval_hours"])
            check_type(argname="argument download_files", value=download_files, expected_type=type_hints["download_files"])
            check_type(argname="argument file_types", value=file_types, expected_type=type_hints["file_types"])
            check_type(argname="argument max_files", value=max_files, expected_type=type_hints["max_files"])
            check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "target_type": target_type,
            "url": url,
        }
        if crawl_interval_hours is not None:
            self._values["crawl_interval_hours"] = crawl_interval_hours
        if download_files is not None:
            self._values["download_files"] = download_files
        if file_types is not None:
            self._values["file_types"] = file_types
        if max_files is not None:
            self._values["max_files"] = max_files
        if max_requests is not None:
            self._values["max_requests"] = max_requests

    @builtins.property
    def target_type(self) -> "CrawlerTargetType":
        '''(experimental) Type of URL to be crawled.

        :stability: experimental
        '''
        result = self._values.get("target_type")
        assert result is not None, "Required property 'target_type' is missing"
        return typing.cast("CrawlerTargetType", result)

    @builtins.property
    def url(self) -> builtins.str:
        '''(experimental) Target URL to be crawled.

        :stability: experimental
        '''
        result = self._values.get("url")
        assert result is not None, "Required property 'url' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def crawl_interval_hours(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Schedule the crawler to run every N hours following the completion of the previous job.

        :default: - not scheduled

        :stability: experimental
        '''
        result = self._values.get("crawl_interval_hours")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def download_files(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Download files from the web site.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("download_files")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def file_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) File types (extensions) to be downloaded.

        :default: - all file types

        :stability: experimental
        '''
        result = self._values.get("file_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def max_files(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum number of files to be downloaded.

        :default: - crawler limit

        :stability: experimental
        '''
        result = self._values.get("max_files")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_requests(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum number of requests to be made by crawler.

        :default: - crawler limit

        :stability: experimental
        '''
        result = self._values.get("max_requests")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrawlerTarget(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/generative-ai-cdk-constructs.CrawlerTargetType")
class CrawlerTargetType(enum.Enum):
    '''
    :stability: experimental
    '''

    WEBSITE = "WEBSITE"
    '''
    :stability: experimental
    '''
    RSS_FEED = "RSS_FEED"
    '''
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.CustomSageMakerEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "container": "container",
        "endpoint_name": "endpointName",
        "instance_type": "instanceType",
        "model_data_url": "modelDataUrl",
        "model_id": "modelId",
        "environment": "environment",
        "instance_count": "instanceCount",
        "model_data_download_timeout_in_seconds": "modelDataDownloadTimeoutInSeconds",
        "role": "role",
        "startup_health_check_timeout_in_seconds": "startupHealthCheckTimeoutInSeconds",
        "volume_size_in_gb": "volumeSizeInGb",
        "vpc_config": "vpcConfig",
    },
)
class CustomSageMakerEndpointProps:
    def __init__(
        self,
        *,
        container: ContainerImage,
        endpoint_name: builtins.str,
        instance_type: "SageMakerInstanceType",
        model_data_url: builtins.str,
        model_id: builtins.str,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        instance_count: typing.Optional[jsii.Number] = None,
        model_data_download_timeout_in_seconds: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
        startup_health_check_timeout_in_seconds: typing.Optional[jsii.Number] = None,
        volume_size_in_gb: typing.Optional[jsii.Number] = None,
        vpc_config: typing.Optional[typing.Union[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param container: 
        :param endpoint_name: 
        :param instance_type: 
        :param model_data_url: 
        :param model_id: 
        :param environment: 
        :param instance_count: 
        :param model_data_download_timeout_in_seconds: 
        :param role: 
        :param startup_health_check_timeout_in_seconds: 
        :param volume_size_in_gb: 
        :param vpc_config: 

        :stability: experimental
        '''
        if isinstance(vpc_config, dict):
            vpc_config = _aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty(**vpc_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64c88e4b8c433fb64fc5ff7b8ba1dfe89b0e7d556fee70434f79fe2f29d94734)
            check_type(argname="argument container", value=container, expected_type=type_hints["container"])
            check_type(argname="argument endpoint_name", value=endpoint_name, expected_type=type_hints["endpoint_name"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument model_data_url", value=model_data_url, expected_type=type_hints["model_data_url"])
            check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
            check_type(argname="argument model_data_download_timeout_in_seconds", value=model_data_download_timeout_in_seconds, expected_type=type_hints["model_data_download_timeout_in_seconds"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument startup_health_check_timeout_in_seconds", value=startup_health_check_timeout_in_seconds, expected_type=type_hints["startup_health_check_timeout_in_seconds"])
            check_type(argname="argument volume_size_in_gb", value=volume_size_in_gb, expected_type=type_hints["volume_size_in_gb"])
            check_type(argname="argument vpc_config", value=vpc_config, expected_type=type_hints["vpc_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container": container,
            "endpoint_name": endpoint_name,
            "instance_type": instance_type,
            "model_data_url": model_data_url,
            "model_id": model_id,
        }
        if environment is not None:
            self._values["environment"] = environment
        if instance_count is not None:
            self._values["instance_count"] = instance_count
        if model_data_download_timeout_in_seconds is not None:
            self._values["model_data_download_timeout_in_seconds"] = model_data_download_timeout_in_seconds
        if role is not None:
            self._values["role"] = role
        if startup_health_check_timeout_in_seconds is not None:
            self._values["startup_health_check_timeout_in_seconds"] = startup_health_check_timeout_in_seconds
        if volume_size_in_gb is not None:
            self._values["volume_size_in_gb"] = volume_size_in_gb
        if vpc_config is not None:
            self._values["vpc_config"] = vpc_config

    @builtins.property
    def container(self) -> ContainerImage:
        '''
        :stability: experimental
        '''
        result = self._values.get("container")
        assert result is not None, "Required property 'container' is missing"
        return typing.cast(ContainerImage, result)

    @builtins.property
    def endpoint_name(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("endpoint_name")
        assert result is not None, "Required property 'endpoint_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> "SageMakerInstanceType":
        '''
        :stability: experimental
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast("SageMakerInstanceType", result)

    @builtins.property
    def model_data_url(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("model_data_url")
        assert result is not None, "Required property 'model_data_url' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def model_id(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("model_id")
        assert result is not None, "Required property 'model_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def instance_count(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("instance_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def model_data_download_timeout_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("model_data_download_timeout_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role]:
        '''
        :stability: experimental
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role], result)

    @builtins.property
    def startup_health_check_timeout_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("startup_health_check_timeout_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def volume_size_in_gb(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("volume_size_in_gb")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def vpc_config(
        self,
    ) -> typing.Optional[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty]:
        '''
        :stability: experimental
        '''
        result = self._values.get("vpc_config")
        return typing.cast(typing.Optional[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomSageMakerEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeepLearningContainerImage(
    ContainerImage,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.DeepLearningContainerImage",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        repository_name: builtins.str,
        tag: builtins.str,
        account_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param repository_name: -
        :param tag: -
        :param account_id: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7d3668d1403c78d9ca09200041cb872341727345f79173e3de3ec628c79a40d2)
            check_type(argname="argument repository_name", value=repository_name, expected_type=type_hints["repository_name"])
            check_type(argname="argument tag", value=tag, expected_type=type_hints["tag"])
            check_type(argname="argument account_id", value=account_id, expected_type=type_hints["account_id"])
        jsii.create(self.__class__, self, [repository_name, tag, account_id])

    @jsii.member(jsii_name="fromDeepLearningContainerImage")
    @builtins.classmethod
    def from_deep_learning_container_image(
        cls,
        repository_name: builtins.str,
        tag: builtins.str,
        account_id: typing.Optional[builtins.str] = None,
    ) -> ContainerImage:
        '''
        :param repository_name: -
        :param tag: -
        :param account_id: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ee4748a769dcb340a4537378546c2adacf2d41a2735729450c9cc171a8ae1af)
            check_type(argname="argument repository_name", value=repository_name, expected_type=type_hints["repository_name"])
            check_type(argname="argument tag", value=tag, expected_type=type_hints["tag"])
            check_type(argname="argument account_id", value=account_id, expected_type=type_hints["account_id"])
        return typing.cast(ContainerImage, jsii.sinvoke(cls, "fromDeepLearningContainerImage", [repository_name, tag, account_id]))

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        grantable: _aws_cdk_aws_iam_ceddda9d.IGrantable,
    ) -> ContainerImageConfig:
        '''
        :param scope: -
        :param grantable: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61d694cb21f225b8854128d356326406cf367ffbc30d0b508a85ebb3ea487f8f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument grantable", value=grantable, expected_type=type_hints["grantable"])
        return typing.cast(ContainerImageConfig, jsii.invoke(self, "bind", [scope, grantable]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_19_0_DEEPSPEED0_7_3_CU113")
    def DJL_INFERENCE_0_19_0_DEEPSPEED0_7_3_CU113(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_19_0_DEEPSPEED0_7_3_CU113"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_20_0_DEEPSPEED0_7_5_CU116")
    def DJL_INFERENCE_0_20_0_DEEPSPEED0_7_5_CU116(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_20_0_DEEPSPEED0_7_5_CU116"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_21_0_DEEPSPEED0_8_0_CU117")
    def DJL_INFERENCE_0_21_0_DEEPSPEED0_8_0_CU117(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_21_0_DEEPSPEED0_8_0_CU117"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_21_0_DEEPSPEED0_8_3_CU117")
    def DJL_INFERENCE_0_21_0_DEEPSPEED0_8_3_CU117(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_21_0_DEEPSPEED0_8_3_CU117"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_21_0_FASTERTRANSFORMER5_3_0_CU117")
    def DJL_INFERENCE_0_21_0_FASTERTRANSFORMER5_3_0_CU117(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_21_0_FASTERTRANSFORMER5_3_0_CU117"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_22_1_DEEPSPEED0_8_3_CU118")
    def DJL_INFERENCE_0_22_1_DEEPSPEED0_8_3_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_22_1_DEEPSPEED0_8_3_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_22_1_DEEPSPEED0_9_2_CU118")
    def DJL_INFERENCE_0_22_1_DEEPSPEED0_9_2_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_22_1_DEEPSPEED0_9_2_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_22_1_FASTERTRANSFORMER5_3_0_CU118")
    def DJL_INFERENCE_0_22_1_FASTERTRANSFORMER5_3_0_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_22_1_FASTERTRANSFORMER5_3_0_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_22_1_NEURONX_SDK2_10_0")
    def DJL_INFERENCE_0_22_1_NEURONX_SDK2_10_0(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_22_1_NEURONX_SDK2_10_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_22_1_NEURONX_SDK2_9_0")
    def DJL_INFERENCE_0_22_1_NEURONX_SDK2_9_0(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_22_1_NEURONX_SDK2_9_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_23_0_DEEPSPEED0_9_5_CU118")
    def DJL_INFERENCE_0_23_0_DEEPSPEED0_9_5_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_23_0_DEEPSPEED0_9_5_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_23_0_FASTERTRANSFORMER5_3_0_CU118")
    def DJL_INFERENCE_0_23_0_FASTERTRANSFORMER5_3_0_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_23_0_FASTERTRANSFORMER5_3_0_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_23_0_NEURONX_SDK2_12_0")
    def DJL_INFERENCE_0_23_0_NEURONX_SDK2_12_0(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_23_0_NEURONX_SDK2_12_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_24_0_DEEPSPEED0_10_0_CU118")
    def DJL_INFERENCE_0_24_0_DEEPSPEED0_10_0_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_24_0_DEEPSPEED0_10_0_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_24_0_FASTERTRANSFORMER5_3_0_CU118")
    def DJL_INFERENCE_0_24_0_FASTERTRANSFORMER5_3_0_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_24_0_FASTERTRANSFORMER5_3_0_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_24_0_NEURONX_SDK2_14_1")
    def DJL_INFERENCE_0_24_0_NEURONX_SDK2_14_1(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_24_0_NEURONX_SDK2_14_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_25_0_DEEPSPEED0_11_0_CU118")
    def DJL_INFERENCE_0_25_0_DEEPSPEED0_11_0_CU118(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_25_0_DEEPSPEED0_11_0_CU118"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_25_0_NEURONX_SDK2_15_0")
    def DJL_INFERENCE_0_25_0_NEURONX_SDK2_15_0(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_25_0_NEURONX_SDK2_15_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_26_0_DEEPSPEED0_12_6_CU121")
    def DJL_INFERENCE_0_26_0_DEEPSPEED0_12_6_CU121(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_26_0_DEEPSPEED0_12_6_CU121"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_26_0_NEURONX_SDK2_16_0")
    def DJL_INFERENCE_0_26_0_NEURONX_SDK2_16_0(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_26_0_NEURONX_SDK2_16_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_27_0_DEEPSPEED0_12_6_CU121")
    def DJL_INFERENCE_0_27_0_DEEPSPEED0_12_6_CU121(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_27_0_DEEPSPEED0_12_6_CU121"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_27_0_NEURONX_SDK2_18_1")
    def DJL_INFERENCE_0_27_0_NEURONX_SDK2_18_1(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_27_0_NEURONX_SDK2_18_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DJL_INFERENCE_0_28_0_NEURONX_SDK2_18_2")
    def DJL_INFERENCE_0_28_0_NEURONX_SDK2_18_2(cls) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "DJL_INFERENCE_0_28_0_NEURONX_SDK2_18_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_10_2_TRANSFORMERS4_17_0_CPU_PY38_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_10_2_TRANSFORMERS4_17_0_CPU_PY38_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_10_2_TRANSFORMERS4_17_0_CPU_PY38_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_10_2_TRANSFORMERS4_17_0_GPU_PY38_CU113_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_10_2_TRANSFORMERS4_17_0_GPU_PY38_CU113_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_10_2_TRANSFORMERS4_17_0_GPU_PY38_CU113_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_13_1_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_13_1_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_13_1_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_13_1_TRANSFORMERS4_26_0_GPU_PY39_CU117_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_13_1_TRANSFORMERS4_26_0_GPU_PY39_CU117_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_13_1_TRANSFORMERS4_26_0_GPU_PY39_CU117_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_7_1_TRANSFORMERS4_6_1_CPU_PY36_UBUNTU18_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_7_1_TRANSFORMERS4_6_1_CPU_PY36_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_7_1_TRANSFORMERS4_6_1_CPU_PY36_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_7_1_TRANSFORMERS4_6_1_GPU_PY36_CU110_UBUNTU18_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_7_1_TRANSFORMERS4_6_1_GPU_PY36_CU110_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_7_1_TRANSFORMERS4_6_1_GPU_PY36_CU110_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_10_2_CPU_PY36_UBUNTU18_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_10_2_CPU_PY36_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_10_2_CPU_PY36_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_10_2_GPU_PY36_CU111_UBUNTU18_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_10_2_GPU_PY36_CU111_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_10_2_GPU_PY36_CU111_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_6_1_CPU_PY36_UBUNTU18_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_6_1_CPU_PY36_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_6_1_CPU_PY36_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_6_1_GPU_PY36_CU111_UBUNTU18_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_6_1_GPU_PY36_CU111_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_8_1_TRANSFORMERS4_6_1_GPU_PY36_CU111_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_10_2_CPU_PY38_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_10_2_CPU_PY38_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_10_2_CPU_PY38_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_10_2_GPU_PY38_CU111_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_10_2_GPU_PY38_CU111_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_10_2_GPU_PY38_CU111_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_11_0_CPU_PY38_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_11_0_CPU_PY38_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_11_0_CPU_PY38_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_11_0_GPU_PY38_CU111_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_11_0_GPU_PY38_CU111_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_9_0_TRANSFORMERS4_11_0_GPU_PY38_CU111_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_9_1_TRANSFORMERS4_12_3_CPU_PY38_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_9_1_TRANSFORMERS4_12_3_CPU_PY38_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_9_1_TRANSFORMERS4_12_3_CPU_PY38_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_1_9_1_TRANSFORMERS4_12_3_GPU_PY38_CU111_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_1_9_1_TRANSFORMERS4_12_3_GPU_PY38_CU111_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_1_9_1_TRANSFORMERS4_12_3_GPU_PY38_CU111_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_2_0_0_TRANSFORMERS4_28_1_CPU_PY310_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_2_0_0_TRANSFORMERS4_28_1_CPU_PY310_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_2_0_0_TRANSFORMERS4_28_1_CPU_PY310_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_2_0_0_TRANSFORMERS4_28_1_GPU_PY310_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_2_0_0_TRANSFORMERS4_28_1_GPU_PY310_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_2_0_0_TRANSFORMERS4_28_1_GPU_PY310_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_2_1_0_TRANSFORMERS4_37_0_CPU_PY310_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_2_1_0_TRANSFORMERS4_37_0_CPU_PY310_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_2_1_0_TRANSFORMERS4_37_0_CPU_PY310_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_2_1_0_TRANSFORMERS4_37_0_GPU_PY310_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_2_1_0_TRANSFORMERS4_37_0_GPU_PY310_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_2_1_0_TRANSFORMERS4_37_0_GPU_PY310_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_0_TRANSFORMERS4_28_1_NEURONX_PY38_SDK2_9_1_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_0_TRANSFORMERS4_28_1_NEURONX_PY38_SDK2_9_1_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_0_TRANSFORMERS4_28_1_NEURONX_PY38_SDK2_9_1_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_1_TRANSFORMERS4_34_1_NEURONX_PY310_SDK2_15_0_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_1_TRANSFORMERS4_34_1_NEURONX_PY310_SDK2_15_0_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_1_TRANSFORMERS4_34_1_NEURONX_PY310_SDK2_15_0_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_1_TRANSFORMERS4_36_2_NEURONX_PY310_SDK2_16_1_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_1_TRANSFORMERS4_36_2_NEURONX_PY310_SDK2_16_1_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_1_13_1_TRANSFORMERS4_36_2_NEURONX_PY310_SDK2_16_1_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_2_1_2_TRANSFORMERS4_36_2_NEURONX_PY310_SDK2_18_0_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_2_1_2_TRANSFORMERS4_36_2_NEURONX_PY310_SDK2_18_0_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_INFERENCE_NEURONX_2_1_2_TRANSFORMERS4_36_2_NEURONX_PY310_SDK2_18_0_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_0_TGI0_6_0_GPU_PY39_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_0_TGI0_6_0_GPU_PY39_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_0_TGI0_6_0_GPU_PY39_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_0_TGI0_8_2_GPU_PY39_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_0_TGI0_8_2_GPU_PY39_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_0_TGI0_8_2_GPU_PY39_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI0_9_3_GPU_PY39_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI0_9_3_GPU_PY39_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI0_9_3_GPU_PY39_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI1_0_3_GPU_PY39_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI1_0_3_GPU_PY39_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI1_0_3_GPU_PY39_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI1_1_0_GPU_PY39_CU118_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI1_1_0_GPU_PY39_CU118_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_0_1_TGI1_1_0_GPU_PY39_CU118_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_2_0_GPU_PY310_CU121_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_2_0_GPU_PY310_CU121_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_2_0_GPU_PY310_CU121_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_3_1_GPU_PY310_CU121_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_3_1_GPU_PY310_CU121_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_3_1_GPU_PY310_CU121_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_3_3_GPU_PY310_CU121_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_3_3_GPU_PY310_CU121_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_3_3_GPU_PY310_CU121_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_0_GPU_PY310_CU121_UBUNTU20_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_0_GPU_PY310_CU121_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_0_GPU_PY310_CU121_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_2_GPU_PY310_CU121_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_2_GPU_PY310_CU121_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_2_GPU_PY310_CU121_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_5_GPU_PY310_CU121_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_5_GPU_PY310_CU121_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI1_4_5_GPU_PY310_CU121_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI2_0_0_GPU_PY310_CU121_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI2_0_0_GPU_PY310_CU121_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI2_0_0_GPU_PY310_CU121_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI2_0_1_GPU_PY310_CU121_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI2_0_1_GPU_PY310_CU121_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_1_1_TGI2_0_1_GPU_PY310_CU121_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_3_0_TGI2_0_2_GPU_PY310_CU121_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_3_0_TGI2_0_2_GPU_PY310_CU121_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_3_0_TGI2_0_2_GPU_PY310_CU121_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_3_0_TGI2_0_3_GPU_PY310_CU121_UBUNTU22_04")
    def HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_3_0_TGI2_0_3_GPU_PY310_CU121_UBUNTU22_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_PYTORCH_TGI_INFERENCE_2_3_0_TGI2_0_3_GPU_PY310_CU121_UBUNTU22_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_0_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_0_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_0_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_0_TRANSFORMERS4_26_0_GPU_PY39_CU112_UBUNTU20_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_0_TRANSFORMERS4_26_0_GPU_PY39_CU112_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_0_TRANSFORMERS4_26_0_GPU_PY39_CU112_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_1_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_1_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_1_TRANSFORMERS4_26_0_CPU_PY39_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_1_TRANSFORMERS4_26_0_GPU_PY39_CU112_UBUNTU20_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_1_TRANSFORMERS4_26_0_GPU_PY39_CU112_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_11_1_TRANSFORMERS4_26_0_GPU_PY39_CU112_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_1_TRANSFORMERS4_6_1_CPU_PY37_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_1_TRANSFORMERS4_6_1_CPU_PY37_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_1_TRANSFORMERS4_6_1_CPU_PY37_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_1_TRANSFORMERS4_6_1_GPU_PY37_CU110_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_1_TRANSFORMERS4_6_1_GPU_PY37_CU110_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_1_TRANSFORMERS4_6_1_GPU_PY37_CU110_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_3_TRANSFORMERS4_10_2_CPU_PY37_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_3_TRANSFORMERS4_10_2_CPU_PY37_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_3_TRANSFORMERS4_10_2_CPU_PY37_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_3_TRANSFORMERS4_10_2_GPU_PY37_CU110_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_3_TRANSFORMERS4_10_2_GPU_PY37_CU110_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_4_3_TRANSFORMERS4_10_2_GPU_PY37_CU110_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_10_2_CPU_PY37_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_10_2_CPU_PY37_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_10_2_CPU_PY37_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_10_2_GPU_PY37_CU112_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_10_2_GPU_PY37_CU112_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_10_2_GPU_PY37_CU112_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_11_0_CPU_PY37_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_11_0_CPU_PY37_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_11_0_CPU_PY37_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_11_0_GPU_PY37_CU112_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_11_0_GPU_PY37_CU112_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_11_0_GPU_PY37_CU112_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_12_3_CPU_PY37_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_12_3_CPU_PY37_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_12_3_CPU_PY37_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_12_3_GPU_PY37_CU112_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_12_3_GPU_PY37_CU112_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_1_TRANSFORMERS4_12_3_GPU_PY37_CU112_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_3_TRANSFORMERS4_12_3_CPU_PY37_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_3_TRANSFORMERS4_12_3_CPU_PY37_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_3_TRANSFORMERS4_12_3_CPU_PY37_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_3_TRANSFORMERS4_12_3_GPU_PY37_CU112_UBUNTU18_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_3_TRANSFORMERS4_12_3_GPU_PY37_CU112_UBUNTU18_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_5_3_TRANSFORMERS4_12_3_GPU_PY37_CU112_UBUNTU18_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_6_3_TRANSFORMERS4_17_0_CPU_PY38_UBUNTU20_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_6_3_TRANSFORMERS4_17_0_CPU_PY38_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_6_3_TRANSFORMERS4_17_0_CPU_PY38_UBUNTU20_04"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TENSORFLOW_INFERENCE_2_6_3_TRANSFORMERS4_17_0_GPU_PY38_CU112_UBUNTU20_04")
    def HUGGINGFACE_TENSORFLOW_INFERENCE_2_6_3_TRANSFORMERS4_17_0_GPU_PY38_CU112_UBUNTU20_04(
        cls,
    ) -> ContainerImage:
        '''
        :stability: experimental
        '''
        return typing.cast(ContainerImage, jsii.sget(cls, "HUGGINGFACE_TENSORFLOW_INFERENCE_2_6_3_TRANSFORMERS4_17_0_GPU_PY38_CU112_UBUNTU20_04"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.DockerLambdaCustomProps",
    jsii_struct_bases=[],
    name_mapping={
        "code": "code",
        "adot_instrumentation": "adotInstrumentation",
        "application_log_level": "applicationLogLevel",
        "architecture": "architecture",
        "code_signing_config": "codeSigningConfig",
        "current_version_options": "currentVersionOptions",
        "dead_letter_queue": "deadLetterQueue",
        "dead_letter_queue_enabled": "deadLetterQueueEnabled",
        "dead_letter_topic": "deadLetterTopic",
        "environment": "environment",
        "environment_encryption": "environmentEncryption",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "events": "events",
        "filesystem": "filesystem",
        "initial_policy": "initialPolicy",
        "insights_version": "insightsVersion",
        "ipv6_allowed_for_dual_stack": "ipv6AllowedForDualStack",
        "log_format": "logFormat",
        "logging_format": "loggingFormat",
        "log_group": "logGroup",
        "log_retention": "logRetention",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "log_retention_role": "logRetentionRole",
        "max_event_age": "maxEventAge",
        "memory_size": "memorySize",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "params_and_secrets": "paramsAndSecrets",
        "profiling": "profiling",
        "profiling_group": "profilingGroup",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "retry_attempts": "retryAttempts",
        "runtime_management_mode": "runtimeManagementMode",
        "snap_start": "snapStart",
        "system_log_level": "systemLogLevel",
        "timeout": "timeout",
    },
)
class DockerLambdaCustomProps:
    def __init__(
        self,
        *,
        code: _aws_cdk_aws_lambda_ceddda9d.DockerImageCode,
        adot_instrumentation: typing.Optional[typing.Union[_aws_cdk_aws_lambda_ceddda9d.AdotInstrumentationConfig, typing.Dict[builtins.str, typing.Any]]] = None,
        application_log_level: typing.Optional[builtins.str] = None,
        architecture: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.Architecture] = None,
        code_signing_config: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.ICodeSigningConfig] = None,
        current_version_options: typing.Optional[typing.Union[_aws_cdk_aws_lambda_ceddda9d.VersionOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        dead_letter_queue: typing.Optional[_aws_cdk_aws_sqs_ceddda9d.IQueue] = None,
        dead_letter_queue_enabled: typing.Optional[builtins.bool] = None,
        dead_letter_topic: typing.Optional[_aws_cdk_aws_sns_ceddda9d.ITopic] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_encryption: typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey] = None,
        ephemeral_storage_size: typing.Optional[_aws_cdk_ceddda9d.Size] = None,
        events: typing.Optional[typing.Sequence[_aws_cdk_aws_lambda_ceddda9d.IEventSource]] = None,
        filesystem: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FileSystem] = None,
        initial_policy: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.PolicyStatement]] = None,
        insights_version: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LambdaInsightsVersion] = None,
        ipv6_allowed_for_dual_stack: typing.Optional[builtins.bool] = None,
        log_format: typing.Optional[builtins.str] = None,
        logging_format: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LoggingFormat] = None,
        log_group: typing.Optional[_aws_cdk_aws_logs_ceddda9d.ILogGroup] = None,
        log_retention: typing.Optional[_aws_cdk_aws_logs_ceddda9d.RetentionDays] = None,
        log_retention_retry_options: typing.Optional[typing.Union[_aws_cdk_aws_lambda_ceddda9d.LogRetentionRetryOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        log_retention_role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
        max_event_age: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        on_failure: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IDestination] = None,
        on_success: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IDestination] = None,
        params_and_secrets: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.ParamsAndSecretsLayerVersion] = None,
        profiling: typing.Optional[builtins.bool] = None,
        profiling_group: typing.Optional[_aws_cdk_aws_codeguruprofiler_ceddda9d.IProfilingGroup] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        runtime_management_mode: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.RuntimeManagementMode] = None,
        snap_start: typing.Optional[_aws_cdk_aws_lambda_ceddda9d.SnapStartConf] = None,
        system_log_level: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_aws_cdk_ceddda9d.Duration] = None,
    ) -> None:
        '''(experimental) DockerLambdaCustomProps.

        :param code: The source code of your Lambda function. You can point to a file in an Amazon Simple Storage Service (Amazon S3) bucket or specify your source code as inline text.
        :param adot_instrumentation: Specify the configuration of AWS Distro for OpenTelemetry (ADOT) instrumentation. Default: - No ADOT instrumentation
        :param application_log_level: Sets the application log level for the function. Default: "INFO"
        :param architecture: The system architectures compatible with this lambda function. Default: Architecture.X86_64
        :param code_signing_config: Code signing config associated with this function. Default: - Not Sign the Code
        :param current_version_options: Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method. Default: - default options as described in ``VersionOptions``
        :param dead_letter_queue: The SQS queue to use if DLQ is enabled. If SNS topic is desired, specify ``deadLetterTopic`` property instead. Default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        :param dead_letter_queue_enabled: Enabled DLQ. If ``deadLetterQueue`` is undefined, an SQS queue with default options will be defined for your Function. Default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        :param dead_letter_topic: The SNS topic to use as a DLQ. Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly. Default: - no SNS topic
        :param environment: Key-value pairs that Lambda caches and makes available for your Lambda functions. Use environment variables to apply configuration changes, such as test and production environment configurations, without changing your Lambda function source code. Default: - No environment variables.
        :param environment_encryption: The AWS KMS key that's used to encrypt your function's environment variables. Default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        :param ephemeral_storage_size: The size of the function’s /tmp directory in MiB. Default: 512 MiB
        :param events: Event sources for this function. You can also add event sources using ``addEventSource``. Default: - No event sources.
        :param filesystem: The filesystem configuration for the lambda function. Default: - will not mount any filesystem
        :param initial_policy: Initial policy statements to add to the created Lambda Role. You can call ``addToRolePolicy`` to the created lambda to add statements post creation. Default: - No policy statements are added to the created Lambda role.
        :param insights_version: Specify the version of CloudWatch Lambda insights to use for monitoring. Default: - No Lambda Insights
        :param ipv6_allowed_for_dual_stack: Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets. Only used if 'vpc' is supplied. Default: false
        :param log_format: Sets the logFormat for the function. Default: "Text"
        :param logging_format: Sets the loggingFormat for the function. Default: LoggingFormat.TEXT
        :param log_group: The log group the function sends logs to. By default, Lambda functions send logs to an automatically created default log group named /aws/lambda/<function name>. However you cannot change the properties of this auto-created log group using the AWS CDK, e.g. you cannot set a different log retention. Use the ``logGroup`` property to create a fully customizable LogGroup ahead of time, and instruct the Lambda function to send logs to it. Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16. If you are deploying to another type of region, please check regional availability first. Default: ``/aws/lambda/${this.functionName}`` - default log group created by Lambda
        :param log_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``INFINITE``. This is a legacy API and we strongly recommend you move away from it if you can. Instead create a fully customizable log group with ``logs.LogGroup`` and use the ``logGroup`` property to instruct the Lambda function to send logs to it. Migrating from ``logRetention`` to ``logGroup`` will cause the name of the log group to change. Users and code and referencing the name verbatim will have to adjust. In AWS CDK code, you can access the log group name directly from the LogGroup construct:: import * as logs from 'aws-cdk-lib/aws-logs'; declare const myLogGroup: logs.LogGroup; myLogGroup.logGroupName; Default: logs.RetentionDays.INFINITE
        :param log_retention_retry_options: When log retention is specified, a custom resource attempts to create the CloudWatch log group. These options control the retry policy when interacting with CloudWatch APIs. This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can. ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it. Default: - Default AWS SDK retry options.
        :param log_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can. ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it. Default: - A new role is created.
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param memory_size: The amount of memory, in MB, that is allocated to your Lambda function. Lambda uses this value to proportionally allocate the amount of CPU power. For more information, see Resource Model in the AWS Lambda Developer Guide. Default: 128
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param params_and_secrets: Specify the configuration of Parameters and Secrets Extension. Default: - No Parameters and Secrets Extension
        :param profiling: Enable profiling. Default: - No profiling.
        :param profiling_group: Profiling Group. Default: - A new profiling group will be created if ``profiling`` is set.
        :param reserved_concurrent_executions: The maximum of concurrent executions you want to reserve for the function. Default: - No specific limit - account limit.
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param runtime_management_mode: Sets the runtime management configuration for a function's version. Default: Auto
        :param snap_start: Enable SnapStart for Lambda Function. SnapStart is currently supported only for Java 11, 17 runtime Default: - No snapstart
        :param system_log_level: Sets the system log level for the function. Default: "INFO"
        :param timeout: The function execution time (in seconds) after which Lambda terminates the function. Because the execution time affects cost, set this value based on the function's expected execution time. Default: Duration.seconds(3)

        :stability: experimental
        '''
        if isinstance(adot_instrumentation, dict):
            adot_instrumentation = _aws_cdk_aws_lambda_ceddda9d.AdotInstrumentationConfig(**adot_instrumentation)
        if isinstance(current_version_options, dict):
            current_version_options = _aws_cdk_aws_lambda_ceddda9d.VersionOptions(**current_version_options)
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = _aws_cdk_aws_lambda_ceddda9d.LogRetentionRetryOptions(**log_retention_retry_options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8887010d20e5a490252cbb670224b3b6cc6d005d6818217fc21859942d9506f)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument adot_instrumentation", value=adot_instrumentation, expected_type=type_hints["adot_instrumentation"])
            check_type(argname="argument application_log_level", value=application_log_level, expected_type=type_hints["application_log_level"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument code_signing_config", value=code_signing_config, expected_type=type_hints["code_signing_config"])
            check_type(argname="argument current_version_options", value=current_version_options, expected_type=type_hints["current_version_options"])
            check_type(argname="argument dead_letter_queue", value=dead_letter_queue, expected_type=type_hints["dead_letter_queue"])
            check_type(argname="argument dead_letter_queue_enabled", value=dead_letter_queue_enabled, expected_type=type_hints["dead_letter_queue_enabled"])
            check_type(argname="argument dead_letter_topic", value=dead_letter_topic, expected_type=type_hints["dead_letter_topic"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument environment_encryption", value=environment_encryption, expected_type=type_hints["environment_encryption"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument events", value=events, expected_type=type_hints["events"])
            check_type(argname="argument filesystem", value=filesystem, expected_type=type_hints["filesystem"])
            check_type(argname="argument initial_policy", value=initial_policy, expected_type=type_hints["initial_policy"])
            check_type(argname="argument insights_version", value=insights_version, expected_type=type_hints["insights_version"])
            check_type(argname="argument ipv6_allowed_for_dual_stack", value=ipv6_allowed_for_dual_stack, expected_type=type_hints["ipv6_allowed_for_dual_stack"])
            check_type(argname="argument log_format", value=log_format, expected_type=type_hints["log_format"])
            check_type(argname="argument logging_format", value=logging_format, expected_type=type_hints["logging_format"])
            check_type(argname="argument log_group", value=log_group, expected_type=type_hints["log_group"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument log_retention_role", value=log_retention_role, expected_type=type_hints["log_retention_role"])
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument params_and_secrets", value=params_and_secrets, expected_type=type_hints["params_and_secrets"])
            check_type(argname="argument profiling", value=profiling, expected_type=type_hints["profiling"])
            check_type(argname="argument profiling_group", value=profiling_group, expected_type=type_hints["profiling_group"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument runtime_management_mode", value=runtime_management_mode, expected_type=type_hints["runtime_management_mode"])
            check_type(argname="argument snap_start", value=snap_start, expected_type=type_hints["snap_start"])
            check_type(argname="argument system_log_level", value=system_log_level, expected_type=type_hints["system_log_level"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
        }
        if adot_instrumentation is not None:
            self._values["adot_instrumentation"] = adot_instrumentation
        if application_log_level is not None:
            self._values["application_log_level"] = application_log_level
        if architecture is not None:
            self._values["architecture"] = architecture
        if code_signing_config is not None:
            self._values["code_signing_config"] = code_signing_config
        if current_version_options is not None:
            self._values["current_version_options"] = current_version_options
        if dead_letter_queue is not None:
            self._values["dead_letter_queue"] = dead_letter_queue
        if dead_letter_queue_enabled is not None:
            self._values["dead_letter_queue_enabled"] = dead_letter_queue_enabled
        if dead_letter_topic is not None:
            self._values["dead_letter_topic"] = dead_letter_topic
        if environment is not None:
            self._values["environment"] = environment
        if environment_encryption is not None:
            self._values["environment_encryption"] = environment_encryption
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if events is not None:
            self._values["events"] = events
        if filesystem is not None:
            self._values["filesystem"] = filesystem
        if initial_policy is not None:
            self._values["initial_policy"] = initial_policy
        if insights_version is not None:
            self._values["insights_version"] = insights_version
        if ipv6_allowed_for_dual_stack is not None:
            self._values["ipv6_allowed_for_dual_stack"] = ipv6_allowed_for_dual_stack
        if log_format is not None:
            self._values["log_format"] = log_format
        if logging_format is not None:
            self._values["logging_format"] = logging_format
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if log_retention_role is not None:
            self._values["log_retention_role"] = log_retention_role
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if params_and_secrets is not None:
            self._values["params_and_secrets"] = params_and_secrets
        if profiling is not None:
            self._values["profiling"] = profiling
        if profiling_group is not None:
            self._values["profiling_group"] = profiling_group
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if runtime_management_mode is not None:
            self._values["runtime_management_mode"] = runtime_management_mode
        if snap_start is not None:
            self._values["snap_start"] = snap_start
        if system_log_level is not None:
            self._values["system_log_level"] = system_log_level
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def code(self) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageCode:
        '''The source code of your Lambda function.

        You can point to a file in an
        Amazon Simple Storage Service (Amazon S3) bucket or specify your source
        code as inline text.
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageCode, result)

    @builtins.property
    def adot_instrumentation(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.AdotInstrumentationConfig]:
        '''Specify the configuration of AWS Distro for OpenTelemetry (ADOT) instrumentation.

        :default: - No ADOT instrumentation
        '''
        result = self._values.get("adot_instrumentation")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.AdotInstrumentationConfig], result)

    @builtins.property
    def application_log_level(self) -> typing.Optional[builtins.str]:
        '''Sets the application log level for the function.

        :default: "INFO"
        '''
        result = self._values.get("application_log_level")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def architecture(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.Architecture]:
        '''The system architectures compatible with this lambda function.

        :default: Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.Architecture], result)

    @builtins.property
    def code_signing_config(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.ICodeSigningConfig]:
        '''Code signing config associated with this function.

        :default: - Not Sign the Code
        '''
        result = self._values.get("code_signing_config")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.ICodeSigningConfig], result)

    @builtins.property
    def current_version_options(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.VersionOptions]:
        '''Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method.

        :default: - default options as described in ``VersionOptions``
        '''
        result = self._values.get("current_version_options")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.VersionOptions], result)

    @builtins.property
    def dead_letter_queue(self) -> typing.Optional[_aws_cdk_aws_sqs_ceddda9d.IQueue]:
        '''The SQS queue to use if DLQ is enabled.

        If SNS topic is desired, specify ``deadLetterTopic`` property instead.

        :default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        '''
        result = self._values.get("dead_letter_queue")
        return typing.cast(typing.Optional[_aws_cdk_aws_sqs_ceddda9d.IQueue], result)

    @builtins.property
    def dead_letter_queue_enabled(self) -> typing.Optional[builtins.bool]:
        '''Enabled DLQ.

        If ``deadLetterQueue`` is undefined,
        an SQS queue with default options will be defined for your Function.

        :default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        '''
        result = self._values.get("dead_letter_queue_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dead_letter_topic(self) -> typing.Optional[_aws_cdk_aws_sns_ceddda9d.ITopic]:
        '''The SNS topic to use as a DLQ.

        Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created
        rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly.

        :default: - no SNS topic
        '''
        result = self._values.get("dead_letter_topic")
        return typing.cast(typing.Optional[_aws_cdk_aws_sns_ceddda9d.ITopic], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that Lambda caches and makes available for your Lambda functions.

        Use environment variables to apply configuration changes, such
        as test and production environment configurations, without changing your
        Lambda function source code.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_encryption(self) -> typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey]:
        '''The AWS KMS key that's used to encrypt your function's environment variables.

        :default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        '''
        result = self._values.get("environment_encryption")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_ceddda9d.IKey], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_aws_cdk_ceddda9d.Size]:
        '''The size of the function’s /tmp directory in MiB.

        :default: 512 MiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Size], result)

    @builtins.property
    def events(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_lambda_ceddda9d.IEventSource]]:
        '''Event sources for this function.

        You can also add event sources using ``addEventSource``.

        :default: - No event sources.
        '''
        result = self._values.get("events")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_lambda_ceddda9d.IEventSource]], result)

    @builtins.property
    def filesystem(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FileSystem]:
        '''The filesystem configuration for the lambda function.

        :default: - will not mount any filesystem
        '''
        result = self._values.get("filesystem")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.FileSystem], result)

    @builtins.property
    def initial_policy(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_iam_ceddda9d.PolicyStatement]]:
        '''Initial policy statements to add to the created Lambda Role.

        You can call ``addToRolePolicy`` to the created lambda to add statements post creation.

        :default: - No policy statements are added to the created Lambda role.
        '''
        result = self._values.get("initial_policy")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_ceddda9d.PolicyStatement]], result)

    @builtins.property
    def insights_version(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LambdaInsightsVersion]:
        '''Specify the version of CloudWatch Lambda insights to use for monitoring.

        :default: - No Lambda Insights
        '''
        result = self._values.get("insights_version")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LambdaInsightsVersion], result)

    @builtins.property
    def ipv6_allowed_for_dual_stack(self) -> typing.Optional[builtins.bool]:
        '''Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets.

        Only used if 'vpc' is supplied.

        :default: false
        '''
        result = self._values.get("ipv6_allowed_for_dual_stack")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def log_format(self) -> typing.Optional[builtins.str]:
        '''Sets the logFormat for the function.

        :default: "Text"
        '''
        result = self._values.get("log_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def logging_format(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LoggingFormat]:
        '''Sets the loggingFormat for the function.

        :default: LoggingFormat.TEXT
        '''
        result = self._values.get("logging_format")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LoggingFormat], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_aws_cdk_aws_logs_ceddda9d.ILogGroup]:
        '''The log group the function sends logs to.

        By default, Lambda functions send logs to an automatically created default log group named /aws/lambda/.
        However you cannot change the properties of this auto-created log group using the AWS CDK, e.g. you cannot set a different log retention.

        Use the ``logGroup`` property to create a fully customizable LogGroup ahead of time, and instruct the Lambda function to send logs to it.

        Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16.
        If you are deploying to another type of region, please check regional availability first.

        :default: ``/aws/lambda/${this.functionName}`` - default log group created by Lambda
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_logs_ceddda9d.ILogGroup], result)

    @builtins.property
    def log_retention(
        self,
    ) -> typing.Optional[_aws_cdk_aws_logs_ceddda9d.RetentionDays]:
        '''The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``INFINITE``.

        This is a legacy API and we strongly recommend you move away from it if you can.
        Instead create a fully customizable log group with ``logs.LogGroup`` and use the ``logGroup`` property
        to instruct the Lambda function to send logs to it.
        Migrating from ``logRetention`` to ``logGroup`` will cause the name of the log group to change.
        Users and code and referencing the name verbatim will have to adjust.

        In AWS CDK code, you can access the log group name directly from the LogGroup construct::

           import * as logs from 'aws-cdk-lib/aws-logs';

           declare const myLogGroup: logs.LogGroup;
           myLogGroup.logGroupName;

        :default: logs.RetentionDays.INFINITE
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_aws_cdk_aws_logs_ceddda9d.RetentionDays], result)

    @builtins.property
    def log_retention_retry_options(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LogRetentionRetryOptions]:
        '''When log retention is specified, a custom resource attempts to create the CloudWatch log group.

        These options control the retry policy when interacting with CloudWatch APIs.

        This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can.
        ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.

        :default: - Default AWS SDK retry options.
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.LogRetentionRetryOptions], result)

    @builtins.property
    def log_retention_role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole]:
        '''The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can.
        ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.

        :default: - A new role is created.
        '''
        result = self._values.get("log_retention_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole], result)

    @builtins.property
    def max_event_age(self) -> typing.Optional[_aws_cdk_ceddda9d.Duration]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Duration], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of memory, in MB, that is allocated to your Lambda function.

        Lambda uses this value to proportionally allocate the amount of CPU
        power. For more information, see Resource Model in the AWS Lambda
        Developer Guide.

        :default: 128
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IDestination]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IDestination], result)

    @builtins.property
    def on_success(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IDestination]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IDestination], result)

    @builtins.property
    def params_and_secrets(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.ParamsAndSecretsLayerVersion]:
        '''Specify the configuration of Parameters and Secrets Extension.

        :default: - No Parameters and Secrets Extension
        '''
        result = self._values.get("params_and_secrets")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.ParamsAndSecretsLayerVersion], result)

    @builtins.property
    def profiling(self) -> typing.Optional[builtins.bool]:
        '''Enable profiling.

        :default: - No profiling.
        '''
        result = self._values.get("profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def profiling_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_codeguruprofiler_ceddda9d.IProfilingGroup]:
        '''Profiling Group.

        :default: - A new profiling group will be created if ``profiling`` is set.
        '''
        result = self._values.get("profiling_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_codeguruprofiler_ceddda9d.IProfilingGroup], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The maximum of concurrent executions you want to reserve for the function.

        :default: - No specific limit - account limit.
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def runtime_management_mode(
        self,
    ) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.RuntimeManagementMode]:
        '''Sets the runtime management configuration for a function's version.

        :default: Auto
        '''
        result = self._values.get("runtime_management_mode")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.RuntimeManagementMode], result)

    @builtins.property
    def snap_start(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.SnapStartConf]:
        '''Enable SnapStart for Lambda Function.

        SnapStart is currently supported only for Java 11, 17 runtime

        :default: - No snapstart
        '''
        result = self._values.get("snap_start")
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.SnapStartConf], result)

    @builtins.property
    def system_log_level(self) -> typing.Optional[builtins.str]:
        '''Sets the system log level for the function.

        :default: "INFO"
        '''
        result = self._values.get("system_log_level")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_aws_cdk_ceddda9d.Duration]:
        '''The function execution time (in seconds) after which Lambda terminates the function.

        Because the execution time affects cost, set this value
        based on the function's expected execution time.

        :default: Duration.seconds(3)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerLambdaCustomProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.HuggingFaceSageMakerEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "container": "container",
        "instance_type": "instanceType",
        "model_id": "modelId",
        "endpoint_name": "endpointName",
        "environment": "environment",
        "instance_count": "instanceCount",
        "role": "role",
        "startup_health_check_timeout_in_seconds": "startupHealthCheckTimeoutInSeconds",
        "vpc_config": "vpcConfig",
    },
)
class HuggingFaceSageMakerEndpointProps:
    def __init__(
        self,
        *,
        container: ContainerImage,
        instance_type: "SageMakerInstanceType",
        model_id: builtins.str,
        endpoint_name: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        instance_count: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
        startup_health_check_timeout_in_seconds: typing.Optional[jsii.Number] = None,
        vpc_config: typing.Optional[typing.Union[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param container: 
        :param instance_type: 
        :param model_id: 
        :param endpoint_name: 
        :param environment: 
        :param instance_count: 
        :param role: 
        :param startup_health_check_timeout_in_seconds: 
        :param vpc_config: 

        :stability: experimental
        '''
        if isinstance(vpc_config, dict):
            vpc_config = _aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty(**vpc_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9abbb731cf44967a6628e353d6c56d0035755b62bf45acdaa8261b4c5048d1f9)
            check_type(argname="argument container", value=container, expected_type=type_hints["container"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
            check_type(argname="argument endpoint_name", value=endpoint_name, expected_type=type_hints["endpoint_name"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument startup_health_check_timeout_in_seconds", value=startup_health_check_timeout_in_seconds, expected_type=type_hints["startup_health_check_timeout_in_seconds"])
            check_type(argname="argument vpc_config", value=vpc_config, expected_type=type_hints["vpc_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container": container,
            "instance_type": instance_type,
            "model_id": model_id,
        }
        if endpoint_name is not None:
            self._values["endpoint_name"] = endpoint_name
        if environment is not None:
            self._values["environment"] = environment
        if instance_count is not None:
            self._values["instance_count"] = instance_count
        if role is not None:
            self._values["role"] = role
        if startup_health_check_timeout_in_seconds is not None:
            self._values["startup_health_check_timeout_in_seconds"] = startup_health_check_timeout_in_seconds
        if vpc_config is not None:
            self._values["vpc_config"] = vpc_config

    @builtins.property
    def container(self) -> ContainerImage:
        '''
        :stability: experimental
        '''
        result = self._values.get("container")
        assert result is not None, "Required property 'container' is missing"
        return typing.cast(ContainerImage, result)

    @builtins.property
    def instance_type(self) -> "SageMakerInstanceType":
        '''
        :stability: experimental
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast("SageMakerInstanceType", result)

    @builtins.property
    def model_id(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("model_id")
        assert result is not None, "Required property 'model_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def endpoint_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("endpoint_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def instance_count(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("instance_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role]:
        '''
        :stability: experimental
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role], result)

    @builtins.property
    def startup_health_check_timeout_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("startup_health_check_timeout_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def vpc_config(
        self,
    ) -> typing.Optional[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty]:
        '''
        :stability: experimental
        '''
        result = self._values.get("vpc_config")
        return typing.cast(typing.Optional[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HuggingFaceSageMakerEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="@cdklabs/generative-ai-cdk-constructs.IInstanceAliase")
class IInstanceAliase(typing_extensions.Protocol):
    '''
    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="aliases")
    def aliases(self) -> typing.Mapping[builtins.str, builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @aliases.setter
    def aliases(self, value: typing.Mapping[builtins.str, builtins.str]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="region")
    def region(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        ...

    @region.setter
    def region(self, value: builtins.str) -> None:
        ...


class _IInstanceAliaseProxy:
    '''
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/generative-ai-cdk-constructs.IInstanceAliase"

    @builtins.property
    @jsii.member(jsii_name="aliases")
    def aliases(self) -> typing.Mapping[builtins.str, builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Mapping[builtins.str, builtins.str], jsii.get(self, "aliases"))

    @aliases.setter
    def aliases(self, value: typing.Mapping[builtins.str, builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a5f72a06047d6784d819d9905a60d528d3adf1757fd113a494d1e278ff756cc0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "aliases", value)

    @builtins.property
    @jsii.member(jsii_name="region")
    def region(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "region"))

    @region.setter
    def region(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09be534ff54d13743d68fbae01a7a6206246397b640b98663316222b3582760e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "region", value)

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInstanceAliase).__jsii_proxy_class__ = lambda : _IInstanceAliaseProxy


@jsii.interface(jsii_type="@cdklabs/generative-ai-cdk-constructs.IInstanceValiant")
class IInstanceValiant(typing_extensions.Protocol):
    '''
    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        ...

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        ...

    @environment.setter
    def environment(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="imageUri")
    def image_uri(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @image_uri.setter
    def image_uri(self, value: typing.Optional[builtins.str]) -> None:
        ...


class _IInstanceValiantProxy:
    '''
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/generative-ai-cdk-constructs.IInstanceValiant"

    @builtins.property
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__784c92c851a2dedf6e80ea471cde9f9490de2aa6021222c9354e77dce1375a9c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceType", value)

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "environment"))

    @environment.setter
    def environment(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9bd10265f2863a8642df6187149bcdb3e5b8d4d4a701e3c0f81e180f79133355)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "environment", value)

    @builtins.property
    @jsii.member(jsii_name="imageUri")
    def image_uri(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "imageUri"))

    @image_uri.setter
    def image_uri(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5208b4cb02ff50a0ebbf048c64aae565c2966812ac05c66bd93c7b3c84aa63ca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "imageUri", value)

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInstanceValiant).__jsii_proxy_class__ = lambda : _IInstanceValiantProxy


@jsii.interface(jsii_type="@cdklabs/generative-ai-cdk-constructs.IJumpStartModelSpec")
class IJumpStartModelSpec(typing_extensions.Protocol):
    '''
    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="defaultInstanceType")
    def default_instance_type(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        ...

    @default_instance_type.setter
    def default_instance_type(self, value: builtins.str) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool]]:
        '''
        :stability: experimental
        '''
        ...

    @environment.setter
    def environment(
        self,
        value: typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="instanceTypes")
    def instance_types(self) -> typing.List[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @instance_types.setter
    def instance_types(self, value: typing.List[builtins.str]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="modelId")
    def model_id(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        ...

    @model_id.setter
    def model_id(self, value: builtins.str) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="requiresEula")
    def requires_eula(self) -> builtins.bool:
        '''
        :stability: experimental
        '''
        ...

    @requires_eula.setter
    def requires_eula(self, value: builtins.bool) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        ...

    @version.setter
    def version(self, value: builtins.str) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="artifactKey")
    def artifact_key(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @artifact_key.setter
    def artifact_key(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="instanceAliases")
    def instance_aliases(self) -> typing.Optional[typing.List[IInstanceAliase]]:
        '''
        :stability: experimental
        '''
        ...

    @instance_aliases.setter
    def instance_aliases(
        self,
        value: typing.Optional[typing.List[IInstanceAliase]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="instanceVariants")
    def instance_variants(self) -> typing.Optional[typing.List[IInstanceValiant]]:
        '''
        :stability: experimental
        '''
        ...

    @instance_variants.setter
    def instance_variants(
        self,
        value: typing.Optional[typing.List[IInstanceValiant]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="modelPackageArns")
    def model_package_arns(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        ...

    @model_package_arns.setter
    def model_package_arns(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="prepackedArtifactKey")
    def prepacked_artifact_key(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @prepacked_artifact_key.setter
    def prepacked_artifact_key(self, value: typing.Optional[builtins.str]) -> None:
        ...


class _IJumpStartModelSpecProxy:
    '''
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/generative-ai-cdk-constructs.IJumpStartModelSpec"

    @builtins.property
    @jsii.member(jsii_name="defaultInstanceType")
    def default_instance_type(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "defaultInstanceType"))

    @default_instance_type.setter
    def default_instance_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a5a0c622f349e4bb3c898657c97ff2fd43f2f1ae081b59b80a923854e9a513f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultInstanceType", value)

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool]], jsii.get(self, "environment"))

    @environment.setter
    def environment(
        self,
        value: typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2073cfea003f3e88c802dd137c85fd6e32f9da1475fd6187a6f87ace6a655672)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "environment", value)

    @builtins.property
    @jsii.member(jsii_name="instanceTypes")
    def instance_types(self) -> typing.List[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "instanceTypes"))

    @instance_types.setter
    def instance_types(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aef2d73f0172f5d063f633b101b7066b0fbe43a8824831786cf98be28ecd7f30)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceTypes", value)

    @builtins.property
    @jsii.member(jsii_name="modelId")
    def model_id(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "modelId"))

    @model_id.setter
    def model_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0aa0d5de10cb8693f8f2a7f933f95ad652052122b9c5fbb256a6a6eb62687607)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "modelId", value)

    @builtins.property
    @jsii.member(jsii_name="requiresEula")
    def requires_eula(self) -> builtins.bool:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "requiresEula"))

    @requires_eula.setter
    def requires_eula(self, value: builtins.bool) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c203346f7b7cd5658001fe31b5db174d97920c7a84748eddb5bb67d041dfe3a0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requiresEula", value)

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "version"))

    @version.setter
    def version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__230bf6b942aecadf2cddb2400e4b8fa93d7b5324764d21f3c3745c62d2c05093)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "version", value)

    @builtins.property
    @jsii.member(jsii_name="artifactKey")
    def artifact_key(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "artifactKey"))

    @artifact_key.setter
    def artifact_key(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e18f266234ad99daa9e9e8bc62ef14e306dceb86e8896985bbb9c334a1a680d4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "artifactKey", value)

    @builtins.property
    @jsii.member(jsii_name="instanceAliases")
    def instance_aliases(self) -> typing.Optional[typing.List[IInstanceAliase]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[IInstanceAliase]], jsii.get(self, "instanceAliases"))

    @instance_aliases.setter
    def instance_aliases(
        self,
        value: typing.Optional[typing.List[IInstanceAliase]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ab8b9681e067fc772bfa8ae21e9b889922f76498043ac3dc59607d18b1c99b6d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceAliases", value)

    @builtins.property
    @jsii.member(jsii_name="instanceVariants")
    def instance_variants(self) -> typing.Optional[typing.List[IInstanceValiant]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[IInstanceValiant]], jsii.get(self, "instanceVariants"))

    @instance_variants.setter
    def instance_variants(
        self,
        value: typing.Optional[typing.List[IInstanceValiant]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a37af151d413380fdbb03c303f6e418d5e5ca5225e04d5264a9a4ea57136b320)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceVariants", value)

    @builtins.property
    @jsii.member(jsii_name="modelPackageArns")
    def model_package_arns(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "modelPackageArns"))

    @model_package_arns.setter
    def model_package_arns(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b5a37e8a274c0f3fc20705440c0c3341b45281869a1e4e347af5d91621fcaa1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "modelPackageArns", value)

    @builtins.property
    @jsii.member(jsii_name="prepackedArtifactKey")
    def prepacked_artifact_key(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "prepackedArtifactKey"))

    @prepacked_artifact_key.setter
    def prepacked_artifact_key(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__03c90f1f4cdb0e78188cc1b2b4752b6e641e3139cd9ddbcd4cf9be7e0e0a09dc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "prepackedArtifactKey", value)

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IJumpStartModelSpec).__jsii_proxy_class__ = lambda : _IJumpStartModelSpecProxy


class JumpStartModel(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.JumpStartModel",
):
    '''
    :stability: experimental
    '''

    def __init__(self, name: builtins.str) -> None:
        '''
        :param name: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__951253bb0ca138e826c6fe2a480e29d1949fa737e804147291d2ca257fc68fbc)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        jsii.create(self.__class__, self, [name])

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, name: builtins.str) -> "JumpStartModel":
        '''
        :param name: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5af7b97de0c920bbdd298f2015ccefdbeb43ecd7756bcf540abe8209e26d7adb)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        return typing.cast("JumpStartModel", jsii.sinvoke(cls, "of", [name]))

    @jsii.member(jsii_name="bind")
    def bind(self) -> IJumpStartModelSpec:
        '''
        :stability: experimental
        '''
        return typing.cast(IJumpStartModelSpec, jsii.invoke(self, "bind", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_BASE_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_BASE_1_0_1")
    def HUGGINGFACE_ASR_WHISPER_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_BASE_2_0_0")
    def HUGGINGFACE_ASR_WHISPER_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_BASE_3_0_0")
    def HUGGINGFACE_ASR_WHISPER_BASE_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_BASE_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_1_0_1")
    def HUGGINGFACE_ASR_WHISPER_LARGE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_2_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_3_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_V2_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_V2_1_0_1")
    def HUGGINGFACE_ASR_WHISPER_LARGE_V2_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_V2_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_V2_2_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_V2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_V2_3_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_V2_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_V2_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_LARGE_V3_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_LARGE_V3_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_LARGE_V3_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_MEDIUM_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_MEDIUM_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_MEDIUM_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_MEDIUM_1_0_1")
    def HUGGINGFACE_ASR_WHISPER_MEDIUM_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_MEDIUM_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_MEDIUM_2_0_0")
    def HUGGINGFACE_ASR_WHISPER_MEDIUM_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_MEDIUM_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_MEDIUM_3_0_0")
    def HUGGINGFACE_ASR_WHISPER_MEDIUM_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_MEDIUM_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_SMALL_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_SMALL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_SMALL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_SMALL_1_0_1")
    def HUGGINGFACE_ASR_WHISPER_SMALL_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_SMALL_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_SMALL_2_0_0")
    def HUGGINGFACE_ASR_WHISPER_SMALL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_SMALL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_SMALL_3_0_0")
    def HUGGINGFACE_ASR_WHISPER_SMALL_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_SMALL_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_TINY_1_0_0")
    def HUGGINGFACE_ASR_WHISPER_TINY_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_TINY_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_TINY_1_0_1")
    def HUGGINGFACE_ASR_WHISPER_TINY_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_TINY_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_TINY_2_0_0")
    def HUGGINGFACE_ASR_WHISPER_TINY_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_TINY_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ASR_WHISPER_TINY_3_0_0")
    def HUGGINGFACE_ASR_WHISPER_TINY_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ASR_WHISPER_TINY_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_1")
    def HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_2")
    def HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_CASED_2_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_1")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_2")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_2_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_1")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_2")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_1")
    def HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_2")
    def HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_EQA_BERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_1")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_2")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_2_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_1")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_2")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_1")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_2")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_2_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_1")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_2")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0")
    def HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_0")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_1")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_2")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_2_0_0")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_1")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_2")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_1")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_2")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILBERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_0")
    def HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_1")
    def HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_2")
    def HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILROBERTA_BASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_DISTILROBERTA_BASE_2_0_0")
    def HUGGINGFACE_EQA_DISTILROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_DISTILROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_1_0_0")
    def HUGGINGFACE_EQA_ROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_1_0_1")
    def HUGGINGFACE_EQA_ROBERTA_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_1_0_2")
    def HUGGINGFACE_EQA_ROBERTA_BASE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_2_0_0")
    def HUGGINGFACE_EQA_ROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0")
    def HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_1")
    def HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_2")
    def HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0")
    def HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_0")
    def HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_1")
    def HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_2")
    def HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_LARGE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_EQA_ROBERTA_LARGE_2_0_0")
    def HUGGINGFACE_EQA_ROBERTA_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_EQA_ROBERTA_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_FILLMASK_BERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_FILLMASK_BERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_FILLMASK_BERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_FILLMASK_BERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_FILLMASK_BERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_FILLMASK_BERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AHXT_LITELLAMA_460M_1T_1_0_0")
    def HUGGINGFACE_LLM_AHXT_LITELLAMA_460_M_1_T_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AHXT_LITELLAMA_460M_1T_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AHXT_LITELLAMA_460M_1T_1_1_0")
    def HUGGINGFACE_LLM_AHXT_LITELLAMA_460_M_1_T_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AHXT_LITELLAMA_460M_1T_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AI_FOREVER_MGPT_1_0_0")
    def HUGGINGFACE_LLM_AI_FOREVER_MGPT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AI_FOREVER_MGPT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AI_FOREVER_MGPT_1_1_0")
    def HUGGINGFACE_LLM_AI_FOREVER_MGPT_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AI_FOREVER_MGPT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ALPINDALE_WIZARD_LM_2_8_22B_1_0_0")
    def HUGGINGFACE_LLM_ALPINDALE_WIZARD_LM_2_8_22_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ALPINDALE_WIZARD_LM_2_8_22B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AMAZON_FALCONLITE_1_0_0")
    def HUGGINGFACE_LLM_AMAZON_FALCONLITE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AMAZON_FALCONLITE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AMAZON_FALCONLITE_1_1_0")
    def HUGGINGFACE_LLM_AMAZON_FALCONLITE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AMAZON_FALCONLITE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AMAZON_FALCONLITE2_1_0_0")
    def HUGGINGFACE_LLM_AMAZON_FALCONLITE2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AMAZON_FALCONLITE2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AMAZON_FALCONLITE2_1_1_0")
    def HUGGINGFACE_LLM_AMAZON_FALCONLITE2_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AMAZON_FALCONLITE2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AMAZON_MISTRALLITE_1_0_0")
    def HUGGINGFACE_LLM_AMAZON_MISTRALLITE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AMAZON_MISTRALLITE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AMAZON_MISTRALLITE_1_1_0")
    def HUGGINGFACE_LLM_AMAZON_MISTRALLITE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AMAZON_MISTRALLITE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_AYA_101_1_0_0")
    def HUGGINGFACE_LLM_AYA_101_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_AYA_101_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BAICHUAN2_7B_BASE_FP16_1_0_0")
    def HUGGINGFACE_LLM_BAICHUAN2_7_B_BASE_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BAICHUAN2_7B_BASE_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BERKELEY_NEST_STARLING_LM_7B_ALPHA_1_0_0")
    def HUGGINGFACE_LLM_BERKELEY_NEST_STARLING_LM_7_B_ALPHA_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BERKELEY_NEST_STARLING_LM_7B_ALPHA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BERKELEY_NEST_STARLING_LM_7B_ALPHA_1_1_0")
    def HUGGINGFACE_LLM_BERKELEY_NEST_STARLING_LM_7_B_ALPHA_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BERKELEY_NEST_STARLING_LM_7B_ALPHA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_1_0_0")
    def HUGGINGFACE_LLM_BILINGUAL_RINNA_4_B_INSTRUCTION_PPO_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_1_1_0")
    def HUGGINGFACE_LLM_BILINGUAL_RINNA_4_B_INSTRUCTION_PPO_BF16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_2_0_0")
    def HUGGINGFACE_LLM_BILINGUAL_RINNA_4_B_INSTRUCTION_PPO_BF16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_2_1_0")
    def HUGGINGFACE_LLM_BILINGUAL_RINNA_4_B_INSTRUCTION_PPO_BF16_2_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_BILINGUAL_RINNA_4B_INSTRUCTION_PPO_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_0_0")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_0_1")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_1_0")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_0")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_1")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_2")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_3")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_4")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_3_0")
    def HUGGINGFACE_LLM_CALM2_7_B_CHAT_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CALM2_7B_CHAT_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_COGNITIVE_DOLPHIN_29_LLAMA3_8B_1_0_0")
    def HUGGINGFACE_LLM_COGNITIVE_DOLPHIN_29_LLAMA3_8_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_COGNITIVE_DOLPHIN_29_LLAMA3_8B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_COHEREFORAI_C4AI_COMMAND_R_PLUS_1_0_0")
    def HUGGINGFACE_LLM_COHEREFORAI_C4_AI_COMMAND_R_PLUS_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_COHEREFORAI_C4AI_COMMAND_R_PLUS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CULTRIX_MISTRALTRIX_V1_1_0_0")
    def HUGGINGFACE_LLM_CULTRIX_MISTRALTRIX_V1_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CULTRIX_MISTRALTRIX_V1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_CULTRIX_MISTRALTRIX_V1_1_1_0")
    def HUGGINGFACE_LLM_CULTRIX_MISTRALTRIX_V1_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_CULTRIX_MISTRALTRIX_V1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DBRX_BASE_1_0_0")
    def HUGGINGFACE_LLM_DBRX_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DBRX_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DBRX_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_DBRX_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DBRX_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DOLPHIN_2_2_1_MISTRAL_7B_1_0_0")
    def HUGGINGFACE_LLM_DOLPHIN_2_2_1_MISTRAL_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DOLPHIN_2_2_1_MISTRAL_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DOLPHIN_2_2_1_MISTRAL_7B_1_1_0")
    def HUGGINGFACE_LLM_DOLPHIN_2_2_1_MISTRAL_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DOLPHIN_2_2_1_MISTRAL_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DOLPHIN_2_5_MIXTRAL_8X7B_1_0_0")
    def HUGGINGFACE_LLM_DOLPHIN_2_5_MIXTRAL_8_X7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DOLPHIN_2_5_MIXTRAL_8X7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DOLPHIN_2_5_MIXTRAL_8X7B_1_1_0")
    def HUGGINGFACE_LLM_DOLPHIN_2_5_MIXTRAL_8_X7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DOLPHIN_2_5_MIXTRAL_8X7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DOLPHIN_2_7_MIXTRAL_8X7B_1_0_0")
    def HUGGINGFACE_LLM_DOLPHIN_2_7_MIXTRAL_8_X7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DOLPHIN_2_7_MIXTRAL_8X7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_DOLPHIN_2_7_MIXTRAL_8X7B_1_1_0")
    def HUGGINGFACE_LLM_DOLPHIN_2_7_MIXTRAL_8_X7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_DOLPHIN_2_7_MIXTRAL_8X7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_1_3B_1_0_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_1_3_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_1_3B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_1_3B_1_1_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_1_3_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_1_3B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_2_7B_1_0_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_2_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_2_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_2_7B_1_1_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_2_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_GPT_NEO_2_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_160M_DEDUPED_1_0_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_160_M_DEDUPED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_160M_DEDUPED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_160M_DEDUPED_1_1_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_160_M_DEDUPED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_160M_DEDUPED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_70M_DEDUPED_1_0_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_70_M_DEDUPED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_70M_DEDUPED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_70M_DEDUPED_1_1_0")
    def HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_70_M_DEDUPED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELEUTHERAI_PYTHIA_70M_DEDUPED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_CHAT_BF16_1_0_0")
    def HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7_B_CHAT_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_CHAT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_CHAT_BF16_1_1_0")
    def HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7_B_CHAT_BF16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_CHAT_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_FAST_CHAT_BF16_1_0_0")
    def HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7_B_FAST_CHAT_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_FAST_CHAT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_FAST_CHAT_BF16_1_1_0")
    def HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7_B_FAST_CHAT_BF16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ELYZA_JAPANESE_LLAMA_2_7B_FAST_CHAT_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_BF16_1_0_0")
    def HUGGINGFACE_LLM_FALCON_180_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_BF16_1_1_0")
    def HUGGINGFACE_LLM_FALCON_180_B_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_BF16_1_2_0")
    def HUGGINGFACE_LLM_FALCON_180_B_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_BF16_1_3_0")
    def HUGGINGFACE_LLM_FALCON_180_B_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_BF16_1_4_0")
    def HUGGINGFACE_LLM_FALCON_180_B_BF16_1_4_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_BF16_1_4_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_BF16_1_5_0")
    def HUGGINGFACE_LLM_FALCON_180_B_BF16_1_5_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_BF16_1_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_0_0")
    def HUGGINGFACE_LLM_FALCON_180_B_CHAT_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_1_0")
    def HUGGINGFACE_LLM_FALCON_180_B_CHAT_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_2_0")
    def HUGGINGFACE_LLM_FALCON_180_B_CHAT_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_3_0")
    def HUGGINGFACE_LLM_FALCON_180_B_CHAT_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_180B_CHAT_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_0_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_1_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_2_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_1")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_2")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_3")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_1_3_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_1_3_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_0_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_0_1")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_0_2")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_1_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_1")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_2")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_3")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_4")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_BF16_2_3_0")
    def HUGGINGFACE_LLM_FALCON_40_B_BF16_2_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_BF16_2_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_0_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_1_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_2_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_1")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_2")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_3")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_1_3_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_1_3_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_0_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_0_1")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_0_2")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_1_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_1")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_2")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_3")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_4")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_3_0")
    def HUGGINGFACE_LLM_FALCON_40_B_INSTRUCT_BF16_2_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_40B_INSTRUCT_BF16_2_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_1_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_1_1")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_2_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_2_1")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_3_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_3_1")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_1_3_2")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_0_1")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_0_2")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_1_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_1")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_2")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_3")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_4")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_2_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_2_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_3_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_4_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_BF16_4_1_0")
    def HUGGINGFACE_LLM_FALCON_7_B_BF16_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_BF16_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_1_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_1_1")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_2_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_2_1")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_3_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_3_1")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_3_2")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_0_1")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_0_2")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_1_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_1")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_2")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_3")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_4")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_2_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_2_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_3_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_4_0_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_4_1_0")
    def HUGGINGFACE_LLM_FALCON_7_B_INSTRUCT_BF16_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON_7B_INSTRUCT_BF16_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_FALCON2_11B_1_0_0")
    def HUGGINGFACE_LLM_FALCON2_11_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_FALCON2_11B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GARAGE_BAIND_PLATYPUS2_7B_1_0_0")
    def HUGGINGFACE_LLM_GARAGE_BAIND_PLATYPUS2_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GARAGE_BAIND_PLATYPUS2_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GARAGE_BAIND_PLATYPUS2_7B_1_1_0")
    def HUGGINGFACE_LLM_GARAGE_BAIND_PLATYPUS2_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GARAGE_BAIND_PLATYPUS2_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_1_0_0")
    def HUGGINGFACE_LLM_GEMMA_2_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_1_1_0")
    def HUGGINGFACE_LLM_GEMMA_2_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_1_1_1")
    def HUGGINGFACE_LLM_GEMMA_2_B_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_1_1_2")
    def HUGGINGFACE_LLM_GEMMA_2_B_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_1_1_3")
    def HUGGINGFACE_LLM_GEMMA_2_B_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_1_2_0")
    def HUGGINGFACE_LLM_GEMMA_2_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_GEMMA_2_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_0")
    def HUGGINGFACE_LLM_GEMMA_2_B_INSTRUCT_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_1")
    def HUGGINGFACE_LLM_GEMMA_2_B_INSTRUCT_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_2")
    def HUGGINGFACE_LLM_GEMMA_2_B_INSTRUCT_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_3")
    def HUGGINGFACE_LLM_GEMMA_2_B_INSTRUCT_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_2_0")
    def HUGGINGFACE_LLM_GEMMA_2_B_INSTRUCT_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_2B_INSTRUCT_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_1_0_0")
    def HUGGINGFACE_LLM_GEMMA_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_1_1_0")
    def HUGGINGFACE_LLM_GEMMA_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_1_1_1")
    def HUGGINGFACE_LLM_GEMMA_7_B_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_1_1_2")
    def HUGGINGFACE_LLM_GEMMA_7_B_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_1_1_3")
    def HUGGINGFACE_LLM_GEMMA_7_B_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_1_2_0")
    def HUGGINGFACE_LLM_GEMMA_7_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_GEMMA_7_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_0")
    def HUGGINGFACE_LLM_GEMMA_7_B_INSTRUCT_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_1")
    def HUGGINGFACE_LLM_GEMMA_7_B_INSTRUCT_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_2")
    def HUGGINGFACE_LLM_GEMMA_7_B_INSTRUCT_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_3")
    def HUGGINGFACE_LLM_GEMMA_7_B_INSTRUCT_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_2_0")
    def HUGGINGFACE_LLM_GEMMA_7_B_INSTRUCT_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GEMMA_7B_INSTRUCT_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_GRADIENTAI_LLAMA_3_8B_INSTRUCT_262K_1_0_0")
    def HUGGINGFACE_LLM_GRADIENTAI_LLAMA_3_8_B_INSTRUCT_262_K_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_GRADIENTAI_LLAMA_3_8B_INSTRUCT_262K_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_ALPHA_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7_B_SFT_ALPHA_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_ALPHA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_ALPHA_1_1_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7_B_SFT_ALPHA_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_ALPHA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_BETA_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7_B_SFT_BETA_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_BETA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_BETA_1_1_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7_B_SFT_BETA_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_MISTRAL_7B_SFT_BETA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_ALPHA_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_ALPHA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_ALPHA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_ALPHA_1_1_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_ALPHA_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_ALPHA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_BETA_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_BETA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_BETA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_BETA_1_1_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_BETA_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_STARCHAT_BETA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_ALPHA_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7_B_ALPHA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_ALPHA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_ALPHA_1_1_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7_B_ALPHA_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_ALPHA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_BETA_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7_B_BETA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_BETA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_BETA_1_1_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7_B_BETA_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_7B_BETA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_ORPO_141B_A35B_V01_1_0_0")
    def HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_ORPO_141_B_A35_B_V01_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_HUGGINGFACEH4_ZEPHYR_ORPO_141B_A35B_V01_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_LLAMA_3_8B_INSTRUCT_GRADIENT_1_0_0")
    def HUGGINGFACE_LLM_LLAMA_3_8_B_INSTRUCT_GRADIENT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_LLAMA_3_8B_INSTRUCT_GRADIENT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_1_0_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_1_1_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_0_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_0_1")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_0_2")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_1_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_2_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_2_1")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_3_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_3_1")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_3_2")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_4_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_4_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_4_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_2_5_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_2_5_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_2_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_2_0_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_INSTRUCT_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_2_1_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_INSTRUCT_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_3_0_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_INSTRUCT_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_3_1_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_INSTRUCT_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_INSTRUCT_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_OPENORCA_GPTQ_1_0_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_OPENORCA_GPTQ_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_OPENORCA_GPTQ_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRAL_7B_OPENORCA_GPTQ_1_1_0")
    def HUGGINGFACE_LLM_MISTRAL_7_B_OPENORCA_GPTQ_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRAL_7B_OPENORCA_GPTQ_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MISTRALAI_MIXTRAL_8X22B_INSTRUCT_V0_1_1_0_0")
    def HUGGINGFACE_LLM_MISTRALAI_MIXTRAL_8_X22_B_INSTRUCT_V0_1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MISTRALAI_MIXTRAL_8X22B_INSTRUCT_V0_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_0_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_0_1")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_1_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_2_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_2_1")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_3_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_3_1")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_3_2")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_4_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_4_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_4_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_1_5_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_1_5_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_1_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_0_1")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_1_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_2_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_2_1")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_3_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_3_1")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_3_2")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_4_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_4_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_4_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_5_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_1_5_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_1_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_GPTQ_1_0_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_GPTQ_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_GPTQ_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_GPTQ_1_1_0")
    def HUGGINGFACE_LLM_MIXTRAL_8_X7_B_INSTRUCT_GPTQ_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_MIXTRAL_8X7B_INSTRUCT_GPTQ_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NEXAAIDEV_OCTOPUS_V2_1_0_0")
    def HUGGINGFACE_LLM_NEXAAIDEV_OCTOPUS_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NEXAAIDEV_OCTOPUS_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NEXUSFLOW_STARLING_LM_7B_BETA_1_0_0")
    def HUGGINGFACE_LLM_NEXUSFLOW_STARLING_LM_7_B_BETA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NEXUSFLOW_STARLING_LM_7B_BETA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_HERMES_2_PRO_LLAMA_3_8B_1_0_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_HERMES_2_PRO_LLAMA_3_8_B_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_HERMES_2_PRO_LLAMA_3_8B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_2_SOLAR_10_7B_1_0_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_2_SOLAR_10_7_B_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_2_SOLAR_10_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_2_SOLAR_10_7B_1_1_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_2_SOLAR_10_7_B_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_2_SOLAR_10_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA_2_7B_1_0_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA_2_7_B_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA_2_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA_2_7B_1_1_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA_2_7_B_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA_2_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13B_1_0_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13_B_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13B_1_1_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13_B_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_YARN_MISTRAL_7B_128K_1_0_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_YARN_MISTRAL_7_B_128_K_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_YARN_MISTRAL_7B_128K_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NOUSRESEARCH_YARN_MISTRAL_7B_128K_1_1_0")
    def HUGGINGFACE_LLM_NOUSRESEARCH_YARN_MISTRAL_7_B_128_K_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NOUSRESEARCH_YARN_MISTRAL_7B_128K_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NVIDIA_LLAMA3_CHATQA_1_5_70B_1_0_0")
    def HUGGINGFACE_LLM_NVIDIA_LLAMA3_CHATQA_1_5_70_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NVIDIA_LLAMA3_CHATQA_1_5_70B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_NVIDIA_LLAMA3_CHATQA_1_5_8B_1_0_0")
    def HUGGINGFACE_LLM_NVIDIA_LLAMA3_CHATQA_1_5_8_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_NVIDIA_LLAMA3_CHATQA_1_5_8B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_OPENLM_RESEARCH_OPEN_LLAMA_7B_V2_1_0_0")
    def HUGGINGFACE_LLM_OPENLM_RESEARCH_OPEN_LLAMA_7_B_V2_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_OPENLM_RESEARCH_OPEN_LLAMA_7B_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_OPENLM_RESEARCH_OPEN_LLAMA_7B_V2_1_1_0")
    def HUGGINGFACE_LLM_OPENLM_RESEARCH_OPEN_LLAMA_7_B_V2_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_OPENLM_RESEARCH_OPEN_LLAMA_7B_V2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_PHI_2_1_0_0")
    def HUGGINGFACE_LLM_PHI_2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_PHI_2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_PHI_3_MINI_128K_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_PHI_3_MINI_128_K_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_PHI_3_MINI_128K_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_PHI_3_MINI_4K_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_PHI_3_MINI_4_K_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_PHI_3_MINI_4K_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_0_0")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_1_0")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_1_1")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_2_0")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_3_0")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_2_0_0")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_2_1_0")
    def HUGGINGFACE_LLM_RINNA_3_6_B_INSTRUCTION_PPO_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_RINNA_3_6B_INSTRUCTION_PPO_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_SEALION_3B_1_0_0")
    def HUGGINGFACE_LLM_SEALION_3_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_SEALION_3B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_SEALION_7B_1_0_0")
    def HUGGINGFACE_LLM_SEALION_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_SEALION_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_SEALION_7B_INSTRUCT_1_0_0")
    def HUGGINGFACE_LLM_SEALION_7_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_SEALION_7B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_SHENZHI_WANG_LLAMA3_8B_CHINESE_CHAT_1_0_0")
    def HUGGINGFACE_LLM_SHENZHI_WANG_LLAMA3_8_B_CHINESE_CHAT_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_SHENZHI_WANG_LLAMA3_8B_CHINESE_CHAT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_STARCODER_1_0_0")
    def HUGGINGFACE_LLM_STARCODER_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_STARCODER_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_STARCODER_1_1_0")
    def HUGGINGFACE_LLM_STARCODER_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_STARCODER_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_STARCODERBASE_1_0_0")
    def HUGGINGFACE_LLM_STARCODERBASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_STARCODERBASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_STARCODERBASE_1_1_0")
    def HUGGINGFACE_LLM_STARCODERBASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_STARCODERBASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TEKNIUM_OPENHERMES_2_MISTRAL_7B_1_0_0")
    def HUGGINGFACE_LLM_TEKNIUM_OPENHERMES_2_MISTRAL_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TEKNIUM_OPENHERMES_2_MISTRAL_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TEKNIUM_OPENHERMES_2_MISTRAL_7B_1_1_0")
    def HUGGINGFACE_LLM_TEKNIUM_OPENHERMES_2_MISTRAL_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TEKNIUM_OPENHERMES_2_MISTRAL_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_THEBLOKE_MISTRAL_7B_OPENORCA_AWQ_1_0_0")
    def HUGGINGFACE_LLM_THEBLOKE_MISTRAL_7_B_OPENORCA_AWQ_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_THEBLOKE_MISTRAL_7B_OPENORCA_AWQ_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_THEBLOKE_MISTRAL_7B_OPENORCA_AWQ_1_1_0")
    def HUGGINGFACE_LLM_THEBLOKE_MISTRAL_7_B_OPENORCA_AWQ_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_THEBLOKE_MISTRAL_7B_OPENORCA_AWQ_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TIIUAE_FALCON_RW_1B_1_0_0")
    def HUGGINGFACE_LLM_TIIUAE_FALCON_RW_1_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TIIUAE_FALCON_RW_1B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TIIUAE_FALCON_RW_1B_1_1_0")
    def HUGGINGFACE_LLM_TIIUAE_FALCON_RW_1_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TIIUAE_FALCON_RW_1B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TINYLLAMA_1_1B_INTERMEDIATE_STEP_1431K_3_1_0_0")
    def HUGGINGFACE_LLM_TINYLLAMA_1_1_B_INTERMEDIATE_STEP_1431_K_3_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TINYLLAMA_1_1B_INTERMEDIATE_STEP_1431K_3_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TINYLLAMA_1_1B_INTERMEDIATE_STEP_1431K_3_1_1_0")
    def HUGGINGFACE_LLM_TINYLLAMA_1_1_B_INTERMEDIATE_STEP_1431_K_3_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TINYLLAMA_1_1B_INTERMEDIATE_STEP_1431K_3_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V0_6_1_0_0")
    def HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1_B_CHAT_V0_6_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V0_6_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V0_6_1_1_0")
    def HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1_B_CHAT_V0_6_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V0_6_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V1_0_1_0_0")
    def HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1_B_CHAT_V1_0_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V1_0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V1_0_1_1_0")
    def HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1_B_CHAT_V1_0_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_TINYLLAMA_TINYLLAMA_1_1B_CHAT_V1_0_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_WRITER_PALMYRA_SMALL_1_0_0")
    def HUGGINGFACE_LLM_WRITER_PALMYRA_SMALL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_WRITER_PALMYRA_SMALL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_WRITER_PALMYRA_SMALL_1_1_0")
    def HUGGINGFACE_LLM_WRITER_PALMYRA_SMALL_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_WRITER_PALMYRA_SMALL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_YI_1_5_34B_1_0_0")
    def HUGGINGFACE_LLM_YI_1_5_34_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_YI_1_5_34B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_YI_1_5_34B_CHAT_1_0_0")
    def HUGGINGFACE_LLM_YI_1_5_34_B_CHAT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_YI_1_5_34B_CHAT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_YI_1_5_6B_1_0_0")
    def HUGGINGFACE_LLM_YI_1_5_6_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_YI_1_5_6B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_YI_1_5_6B_CHAT_1_0_0")
    def HUGGINGFACE_LLM_YI_1_5_6_B_CHAT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_YI_1_5_6B_CHAT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_YI_1_5_9B_1_0_0")
    def HUGGINGFACE_LLM_YI_1_5_9_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_YI_1_5_9B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_YI_1_5_9B_CHAT_1_0_0")
    def HUGGINGFACE_LLM_YI_1_5_9_B_CHAT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_YI_1_5_9B_CHAT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_LLM_ZEPHYR_7B_GEMMA_1_0_0")
    def HUGGINGFACE_LLM_ZEPHYR_7_B_GEMMA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_LLM_ZEPHYR_7B_GEMMA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_1_0_0")
    def HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_1_1_0")
    def HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_2_0_0")
    def HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_NER_DISTILBERT_BASE_CASED_FINETUNED_CONLL03_ENGLISH_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_1_0_0")
    def HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_1_1_0")
    def HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_2_0_0")
    def HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_NER_DISTILBERT_BASE_UNCASED_FINETUNED_CONLL03_ENGLISH_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_1_0")
    def HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_1_1")
    def HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_ALL_MINILM_L6_V2_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_V1_5_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_V1_5_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_V1_5_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_V1_5_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_V1_5_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_BASE_EN_V1_5_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_V1_5_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_V1_5_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_V1_5_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_V1_5_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_V1_5_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_EN_V1_5_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_ZH_V1_5_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_ZH_V1_5_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_ZH_V1_5_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_ZH_V1_5_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_ZH_V1_5_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_LARGE_ZH_V1_5_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_M3_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_M3_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_M3_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_M3_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_M3_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_M3_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_V1_5_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_V1_5_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_V1_5_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_V1_5_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_V1_5_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_BGE_SMALL_EN_V1_5_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_BASE_V2_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_LARGE_V2_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_E5_SMALL_V2_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_BASE_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_3_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_LARGE_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_GTE_SMALL_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_2_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_BASE_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_1_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_1_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_2_0_0")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_2_0_1")
    def HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_2_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SENTENCESIMILARITY_MULTILINGUAL_E5_LARGE_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_1_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_1_1_0")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_0")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_1")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_2")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_3")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_CASED_2_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_1_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_1")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_2")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_3")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_2_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_1_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_1")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_2")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_3")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_1_0")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_0")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_1")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_2")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_3")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_SPC_BERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_1_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_1_1_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_1")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_2")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_3")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_2_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_1_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_1")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_2")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_3")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_1_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_1")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_2")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_3")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_2_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_1_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_1")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_2")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_3")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0")
    def HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_0_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_1_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_1")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_2")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_3")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_2_0_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_1_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_1")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_2")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_3")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_1_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_1")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_2")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_3")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILBERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_0_0")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_1_0")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_0")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_1")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_2")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_3")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_DISTILROBERTA_BASE_2_0_0")
    def HUGGINGFACE_SPC_DISTILROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_DISTILROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_1_0_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_1_1_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_1_2_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_1_2_1")
    def HUGGINGFACE_SPC_ROBERTA_BASE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_1_2_2")
    def HUGGINGFACE_SPC_ROBERTA_BASE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_1_2_3")
    def HUGGINGFACE_SPC_ROBERTA_BASE_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_2_0_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_1_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_1")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_2")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_3")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0")
    def HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_1_0_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_1_1_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_1")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_2")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_3")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_2_0_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_1_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_1")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_2")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_3")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_2_0_0")
    def HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_ROBERTA_LARGE_OPENAI_DETECTOR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_0_0")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_1_0")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_0")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_1")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_2")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_3")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_2_0_0")
    def HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_CLM_ENDE_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_1_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_1")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_2")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_3")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_2_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENDE_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_1_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_1")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_2")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_3")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_2_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_ENRO_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_1_0")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_0")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_1")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_2")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_3")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_2_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_TLM_XNLI15_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_1_0")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_0")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_1")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_2")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_3")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_2_0_0")
    def HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SPC_XLM_MLM_XNLI15_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BART_LARGE_CNN_SAMSUM_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_BERT_SMALL2_BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_BERT_SMALL2_BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_BERT_SMALL2_BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_BERT_SMALL2_BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BERT_SMALL2BERT_SMALL_FINETUNED_CNN_DAILY_MAIL_SUMMARIZATION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_ARXIV_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_BIGBIRD_PEGASUS_LARGE_PUBMED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_12_6_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_CNN_6_6_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_1_1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_1_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_2_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_2_0_0")
    def HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_SUMMARIZATION_DISTILBART_XSUM_12_3_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_CASED_1_0_0")
    def HUGGINGFACE_TC_BERT_BASE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_CASED_1_0_1")
    def HUGGINGFACE_TC_BERT_BASE_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_CASED_1_0_2")
    def HUGGINGFACE_TC_BERT_BASE_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_CASED_2_0_0")
    def HUGGINGFACE_TC_BERT_BASE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_0")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_1")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_2")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_2_0_0")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_1")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_2")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0")
    def HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_MULTILINGUAL_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_1")
    def HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_2")
    def HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_TC_BERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_1")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_2")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_2_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_1")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_2")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_CASED_WHOLE_WORD_MASKING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_1")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_2")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_2_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_1")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_2")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0")
    def HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_BERT_LARGE_UNCASED_WHOLE_WORD_MASKING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_0")
    def HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_1")
    def HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_2")
    def HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_CASED_2_0_0")
    def HUGGINGFACE_TC_DISTILBERT_BASE_CASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0")
    def HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_1")
    def HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_2")
    def HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0")
    def HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_MULTILINGUAL_CASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_0")
    def HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_1")
    def HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_2")
    def HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_2_0_0")
    def HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILBERT_BASE_UNCASED_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_0")
    def HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_1")
    def HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_2")
    def HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILROBERTA_BASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_DISTILROBERTA_BASE_2_0_0")
    def HUGGINGFACE_TC_DISTILROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_DISTILROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_MODELS_1_0_0")
    def HUGGINGFACE_TC_MODELS_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_MODELS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_MODELS_1_0_1")
    def HUGGINGFACE_TC_MODELS_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_MODELS_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_MODELS_1_0_2")
    def HUGGINGFACE_TC_MODELS_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_MODELS_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_MODELS_2_0_0")
    def HUGGINGFACE_TC_MODELS_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_MODELS_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_1_0_0")
    def HUGGINGFACE_TC_ROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_1_0_1")
    def HUGGINGFACE_TC_ROBERTA_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_1_0_2")
    def HUGGINGFACE_TC_ROBERTA_BASE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_2_0_0")
    def HUGGINGFACE_TC_ROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0")
    def HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_1")
    def HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_2")
    def HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0")
    def HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_BASE_OPENAI_DETECTOR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_1_0_0")
    def HUGGINGFACE_TC_ROBERTA_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_1_0_1")
    def HUGGINGFACE_TC_ROBERTA_LARGE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_1_0_2")
    def HUGGINGFACE_TC_ROBERTA_LARGE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_2_0_0")
    def HUGGINGFACE_TC_ROBERTA_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_0")
    def HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_1")
    def HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_2")
    def HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_2_0_0")
    def HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_ROBERTA_LARGE_OPENAI_DETECTOR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_0")
    def HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_1")
    def HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_2")
    def HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_CLM_ENDE_1024_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_CLM_ENDE_1024_2_0_0")
    def HUGGINGFACE_TC_XLM_CLM_ENDE_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_CLM_ENDE_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_0")
    def HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_1")
    def HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_2")
    def HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENDE_1024_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENDE_1024_2_0_0")
    def HUGGINGFACE_TC_XLM_MLM_ENDE_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENDE_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_0")
    def HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_1")
    def HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_2")
    def HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENRO_1024_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_ENRO_1024_2_0_0")
    def HUGGINGFACE_TC_XLM_MLM_ENRO_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_ENRO_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_0")
    def HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_1")
    def HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_2")
    def HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_2_0_0")
    def HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TC_XLM_MLM_TLM_XNLI15_1024_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_BART4_CSC_BASE_CHINESE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BART4CSC_BASE_CHINESE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_BNB_INT8_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_BNB_INT8_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_BNB_INT8_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_BNB_INT8_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_BNB_INT8_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_BNB_INT8_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_BNB_INT8_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_BNB_INT8_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_BNB_INT8_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_FP16_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_FP16_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_FP16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_FP16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_FP16_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_BIGSCIENCE_T0_PP_FP16_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_BIGSCIENCE_T0PP_FP16_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_4")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_5")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_2_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_2_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_3_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_3_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_3_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_BASE_SAMSUM_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_BASE_SAMSUM_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_4")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_5")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_6")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_1_6(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_1_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_LARGE_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_LARGE_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_4")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_5")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_2_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_2_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_3_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_3_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_3_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_3_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_1_3_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_1_3_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_SMALL_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_SMALL_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_3")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_4")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_5")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_6")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_1_6(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_1_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XL_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XL_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_BNB_INT8_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_BNB_INT8_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_BNB_INT8_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_BNB_INT8_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_BNB_INT8_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_FP16_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_FP16_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_FP16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_FP16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_FP16_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_T5_XXL_FP16_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_T5_XXL_FP16_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_UL2_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_UL2_BF16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_1_1_2")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_UL2_BF16_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_UL2_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_FLAN_UL2_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_FLAN_UL2_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_PEGASUS_PARAPHRASE_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_PEGASUS_PARAPHRASE_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_QCPG_SENTENCES_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_QCPG_SENTENCES_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_0")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_1")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_2")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_3")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_1_0")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_1_1")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_2_0")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_2_1")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_2_2")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_2_0_0")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_2_1_0")
    def HUGGINGFACE_TEXT2_TEXT_T5_ONE_LINE_SUMMARY_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXT2TEXT_T5_ONE_LINE_SUMMARY_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTEMBEDDING_ALL_MINILM_L6_V2_1_0_0")
    def HUGGINGFACE_TEXTEMBEDDING_ALL_MINILM_L6_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTEMBEDDING_ALL_MINILM_L6_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_2")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_3")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_2_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_3_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B1_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B1_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_2")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_3")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_2_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_3_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_1_B7_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_1B7_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_2")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_3")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_2_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_3_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOM_560M_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOM_560_M_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOM_560M_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B1_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B1_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_1_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B1_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B1_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B1_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B1_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B1_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B7_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B7_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_1_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B7_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B7_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B7_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_1_B7_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_1B7_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_560_M_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_560_M_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_1_1")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_560_M_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_560_M_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_560_M_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_BLOOMZ_560_M_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_BLOOMZ_560M_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_1")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_2")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_3")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_3_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_4_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_4_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_4_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_4_1")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_4_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_4_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_5_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_5_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_1_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DISTILGPT2_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_DISTILGPT2_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DISTILGPT2_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12_B_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12_B_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12_B_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_12B_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3_B_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3_B_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3_B_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_3B_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7_B_BF16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7_B_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7_B_BF16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_DOLLY_V2_7B_BF16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_FALCON_40B_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_FALCON_40_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_FALCON_40B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_FALCON_40B_INSTRUCT_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_FALCON_40_B_INSTRUCT_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_FALCON_40B_INSTRUCT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_FALCON_7B_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_FALCON_7_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_FALCON_7B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_FALCON_7B_INSTRUCT_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_FALCON_7_B_INSTRUCT_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_FALCON_7B_INSTRUCT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_2_1")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_2_2")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_2_3")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_3_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_4_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_4_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_4_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_4_1")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_4_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_4_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_1_5_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_1_5_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_1_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_GPT2_2_1_0")
    def HUGGINGFACE_TEXTGENERATION_GPT2_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_GPT2_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_0_1")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_0_2")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_2_1")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_MODELS_1_3_0")
    def HUGGINGFACE_TEXTGENERATION_MODELS_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_MODELS_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_0_0")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_1_0")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_1_1")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_2_0")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_2_0_0")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_3_0_0")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_3_1_0")
    def HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION_OPEN_LLAMA_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_176B_INT8_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_176_B_INT8_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_176B_INT8_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_176B_INT8_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_176_B_INT8_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_176B_INT8_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_176B_INT8_1_0_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_176_B_INT8_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_176B_INT8_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_3_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_3B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOM_7_B1_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOM_7B1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176_B_FP16_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_1_0_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176_B_FP16_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_2_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176_B_FP16_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_176B_FP16_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3_B_FP16_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_3B_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7_B1_FP16_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_BLOOMZ_7B1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_2_XL_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_1_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_3")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_4")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_1_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_3")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_4")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_2_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_J_6_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_J_6B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_1_3B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125_M_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_125M_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_2_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_2_1")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_2_2")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_3_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_GPT_NEO_2_7B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_LIGHTGPT_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_BF16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_BF16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_BF16_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_BF16_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_BF16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_INSTRUCT_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_INSTRUCT_BF16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_INSTRUCT_BF16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_INSTRUCT_BF16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_INSTRUCT_BF16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_STORYWRITER_BF16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_STORYWRITER_BF16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_STORYWRITER_BF16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_MPT_7_B_STORYWRITER_BF16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_MPT_7B_STORYWRITER_BF16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3_B_V1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3_B_V1_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3_B_V1_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3_B_V1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3_B_V1_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3_B_V1_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_3B_V1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7_B_V1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7_B_V1_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7_B_V1_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7_B_V1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7_B_V1_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7_B_V1_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_BASE_7B_V1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3_B_V1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3_B_V1_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3_B_V1_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3_B_V1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3_B_V1_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3_B_V1_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_3B_V1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7_B_V1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7_B_V1_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7_B_V1_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7_B_V1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7_B_V1_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7_B_V1_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_CHAT_7B_V1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3_B_V1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3_B_V1_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3_B_V1_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3_B_V1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3_B_V1_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3_B_V1_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_3B_V1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7_B_V1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7_B_V1_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_1_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7_B_V1_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7_B_V1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7_B_V1_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7_B_V1_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION1_REDPAJAMA_INCITE_INSTRUCT_7B_V1_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20_B_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20_B_FP16_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20_B_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20_B_FP16_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20_B_FP16_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOX_20B_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_1_0_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20_B_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_1_0_1")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20_B_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_2_0_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20_B_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_3_0_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20_B_FP16_3_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_3_1_0")
    def HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20_B_FP16_3_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TEXTGENERATION2_GPT_NEOXT_CHAT_BASE_20B_FP16_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_1_0_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_1_1_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_2_0_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_EN_ES_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_1_0_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_1_1_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_2_0_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_EN_VI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_OPUS_MT_MUL_EN_1_0_0")
    def HUGGINGFACE_TRANSLATION_OPUS_MT_MUL_EN_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_OPUS_MT_MUL_EN_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_BASE_1_0_0")
    def HUGGINGFACE_TRANSLATION_T5_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_BASE_1_1_0")
    def HUGGINGFACE_TRANSLATION_T5_BASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_BASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_BASE_2_0_0")
    def HUGGINGFACE_TRANSLATION_T5_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_LARGE_1_0_0")
    def HUGGINGFACE_TRANSLATION_T5_LARGE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_LARGE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_LARGE_1_1_0")
    def HUGGINGFACE_TRANSLATION_T5_LARGE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_LARGE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_LARGE_2_0_0")
    def HUGGINGFACE_TRANSLATION_T5_LARGE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_LARGE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_SMALL_1_0_0")
    def HUGGINGFACE_TRANSLATION_T5_SMALL_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_SMALL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_SMALL_1_1_0")
    def HUGGINGFACE_TRANSLATION_T5_SMALL_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_SMALL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TRANSLATION_T5_SMALL_2_0_0")
    def HUGGINGFACE_TRANSLATION_T5_SMALL_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TRANSLATION_T5_SMALL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_22H_VINTEDOIS_DIFFUSION_V0_1_1_0_0")
    def HUGGINGFACE_TXT2_IMG_22_H_VINTEDOIS_DIFFUSION_V0_1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_22H_VINTEDOIS_DIFFUSION_V0_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_22H_VINTEDOIS_DIFFUSION_V0_1_1_1_0")
    def HUGGINGFACE_TXT2_IMG_22_H_VINTEDOIS_DIFFUSION_V0_1_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_22H_VINTEDOIS_DIFFUSION_V0_1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_22H_VINTEDOIS_DIFFUSION_V0_1_2_0_0")
    def HUGGINGFACE_TXT2_IMG_22_H_VINTEDOIS_DIFFUSION_V0_1_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_22H_VINTEDOIS_DIFFUSION_V0_1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AKIKAGURA_MKGEN_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_AKIKAGURA_MKGEN_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AKIKAGURA_MKGEN_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AKIKAGURA_MKGEN_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_AKIKAGURA_MKGEN_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AKIKAGURA_MKGEN_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AKIKAGURA_MKGEN_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_AKIKAGURA_MKGEN_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AKIKAGURA_MKGEN_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES_FASTDB_4800_1_0_0")
    def HUGGINGFACE_TXT2_IMG_ALXDFY_NOGGLES_FASTDB_4800_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES_FASTDB_4800_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES_FASTDB_4800_1_1_0")
    def HUGGINGFACE_TXT2_IMG_ALXDFY_NOGGLES_FASTDB_4800_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES_FASTDB_4800_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES_FASTDB_4800_2_0_0")
    def HUGGINGFACE_TXT2_IMG_ALXDFY_NOGGLES_FASTDB_4800_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES_FASTDB_4800_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES9000_1_0_0")
    def HUGGINGFACE_TXT2_IMG_ALXDFY_NOGGLES9000_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES9000_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES9000_1_1_0")
    def HUGGINGFACE_TXT2_IMG_ALXDFY_NOGGLES9000_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES9000_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES9000_2_0_0")
    def HUGGINGFACE_TXT2_IMG_ALXDFY_NOGGLES9000_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ALXDFY_NOGGLES9000_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ANDITE_ANYTHING_V4_0_1_0_0")
    def HUGGINGFACE_TXT2_IMG_ANDITE_ANYTHING_V4_0_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ANDITE_ANYTHING_V4_0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ANDITE_ANYTHING_V4_0_1_1_0")
    def HUGGINGFACE_TXT2_IMG_ANDITE_ANYTHING_V4_0_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ANDITE_ANYTHING_V4_0_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ANDITE_ANYTHING_V4_0_2_0_0")
    def HUGGINGFACE_TXT2_IMG_ANDITE_ANYTHING_V4_0_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ANDITE_ANYTHING_V4_0_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_1_0_0")
    def HUGGINGFACE_TXT2_IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_1_1_0")
    def HUGGINGFACE_TXT2_IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_2_0_0")
    def HUGGINGFACE_TXT2_IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ASTRALITEHEART_PONY_DIFFUSION_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_1_0_0")
    def HUGGINGFACE_TXT2_IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_1_1_0")
    def HUGGINGFACE_TXT2_IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_2_0_0")
    def HUGGINGFACE_TXT2_IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AVRIK_ABSTRACT_ANIM_SPRITESHEETS_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AYBEECEEDEE_KNOLLINGCASE_1_0_0")
    def HUGGINGFACE_TXT2_IMG_AYBEECEEDEE_KNOLLINGCASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AYBEECEEDEE_KNOLLINGCASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AYBEECEEDEE_KNOLLINGCASE_1_1_0")
    def HUGGINGFACE_TXT2_IMG_AYBEECEEDEE_KNOLLINGCASE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AYBEECEEDEE_KNOLLINGCASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_AYBEECEEDEE_KNOLLINGCASE_2_0_0")
    def HUGGINGFACE_TXT2_IMG_AYBEECEEDEE_KNOLLINGCASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_AYBEECEEDEE_KNOLLINGCASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BINGSU_MY_K_ANYTHING_V3_0_1_0_0")
    def HUGGINGFACE_TXT2_IMG_BINGSU_MY_K_ANYTHING_V3_0_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BINGSU_MY_K_ANYTHING_V3_0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BINGSU_MY_K_ANYTHING_V3_0_1_1_0")
    def HUGGINGFACE_TXT2_IMG_BINGSU_MY_K_ANYTHING_V3_0_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BINGSU_MY_K_ANYTHING_V3_0_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BINGSU_MY_K_ANYTHING_V3_0_2_0_0")
    def HUGGINGFACE_TXT2_IMG_BINGSU_MY_K_ANYTHING_V3_0_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BINGSU_MY_K_ANYTHING_V3_0_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_1_0_0")
    def HUGGINGFACE_TXT2_IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_1_1_0")
    def HUGGINGFACE_TXT2_IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_2_0_0")
    def HUGGINGFACE_TXT2_IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BINGSU_MY_KOREAN_STABLE_DIFFUSION_V1_5_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_1_0_0")
    def HUGGINGFACE_TXT2_IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_1_1_0")
    def HUGGINGFACE_TXT2_IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_2_0_0")
    def HUGGINGFACE_TXT2_IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_BUNTOPSIH_NOVGORANSTEFANOVSKI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_1_0_0")
    def HUGGINGFACE_TXT2_IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_1_1_0")
    def HUGGINGFACE_TXT2_IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_2_0_0")
    def HUGGINGFACE_TXT2_IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CLAUDFUEN_PHOTOREALISTIC_FUEN_V1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CODER119_VECTORARTZ_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_CODER119_VECTORARTZ_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CODER119_VECTORARTZ_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CODER119_VECTORARTZ_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_CODER119_VECTORARTZ_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CODER119_VECTORARTZ_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CODER119_VECTORARTZ_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_CODER119_VECTORARTZ_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CODER119_VECTORARTZ_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CONFLICTX_COMPLEX_LINEART_1_0_0")
    def HUGGINGFACE_TXT2_IMG_CONFLICTX_COMPLEX_LINEART_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CONFLICTX_COMPLEX_LINEART_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CONFLICTX_COMPLEX_LINEART_1_1_0")
    def HUGGINGFACE_TXT2_IMG_CONFLICTX_COMPLEX_LINEART_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CONFLICTX_COMPLEX_LINEART_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_CONFLICTX_COMPLEX_LINEART_2_0_0")
    def HUGGINGFACE_TXT2_IMG_CONFLICTX_COMPLEX_LINEART_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_CONFLICTX_COMPLEX_LINEART_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_CATS_MUSICAL_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_JWST_DEEP_SPACE_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_TRON_LEGACY_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DALLINMACKAY_VAN_GOGH_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DGSPITZER_CYBERPUNK_ANIME_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_1_0_0")
    def HUGGINGFACE_TXT2_IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_1_1_0")
    def HUGGINGFACE_TXT2_IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_2_0_0")
    def HUGGINGFACE_TXT2_IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_DREAMLIKE_ART_DREAMLIKE_DIFFUSION_1_0_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EIMISS_EIMISANIMEDIFFUSION_1_0V_1_0_0")
    def HUGGINGFACE_TXT2_IMG_EIMISS_EIMISANIMEDIFFUSION_1_0_V_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EIMISS_EIMISANIMEDIFFUSION_1_0V_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EIMISS_EIMISANIMEDIFFUSION_1_0V_1_1_0")
    def HUGGINGFACE_TXT2_IMG_EIMISS_EIMISANIMEDIFFUSION_1_0_V_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EIMISS_EIMISANIMEDIFFUSION_1_0V_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EIMISS_EIMISANIMEDIFFUSION_1_0V_2_0_0")
    def HUGGINGFACE_TXT2_IMG_EIMISS_EIMISANIMEDIFFUSION_1_0_V_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EIMISS_EIMISANIMEDIFFUSION_1_0V_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ENVVI_INKPUNK_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_ENVVI_INKPUNK_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ENVVI_INKPUNK_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ENVVI_INKPUNK_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_ENVVI_INKPUNK_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ENVVI_INKPUNK_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_ENVVI_INKPUNK_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_ENVVI_INKPUNK_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_ENVVI_INKPUNK_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EVEL_YOSHIN_1_0_0")
    def HUGGINGFACE_TXT2_IMG_EVEL_YOSHIN_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EVEL_YOSHIN_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EVEL_YOSHIN_1_1_0")
    def HUGGINGFACE_TXT2_IMG_EVEL_YOSHIN_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EVEL_YOSHIN_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EVEL_YOSHIN_2_0_0")
    def HUGGINGFACE_TXT2_IMG_EVEL_YOSHIN_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EVEL_YOSHIN_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_1_0_0")
    def HUGGINGFACE_TXT2_IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_1_1_0")
    def HUGGINGFACE_TXT2_IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_2_0_0")
    def HUGGINGFACE_TXT2_IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_EXTRAPHY_MUSTAFA_KEMAL_ATATURK_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_1_0_0")
    def HUGGINGFACE_TXT2_IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_1_1_0")
    def HUGGINGFACE_TXT2_IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_2_0_0")
    def HUGGINGFACE_TXT2_IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FFFILONI_MR_MEN_AND_LITTLE_MISSES_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_ELRISITAS_1_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_ELRISITAS_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_ELRISITAS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_ELRISITAS_1_1_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_ELRISITAS_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_ELRISITAS_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_ELRISITAS_2_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_ELRISITAS_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_ELRISITAS_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_1_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_1_1_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_2_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_BALLOONART_MODEL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_1_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_1_1_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_2_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_MICROSCOPIC_MODEL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_1_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_1_1_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_2_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_PAPERCUT_MODEL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_1_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_1_1_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_2_0_0")
    def HUGGINGFACE_TXT2_IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_FICTIVERSE_STABLE_DIFFUSION_VOXELART_MODEL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_HAOR_EVT_V3_1_0_0")
    def HUGGINGFACE_TXT2_IMG_HAOR_EVT_V3_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_HAOR_EVT_V3_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_HAOR_EVT_V3_1_1_0")
    def HUGGINGFACE_TXT2_IMG_HAOR_EVT_V3_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_HAOR_EVT_V3_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_HAOR_EVT_V3_2_0_0")
    def HUGGINGFACE_TXT2_IMG_HAOR_EVT_V3_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_HAOR_EVT_V3_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_HASSANBLEND_HASSANBLEND1_4_1_0_0")
    def HUGGINGFACE_TXT2_IMG_HASSANBLEND_HASSANBLEND1_4_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_HASSANBLEND_HASSANBLEND1_4_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_HASSANBLEND_HASSANBLEND1_4_1_1_0")
    def HUGGINGFACE_TXT2_IMG_HASSANBLEND_HASSANBLEND1_4_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_HASSANBLEND_HASSANBLEND1_4_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_HASSANBLEND_HASSANBLEND1_4_2_0_0")
    def HUGGINGFACE_TXT2_IMG_HASSANBLEND_HASSANBLEND1_4_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_HASSANBLEND_HASSANBLEND1_4_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_EN_V0_1_1_0_0")
    def HUGGINGFACE_TXT2_IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1_B_CHINESE_EN_V0_1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_EN_V0_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_EN_V0_1_1_1_0")
    def HUGGINGFACE_TXT2_IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1_B_CHINESE_EN_V0_1_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_EN_V0_1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_EN_V0_1_2_0_0")
    def HUGGINGFACE_TXT2_IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1_B_CHINESE_EN_V0_1_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_EN_V0_1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_V0_1_1_0_0")
    def HUGGINGFACE_TXT2_IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1_B_CHINESE_V0_1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_V0_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_V0_1_1_1_0")
    def HUGGINGFACE_TXT2_IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1_B_CHINESE_V0_1_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_V0_1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_V0_1_2_0_0")
    def HUGGINGFACE_TXT2_IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1_B_CHINESE_V0_1_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IDEA_CCNL_TAIYI_STABLE_DIFFUSION_1B_CHINESE_V0_1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IFANSNEK_JOHNDIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_IFANSNEK_JOHNDIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IFANSNEK_JOHNDIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IFANSNEK_JOHNDIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_IFANSNEK_JOHNDIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IFANSNEK_JOHNDIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_IFANSNEK_JOHNDIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_IFANSNEK_JOHNDIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_IFANSNEK_JOHNDIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_JERSONM89_AVATAR_1_0_0")
    def HUGGINGFACE_TXT2_IMG_JERSONM89_AVATAR_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_JERSONM89_AVATAR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_JERSONM89_AVATAR_1_1_0")
    def HUGGINGFACE_TXT2_IMG_JERSONM89_AVATAR_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_JERSONM89_AVATAR_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_JERSONM89_AVATAR_2_0_0")
    def HUGGINGFACE_TXT2_IMG_JERSONM89_AVATAR_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_JERSONM89_AVATAR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_1_0_0")
    def HUGGINGFACE_TXT2_IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_1_1_0")
    def HUGGINGFACE_TXT2_IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_2_0_0")
    def HUGGINGFACE_TXT2_IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_JVKAPE_ICONSMI_APPICONSMODELFORSD_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_KATAKANA_2D_MIX_1_0_0")
    def HUGGINGFACE_TXT2_IMG_KATAKANA_2_D_MIX_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_KATAKANA_2D_MIX_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_KATAKANA_2D_MIX_1_1_0")
    def HUGGINGFACE_TXT2_IMG_KATAKANA_2_D_MIX_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_KATAKANA_2D_MIX_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_KATAKANA_2D_MIX_2_0_0")
    def HUGGINGFACE_TXT2_IMG_KATAKANA_2_D_MIX_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_KATAKANA_2D_MIX_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LACAMBRE_VULVINE_LOOK_V02_1_0_0")
    def HUGGINGFACE_TXT2_IMG_LACAMBRE_VULVINE_LOOK_V02_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LACAMBRE_VULVINE_LOOK_V02_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LACAMBRE_VULVINE_LOOK_V02_1_1_0")
    def HUGGINGFACE_TXT2_IMG_LACAMBRE_VULVINE_LOOK_V02_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LACAMBRE_VULVINE_LOOK_V02_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LACAMBRE_VULVINE_LOOK_V02_2_0_0")
    def HUGGINGFACE_TXT2_IMG_LACAMBRE_VULVINE_LOOK_V02_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LACAMBRE_VULVINE_LOOK_V02_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LANGBOAT_GUOHUA_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_LANGBOAT_GUOHUA_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LANGBOAT_GUOHUA_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LANGBOAT_GUOHUA_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_LANGBOAT_GUOHUA_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LANGBOAT_GUOHUA_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LANGBOAT_GUOHUA_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_LANGBOAT_GUOHUA_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LANGBOAT_GUOHUA_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LINAQRUF_ANYTHING_V3_0_1_0_0")
    def HUGGINGFACE_TXT2_IMG_LINAQRUF_ANYTHING_V3_0_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LINAQRUF_ANYTHING_V3_0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LINAQRUF_ANYTHING_V3_0_1_1_0")
    def HUGGINGFACE_TXT2_IMG_LINAQRUF_ANYTHING_V3_0_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LINAQRUF_ANYTHING_V3_0_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_LINAQRUF_ANYTHING_V3_0_2_0_0")
    def HUGGINGFACE_TXT2_IMG_LINAQRUF_ANYTHING_V3_0_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_LINAQRUF_ANYTHING_V3_0_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MIKESMODELS_WALTZ_WITH_BASHIR_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITCHTECH_KLINGON_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_MITCHTECH_KLINGON_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITCHTECH_KLINGON_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITCHTECH_KLINGON_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_MITCHTECH_KLINGON_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITCHTECH_KLINGON_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITCHTECH_KLINGON_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_MITCHTECH_KLINGON_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITCHTECH_KLINGON_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITCHTECH_VULCAN_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_MITCHTECH_VULCAN_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITCHTECH_VULCAN_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITCHTECH_VULCAN_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_MITCHTECH_VULCAN_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITCHTECH_VULCAN_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITCHTECH_VULCAN_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_MITCHTECH_VULCAN_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITCHTECH_VULCAN_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITSUA_MITSUA_DIFFUSION_CC0_1_0_0")
    def HUGGINGFACE_TXT2_IMG_MITSUA_MITSUA_DIFFUSION_CC0_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITSUA_MITSUA_DIFFUSION_CC0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITSUA_MITSUA_DIFFUSION_CC0_1_1_0")
    def HUGGINGFACE_TXT2_IMG_MITSUA_MITSUA_DIFFUSION_CC0_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITSUA_MITSUA_DIFFUSION_CC0_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_MITSUA_MITSUA_DIFFUSION_CC0_2_0_0")
    def HUGGINGFACE_TXT2_IMG_MITSUA_MITSUA_DIFFUSION_CC0_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_MITSUA_MITSUA_DIFFUSION_CC0_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NACLBIT_TRINART_STABLE_DIFFUSION_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCANE_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ARCANE_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCANE_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCANE_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ARCANE_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCANE_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCANE_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ARCANE_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCANE_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCHER_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ARCHER_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCHER_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCHER_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ARCHER_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCHER_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCHER_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ARCHER_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ARCHER_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_CLASSIC_ANIM_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_ELDEN_RING_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_FUTURE_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_FUTURE_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_FUTURE_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_FUTURE_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_FUTURE_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_FUTURE_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_FUTURE_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_FUTURE_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_FUTURE_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_GHIBLI_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_GHIBLI_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_GHIBLI_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_GHIBLI_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_GHIBLI_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_GHIBLI_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_GHIBLI_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_GHIBLI_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_GHIBLI_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_MO_DI_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_MO_DI_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_MO_DI_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_MO_DI_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_MO_DI_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_MO_DI_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_MO_DI_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_MO_DI_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_MO_DI_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_NITRO_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_NITRO_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_NITRO_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_NITRO_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_NITRO_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_NITRO_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_NITRO_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_NITRO_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_NITRO_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_REDSHIFT_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_REDSHIFT_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_REDSHIFT_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_REDSHIFT_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_REDSHIFT_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_REDSHIFT_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_REDSHIFT_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_REDSHIFT_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_REDSHIFT_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NITROSOCKE_SPIDER_VERSE_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NOUSR_ROBO_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_NOUSR_ROBO_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NOUSR_ROBO_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NOUSR_ROBO_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_NOUSR_ROBO_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NOUSR_ROBO_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_NOUSR_ROBO_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_NOUSR_ROBO_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_NOUSR_ROBO_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_OGKALU_COMIC_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_OGKALU_COMIC_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_OGKALU_COMIC_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_OGKALU_COMIC_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_OGKALU_COMIC_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_OGKALU_COMIC_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_OGKALU_COMIC_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_OGKALU_COMIC_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_OGKALU_COMIC_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_OPENJOURNEY_OPENJOURNEY_1_0_0")
    def HUGGINGFACE_TXT2_IMG_OPENJOURNEY_OPENJOURNEY_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_OPENJOURNEY_OPENJOURNEY_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_OPENJOURNEY_OPENJOURNEY_1_1_0")
    def HUGGINGFACE_TXT2_IMG_OPENJOURNEY_OPENJOURNEY_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_OPENJOURNEY_OPENJOURNEY_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_OPENJOURNEY_OPENJOURNEY_2_0_0")
    def HUGGINGFACE_TXT2_IMG_OPENJOURNEY_OPENJOURNEY_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_OPENJOURNEY_OPENJOURNEY_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_1_0_0")
    def HUGGINGFACE_TXT2_IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_1_1_0")
    def HUGGINGFACE_TXT2_IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_2_0_0")
    def HUGGINGFACE_TXT2_IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PIESPOSITO_OPENPOTIONBOTTLE_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PLASMO_VOXEL_ISH_1_0_0")
    def HUGGINGFACE_TXT2_IMG_PLASMO_VOXEL_ISH_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PLASMO_VOXEL_ISH_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PLASMO_VOXEL_ISH_1_1_0")
    def HUGGINGFACE_TXT2_IMG_PLASMO_VOXEL_ISH_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PLASMO_VOXEL_ISH_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PLASMO_VOXEL_ISH_2_0_0")
    def HUGGINGFACE_TXT2_IMG_PLASMO_VOXEL_ISH_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PLASMO_VOXEL_ISH_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PLASMO_WOOLITIZE_1_0_0")
    def HUGGINGFACE_TXT2_IMG_PLASMO_WOOLITIZE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PLASMO_WOOLITIZE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PLASMO_WOOLITIZE_1_1_0")
    def HUGGINGFACE_TXT2_IMG_PLASMO_WOOLITIZE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PLASMO_WOOLITIZE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PLASMO_WOOLITIZE_2_0_0")
    def HUGGINGFACE_TXT2_IMG_PLASMO_WOOLITIZE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PLASMO_WOOLITIZE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROGAMERGOV_MIN_ILLUST_BACKGROUND_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROMPTHERO_LINKEDIN_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_PROMPTHERO_LINKEDIN_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROMPTHERO_LINKEDIN_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROMPTHERO_LINKEDIN_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_PROMPTHERO_LINKEDIN_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROMPTHERO_LINKEDIN_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROMPTHERO_LINKEDIN_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_PROMPTHERO_LINKEDIN_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROMPTHERO_LINKEDIN_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROMPTHERO_OPENJOURNEY_1_0_0")
    def HUGGINGFACE_TXT2_IMG_PROMPTHERO_OPENJOURNEY_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROMPTHERO_OPENJOURNEY_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROMPTHERO_OPENJOURNEY_1_1_0")
    def HUGGINGFACE_TXT2_IMG_PROMPTHERO_OPENJOURNEY_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROMPTHERO_OPENJOURNEY_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_PROMPTHERO_OPENJOURNEY_2_0_0")
    def HUGGINGFACE_TXT2_IMG_PROMPTHERO_OPENJOURNEY_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_PROMPTHERO_OPENJOURNEY_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_QILEX_MAGIC_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_QILEX_MAGIC_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_QILEX_MAGIC_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_QILEX_MAGIC_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_QILEX_MAGIC_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_QILEX_MAGIC_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_QILEX_MAGIC_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_QILEX_MAGIC_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_QILEX_MAGIC_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_1_0_0")
    def HUGGINGFACE_TXT2_IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_1_1_0")
    def HUGGINGFACE_TXT2_IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_2_0_0")
    def HUGGINGFACE_TXT2_IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RABIDGREMLIN_SD_DB_EPIC_SPACE_MACHINE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RAYHELL_POPUPBOOK_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_RAYHELL_POPUPBOOK_DIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RAYHELL_POPUPBOOK_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RAYHELL_POPUPBOOK_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_RAYHELL_POPUPBOOK_DIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RAYHELL_POPUPBOOK_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RAYHELL_POPUPBOOK_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_RAYHELL_POPUPBOOK_DIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RAYHELL_POPUPBOOK_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_1_0_0")
    def HUGGINGFACE_TXT2_IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_1_1_0")
    def HUGGINGFACE_TXT2_IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_2_0_0")
    def HUGGINGFACE_TXT2_IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_RUNWAYML_STABLE_DIFFUSION_V1_5_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_S3NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_S3_NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_S3NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_S3NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_S3_NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_S3NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_S3NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_S3_NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_S3NH_BEKSINSKI_STYLE_STABLE_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_1_0_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_1_1_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_2_0_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_ORIGINAL_CHARACTER_CYCLPS_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_1_0_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_1_1_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_2_0_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_PERSONA_5_SHIGENORI_STYLE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_1_0_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_1_1_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_2_0_0")
    def HUGGINGFACE_TXT2_IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SD_DREAMBOOTH_LIBRARY_SERAPHM_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SHIRAYU_SD_TOHOKU_V1_1_0_0")
    def HUGGINGFACE_TXT2_IMG_SHIRAYU_SD_TOHOKU_V1_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SHIRAYU_SD_TOHOKU_V1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SHIRAYU_SD_TOHOKU_V1_1_1_0")
    def HUGGINGFACE_TXT2_IMG_SHIRAYU_SD_TOHOKU_V1_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SHIRAYU_SD_TOHOKU_V1_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_SHIRAYU_SD_TOHOKU_V1_2_0_0")
    def HUGGINGFACE_TXT2_IMG_SHIRAYU_SD_TOHOKU_V1_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_SHIRAYU_SD_TOHOKU_V1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_THELASTBEN_HRRZG_STYLE_768PX_1_0_0")
    def HUGGINGFACE_TXT2_IMG_THELASTBEN_HRRZG_STYLE_768_PX_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_THELASTBEN_HRRZG_STYLE_768PX_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_THELASTBEN_HRRZG_STYLE_768PX_1_1_0")
    def HUGGINGFACE_TXT2_IMG_THELASTBEN_HRRZG_STYLE_768_PX_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_THELASTBEN_HRRZG_STYLE_768PX_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_THELASTBEN_HRRZG_STYLE_768PX_2_0_0")
    def HUGGINGFACE_TXT2_IMG_THELASTBEN_HRRZG_STYLE_768_PX_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_THELASTBEN_HRRZG_STYLE_768PX_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TIMOTHEPEARCE_GINA_THE_CAT_1_0_0")
    def HUGGINGFACE_TXT2_IMG_TIMOTHEPEARCE_GINA_THE_CAT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TIMOTHEPEARCE_GINA_THE_CAT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TIMOTHEPEARCE_GINA_THE_CAT_1_1_0")
    def HUGGINGFACE_TXT2_IMG_TIMOTHEPEARCE_GINA_THE_CAT_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TIMOTHEPEARCE_GINA_THE_CAT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TIMOTHEPEARCE_GINA_THE_CAT_2_0_0")
    def HUGGINGFACE_TXT2_IMG_TIMOTHEPEARCE_GINA_THE_CAT_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TIMOTHEPEARCE_GINA_THE_CAT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TRYSTAR_CLONEDIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_TRYSTAR_CLONEDIFFUSION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TRYSTAR_CLONEDIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TRYSTAR_CLONEDIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_TRYSTAR_CLONEDIFFUSION_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TRYSTAR_CLONEDIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TRYSTAR_CLONEDIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_TRYSTAR_CLONEDIFFUSION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TRYSTAR_CLONEDIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TUWONGA_DBLUTH_1_0_0")
    def HUGGINGFACE_TXT2_IMG_TUWONGA_DBLUTH_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TUWONGA_DBLUTH_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TUWONGA_DBLUTH_1_1_0")
    def HUGGINGFACE_TXT2_IMG_TUWONGA_DBLUTH_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TUWONGA_DBLUTH_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TUWONGA_DBLUTH_2_0_0")
    def HUGGINGFACE_TXT2_IMG_TUWONGA_DBLUTH_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TUWONGA_DBLUTH_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TUWONGA_ROTOSCOPEE_1_0_0")
    def HUGGINGFACE_TXT2_IMG_TUWONGA_ROTOSCOPEE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TUWONGA_ROTOSCOPEE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TUWONGA_ROTOSCOPEE_1_1_0")
    def HUGGINGFACE_TXT2_IMG_TUWONGA_ROTOSCOPEE_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TUWONGA_ROTOSCOPEE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_TUWONGA_ROTOSCOPEE_2_0_0")
    def HUGGINGFACE_TXT2_IMG_TUWONGA_ROTOSCOPEE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_TUWONGA_ROTOSCOPEE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_1_0_0")
    def HUGGINGFACE_TXT2_IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_1_1_0")
    def HUGGINGFACE_TXT2_IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_2_0_0")
    def HUGGINGFACE_TXT2_IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_VOLRATH50_FANTASY_CARD_DIFFUSION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_1_0_0")
    def HUGGINGFACE_TXT2_IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_1_1_0")
    def HUGGINGFACE_TXT2_IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_TXT2IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_2_0_0")
    def HUGGINGFACE_TXT2_IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_TXT2IMG_YAYAB_SD_ONEPIECE_DIFFUSERS4_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DEBERTA_BASE_1_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DEBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DEBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DEBERTA_BASE_2_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DEBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DEBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DISTILROBERTA_BASE_1_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DISTILROBERTA_BASE_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DISTILROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DISTILROBERTA_BASE_2_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DISTILROBERTA_BASE_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_DISTILROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_MINILM2_L6_H768_1_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_MINILM2_L6_H768_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_MINILM2_L6_H768_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_MINILM2_L6_H768_2_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_MINILM2_L6_H768_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_MINILM2_L6_H768_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_ROBERTA_BASE_1_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_ROBERTA_BASE_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_ROBERTA_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_ROBERTA_BASE_2_0_0")
    def HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_ROBERTA_BASE_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_CROSS_ENCODER_NLI_ROBERTA_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_DIGITALEPIDEMIOLOGYLAB_COVID_TWITTER_BERT_V2_MNLI_1_0_0")
    def HUGGINGFACE_ZSTC_DIGITALEPIDEMIOLOGYLAB_COVID_TWITTER_BERT_V2_MNLI_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_DIGITALEPIDEMIOLOGYLAB_COVID_TWITTER_BERT_V2_MNLI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_DIGITALEPIDEMIOLOGYLAB_COVID_TWITTER_BERT_V2_MNLI_2_0_0")
    def HUGGINGFACE_ZSTC_DIGITALEPIDEMIOLOGYLAB_COVID_TWITTER_BERT_V2_MNLI_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_DIGITALEPIDEMIOLOGYLAB_COVID_TWITTER_BERT_V2_MNLI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_ELELDAR_THEME_CLASSIFICATION_1_0_0")
    def HUGGINGFACE_ZSTC_ELELDAR_THEME_CLASSIFICATION_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_ELELDAR_THEME_CLASSIFICATION_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_ELELDAR_THEME_CLASSIFICATION_2_0_0")
    def HUGGINGFACE_ZSTC_ELELDAR_THEME_CLASSIFICATION_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_ELELDAR_THEME_CLASSIFICATION_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_ALLNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_ALLNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_ALLNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_ALLNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_ALLNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_ALLNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_MULTINLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_MULTINLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_MULTINLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_MULTINLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_MULTINLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_MULTINLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_SNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_SNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_SNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_SNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_SNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_MULTILINGUAL_CASED_SNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_ALLNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_ALLNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_ALLNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_ALLNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_ALLNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_ALLNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_MULTINLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_MULTINLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_MULTINLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_MULTINLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_MULTINLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_MULTINLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_SNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_SNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_SNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_SNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_SNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_BERT_BASE_TURKISH_CASED_SNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_ALLNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_ALLNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_ALLNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_ALLNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_ALLNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_ALLNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_MULTINLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_MULTINLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_MULTINLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_MULTINLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_MULTINLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_MULTINLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_SNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_SNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_SNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_SNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_SNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_CONVBERT_BASE_TURKISH_MC4_CASED_SNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_ALLNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_ALLNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_ALLNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_ALLNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_ALLNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_ALLNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_MULTINLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_MULTINLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_MULTINLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_MULTINLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_MULTINLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_MULTINLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_SNLI_TR_1_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_SNLI_TR_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_SNLI_TR_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_SNLI_TR_2_0_0")
    def HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_SNLI_TR_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_EMRECAN_DISTILBERT_BASE_TURKISH_CASED_SNLI_TR_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_FACEBOOK_BART_LARGE_MNLI_1_0_0")
    def HUGGINGFACE_ZSTC_FACEBOOK_BART_LARGE_MNLI_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_FACEBOOK_BART_LARGE_MNLI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_FACEBOOK_BART_LARGE_MNLI_2_0_0")
    def HUGGINGFACE_ZSTC_FACEBOOK_BART_LARGE_MNLI_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_FACEBOOK_BART_LARGE_MNLI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_JIVA_XLM_ROBERTA_LARGE_IT_MNLI_1_0_0")
    def HUGGINGFACE_ZSTC_JIVA_XLM_ROBERTA_LARGE_IT_MNLI_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_JIVA_XLM_ROBERTA_LARGE_IT_MNLI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_JIVA_XLM_ROBERTA_LARGE_IT_MNLI_2_0_0")
    def HUGGINGFACE_ZSTC_JIVA_XLM_ROBERTA_LARGE_IT_MNLI_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_JIVA_XLM_ROBERTA_LARGE_IT_MNLI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_LIGHTETERNAL_NLI_XLM_R_GREEK_1_0_0")
    def HUGGINGFACE_ZSTC_LIGHTETERNAL_NLI_XLM_R_GREEK_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_LIGHTETERNAL_NLI_XLM_R_GREEK_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_LIGHTETERNAL_NLI_XLM_R_GREEK_2_0_0")
    def HUGGINGFACE_ZSTC_LIGHTETERNAL_NLI_XLM_R_GREEK_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_LIGHTETERNAL_NLI_XLM_R_GREEK_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_MORITZLAURER_DEBERTA_V3_LARGE_MNLI_FEVER_ANLI_LING_WANLI_1_0_0")
    def HUGGINGFACE_ZSTC_MORITZLAURER_DEBERTA_V3_LARGE_MNLI_FEVER_ANLI_LING_WANLI_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_MORITZLAURER_DEBERTA_V3_LARGE_MNLI_FEVER_ANLI_LING_WANLI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_MORITZLAURER_DEBERTA_V3_LARGE_MNLI_FEVER_ANLI_LING_WANLI_2_0_0")
    def HUGGINGFACE_ZSTC_MORITZLAURER_DEBERTA_V3_LARGE_MNLI_FEVER_ANLI_LING_WANLI_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_MORITZLAURER_DEBERTA_V3_LARGE_MNLI_FEVER_ANLI_LING_WANLI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_MORITZLAURER_MDEBERTA_V3_BASE_XNLI_MULTILINGUAL_NLI_2MIL7_1_0_0")
    def HUGGINGFACE_ZSTC_MORITZLAURER_MDEBERTA_V3_BASE_XNLI_MULTILINGUAL_NLI_2_MIL7_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_MORITZLAURER_MDEBERTA_V3_BASE_XNLI_MULTILINGUAL_NLI_2MIL7_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_MORITZLAURER_MDEBERTA_V3_BASE_XNLI_MULTILINGUAL_NLI_2MIL7_2_0_0")
    def HUGGINGFACE_ZSTC_MORITZLAURER_MDEBERTA_V3_BASE_XNLI_MULTILINGUAL_NLI_2_MIL7_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_MORITZLAURER_MDEBERTA_V3_BASE_XNLI_MULTILINGUAL_NLI_2MIL7_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_NARSIL_BART_LARGE_MNLI_OPTI_1_0_0")
    def HUGGINGFACE_ZSTC_NARSIL_BART_LARGE_MNLI_OPTI_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_NARSIL_BART_LARGE_MNLI_OPTI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_NARSIL_BART_LARGE_MNLI_OPTI_2_0_0")
    def HUGGINGFACE_ZSTC_NARSIL_BART_LARGE_MNLI_OPTI_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_NARSIL_BART_LARGE_MNLI_OPTI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_NARSIL_DEBERTA_LARGE_MNLI_ZERO_CLS_1_0_0")
    def HUGGINGFACE_ZSTC_NARSIL_DEBERTA_LARGE_MNLI_ZERO_CLS_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_NARSIL_DEBERTA_LARGE_MNLI_ZERO_CLS_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_NARSIL_DEBERTA_LARGE_MNLI_ZERO_CLS_2_0_0")
    def HUGGINGFACE_ZSTC_NARSIL_DEBERTA_LARGE_MNLI_ZERO_CLS_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_NARSIL_DEBERTA_LARGE_MNLI_ZERO_CLS_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_NAVTECA_BART_LARGE_MNLI_1_0_0")
    def HUGGINGFACE_ZSTC_NAVTECA_BART_LARGE_MNLI_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_NAVTECA_BART_LARGE_MNLI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_NAVTECA_BART_LARGE_MNLI_2_0_0")
    def HUGGINGFACE_ZSTC_NAVTECA_BART_LARGE_MNLI_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_NAVTECA_BART_LARGE_MNLI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_RECOGNAI_BERT_BASE_SPANISH_WWM_CASED_XNLI_1_0_0")
    def HUGGINGFACE_ZSTC_RECOGNAI_BERT_BASE_SPANISH_WWM_CASED_XNLI_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_RECOGNAI_BERT_BASE_SPANISH_WWM_CASED_XNLI_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_RECOGNAI_BERT_BASE_SPANISH_WWM_CASED_XNLI_2_0_0")
    def HUGGINGFACE_ZSTC_RECOGNAI_BERT_BASE_SPANISH_WWM_CASED_XNLI_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_RECOGNAI_BERT_BASE_SPANISH_WWM_CASED_XNLI_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_MEDIUM_1_0_0")
    def HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_MEDIUM_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_MEDIUM_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_MEDIUM_2_0_0")
    def HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_MEDIUM_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_MEDIUM_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_SMALL_1_0_0")
    def HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_SMALL_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_SMALL_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_SMALL_2_0_0")
    def HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_SMALL_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "HUGGINGFACE_ZSTC_RECOGNAI_ZEROSHOT_SELECTRA_SMALL_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_0_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_2")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_3")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_4")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_5")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_6")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_6(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_7")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_7(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_2_1_8")
    def META_TEXTGENERATION_LLAMA_2_13_B_2_1_8(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_2_1_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_0_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_0_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_0_2")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_0_3")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_0_4")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_1_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_3_2_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_3_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_3_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_4_0_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_4_1_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_4_1_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_4_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_4_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_1_1_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_1_2_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_1_2_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_2_0_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_2_0_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_2_0_2")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_2_0_3")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_2_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_2_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_2_0_4")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_2_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_2_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_0_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_0_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_0_2")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_0_3")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_0_4")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_1_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_3_2_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_3_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_3_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_4_0_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_4_1_0")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_13B_F_4_1_1")
    def META_TEXTGENERATION_LLAMA_2_13_B_F_4_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_13B_F_4_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_1_1_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_1_2_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_2")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_3")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_4")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_5")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_6")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_6(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_2_0_7")
    def META_TEXTGENERATION_LLAMA_2_70_B_2_0_7(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_2_0_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_0_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_0_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_0_2")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_0_3")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_0_4")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_1_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_3_1_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_4_0_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_4_1_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_4_1_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_4_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_4_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_1_1_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_1_2_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_1_2_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_2_0_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_2_0_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_2_0_2")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_2_0_3")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_2_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_2_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_2_0_4")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_2_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_2_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_0_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_0_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_0_2")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_0_3")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_0_4")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_1_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_3_1_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_4_0_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_4_1_0")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_70B_F_4_1_1")
    def META_TEXTGENERATION_LLAMA_2_70_B_F_4_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_70B_F_4_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_0_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_2")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_3")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_4")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_5")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_5(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_6")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_6(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_7")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_7(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_2_1_8")
    def META_TEXTGENERATION_LLAMA_2_7_B_2_1_8(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_2_1_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_0_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_0_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_0_2")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_0_3")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_0_4")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_1_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_3_2_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_3_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_3_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_4_0_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_4_1_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_4_1_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_4_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_4_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_1_1_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_1_2_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_1_2_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_2_0_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_2_0_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_2_0_2")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_2_0_3")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_2_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_2_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_2_0_4")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_2_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_2_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_0_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_0_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_0_2")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_0_3")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_0_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_0_4")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_0_4(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_1_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_3_2_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_3_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_3_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_4_0_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_4_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_4_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_4_1_0")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_4_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_4_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_2_7B_F_4_1_1")
    def META_TEXTGENERATION_LLAMA_2_7_B_F_4_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_2_7B_F_4_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_1_0_0")
    def META_TEXTGENERATION_LLAMA_3_70_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_2_0_0")
    def META_TEXTGENERATION_LLAMA_3_70_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_2_0_1")
    def META_TEXTGENERATION_LLAMA_3_70_B_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_2_0_2")
    def META_TEXTGENERATION_LLAMA_3_70_B_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_1_0_0")
    def META_TEXTGENERATION_LLAMA_3_70_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_2_0_0")
    def META_TEXTGENERATION_LLAMA_3_70_B_INSTRUCT_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_2_0_1")
    def META_TEXTGENERATION_LLAMA_3_70_B_INSTRUCT_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_2_0_2")
    def META_TEXTGENERATION_LLAMA_3_70_B_INSTRUCT_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_70B_INSTRUCT_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_1_0_0")
    def META_TEXTGENERATION_LLAMA_3_8_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_2_0_0")
    def META_TEXTGENERATION_LLAMA_3_8_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_2_0_1")
    def META_TEXTGENERATION_LLAMA_3_8_B_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_2_0_2")
    def META_TEXTGENERATION_LLAMA_3_8_B_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_1_0_0")
    def META_TEXTGENERATION_LLAMA_3_8_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_2_0_0")
    def META_TEXTGENERATION_LLAMA_3_8_B_INSTRUCT_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_2_0_1")
    def META_TEXTGENERATION_LLAMA_3_8_B_INSTRUCT_2_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_2_0_2")
    def META_TEXTGENERATION_LLAMA_3_8_B_INSTRUCT_2_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_3_8B_INSTRUCT_2_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_3_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_3_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_3_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_INSTRUCT_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_INSTRUCT_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_INSTRUCT_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_INSTRUCT_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_INSTRUCT_2_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_INSTRUCT_2_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_INSTRUCT_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_3_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_3_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_3_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_13_B_PYTHON_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_13B_PYTHON_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_3_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_3_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_3_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_INSTRUCT_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_INSTRUCT_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_INSTRUCT_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_INSTRUCT_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_INSTRUCT_2_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_INSTRUCT_2_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_INSTRUCT_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_3_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_3_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_3_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_34_B_PYTHON_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_34B_PYTHON_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_INSTRUCT_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_INSTRUCT_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_INSTRUCT_1_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_INSTRUCT_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_INSTRUCT_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_INSTRUCT_1_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_INSTRUCT_1_2_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_INSTRUCT_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_70_B_PYTHON_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_70B_PYTHON_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_3_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_3_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_3_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_INSTRUCT_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_INSTRUCT_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_INSTRUCT_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_INSTRUCT_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_INSTRUCT_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_INSTRUCT_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_INSTRUCT_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_1_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_1_0_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_1_0_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_1_0_2")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_1_0_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_2_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_2_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_2_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_2_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_2_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_2_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_3_0_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_3_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_3_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_3_1_0")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_3_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_3_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_3_1_1")
    def META_TEXTGENERATION_LLAMA_CODELLAMA_7_B_PYTHON_3_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_CODELLAMA_7B_PYTHON_3_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_GUARD_7B_1_0_0")
    def META_TEXTGENERATION_LLAMA_GUARD_7_B_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_GUARD_7B_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_GUARD_7B_1_1_0")
    def META_TEXTGENERATION_LLAMA_GUARD_7_B_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_GUARD_7B_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="META_TEXTGENERATION_LLAMA_GUARD_7B_1_2_0")
    def META_TEXTGENERATION_LLAMA_GUARD_7_B_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "META_TEXTGENERATION_LLAMA_GUARD_7B_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_2_DEPTH_FP16_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_2_DEPTH_FP16_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_2_DEPTH_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_2_DEPTH_FP16_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_2_DEPTH_FP16_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_2_DEPTH_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_FP16_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_FP16_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_FP16_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_FP16_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V1_5_CONTROLNET_V1_1_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_FP16_1_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_FP16_2_0_0")
    def MODEL_DEPTH2_IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_DEPTH2IMG_STABLE_DIFFUSION_V2_1_CONTROLNET_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_V2_1_1_0_0")
    def MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_V2_1_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_V2_1_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0_1_0_0")
    def MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0_1_0_1")
    def MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_IMAGEGENERATION_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_1_0_0")
    def MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_1_1_0")
    def MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_2_0_0")
    def MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_1_0_0")
    def MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_1_1_0")
    def MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_2_0_0")
    def MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_RUNWAYML_STABLE_DIFFUSION_INPAINTING_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_1_0_0")
    def MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_1_1_0")
    def MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_2_0_0")
    def MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_1_0_0")
    def MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_1_1_0")
    def MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_2_0_0")
    def MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_INPAINTING_STABILITYAI_STABLE_DIFFUSION_2_INPAINTING_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TEXTGENERATIONJP_JAPANESE_STABLELM_INSTRUCT_ALPHA_7B_V2_1_0_0")
    def MODEL_TEXTGENERATIONJP_JAPANESE_STABLELM_INSTRUCT_ALPHA_7_B_V2_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TEXTGENERATIONJP_JAPANESE_STABLELM_INSTRUCT_ALPHA_7B_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_1_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_1")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_2")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_3")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_3_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_3_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_1_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_2_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_1")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_2")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_3")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_1_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_2_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V1_4_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_1")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_1(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_2")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_2(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_3")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_3(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_2_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_2_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_1")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_2")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_3")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_4")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_4(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_0_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_1")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_2")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_3")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_1_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_2_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_1_BASE_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_2_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_2_0_0(cls) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_1")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_1(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_2")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_2(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_3")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_3(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_0_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_1_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_2_0_0")
    def MODEL_TXT2_IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_TXT2IMG_STABILITYAI_STABLE_DIFFUSION_V2_FP16_2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_1_0_0")
    def MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_1_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_1_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_1_1_0")
    def MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_1_1_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_1_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_2_0_0")
    def MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_2_0_0(
        cls,
    ) -> "JumpStartModel":
        '''
        :stability: experimental
        '''
        return typing.cast("JumpStartModel", jsii.sget(cls, "MODEL_UPSCALING_STABILITYAI_STABLE_DIFFUSION_X4_UPSCALER_FP16_2_0_0"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.JumpStartSageMakerEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "endpoint_name": "endpointName",
        "model": "model",
        "accept_eula": "acceptEula",
        "environment": "environment",
        "instance_count": "instanceCount",
        "instance_type": "instanceType",
        "role": "role",
        "startup_health_check_timeout_in_seconds": "startupHealthCheckTimeoutInSeconds",
        "vpc_config": "vpcConfig",
    },
)
class JumpStartSageMakerEndpointProps:
    def __init__(
        self,
        *,
        endpoint_name: builtins.str,
        model: JumpStartModel,
        accept_eula: typing.Optional[builtins.bool] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        instance_count: typing.Optional[jsii.Number] = None,
        instance_type: typing.Optional["SageMakerInstanceType"] = None,
        role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role] = None,
        startup_health_check_timeout_in_seconds: typing.Optional[jsii.Number] = None,
        vpc_config: typing.Optional[typing.Union[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param endpoint_name: 
        :param model: 
        :param accept_eula: 
        :param environment: 
        :param instance_count: 
        :param instance_type: 
        :param role: 
        :param startup_health_check_timeout_in_seconds: 
        :param vpc_config: 

        :stability: experimental
        '''
        if isinstance(vpc_config, dict):
            vpc_config = _aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty(**vpc_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a21e4787a48ba458913fa109830cd5d13f8be4ce00374d6d25306a7755c73997)
            check_type(argname="argument endpoint_name", value=endpoint_name, expected_type=type_hints["endpoint_name"])
            check_type(argname="argument model", value=model, expected_type=type_hints["model"])
            check_type(argname="argument accept_eula", value=accept_eula, expected_type=type_hints["accept_eula"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument startup_health_check_timeout_in_seconds", value=startup_health_check_timeout_in_seconds, expected_type=type_hints["startup_health_check_timeout_in_seconds"])
            check_type(argname="argument vpc_config", value=vpc_config, expected_type=type_hints["vpc_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "endpoint_name": endpoint_name,
            "model": model,
        }
        if accept_eula is not None:
            self._values["accept_eula"] = accept_eula
        if environment is not None:
            self._values["environment"] = environment
        if instance_count is not None:
            self._values["instance_count"] = instance_count
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if role is not None:
            self._values["role"] = role
        if startup_health_check_timeout_in_seconds is not None:
            self._values["startup_health_check_timeout_in_seconds"] = startup_health_check_timeout_in_seconds
        if vpc_config is not None:
            self._values["vpc_config"] = vpc_config

    @builtins.property
    def endpoint_name(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("endpoint_name")
        assert result is not None, "Required property 'endpoint_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def model(self) -> JumpStartModel:
        '''
        :stability: experimental
        '''
        result = self._values.get("model")
        assert result is not None, "Required property 'model' is missing"
        return typing.cast(JumpStartModel, result)

    @builtins.property
    def accept_eula(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("accept_eula")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def instance_count(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("instance_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instance_type(self) -> typing.Optional["SageMakerInstanceType"]:
        '''
        :stability: experimental
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional["SageMakerInstanceType"], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role]:
        '''
        :stability: experimental
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.Role], result)

    @builtins.property
    def startup_health_check_timeout_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''
        :stability: experimental
        '''
        result = self._values.get("startup_health_check_timeout_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def vpc_config(
        self,
    ) -> typing.Optional[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty]:
        '''
        :stability: experimental
        '''
        result = self._values.get("vpc_config")
        return typing.cast(typing.Optional[_aws_cdk_aws_sagemaker_ceddda9d.CfnModel.VpcConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JumpStartSageMakerEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class LangchainCommonDepsLayer(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.LangchainCommonDepsLayer",
):
    '''
    :stability: experimental
    :summary: The LangchainCommonDepsLayer class.
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        architecture: _aws_cdk_aws_lambda_ceddda9d.Architecture,
        runtime: _aws_cdk_aws_lambda_ceddda9d.Runtime,
        additional_packages: typing.Optional[typing.Sequence[builtins.str]] = None,
        auto_upgrade: typing.Optional[builtins.bool] = None,
        local: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a a scope-unique id.
        :param architecture: (experimental) Required. Lambda function architecture compatible with this Layer.
        :param runtime: (experimental) Required. Lambda function runtime compatible with this Layer.
        :param additional_packages: (experimental) A prop allowing additional python pip libraries to be installed with this langchain layer. Default: - none
        :param auto_upgrade: (experimental) Optional: Add '--upgrade' to pip install requirements.txt In case of a LangchainCommonLayer, this parameter is not used. Default: - false
        :param local: (experimental) Optional: Local compute will be used when installing requirements.txt. By default, a docker container will be spun up to install requirements. To override this behavior, use the python alias string of ``python`` or ``python3`` The string value will be the python alias used to install requirements. Default: - none
        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: This construct creates a lambda layer loaded with relevant libraries to run genai applications. Libraries include boto3, botocore, requests, requests-aws4auth, langchain, opensearch-py and openai.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5887bf17d5f030e30d4d81e59cc58f7736b547350a7fc34d4824591bab542b31)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = LangchainLayerProps(
            architecture=architecture,
            runtime=runtime,
            additional_packages=additional_packages,
            auto_upgrade=auto_upgrade,
            local=local,
            description=description,
            layer_version_name=layer_version_name,
            license=license,
            removal_policy=removal_policy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="layer")
    def layer(self) -> _aws_cdk_aws_lambda_ceddda9d.LayerVersion:
        '''(experimental) Returns the instance of lambda.LayerVersion created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.LayerVersion, jsii.get(self, "layer"))


class LangchainCommonLayer(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.LangchainCommonLayer",
):
    '''
    :stability: experimental
    :summary: LangchainCommonLayer allows developers to instantiate a llm client adapter on bedrock, sagemaker or openai following best practise.

    Example::

        import boto3
        from genai_core.adapters.registry import registry
        
        adapter = registry.get_adapter(f"{provider}.{model_id}")
        bedrock_client = boto3.client('bedrock-runtime')
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        compatible_architectures: typing.Optional[typing.Sequence[_aws_cdk_aws_lambda_ceddda9d.Architecture]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence[_aws_cdk_aws_lambda_ceddda9d.Runtime]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a a scope-unique id.
        :param compatible_architectures: The system architectures compatible with this layer. Default: [Architecture.X86_64]
        :param compatible_runtimes: The runtimes compatible with this Layer. Default: - All runtimes are supported.
        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: This construct allows developers to instantiate a llm client adapter on bedrock, sagemaker or openai following best practise.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9da98e3b66ac7e9145afc01d0bcd18f45c1684ff1bc89ccc645b65e2edc8f13a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AdapterProps(
            compatible_architectures=compatible_architectures,
            compatible_runtimes=compatible_runtimes,
            description=description,
            layer_version_name=layer_version_name,
            license=license,
            removal_policy=removal_policy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="layer")
    def layer(self) -> _aws_cdk_aws_lambda_ceddda9d.LayerVersion:
        '''(experimental) Returns the instance of lambda.LayerVersion created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.LayerVersion, jsii.get(self, "layer"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.LangchainProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "layer_version_name": "layerVersionName",
        "license": "license",
        "removal_policy": "removalPolicy",
    },
)
class LangchainProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        layer_version_name: typing.Optional[builtins.str] = None,
        license: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy] = None,
    ) -> None:
        '''(experimental) LangchainProps.

        :param description: The description the this Lambda Layer. Default: - No description.
        :param layer_version_name: The name of the layer. Default: - A name will be generated.
        :param license: The SPDX licence identifier or URL to the license file for this layer. Default: - No license information will be recorded.
        :param removal_policy: Whether to retain this version of the layer when a new version is added or when the stack is deleted. Default: RemovalPolicy.DESTROY

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52d31c4ca4bd0cf5b3967d59cb772ea252af908e216abb9db44bcadbdcb0c948)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument layer_version_name", value=layer_version_name, expected_type=type_hints["layer_version_name"])
            check_type(argname="argument license", value=license, expected_type=type_hints["license"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if layer_version_name is not None:
            self._values["layer_version_name"] = layer_version_name
        if license is not None:
            self._values["license"] = license
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description the this Lambda Layer.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layer_version_name(self) -> typing.Optional[builtins.str]:
        '''The name of the layer.

        :default: - A name will be generated.
        '''
        result = self._values.get("layer_version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license(self) -> typing.Optional[builtins.str]:
        '''The SPDX licence identifier or URL to the license file for this layer.

        :default: - No license information will be recorded.
        '''
        result = self._values.get("license")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy]:
        '''Whether to retain this version of the layer when a new version is added or when the stack is deleted.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.RemovalPolicy], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LangchainProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class QaAppsyncOpensearch(
    BaseClass,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.QaAppsyncOpensearch",
):
    '''
    :stability: experimental
    :summary: The QaAppsyncOpensearch class.
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        open_search_index_name: builtins.str,
        bucket_inputs_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_input_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_opensearch_domain: typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain] = None,
        existing_opensearch_serverless_collection: typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        lambda_provisioned_concurrency: typing.Optional[jsii.Number] = None,
        observability: typing.Optional[builtins.bool] = None,
        open_search_secret: typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret] = None,
        stage: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a a scope-unique id.
        :param cognito_user_pool: (experimental) Cognito user pool used for authentication. Default: - None
        :param open_search_index_name: (experimental) Data Index name for the OpenSearch Service. Default: - None
        :param bucket_inputs_assets_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingInputAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param custom_docker_lambda_props: (experimental) Optional. Allows to provide custom lambda code and settings instead of the existing
        :param existing_bus_interface: (experimental) Optional Existing instance of an EventBridge bus. If not provided, the construct will create one. Default: - None
        :param existing_input_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketInputsAssetsProps`` will cause an error. Default: - None
        :param existing_merged_api: (experimental) Existing merged Appsync GraphQL api. Default: - None
        :param existing_opensearch_domain: (experimental) Existing Amazon OpenSearch Service domain. Default: - None
        :param existing_opensearch_serverless_collection: (experimental) Existing Amazon Amazon OpenSearch Serverless collection. Default: - None
        :param existing_security_group: (experimental) Optional existing security group allowing access to opensearch. Used by the lambda functions built by this construct. If not provided, the construct will create one. Default: - none
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param lambda_provisioned_concurrency: (experimental) Optional. Allows a user to configure Lambda provisioned concurrency for consistent performance
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practive to enable by default. Default: - true
        :param open_search_secret: (experimental) Optional. SecretsManager secret to authenticate against the OpenSearch Service domain if domain is configured with Username/Password. Default: - None
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the Lambda functions the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: Constructs a new instance of the RagAppsyncStepfnOpensearch class.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87235dce222bf37474e367e0b5284fe8ad89c007d7c7d06682de5ff0d56df01b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = QaAppsyncOpensearchProps(
            cognito_user_pool=cognito_user_pool,
            open_search_index_name=open_search_index_name,
            bucket_inputs_assets_props=bucket_inputs_assets_props,
            custom_docker_lambda_props=custom_docker_lambda_props,
            existing_bus_interface=existing_bus_interface,
            existing_input_assets_bucket_obj=existing_input_assets_bucket_obj,
            existing_merged_api=existing_merged_api,
            existing_opensearch_domain=existing_opensearch_domain,
            existing_opensearch_serverless_collection=existing_opensearch_serverless_collection,
            existing_security_group=existing_security_group,
            existing_vpc=existing_vpc,
            lambda_provisioned_concurrency=lambda_provisioned_concurrency,
            observability=observability,
            open_search_secret=open_search_secret,
            stage=stage,
            vpc_props=vpc_props,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="graphqlApi")
    def graphql_api(self) -> _aws_cdk_aws_appsync_ceddda9d.IGraphqlApi:
        '''(experimental) Returns an instance of appsync.IGraphqlApi created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_appsync_ceddda9d.IGraphqlApi, jsii.get(self, "graphqlApi"))

    @builtins.property
    @jsii.member(jsii_name="qaBus")
    def qa_bus(self) -> _aws_cdk_aws_events_ceddda9d.IEventBus:
        '''(experimental) Returns the instance of events.IEventBus used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_events_ceddda9d.IEventBus, jsii.get(self, "qaBus"))

    @builtins.property
    @jsii.member(jsii_name="qaLambdaFunction")
    def qa_lambda_function(self) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of appsync.IGraphqlApi created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "qaLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="s3InputAssetsBucketInterface")
    def s3_input_assets_bucket_interface(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns an instance of s3.IBucket created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "s3InputAssetsBucketInterface"))

    @builtins.property
    @jsii.member(jsii_name="securityGroup")
    def security_group(self) -> _aws_cdk_aws_ec2_ceddda9d.ISecurityGroup:
        '''(experimental) Returns the instance of ec2.ISecurityGroup used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup, jsii.get(self, "securityGroup"))

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _aws_cdk_aws_ec2_ceddda9d.IVpc:
        '''(experimental) Returns the instance of ec2.IVpc used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.IVpc, jsii.get(self, "vpc"))

    @builtins.property
    @jsii.member(jsii_name="s3InputAssetsBucket")
    def s3_input_assets_bucket(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket]:
        '''(experimental) Returns an instance of s3.Bucket created by the construct. IMPORTANT: If existingInputAssetsBucketObj was provided in Pattern Construct Props, this property will be undefined.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket], jsii.get(self, "s3InputAssetsBucket"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.QaAppsyncOpensearchProps",
    jsii_struct_bases=[],
    name_mapping={
        "cognito_user_pool": "cognitoUserPool",
        "open_search_index_name": "openSearchIndexName",
        "bucket_inputs_assets_props": "bucketInputsAssetsProps",
        "custom_docker_lambda_props": "customDockerLambdaProps",
        "existing_bus_interface": "existingBusInterface",
        "existing_input_assets_bucket_obj": "existingInputAssetsBucketObj",
        "existing_merged_api": "existingMergedApi",
        "existing_opensearch_domain": "existingOpensearchDomain",
        "existing_opensearch_serverless_collection": "existingOpensearchServerlessCollection",
        "existing_security_group": "existingSecurityGroup",
        "existing_vpc": "existingVpc",
        "lambda_provisioned_concurrency": "lambdaProvisionedConcurrency",
        "observability": "observability",
        "open_search_secret": "openSearchSecret",
        "stage": "stage",
        "vpc_props": "vpcProps",
    },
)
class QaAppsyncOpensearchProps:
    def __init__(
        self,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        open_search_index_name: builtins.str,
        bucket_inputs_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_input_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_opensearch_domain: typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain] = None,
        existing_opensearch_serverless_collection: typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        lambda_provisioned_concurrency: typing.Optional[jsii.Number] = None,
        observability: typing.Optional[builtins.bool] = None,
        open_search_secret: typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret] = None,
        stage: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) The properties for the QaAppsyncOpensearchProps class.

        :param cognito_user_pool: (experimental) Cognito user pool used for authentication. Default: - None
        :param open_search_index_name: (experimental) Data Index name for the OpenSearch Service. Default: - None
        :param bucket_inputs_assets_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingInputAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param custom_docker_lambda_props: (experimental) Optional. Allows to provide custom lambda code and settings instead of the existing
        :param existing_bus_interface: (experimental) Optional Existing instance of an EventBridge bus. If not provided, the construct will create one. Default: - None
        :param existing_input_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketInputsAssetsProps`` will cause an error. Default: - None
        :param existing_merged_api: (experimental) Existing merged Appsync GraphQL api. Default: - None
        :param existing_opensearch_domain: (experimental) Existing Amazon OpenSearch Service domain. Default: - None
        :param existing_opensearch_serverless_collection: (experimental) Existing Amazon Amazon OpenSearch Serverless collection. Default: - None
        :param existing_security_group: (experimental) Optional existing security group allowing access to opensearch. Used by the lambda functions built by this construct. If not provided, the construct will create one. Default: - none
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param lambda_provisioned_concurrency: (experimental) Optional. Allows a user to configure Lambda provisioned concurrency for consistent performance
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practive to enable by default. Default: - true
        :param open_search_secret: (experimental) Optional. SecretsManager secret to authenticate against the OpenSearch Service domain if domain is configured with Username/Password. Default: - None
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the Lambda functions the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        '''
        if isinstance(bucket_inputs_assets_props, dict):
            bucket_inputs_assets_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**bucket_inputs_assets_props)
        if isinstance(custom_docker_lambda_props, dict):
            custom_docker_lambda_props = DockerLambdaCustomProps(**custom_docker_lambda_props)
        if isinstance(vpc_props, dict):
            vpc_props = _aws_cdk_aws_ec2_ceddda9d.VpcProps(**vpc_props)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff2c8f0d402dcb481b313b3fe2800ddb797d454c33c358eadd45c7a626be6592)
            check_type(argname="argument cognito_user_pool", value=cognito_user_pool, expected_type=type_hints["cognito_user_pool"])
            check_type(argname="argument open_search_index_name", value=open_search_index_name, expected_type=type_hints["open_search_index_name"])
            check_type(argname="argument bucket_inputs_assets_props", value=bucket_inputs_assets_props, expected_type=type_hints["bucket_inputs_assets_props"])
            check_type(argname="argument custom_docker_lambda_props", value=custom_docker_lambda_props, expected_type=type_hints["custom_docker_lambda_props"])
            check_type(argname="argument existing_bus_interface", value=existing_bus_interface, expected_type=type_hints["existing_bus_interface"])
            check_type(argname="argument existing_input_assets_bucket_obj", value=existing_input_assets_bucket_obj, expected_type=type_hints["existing_input_assets_bucket_obj"])
            check_type(argname="argument existing_merged_api", value=existing_merged_api, expected_type=type_hints["existing_merged_api"])
            check_type(argname="argument existing_opensearch_domain", value=existing_opensearch_domain, expected_type=type_hints["existing_opensearch_domain"])
            check_type(argname="argument existing_opensearch_serverless_collection", value=existing_opensearch_serverless_collection, expected_type=type_hints["existing_opensearch_serverless_collection"])
            check_type(argname="argument existing_security_group", value=existing_security_group, expected_type=type_hints["existing_security_group"])
            check_type(argname="argument existing_vpc", value=existing_vpc, expected_type=type_hints["existing_vpc"])
            check_type(argname="argument lambda_provisioned_concurrency", value=lambda_provisioned_concurrency, expected_type=type_hints["lambda_provisioned_concurrency"])
            check_type(argname="argument observability", value=observability, expected_type=type_hints["observability"])
            check_type(argname="argument open_search_secret", value=open_search_secret, expected_type=type_hints["open_search_secret"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument vpc_props", value=vpc_props, expected_type=type_hints["vpc_props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cognito_user_pool": cognito_user_pool,
            "open_search_index_name": open_search_index_name,
        }
        if bucket_inputs_assets_props is not None:
            self._values["bucket_inputs_assets_props"] = bucket_inputs_assets_props
        if custom_docker_lambda_props is not None:
            self._values["custom_docker_lambda_props"] = custom_docker_lambda_props
        if existing_bus_interface is not None:
            self._values["existing_bus_interface"] = existing_bus_interface
        if existing_input_assets_bucket_obj is not None:
            self._values["existing_input_assets_bucket_obj"] = existing_input_assets_bucket_obj
        if existing_merged_api is not None:
            self._values["existing_merged_api"] = existing_merged_api
        if existing_opensearch_domain is not None:
            self._values["existing_opensearch_domain"] = existing_opensearch_domain
        if existing_opensearch_serverless_collection is not None:
            self._values["existing_opensearch_serverless_collection"] = existing_opensearch_serverless_collection
        if existing_security_group is not None:
            self._values["existing_security_group"] = existing_security_group
        if existing_vpc is not None:
            self._values["existing_vpc"] = existing_vpc
        if lambda_provisioned_concurrency is not None:
            self._values["lambda_provisioned_concurrency"] = lambda_provisioned_concurrency
        if observability is not None:
            self._values["observability"] = observability
        if open_search_secret is not None:
            self._values["open_search_secret"] = open_search_secret
        if stage is not None:
            self._values["stage"] = stage
        if vpc_props is not None:
            self._values["vpc_props"] = vpc_props

    @builtins.property
    def cognito_user_pool(self) -> _aws_cdk_aws_cognito_ceddda9d.IUserPool:
        '''(experimental) Cognito user pool used for authentication.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("cognito_user_pool")
        assert result is not None, "Required property 'cognito_user_pool' is missing"
        return typing.cast(_aws_cdk_aws_cognito_ceddda9d.IUserPool, result)

    @builtins.property
    def open_search_index_name(self) -> builtins.str:
        '''(experimental) Data Index name for the OpenSearch Service.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("open_search_index_name")
        assert result is not None, "Required property 'open_search_index_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def bucket_inputs_assets_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional user provided props to override the default props for the S3 Bucket.

        Providing both this and ``existingInputAssetsBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("bucket_inputs_assets_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def custom_docker_lambda_props(self) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def existing_bus_interface(
        self,
    ) -> typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus]:
        '''(experimental) Optional Existing instance of an EventBridge bus.

        If not provided, the construct will create one.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_bus_interface")
        return typing.cast(typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus], result)

    @builtins.property
    def existing_input_assets_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Existing instance of S3 Bucket object, providing both this and ``bucketInputsAssetsProps`` will cause an error.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_input_assets_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_merged_api(
        self,
    ) -> typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi]:
        '''(experimental) Existing merged Appsync GraphQL api.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_merged_api")
        return typing.cast(typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi], result)

    @builtins.property
    def existing_opensearch_domain(
        self,
    ) -> typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain]:
        '''(experimental) Existing Amazon OpenSearch Service domain.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_opensearch_domain")
        return typing.cast(typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain], result)

    @builtins.property
    def existing_opensearch_serverless_collection(
        self,
    ) -> typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection]:
        '''(experimental) Existing Amazon Amazon OpenSearch Serverless collection.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_opensearch_serverless_collection")
        return typing.cast(typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection], result)

    @builtins.property
    def existing_security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup]:
        '''(experimental) Optional existing security group allowing access to opensearch.

        Used by the lambda functions
        built by this construct. If not provided, the construct will create one.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup], result)

    @builtins.property
    def existing_vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc]:
        '''(experimental) Optional An existing VPC in which to deploy the construct.

        Providing both this and
        vpcProps is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc], result)

    @builtins.property
    def lambda_provisioned_concurrency(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Optional.

        Allows a user to configure
        Lambda provisioned concurrency for consistent performance

        :stability: experimental
        '''
        result = self._values.get("lambda_provisioned_concurrency")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def observability(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable observability.

        Warning: associated cost with the services
        used. Best practive to enable by default.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("observability")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def open_search_secret(
        self,
    ) -> typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret]:
        '''(experimental) Optional.

        SecretsManager secret to authenticate against the OpenSearch Service domain if
        domain is configured with Username/Password.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("open_search_secret")
        return typing.cast(typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_props(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps]:
        '''(experimental) Optional custom properties for a VPC the construct will create.

        This VPC will
        be used by the Lambda functions the construct creates. Providing
        both this and existingVpc is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("vpc_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "QaAppsyncOpensearchProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RagAppsyncStepfnOpensearch(
    BaseClass,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.RagAppsyncStepfnOpensearch",
):
    '''
    :stability: experimental
    :summary: The RagAppsyncStepfnOpensearch class.
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        open_search_index_name: builtins.str,
        bucket_inputs_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        bucket_processed_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_embeddings_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_file_transformer_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_input_validation_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_input_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_opensearch_domain: typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain] = None,
        existing_opensearch_serverless_collection: typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection] = None,
        existing_processed_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        observability: typing.Optional[builtins.bool] = None,
        open_search_secret: typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret] = None,
        stage: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a scope-unique id.
        :param cognito_user_pool: (experimental) Cognito user pool used for authentication. Default: - None
        :param open_search_index_name: (experimental) Index name for the OpenSearch Service. Default: - None
        :param bucket_inputs_assets_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingInputAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param bucket_processed_assets_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingProcessedAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param custom_embeddings_docker_lambda_props: (experimental) Optional. Allows to provide Embeddings custom lambda code and settings instead of the existing
        :param custom_file_transformer_docker_lambda_props: (experimental) Optional. Allows to provide File Transformer custom lambda code and settings instead of the existing
        :param custom_input_validation_docker_lambda_props: (experimental) Optional. Allows to provide Input Validation custom lambda code and settings instead of the existing
        :param existing_bus_interface: (experimental) Optional Existing instance of an EventBridge bus. If not provided, the construct will create one. Default: - None
        :param existing_input_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketInputsAssetsProps`` will cause an error. Default: - None
        :param existing_merged_api: (experimental) Existing merged Appsync GraphQL api. Default: - None
        :param existing_opensearch_domain: (experimental) Optional existing Amazon OpenSearch Service domain. Default: - None
        :param existing_opensearch_serverless_collection: (experimental) Optional existing Amazon Amazon OpenSearch Serverless collection. Default: - None
        :param existing_processed_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketProcessedAssetsProps`` will cause an error. Default: - None
        :param existing_security_group: (experimental) Optional existing security group allowing access to opensearch. Used by the lambda functions built by this construct. If not provided, the construct will create one. Default: - none
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param open_search_secret: (experimental) Optional. SecretsManager secret to authenticate against the OpenSearch Service domain if domain is configured with Username/Password. Default: - None
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the Lambda functions the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: Constructs a new instance of the RagAppsyncStepfnOpensearch class.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b47682f30dcd4f711daa26c544bfc9e0888d2931d0711782d9e99f42de2ba10)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = RagAppsyncStepfnOpensearchProps(
            cognito_user_pool=cognito_user_pool,
            open_search_index_name=open_search_index_name,
            bucket_inputs_assets_props=bucket_inputs_assets_props,
            bucket_processed_assets_props=bucket_processed_assets_props,
            custom_embeddings_docker_lambda_props=custom_embeddings_docker_lambda_props,
            custom_file_transformer_docker_lambda_props=custom_file_transformer_docker_lambda_props,
            custom_input_validation_docker_lambda_props=custom_input_validation_docker_lambda_props,
            existing_bus_interface=existing_bus_interface,
            existing_input_assets_bucket_obj=existing_input_assets_bucket_obj,
            existing_merged_api=existing_merged_api,
            existing_opensearch_domain=existing_opensearch_domain,
            existing_opensearch_serverless_collection=existing_opensearch_serverless_collection,
            existing_processed_assets_bucket_obj=existing_processed_assets_bucket_obj,
            existing_security_group=existing_security_group,
            existing_vpc=existing_vpc,
            observability=observability,
            open_search_secret=open_search_secret,
            stage=stage,
            vpc_props=vpc_props,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="embeddingsLambdaFunction")
    def embeddings_lambda_function(
        self,
    ) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of lambda.DockerImageFunction used for the embeddings job created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "embeddingsLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="fileTransformerLambdaFunction")
    def file_transformer_lambda_function(
        self,
    ) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of lambda.DockerImageFunction used for the file transformer job created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "fileTransformerLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="graphqlApi")
    def graphql_api(self) -> _aws_cdk_aws_appsync_ceddda9d.IGraphqlApi:
        '''(experimental) Returns an instance of appsync.IGraphqlApi created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_appsync_ceddda9d.IGraphqlApi, jsii.get(self, "graphqlApi"))

    @builtins.property
    @jsii.member(jsii_name="ingestionBus")
    def ingestion_bus(self) -> _aws_cdk_aws_events_ceddda9d.IEventBus:
        '''(experimental) Returns the instance of events.IEventBus used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_events_ceddda9d.IEventBus, jsii.get(self, "ingestionBus"))

    @builtins.property
    @jsii.member(jsii_name="inputValidationLambdaFunction")
    def input_validation_lambda_function(
        self,
    ) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of lambda.DockerImageFunction used for the input validation job created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "inputValidationLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="s3InputAssetsBucketInterface")
    def s3_input_assets_bucket_interface(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns an instance of s3.IBucket created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "s3InputAssetsBucketInterface"))

    @builtins.property
    @jsii.member(jsii_name="s3ProcessedAssetsBucketInterface")
    def s3_processed_assets_bucket_interface(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns an instance of s3.IBucket created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "s3ProcessedAssetsBucketInterface"))

    @builtins.property
    @jsii.member(jsii_name="securityGroup")
    def security_group(self) -> _aws_cdk_aws_ec2_ceddda9d.ISecurityGroup:
        '''(experimental) Returns the instance of ec2.ISecurityGroup used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup, jsii.get(self, "securityGroup"))

    @builtins.property
    @jsii.member(jsii_name="stateMachine")
    def state_machine(self) -> _aws_cdk_aws_stepfunctions_ceddda9d.StateMachine:
        '''(experimental) Returns an instance of stepfn.StateMachine created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_stepfunctions_ceddda9d.StateMachine, jsii.get(self, "stateMachine"))

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _aws_cdk_aws_ec2_ceddda9d.IVpc:
        '''(experimental) Returns the instance of ec2.IVpc used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.IVpc, jsii.get(self, "vpc"))

    @builtins.property
    @jsii.member(jsii_name="s3InputAssetsBucket")
    def s3_input_assets_bucket(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket]:
        '''(experimental) Returns an instance of s3.Bucket created by the construct. IMPORTANT: If existingInputAssetsBucketObj was provided in Pattern Construct Props, this property will be undefined.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket], jsii.get(self, "s3InputAssetsBucket"))

    @builtins.property
    @jsii.member(jsii_name="s3ProcessedAssetsBucket")
    def s3_processed_assets_bucket(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket]:
        '''(experimental) Returns an instance of s3.Bucket created by the construct. IMPORTANT: If existingProcessedAssetsBucketObj was provided in Pattern Construct Props, this property will be undefined.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.Bucket], jsii.get(self, "s3ProcessedAssetsBucket"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.RagAppsyncStepfnOpensearchProps",
    jsii_struct_bases=[],
    name_mapping={
        "cognito_user_pool": "cognitoUserPool",
        "open_search_index_name": "openSearchIndexName",
        "bucket_inputs_assets_props": "bucketInputsAssetsProps",
        "bucket_processed_assets_props": "bucketProcessedAssetsProps",
        "custom_embeddings_docker_lambda_props": "customEmbeddingsDockerLambdaProps",
        "custom_file_transformer_docker_lambda_props": "customFileTransformerDockerLambdaProps",
        "custom_input_validation_docker_lambda_props": "customInputValidationDockerLambdaProps",
        "existing_bus_interface": "existingBusInterface",
        "existing_input_assets_bucket_obj": "existingInputAssetsBucketObj",
        "existing_merged_api": "existingMergedApi",
        "existing_opensearch_domain": "existingOpensearchDomain",
        "existing_opensearch_serverless_collection": "existingOpensearchServerlessCollection",
        "existing_processed_assets_bucket_obj": "existingProcessedAssetsBucketObj",
        "existing_security_group": "existingSecurityGroup",
        "existing_vpc": "existingVpc",
        "observability": "observability",
        "open_search_secret": "openSearchSecret",
        "stage": "stage",
        "vpc_props": "vpcProps",
    },
)
class RagAppsyncStepfnOpensearchProps:
    def __init__(
        self,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        open_search_index_name: builtins.str,
        bucket_inputs_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        bucket_processed_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_embeddings_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_file_transformer_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_input_validation_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_input_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_opensearch_domain: typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain] = None,
        existing_opensearch_serverless_collection: typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection] = None,
        existing_processed_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        observability: typing.Optional[builtins.bool] = None,
        open_search_secret: typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret] = None,
        stage: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''(experimental) The properties for the RagAppsyncStepfnOpensearchProps class.

        :param cognito_user_pool: (experimental) Cognito user pool used for authentication. Default: - None
        :param open_search_index_name: (experimental) Index name for the OpenSearch Service. Default: - None
        :param bucket_inputs_assets_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingInputAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param bucket_processed_assets_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingProcessedAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param custom_embeddings_docker_lambda_props: (experimental) Optional. Allows to provide Embeddings custom lambda code and settings instead of the existing
        :param custom_file_transformer_docker_lambda_props: (experimental) Optional. Allows to provide File Transformer custom lambda code and settings instead of the existing
        :param custom_input_validation_docker_lambda_props: (experimental) Optional. Allows to provide Input Validation custom lambda code and settings instead of the existing
        :param existing_bus_interface: (experimental) Optional Existing instance of an EventBridge bus. If not provided, the construct will create one. Default: - None
        :param existing_input_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketInputsAssetsProps`` will cause an error. Default: - None
        :param existing_merged_api: (experimental) Existing merged Appsync GraphQL api. Default: - None
        :param existing_opensearch_domain: (experimental) Optional existing Amazon OpenSearch Service domain. Default: - None
        :param existing_opensearch_serverless_collection: (experimental) Optional existing Amazon Amazon OpenSearch Serverless collection. Default: - None
        :param existing_processed_assets_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketProcessedAssetsProps`` will cause an error. Default: - None
        :param existing_security_group: (experimental) Optional existing security group allowing access to opensearch. Used by the lambda functions built by this construct. If not provided, the construct will create one. Default: - none
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param open_search_secret: (experimental) Optional. SecretsManager secret to authenticate against the OpenSearch Service domain if domain is configured with Username/Password. Default: - None
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the Lambda functions the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        '''
        if isinstance(bucket_inputs_assets_props, dict):
            bucket_inputs_assets_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**bucket_inputs_assets_props)
        if isinstance(bucket_processed_assets_props, dict):
            bucket_processed_assets_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**bucket_processed_assets_props)
        if isinstance(custom_embeddings_docker_lambda_props, dict):
            custom_embeddings_docker_lambda_props = DockerLambdaCustomProps(**custom_embeddings_docker_lambda_props)
        if isinstance(custom_file_transformer_docker_lambda_props, dict):
            custom_file_transformer_docker_lambda_props = DockerLambdaCustomProps(**custom_file_transformer_docker_lambda_props)
        if isinstance(custom_input_validation_docker_lambda_props, dict):
            custom_input_validation_docker_lambda_props = DockerLambdaCustomProps(**custom_input_validation_docker_lambda_props)
        if isinstance(vpc_props, dict):
            vpc_props = _aws_cdk_aws_ec2_ceddda9d.VpcProps(**vpc_props)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84410e9ddf3c6c72a9fafb245544fa73e4c8020dfb3e51d2f31e3b9c8bfa846f)
            check_type(argname="argument cognito_user_pool", value=cognito_user_pool, expected_type=type_hints["cognito_user_pool"])
            check_type(argname="argument open_search_index_name", value=open_search_index_name, expected_type=type_hints["open_search_index_name"])
            check_type(argname="argument bucket_inputs_assets_props", value=bucket_inputs_assets_props, expected_type=type_hints["bucket_inputs_assets_props"])
            check_type(argname="argument bucket_processed_assets_props", value=bucket_processed_assets_props, expected_type=type_hints["bucket_processed_assets_props"])
            check_type(argname="argument custom_embeddings_docker_lambda_props", value=custom_embeddings_docker_lambda_props, expected_type=type_hints["custom_embeddings_docker_lambda_props"])
            check_type(argname="argument custom_file_transformer_docker_lambda_props", value=custom_file_transformer_docker_lambda_props, expected_type=type_hints["custom_file_transformer_docker_lambda_props"])
            check_type(argname="argument custom_input_validation_docker_lambda_props", value=custom_input_validation_docker_lambda_props, expected_type=type_hints["custom_input_validation_docker_lambda_props"])
            check_type(argname="argument existing_bus_interface", value=existing_bus_interface, expected_type=type_hints["existing_bus_interface"])
            check_type(argname="argument existing_input_assets_bucket_obj", value=existing_input_assets_bucket_obj, expected_type=type_hints["existing_input_assets_bucket_obj"])
            check_type(argname="argument existing_merged_api", value=existing_merged_api, expected_type=type_hints["existing_merged_api"])
            check_type(argname="argument existing_opensearch_domain", value=existing_opensearch_domain, expected_type=type_hints["existing_opensearch_domain"])
            check_type(argname="argument existing_opensearch_serverless_collection", value=existing_opensearch_serverless_collection, expected_type=type_hints["existing_opensearch_serverless_collection"])
            check_type(argname="argument existing_processed_assets_bucket_obj", value=existing_processed_assets_bucket_obj, expected_type=type_hints["existing_processed_assets_bucket_obj"])
            check_type(argname="argument existing_security_group", value=existing_security_group, expected_type=type_hints["existing_security_group"])
            check_type(argname="argument existing_vpc", value=existing_vpc, expected_type=type_hints["existing_vpc"])
            check_type(argname="argument observability", value=observability, expected_type=type_hints["observability"])
            check_type(argname="argument open_search_secret", value=open_search_secret, expected_type=type_hints["open_search_secret"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument vpc_props", value=vpc_props, expected_type=type_hints["vpc_props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cognito_user_pool": cognito_user_pool,
            "open_search_index_name": open_search_index_name,
        }
        if bucket_inputs_assets_props is not None:
            self._values["bucket_inputs_assets_props"] = bucket_inputs_assets_props
        if bucket_processed_assets_props is not None:
            self._values["bucket_processed_assets_props"] = bucket_processed_assets_props
        if custom_embeddings_docker_lambda_props is not None:
            self._values["custom_embeddings_docker_lambda_props"] = custom_embeddings_docker_lambda_props
        if custom_file_transformer_docker_lambda_props is not None:
            self._values["custom_file_transformer_docker_lambda_props"] = custom_file_transformer_docker_lambda_props
        if custom_input_validation_docker_lambda_props is not None:
            self._values["custom_input_validation_docker_lambda_props"] = custom_input_validation_docker_lambda_props
        if existing_bus_interface is not None:
            self._values["existing_bus_interface"] = existing_bus_interface
        if existing_input_assets_bucket_obj is not None:
            self._values["existing_input_assets_bucket_obj"] = existing_input_assets_bucket_obj
        if existing_merged_api is not None:
            self._values["existing_merged_api"] = existing_merged_api
        if existing_opensearch_domain is not None:
            self._values["existing_opensearch_domain"] = existing_opensearch_domain
        if existing_opensearch_serverless_collection is not None:
            self._values["existing_opensearch_serverless_collection"] = existing_opensearch_serverless_collection
        if existing_processed_assets_bucket_obj is not None:
            self._values["existing_processed_assets_bucket_obj"] = existing_processed_assets_bucket_obj
        if existing_security_group is not None:
            self._values["existing_security_group"] = existing_security_group
        if existing_vpc is not None:
            self._values["existing_vpc"] = existing_vpc
        if observability is not None:
            self._values["observability"] = observability
        if open_search_secret is not None:
            self._values["open_search_secret"] = open_search_secret
        if stage is not None:
            self._values["stage"] = stage
        if vpc_props is not None:
            self._values["vpc_props"] = vpc_props

    @builtins.property
    def cognito_user_pool(self) -> _aws_cdk_aws_cognito_ceddda9d.IUserPool:
        '''(experimental) Cognito user pool used for authentication.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("cognito_user_pool")
        assert result is not None, "Required property 'cognito_user_pool' is missing"
        return typing.cast(_aws_cdk_aws_cognito_ceddda9d.IUserPool, result)

    @builtins.property
    def open_search_index_name(self) -> builtins.str:
        '''(experimental) Index name for the OpenSearch Service.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("open_search_index_name")
        assert result is not None, "Required property 'open_search_index_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def bucket_inputs_assets_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional user provided props to override the default props for the S3 Bucket.

        Providing both this and ``existingInputAssetsBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("bucket_inputs_assets_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def bucket_processed_assets_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional user provided props to override the default props for the S3 Bucket.

        Providing both this and ``existingProcessedAssetsBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("bucket_processed_assets_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def custom_embeddings_docker_lambda_props(
        self,
    ) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide Embeddings custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_embeddings_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def custom_file_transformer_docker_lambda_props(
        self,
    ) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide File Transformer custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_file_transformer_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def custom_input_validation_docker_lambda_props(
        self,
    ) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide Input Validation custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_input_validation_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def existing_bus_interface(
        self,
    ) -> typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus]:
        '''(experimental) Optional Existing instance of an EventBridge bus.

        If not provided, the construct will create one.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_bus_interface")
        return typing.cast(typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus], result)

    @builtins.property
    def existing_input_assets_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Existing instance of S3 Bucket object, providing both this and ``bucketInputsAssetsProps`` will cause an error.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_input_assets_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_merged_api(
        self,
    ) -> typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi]:
        '''(experimental) Existing merged Appsync GraphQL api.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_merged_api")
        return typing.cast(typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi], result)

    @builtins.property
    def existing_opensearch_domain(
        self,
    ) -> typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain]:
        '''(experimental) Optional existing Amazon OpenSearch Service domain.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_opensearch_domain")
        return typing.cast(typing.Optional[_aws_cdk_aws_opensearchservice_ceddda9d.IDomain], result)

    @builtins.property
    def existing_opensearch_serverless_collection(
        self,
    ) -> typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection]:
        '''(experimental) Optional existing Amazon Amazon OpenSearch Serverless collection.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_opensearch_serverless_collection")
        return typing.cast(typing.Optional[_aws_cdk_aws_opensearchserverless_ceddda9d.CfnCollection], result)

    @builtins.property
    def existing_processed_assets_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Existing instance of S3 Bucket object, providing both this and ``bucketProcessedAssetsProps`` will cause an error.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_processed_assets_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup]:
        '''(experimental) Optional existing security group allowing access to opensearch.

        Used by the lambda functions
        built by this construct. If not provided, the construct will create one.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup], result)

    @builtins.property
    def existing_vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc]:
        '''(experimental) Optional An existing VPC in which to deploy the construct.

        Providing both this and
        vpcProps is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc], result)

    @builtins.property
    def observability(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable observability.

        Warning: associated cost with the services
        used. Best practice to enable by default.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("observability")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def open_search_secret(
        self,
    ) -> typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret]:
        '''(experimental) Optional.

        SecretsManager secret to authenticate against the OpenSearch Service domain if
        domain is configured with Username/Password.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("open_search_secret")
        return typing.cast(typing.Optional[_aws_cdk_aws_secretsmanager_ceddda9d.ISecret], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_props(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps]:
        '''(experimental) Optional custom properties for a VPC the construct will create.

        This VPC will
        be used by the Lambda functions the construct creates. Providing
        both this and existingVpc is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("vpc_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RagAppsyncStepfnOpensearchProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SageMakerEndpointBase(
    BaseClass,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.SageMakerEndpointBase",
):
    '''
    :stability: experimental
    '''

    def __init__(self, scope: _constructs_77d1e7e8.Construct, id: builtins.str) -> None:
        '''
        :param scope: -
        :param id: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c37da07d50996bf3815df5522cdacee052c06fde9388dfbbfe2234481a961908)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="createSageMakerRole")
    def _create_sage_maker_role(self) -> _aws_cdk_aws_iam_ceddda9d.Role:
        '''
        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_iam_ceddda9d.Role, jsii.invoke(self, "createSageMakerRole", []))


class _SageMakerEndpointBaseProxy(SageMakerEndpointBase):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, SageMakerEndpointBase).__jsii_proxy_class__ = lambda : _SageMakerEndpointBaseProxy


class SageMakerInstanceType(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.SageMakerInstanceType",
):
    '''(experimental) Supported instance types for SageMaker instance-based production variants.

    :stability: experimental
    '''

    def __init__(self, instance_type: builtins.str) -> None:
        '''
        :param instance_type: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f06ce0cc4b08453b7ab410c78b48041f013a51e2dfa884a02fd5c6e2128fe4e0)
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
        jsii.create(self.__class__, self, [instance_type])

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, instance_type: builtins.str) -> "SageMakerInstanceType":
        '''(experimental) Builds an InstanceType from a given string or token (such as a CfnParameter).

        :param instance_type: An instance type as string.

        :return: A strongly typed InstanceType

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f58c993f4fbd1d7d4666ae23b06ee672f893b6774b448ab48869abff7e44114)
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
        return typing.cast("SageMakerInstanceType", jsii.sinvoke(cls, "of", [instance_type]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''(experimental) Return the instance type as a string.

        :return: The instance type as a string

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C4_2XLARGE")
    def ML_C4_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c4.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C4_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C4_4XLARGE")
    def ML_C4_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c4.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C4_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C4_8XLARGE")
    def ML_C4_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c4.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C4_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C4_LARGE")
    def ML_C4_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c4.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C4_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C4_XLARGE")
    def ML_C4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c4.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C4_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5_18XLARGE")
    def ML_C5_18_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5.18xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5_18XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5_2XLARGE")
    def ML_C5_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5_4XLARGE")
    def ML_C5_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5_9XLARGE")
    def ML_C5_9_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5.9xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5_9XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5_LARGE")
    def ML_C5_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5_XLARGE")
    def ML_C5_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5D_18XLARGE")
    def ML_C5_D_18_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5d.18xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5D_18XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5D_2XLARGE")
    def ML_C5_D_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5d.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5D_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5D_4XLARGE")
    def ML_C5_D_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5d.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5D_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5D_9XLARGE")
    def ML_C5_D_9_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5d.9xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5D_9XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5D_LARGE")
    def ML_C5_D_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5d.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5D_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C5D_XLARGE")
    def ML_C5_D_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c5d.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C5D_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_12XLARGE")
    def ML_C6_I_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_16XLARGE")
    def ML_C6_I_16_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.16xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_16XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_24XLARGE")
    def ML_C6_I_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_2XLARGE")
    def ML_C6_I_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_32XLARGE")
    def ML_C6_I_32_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.32xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_32XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_4XLARGE")
    def ML_C6_I_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_8XLARGE")
    def ML_C6_I_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_LARGE")
    def ML_C6_I_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_C6I_XLARGE")
    def ML_C6_I_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.c6i.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_C6I_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G4DN_12XLARGE")
    def ML_G4_DN_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g4dn.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G4DN_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G4DN_16XLARGE")
    def ML_G4_DN_16_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g4dn.16xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G4DN_16XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G4DN_2XLARGE")
    def ML_G4_DN_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g4dn.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G4DN_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G4DN_4XLARGE")
    def ML_G4_DN_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g4dn.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G4DN_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G4DN_8XLARGE")
    def ML_G4_DN_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g4dn.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G4DN_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G4DN_XLARGE")
    def ML_G4_DN_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g4dn.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G4DN_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_12XLARGE")
    def ML_G5_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_16XLARGE")
    def ML_G5_16_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.16xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_16XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_24XLARGE")
    def ML_G5_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_2XLARGE")
    def ML_G5_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_48XLARGE")
    def ML_G5_48_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.48xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_48XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_4XLARGE")
    def ML_G5_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_8XLARGE")
    def ML_G5_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_G5_XLARGE")
    def ML_G5_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.g5.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_G5_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF1_24XLARGE")
    def ML_INF1_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf1.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF1_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF1_2XLARGE")
    def ML_INF1_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf1.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF1_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF1_6XLARGE")
    def ML_INF1_6_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf1.6xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF1_6XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF1_XLARGE")
    def ML_INF1_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf1.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF1_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF2_24XLARGE")
    def ML_INF2_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf2.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF2_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF2_48XLARGE")
    def ML_INF2_48_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf2.48xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF2_48XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF2_8XLARGE")
    def ML_INF2_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf2.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF2_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_INF2_XLARGE")
    def ML_INF2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.inf2.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_INF2_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M4_10XLARGE")
    def ML_M4_10_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m4.10xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M4_10XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M4_16XLARGE")
    def ML_M4_16_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m4.16xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M4_16XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M4_2XLARGE")
    def ML_M4_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m4.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M4_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M4_4XLARGE")
    def ML_M4_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m4.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M4_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M4_XLARGE")
    def ML_M4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m4.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M4_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5_12XLARGE")
    def ML_M5_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5_24XLARGE")
    def ML_M5_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5_2XLARGE")
    def ML_M5_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5_4XLARGE")
    def ML_M5_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5_LARGE")
    def ML_M5_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5_XLARGE")
    def ML_M5_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5D_12XLARGE")
    def ML_M5_D_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5d.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5D_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5D_24XLARGE")
    def ML_M5_D_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5d.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5D_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5D_2XLARGE")
    def ML_M5_D_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5d.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5D_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5D_4XLARGE")
    def ML_M5_D_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5d.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5D_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5D_LARGE")
    def ML_M5_D_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5d.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5D_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_M5D_XLARGE")
    def ML_M5_D_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.m5d.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_M5D_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P2_16XLARGE")
    def ML_P2_16_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p2.16xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P2_16XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P2_8XLARGE")
    def ML_P2_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p2.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P2_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P2_XLARGE")
    def ML_P2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p2.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P2_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P3_16XLARGE")
    def ML_P3_16_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p3.16xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P3_16XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P3_2XLARGE")
    def ML_P3_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p3.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P3_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P3_8XLARGE")
    def ML_P3_8_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p3.8xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P3_8XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_P4D_24XLARGE")
    def ML_P4_D_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.p4d.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_P4D_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5_12XLARGE")
    def ML_R5_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5_24XLARGE")
    def ML_R5_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5_2XLARGE")
    def ML_R5_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5_4XLARGE")
    def ML_R5_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5_LARGE")
    def ML_R5_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5_XLARGE")
    def ML_R5_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5D_12XLARGE")
    def ML_R5_D_12_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5d.12xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5D_12XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5D_24XLARGE")
    def ML_R5_D_24_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5d.24xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5D_24XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5D_2XLARGE")
    def ML_R5_D_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5d.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5D_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5D_4XLARGE")
    def ML_R5_D_4_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5d.4xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5D_4XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5D_LARGE")
    def ML_R5_D_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5d.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5D_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_R5D_XLARGE")
    def ML_R5_D_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.r5d.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_R5D_XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_T2_2XLARGE")
    def ML_T2_2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.t2.2xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_T2_2XLARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_T2_LARGE")
    def ML_T2_LARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.t2.large.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_T2_LARGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_T2_MEDIUM")
    def ML_T2_MEDIUM(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.t2.medium.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_T2_MEDIUM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ML_T2_XLARGE")
    def ML_T2_XLARGE(cls) -> "SageMakerInstanceType":
        '''(experimental) ml.t2.xlarge.

        :stability: experimental
        '''
        return typing.cast("SageMakerInstanceType", jsii.sget(cls, "ML_T2_XLARGE"))


class SummarizationAppsyncStepfn(
    BaseClass,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.SummarizationAppsyncStepfn",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        bucket_inputs_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        bucket_processed_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_document_reader_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_input_validation_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_summary_generator_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        event_bus_props: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventBusProps, typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_input_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_processed_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        is_file_transformation_required: typing.Optional[builtins.str] = None,
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
        summary_api_name: typing.Optional[builtins.str] = None,
        summary_chain_type: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a a scope-unique id.
        :param cognito_user_pool: (experimental) Required. Cognito user pool used for authentication. Default: - None
        :param bucket_inputs_assets_props: (experimental) Optional. User provided props to override the default props for the S3 Bucket. Providing both this and ``existingInputAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param bucket_processed_assets_props: (experimental) Optional. User provided props to override the default props for the S3 Bucket. Providing both this and ``existingProcessedAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param custom_document_reader_docker_lambda_props: (experimental) Optional. Allows to provide Embeddings custom lambda code and settings instead of the existing
        :param custom_input_validation_docker_lambda_props: (experimental) Optional. Allows to provide Input Validation custom lambda code and settings instead of the existing
        :param custom_summary_generator_docker_lambda_props: (experimental) Optional. Allows to provide File Transformer custom lambda code and settings instead of the existing
        :param event_bus_props: (experimental) Optional. A new custom EventBus is created with provided props. Providing existingEventBusInterface and eventBusProps both will result in validation error. Default: - None
        :param existing_bus_interface: (experimental) Optional. Existing instance of EventBus. The summary construct integrate appsync with event bridge' to route the request to step functions. Default: - None
        :param existing_input_assets_bucket_obj: (experimental) Optional. Existing s3 Bucket to store the input document which needs to be summarized. pdf is the supported input document format. If transformed (txt format) file is available then this bucket is optional. Default: - None
        :param existing_merged_api: (experimental) Optional - Existing merged Appsync GraphQL api. Default: - None
        :param existing_processed_assets_bucket_obj: (experimental) Optional. This bucket stores the transformed (txt) assets for generating summary. If None is provided then this contruct will create one. Default: - None
        :param existing_security_group: (experimental) Optional. Security group for the lambda function which this construct will use. If no exisiting security group is provided it will create one from the vpc. Default: - none
        :param existing_vpc: (experimental) Optional. An existing VPC can be used to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param is_file_transformation_required: (experimental) Optional. The summary construct transform the input document into txt format. If the transformation is not required then this flag can be set to false. If set to true then a transformed asset bucket is created which transform the input document from input asset bucket to txt format. Default: - False
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param summary_api_name: (experimental) Optional. User provided Name for summary api on appsync. A graphql api will be created by this construct with this name. Default: 'summaryApi'
        :param summary_chain_type: (experimental) Optional. Chain type defines how to pass the document to LLM. there are three types of chain types. Stuff: Simply "stuff" all your documents into a single prompt. Map-reduce: Summarize each document on it's own in a "map" step and then "reduce" the summaries into a final summary Refine : This constructs a response by looping over the input documents and iteratively updating its answer Default: - Stuff
        :param vpc_props: (experimental) Optional. The construct creates a custom VPC based on vpcProps. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: Constructs a new instance of the SummarizationAppsyncStepfn class.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e3d8b9e643131ffd22e254e28d1be23095a03faa91dc469f13c16edf293c7af)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = SummarizationAppsyncStepfnProps(
            cognito_user_pool=cognito_user_pool,
            bucket_inputs_assets_props=bucket_inputs_assets_props,
            bucket_processed_assets_props=bucket_processed_assets_props,
            custom_document_reader_docker_lambda_props=custom_document_reader_docker_lambda_props,
            custom_input_validation_docker_lambda_props=custom_input_validation_docker_lambda_props,
            custom_summary_generator_docker_lambda_props=custom_summary_generator_docker_lambda_props,
            event_bus_props=event_bus_props,
            existing_bus_interface=existing_bus_interface,
            existing_input_assets_bucket_obj=existing_input_assets_bucket_obj,
            existing_merged_api=existing_merged_api,
            existing_processed_assets_bucket_obj=existing_processed_assets_bucket_obj,
            existing_security_group=existing_security_group,
            existing_vpc=existing_vpc,
            is_file_transformation_required=is_file_transformation_required,
            observability=observability,
            stage=stage,
            summary_api_name=summary_api_name,
            summary_chain_type=summary_chain_type,
            vpc_props=vpc_props,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="documentReaderLambdaFunction")
    def document_reader_lambda_function(
        self,
    ) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of lambda.DockerImageFunction used for the document reading job created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "documentReaderLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="eventBridgeBus")
    def event_bridge_bus(self) -> _aws_cdk_aws_events_ceddda9d.IEventBus:
        '''(experimental) Returns an instance of events.IEventBus created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_events_ceddda9d.IEventBus, jsii.get(self, "eventBridgeBus"))

    @builtins.property
    @jsii.member(jsii_name="graphqlApi")
    def graphql_api(self) -> _aws_cdk_aws_appsync_ceddda9d.IGraphqlApi:
        '''(experimental) Returns an instance of appsync.CfnGraphQLApi for summary created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_appsync_ceddda9d.IGraphqlApi, jsii.get(self, "graphqlApi"))

    @builtins.property
    @jsii.member(jsii_name="graphqlApiId")
    def graphql_api_id(self) -> builtins.str:
        '''(experimental) Graphql Api Id value.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "graphqlApiId"))

    @builtins.property
    @jsii.member(jsii_name="graphqlUrl")
    def graphql_url(self) -> builtins.str:
        '''(experimental) Graphql Url value.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "graphqlUrl"))

    @builtins.property
    @jsii.member(jsii_name="inputAssetBucket")
    def input_asset_bucket(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns the instance of s3.IBucket used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "inputAssetBucket"))

    @builtins.property
    @jsii.member(jsii_name="inputValidationLambdaFunction")
    def input_validation_lambda_function(
        self,
    ) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of lambda.DockerImageFunction used for the input validation job created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "inputValidationLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="processedAssetBucket")
    def processed_asset_bucket(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns the instance of s3.IBucket used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "processedAssetBucket"))

    @builtins.property
    @jsii.member(jsii_name="securityGroup")
    def security_group(self) -> _aws_cdk_aws_ec2_ceddda9d.ISecurityGroup:
        '''(experimental) Returns the instance of ec2.ISecurityGroup used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup, jsii.get(self, "securityGroup"))

    @builtins.property
    @jsii.member(jsii_name="stateMachine")
    def state_machine(self) -> _aws_cdk_aws_stepfunctions_ceddda9d.StateMachine:
        '''(experimental) Step function.

        :default: - fieldLogLevel - None

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_stepfunctions_ceddda9d.StateMachine, jsii.get(self, "stateMachine"))

    @builtins.property
    @jsii.member(jsii_name="summaryGeneratorLambdaFunction")
    def summary_generator_lambda_function(
        self,
    ) -> _aws_cdk_aws_lambda_ceddda9d.DockerImageFunction:
        '''(experimental) Returns an instance of lambda.DockerImageFunction used for the summary generation job created by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_lambda_ceddda9d.DockerImageFunction, jsii.get(self, "summaryGeneratorLambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _aws_cdk_aws_ec2_ceddda9d.IVpc:
        '''(experimental) Returns the instance of ec2.IVpc used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.IVpc, jsii.get(self, "vpc"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.SummarizationAppsyncStepfnProps",
    jsii_struct_bases=[],
    name_mapping={
        "cognito_user_pool": "cognitoUserPool",
        "bucket_inputs_assets_props": "bucketInputsAssetsProps",
        "bucket_processed_assets_props": "bucketProcessedAssetsProps",
        "custom_document_reader_docker_lambda_props": "customDocumentReaderDockerLambdaProps",
        "custom_input_validation_docker_lambda_props": "customInputValidationDockerLambdaProps",
        "custom_summary_generator_docker_lambda_props": "customSummaryGeneratorDockerLambdaProps",
        "event_bus_props": "eventBusProps",
        "existing_bus_interface": "existingBusInterface",
        "existing_input_assets_bucket_obj": "existingInputAssetsBucketObj",
        "existing_merged_api": "existingMergedApi",
        "existing_processed_assets_bucket_obj": "existingProcessedAssetsBucketObj",
        "existing_security_group": "existingSecurityGroup",
        "existing_vpc": "existingVpc",
        "is_file_transformation_required": "isFileTransformationRequired",
        "observability": "observability",
        "stage": "stage",
        "summary_api_name": "summaryApiName",
        "summary_chain_type": "summaryChainType",
        "vpc_props": "vpcProps",
    },
)
class SummarizationAppsyncStepfnProps:
    def __init__(
        self,
        *,
        cognito_user_pool: _aws_cdk_aws_cognito_ceddda9d.IUserPool,
        bucket_inputs_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        bucket_processed_assets_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_document_reader_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_input_validation_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_summary_generator_docker_lambda_props: typing.Optional[typing.Union[DockerLambdaCustomProps, typing.Dict[builtins.str, typing.Any]]] = None,
        event_bus_props: typing.Optional[typing.Union[_aws_cdk_aws_events_ceddda9d.EventBusProps, typing.Dict[builtins.str, typing.Any]]] = None,
        existing_bus_interface: typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus] = None,
        existing_input_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_merged_api: typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi] = None,
        existing_processed_assets_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_security_group: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        is_file_transformation_required: typing.Optional[builtins.str] = None,
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
        summary_api_name: typing.Optional[builtins.str] = None,
        summary_chain_type: typing.Optional[builtins.str] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param cognito_user_pool: (experimental) Required. Cognito user pool used for authentication. Default: - None
        :param bucket_inputs_assets_props: (experimental) Optional. User provided props to override the default props for the S3 Bucket. Providing both this and ``existingInputAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param bucket_processed_assets_props: (experimental) Optional. User provided props to override the default props for the S3 Bucket. Providing both this and ``existingProcessedAssetsBucketObj`` will cause an error. Default: - Default props are used
        :param custom_document_reader_docker_lambda_props: (experimental) Optional. Allows to provide Embeddings custom lambda code and settings instead of the existing
        :param custom_input_validation_docker_lambda_props: (experimental) Optional. Allows to provide Input Validation custom lambda code and settings instead of the existing
        :param custom_summary_generator_docker_lambda_props: (experimental) Optional. Allows to provide File Transformer custom lambda code and settings instead of the existing
        :param event_bus_props: (experimental) Optional. A new custom EventBus is created with provided props. Providing existingEventBusInterface and eventBusProps both will result in validation error. Default: - None
        :param existing_bus_interface: (experimental) Optional. Existing instance of EventBus. The summary construct integrate appsync with event bridge' to route the request to step functions. Default: - None
        :param existing_input_assets_bucket_obj: (experimental) Optional. Existing s3 Bucket to store the input document which needs to be summarized. pdf is the supported input document format. If transformed (txt format) file is available then this bucket is optional. Default: - None
        :param existing_merged_api: (experimental) Optional - Existing merged Appsync GraphQL api. Default: - None
        :param existing_processed_assets_bucket_obj: (experimental) Optional. This bucket stores the transformed (txt) assets for generating summary. If None is provided then this contruct will create one. Default: - None
        :param existing_security_group: (experimental) Optional. Security group for the lambda function which this construct will use. If no exisiting security group is provided it will create one from the vpc. Default: - none
        :param existing_vpc: (experimental) Optional. An existing VPC can be used to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param is_file_transformation_required: (experimental) Optional. The summary construct transform the input document into txt format. If the transformation is not required then this flag can be set to false. If set to true then a transformed asset bucket is created which transform the input document from input asset bucket to txt format. Default: - False
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param summary_api_name: (experimental) Optional. User provided Name for summary api on appsync. A graphql api will be created by this construct with this name. Default: 'summaryApi'
        :param summary_chain_type: (experimental) Optional. Chain type defines how to pass the document to LLM. there are three types of chain types. Stuff: Simply "stuff" all your documents into a single prompt. Map-reduce: Summarize each document on it's own in a "map" step and then "reduce" the summaries into a final summary Refine : This constructs a response by looping over the input documents and iteratively updating its answer Default: - Stuff
        :param vpc_props: (experimental) Optional. The construct creates a custom VPC based on vpcProps. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        '''
        if isinstance(bucket_inputs_assets_props, dict):
            bucket_inputs_assets_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**bucket_inputs_assets_props)
        if isinstance(bucket_processed_assets_props, dict):
            bucket_processed_assets_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**bucket_processed_assets_props)
        if isinstance(custom_document_reader_docker_lambda_props, dict):
            custom_document_reader_docker_lambda_props = DockerLambdaCustomProps(**custom_document_reader_docker_lambda_props)
        if isinstance(custom_input_validation_docker_lambda_props, dict):
            custom_input_validation_docker_lambda_props = DockerLambdaCustomProps(**custom_input_validation_docker_lambda_props)
        if isinstance(custom_summary_generator_docker_lambda_props, dict):
            custom_summary_generator_docker_lambda_props = DockerLambdaCustomProps(**custom_summary_generator_docker_lambda_props)
        if isinstance(event_bus_props, dict):
            event_bus_props = _aws_cdk_aws_events_ceddda9d.EventBusProps(**event_bus_props)
        if isinstance(vpc_props, dict):
            vpc_props = _aws_cdk_aws_ec2_ceddda9d.VpcProps(**vpc_props)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4c70798ec1d57933ac8ff9b3e4952de2c9f173958f717d396052ee110b0045bc)
            check_type(argname="argument cognito_user_pool", value=cognito_user_pool, expected_type=type_hints["cognito_user_pool"])
            check_type(argname="argument bucket_inputs_assets_props", value=bucket_inputs_assets_props, expected_type=type_hints["bucket_inputs_assets_props"])
            check_type(argname="argument bucket_processed_assets_props", value=bucket_processed_assets_props, expected_type=type_hints["bucket_processed_assets_props"])
            check_type(argname="argument custom_document_reader_docker_lambda_props", value=custom_document_reader_docker_lambda_props, expected_type=type_hints["custom_document_reader_docker_lambda_props"])
            check_type(argname="argument custom_input_validation_docker_lambda_props", value=custom_input_validation_docker_lambda_props, expected_type=type_hints["custom_input_validation_docker_lambda_props"])
            check_type(argname="argument custom_summary_generator_docker_lambda_props", value=custom_summary_generator_docker_lambda_props, expected_type=type_hints["custom_summary_generator_docker_lambda_props"])
            check_type(argname="argument event_bus_props", value=event_bus_props, expected_type=type_hints["event_bus_props"])
            check_type(argname="argument existing_bus_interface", value=existing_bus_interface, expected_type=type_hints["existing_bus_interface"])
            check_type(argname="argument existing_input_assets_bucket_obj", value=existing_input_assets_bucket_obj, expected_type=type_hints["existing_input_assets_bucket_obj"])
            check_type(argname="argument existing_merged_api", value=existing_merged_api, expected_type=type_hints["existing_merged_api"])
            check_type(argname="argument existing_processed_assets_bucket_obj", value=existing_processed_assets_bucket_obj, expected_type=type_hints["existing_processed_assets_bucket_obj"])
            check_type(argname="argument existing_security_group", value=existing_security_group, expected_type=type_hints["existing_security_group"])
            check_type(argname="argument existing_vpc", value=existing_vpc, expected_type=type_hints["existing_vpc"])
            check_type(argname="argument is_file_transformation_required", value=is_file_transformation_required, expected_type=type_hints["is_file_transformation_required"])
            check_type(argname="argument observability", value=observability, expected_type=type_hints["observability"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument summary_api_name", value=summary_api_name, expected_type=type_hints["summary_api_name"])
            check_type(argname="argument summary_chain_type", value=summary_chain_type, expected_type=type_hints["summary_chain_type"])
            check_type(argname="argument vpc_props", value=vpc_props, expected_type=type_hints["vpc_props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cognito_user_pool": cognito_user_pool,
        }
        if bucket_inputs_assets_props is not None:
            self._values["bucket_inputs_assets_props"] = bucket_inputs_assets_props
        if bucket_processed_assets_props is not None:
            self._values["bucket_processed_assets_props"] = bucket_processed_assets_props
        if custom_document_reader_docker_lambda_props is not None:
            self._values["custom_document_reader_docker_lambda_props"] = custom_document_reader_docker_lambda_props
        if custom_input_validation_docker_lambda_props is not None:
            self._values["custom_input_validation_docker_lambda_props"] = custom_input_validation_docker_lambda_props
        if custom_summary_generator_docker_lambda_props is not None:
            self._values["custom_summary_generator_docker_lambda_props"] = custom_summary_generator_docker_lambda_props
        if event_bus_props is not None:
            self._values["event_bus_props"] = event_bus_props
        if existing_bus_interface is not None:
            self._values["existing_bus_interface"] = existing_bus_interface
        if existing_input_assets_bucket_obj is not None:
            self._values["existing_input_assets_bucket_obj"] = existing_input_assets_bucket_obj
        if existing_merged_api is not None:
            self._values["existing_merged_api"] = existing_merged_api
        if existing_processed_assets_bucket_obj is not None:
            self._values["existing_processed_assets_bucket_obj"] = existing_processed_assets_bucket_obj
        if existing_security_group is not None:
            self._values["existing_security_group"] = existing_security_group
        if existing_vpc is not None:
            self._values["existing_vpc"] = existing_vpc
        if is_file_transformation_required is not None:
            self._values["is_file_transformation_required"] = is_file_transformation_required
        if observability is not None:
            self._values["observability"] = observability
        if stage is not None:
            self._values["stage"] = stage
        if summary_api_name is not None:
            self._values["summary_api_name"] = summary_api_name
        if summary_chain_type is not None:
            self._values["summary_chain_type"] = summary_chain_type
        if vpc_props is not None:
            self._values["vpc_props"] = vpc_props

    @builtins.property
    def cognito_user_pool(self) -> _aws_cdk_aws_cognito_ceddda9d.IUserPool:
        '''(experimental) Required.

        Cognito user pool used for authentication.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("cognito_user_pool")
        assert result is not None, "Required property 'cognito_user_pool' is missing"
        return typing.cast(_aws_cdk_aws_cognito_ceddda9d.IUserPool, result)

    @builtins.property
    def bucket_inputs_assets_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional.

        User provided props to override the default props for the S3 Bucket.
        Providing both this and ``existingInputAssetsBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("bucket_inputs_assets_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def bucket_processed_assets_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional.

        User provided props to override the default props for the S3 Bucket.
        Providing both this and ``existingProcessedAssetsBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("bucket_processed_assets_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def custom_document_reader_docker_lambda_props(
        self,
    ) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide Embeddings custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_document_reader_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def custom_input_validation_docker_lambda_props(
        self,
    ) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide Input Validation custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_input_validation_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def custom_summary_generator_docker_lambda_props(
        self,
    ) -> typing.Optional[DockerLambdaCustomProps]:
        '''(experimental) Optional.

        Allows to provide File Transformer custom lambda code
        and settings instead of the existing

        :stability: experimental
        '''
        result = self._values.get("custom_summary_generator_docker_lambda_props")
        return typing.cast(typing.Optional[DockerLambdaCustomProps], result)

    @builtins.property
    def event_bus_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_events_ceddda9d.EventBusProps]:
        '''(experimental) Optional.

        A new custom EventBus is created with provided props.
        Providing existingEventBusInterface and eventBusProps both will result in validation error.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("event_bus_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_events_ceddda9d.EventBusProps], result)

    @builtins.property
    def existing_bus_interface(
        self,
    ) -> typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus]:
        '''(experimental) Optional.

        Existing instance of EventBus. The summary construct integrate appsync with event bridge'
        to route the request to step functions.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_bus_interface")
        return typing.cast(typing.Optional[_aws_cdk_aws_events_ceddda9d.IEventBus], result)

    @builtins.property
    def existing_input_assets_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Optional.

        Existing s3 Bucket to store the input document which needs to be summarized.
        pdf is the supported input document format. If transformed (txt format) file is
        available then this bucket is optional.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_input_assets_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_merged_api(
        self,
    ) -> typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi]:
        '''(experimental) Optional - Existing merged Appsync GraphQL api.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_merged_api")
        return typing.cast(typing.Optional[_aws_cdk_aws_appsync_ceddda9d.CfnGraphQLApi], result)

    @builtins.property
    def existing_processed_assets_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Optional.

        This bucket stores the transformed (txt) assets for generating summary.
        If None is provided then this contruct will create one.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_processed_assets_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_security_group(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup]:
        '''(experimental) Optional.

        Security group for the lambda function which this construct will use.
        If no exisiting security group is provided it will create one from the vpc.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_security_group")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.ISecurityGroup], result)

    @builtins.property
    def existing_vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc]:
        '''(experimental) Optional.

        An existing VPC can be used to deploy the construct.
        Providing both this and vpcProps is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc], result)

    @builtins.property
    def is_file_transformation_required(self) -> typing.Optional[builtins.str]:
        '''(experimental) Optional.

        The summary construct transform the input document into txt format. If the
        transformation is not required then this flag can be set to false. If set to true
        then a transformed asset bucket is created which transform the input document from
        input asset bucket to txt format.

        :default: - False

        :stability: experimental
        '''
        result = self._values.get("is_file_transformation_required")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def observability(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable observability.

        Warning: associated cost with the services
        used. Best practice to enable by default.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("observability")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def summary_api_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Optional.

        User provided Name for summary api on appsync.
        A graphql api will be created by this construct with this name.

        :default: 'summaryApi'

        :stability: experimental
        '''
        result = self._values.get("summary_api_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def summary_chain_type(self) -> typing.Optional[builtins.str]:
        '''(experimental) Optional.

        Chain type defines how to pass the document to LLM.
        there are three types of chain types.
        Stuff: Simply "stuff" all your documents into a single prompt.
        Map-reduce: Summarize each document on it's own in a "map" step and then "reduce" the summaries into a final summary
        Refine :  This constructs a response by looping over the input documents and iteratively updating its answer

        :default: - Stuff

        :stability: experimental
        '''
        result = self._values.get("summary_chain_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_props(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps]:
        '''(experimental) Optional.

        The construct creates a custom VPC based on vpcProps.
        Providing both this and existingVpc is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("vpc_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SummarizationAppsyncStepfnProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class WebCrawler(
    BaseClass,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/generative-ai-cdk-constructs.WebCrawler",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        bucket_output_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        enable_lambda_crawler: typing.Optional[builtins.bool] = None,
        existing_output_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Union[CrawlerTarget, typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param scope: - represents the scope for all the resources.
        :param id: - this is a a scope-unique id.
        :param bucket_output_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingOutputBucketObj`` will cause an error. Default: - Default props are used
        :param enable_lambda_crawler: (experimental) Deploy Lambda crawler. Default: - false
        :param existing_output_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketOutputProps`` will cause an error. Default: - None
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param targets: (experimental) Targets to be crawled. Default: - none
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the compute resources the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        :access: public
        :since: 0.0.0
        :summary: Constructs a new instance of the WebCrawler class.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__83cae1407f7e6b93373e9c0ed87f392b4683d050d05805cdb41420ce9fb62b90)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = WebCrawlerProps(
            bucket_output_props=bucket_output_props,
            enable_lambda_crawler=enable_lambda_crawler,
            existing_output_bucket_obj=existing_output_bucket_obj,
            existing_vpc=existing_vpc,
            observability=observability,
            stage=stage,
            targets=targets,
            vpc_props=vpc_props,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="dataBucket")
    def data_bucket(self) -> _aws_cdk_aws_s3_ceddda9d.IBucket:
        '''(experimental) Returns the instance of S3 bucket used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_s3_ceddda9d.IBucket, jsii.get(self, "dataBucket"))

    @builtins.property
    @jsii.member(jsii_name="jobQueue")
    def job_queue(self) -> _aws_cdk_aws_batch_ceddda9d.IJobQueue:
        '''(experimental) Returns the instance of JobQueue used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_batch_ceddda9d.IJobQueue, jsii.get(self, "jobQueue"))

    @builtins.property
    @jsii.member(jsii_name="jobsTable")
    def jobs_table(self) -> _aws_cdk_aws_dynamodb_ceddda9d.ITable:
        '''(experimental) Returns the instance of Jobs DynamoDB table.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_dynamodb_ceddda9d.ITable, jsii.get(self, "jobsTable"))

    @builtins.property
    @jsii.member(jsii_name="lambdaCrawlerApiSchemaPath")
    def lambda_crawler_api_schema_path(self) -> builtins.str:
        '''(experimental) Lambda crawler API schema path.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "lambdaCrawlerApiSchemaPath"))

    @builtins.property
    @jsii.member(jsii_name="snsTopic")
    def sns_topic(self) -> _aws_cdk_aws_sns_ceddda9d.ITopic:
        '''(experimental) Returns the instance of SNS Topic used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_sns_ceddda9d.ITopic, jsii.get(self, "snsTopic"))

    @builtins.property
    @jsii.member(jsii_name="targetsTable")
    def targets_table(self) -> _aws_cdk_aws_dynamodb_ceddda9d.ITable:
        '''(experimental) Returns the instance of Targets DynamoDB table.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_dynamodb_ceddda9d.ITable, jsii.get(self, "targetsTable"))

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _aws_cdk_aws_ec2_ceddda9d.IVpc:
        '''(experimental) Returns the instance of ec2.IVpc used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_ec2_ceddda9d.IVpc, jsii.get(self, "vpc"))

    @builtins.property
    @jsii.member(jsii_name="webCrawlerJobDefinition")
    def web_crawler_job_definition(self) -> _aws_cdk_aws_batch_ceddda9d.IJobDefinition:
        '''(experimental) Returns the instance of JobDefinition used by the construct.

        :stability: experimental
        '''
        return typing.cast(_aws_cdk_aws_batch_ceddda9d.IJobDefinition, jsii.get(self, "webCrawlerJobDefinition"))

    @builtins.property
    @jsii.member(jsii_name="lambdaCrawler")
    def lambda_crawler(self) -> typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction]:
        '''(experimental) Lambda crawler.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_lambda_ceddda9d.IFunction], jsii.get(self, "lambdaCrawler"))


@jsii.data_type(
    jsii_type="@cdklabs/generative-ai-cdk-constructs.WebCrawlerProps",
    jsii_struct_bases=[],
    name_mapping={
        "bucket_output_props": "bucketOutputProps",
        "enable_lambda_crawler": "enableLambdaCrawler",
        "existing_output_bucket_obj": "existingOutputBucketObj",
        "existing_vpc": "existingVpc",
        "observability": "observability",
        "stage": "stage",
        "targets": "targets",
        "vpc_props": "vpcProps",
    },
)
class WebCrawlerProps:
    def __init__(
        self,
        *,
        bucket_output_props: typing.Optional[typing.Union[_aws_cdk_aws_s3_ceddda9d.BucketProps, typing.Dict[builtins.str, typing.Any]]] = None,
        enable_lambda_crawler: typing.Optional[builtins.bool] = None,
        existing_output_bucket_obj: typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket] = None,
        existing_vpc: typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc] = None,
        observability: typing.Optional[builtins.bool] = None,
        stage: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Union[CrawlerTarget, typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_props: typing.Optional[typing.Union[_aws_cdk_aws_ec2_ceddda9d.VpcProps, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param bucket_output_props: (experimental) Optional user provided props to override the default props for the S3 Bucket. Providing both this and ``existingOutputBucketObj`` will cause an error. Default: - Default props are used
        :param enable_lambda_crawler: (experimental) Deploy Lambda crawler. Default: - false
        :param existing_output_bucket_obj: (experimental) Existing instance of S3 Bucket object, providing both this and ``bucketOutputProps`` will cause an error. Default: - None
        :param existing_vpc: (experimental) Optional An existing VPC in which to deploy the construct. Providing both this and vpcProps is an error. Default: - none
        :param observability: (experimental) Enable observability. Warning: associated cost with the services used. Best practice to enable by default. Default: - true
        :param stage: (experimental) Value will be appended to resources name. Default: - _dev
        :param targets: (experimental) Targets to be crawled. Default: - none
        :param vpc_props: (experimental) Optional custom properties for a VPC the construct will create. This VPC will be used by the compute resources the construct creates. Providing both this and existingVpc is an error. Default: - none

        :stability: experimental
        '''
        if isinstance(bucket_output_props, dict):
            bucket_output_props = _aws_cdk_aws_s3_ceddda9d.BucketProps(**bucket_output_props)
        if isinstance(vpc_props, dict):
            vpc_props = _aws_cdk_aws_ec2_ceddda9d.VpcProps(**vpc_props)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4c7ff738b5e2ff1b0ad3e736aef617467e1201e08229fcdd687a71a7c2668dd)
            check_type(argname="argument bucket_output_props", value=bucket_output_props, expected_type=type_hints["bucket_output_props"])
            check_type(argname="argument enable_lambda_crawler", value=enable_lambda_crawler, expected_type=type_hints["enable_lambda_crawler"])
            check_type(argname="argument existing_output_bucket_obj", value=existing_output_bucket_obj, expected_type=type_hints["existing_output_bucket_obj"])
            check_type(argname="argument existing_vpc", value=existing_vpc, expected_type=type_hints["existing_vpc"])
            check_type(argname="argument observability", value=observability, expected_type=type_hints["observability"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument targets", value=targets, expected_type=type_hints["targets"])
            check_type(argname="argument vpc_props", value=vpc_props, expected_type=type_hints["vpc_props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bucket_output_props is not None:
            self._values["bucket_output_props"] = bucket_output_props
        if enable_lambda_crawler is not None:
            self._values["enable_lambda_crawler"] = enable_lambda_crawler
        if existing_output_bucket_obj is not None:
            self._values["existing_output_bucket_obj"] = existing_output_bucket_obj
        if existing_vpc is not None:
            self._values["existing_vpc"] = existing_vpc
        if observability is not None:
            self._values["observability"] = observability
        if stage is not None:
            self._values["stage"] = stage
        if targets is not None:
            self._values["targets"] = targets
        if vpc_props is not None:
            self._values["vpc_props"] = vpc_props

    @builtins.property
    def bucket_output_props(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps]:
        '''(experimental) Optional user provided props to override the default props for the S3 Bucket.

        Providing both this and ``existingOutputBucketObj`` will cause an error.

        :default: - Default props are used

        :stability: experimental
        '''
        result = self._values.get("bucket_output_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.BucketProps], result)

    @builtins.property
    def enable_lambda_crawler(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Deploy Lambda crawler.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("enable_lambda_crawler")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def existing_output_bucket_obj(
        self,
    ) -> typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket]:
        '''(experimental) Existing instance of S3 Bucket object, providing both this and ``bucketOutputProps`` will cause an error.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("existing_output_bucket_obj")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_ceddda9d.IBucket], result)

    @builtins.property
    def existing_vpc(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc]:
        '''(experimental) Optional An existing VPC in which to deploy the construct.

        Providing both this and
        vpcProps is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("existing_vpc")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.IVpc], result)

    @builtins.property
    def observability(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable observability.

        Warning: associated cost with the services
        used. Best practice to enable by default.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("observability")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''(experimental) Value will be appended to resources name.

        :default: - _dev

        :stability: experimental
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def targets(self) -> typing.Optional[typing.List[CrawlerTarget]]:
        '''(experimental) Targets to be crawled.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("targets")
        return typing.cast(typing.Optional[typing.List[CrawlerTarget]], result)

    @builtins.property
    def vpc_props(self) -> typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps]:
        '''(experimental) Optional custom properties for a VPC the construct will create.

        This VPC will
        be used by the compute resources the construct creates. Providing
        both this and existingVpc is an error.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("vpc_props")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_ceddda9d.VpcProps], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "We