"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebCrawler = exports.CrawlerTargetType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const crypto = require("crypto");
const path = require("path");
const url = require("url");
const cdk = require("aws-cdk-lib");
const batch = require("aws-cdk-lib/aws-batch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const ec2 = require("aws-cdk-lib/aws-ec2");
const aws_ecr_assets = require("aws-cdk-lib/aws-ecr-assets");
const ecs = require("aws-cdk-lib/aws-ecs");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const sns = require("aws-cdk-lib/aws-sns");
const cr = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const base_class_2 = require("../../../common/base-class/base-class");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const vpc_helper = require("../../../common/helpers/vpc-helper");
const types_1 = require("../../../patterns/gen-ai/aws-rag-appsync-stepfn-kendra/types");
var CrawlerTargetType;
(function (CrawlerTargetType) {
    CrawlerTargetType["WEBSITE"] = "website";
    CrawlerTargetType["RSS_FEED"] = "rss_feed";
})(CrawlerTargetType || (exports.CrawlerTargetType = CrawlerTargetType = {}));
class WebCrawler extends base_class_2.BaseClass {
    /**
     * @summary Constructs a new instance of the WebCrawler class.
     * @param {Construct} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {WebCrawlerProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingOutputBucketObj,
            bucketProps: props.bucketOutputProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else if (props.vpcProps) {
            this.vpc = new ec2.Vpc(this, 'webCrawlerVpc', props.vpcProps);
        }
        else {
            this.vpc = new ec2.Vpc(this, 'webCrawlerVpc', {
                createInternetGateway: true,
                natGateways: 1,
            });
        }
        // add VPC endpoints for the compute environment
        vpc_helper.AddAwsServiceEndpoint(this, this.vpc, types_1.ServiceEndpointTypeEnum.ECR_API);
        vpc_helper.AddAwsServiceEndpoint(this, this.vpc, types_1.ServiceEndpointTypeEnum.ECR_DKR);
        vpc_helper.AddAwsServiceEndpoint(this, this.vpc, types_1.ServiceEndpointTypeEnum.S3);
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'webCrawlerConstructLogGroup');
        const role = new iam.Role(this, 'webCrawlerConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        const baseProps = {
            stage: props.stage,
            constructName: base_class_1.ConstructName.AWSWEBCRAWLER,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        const targetsTable = new dynamodb.Table(this, 'targetsTable', {
            partitionKey: { name: 'target_url', type: dynamodb.AttributeType.STRING },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            pointInTimeRecovery: true,
            encryption: dynamodb.TableEncryption.AWS_MANAGED,
        });
        const jobsTable = new dynamodb.Table(this, 'jobsTable', {
            partitionKey: { name: 'target_url', type: dynamodb.AttributeType.STRING },
            sortKey: { name: 'job_id', type: dynamodb.AttributeType.STRING },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            pointInTimeRecovery: true,
            encryption: dynamodb.TableEncryption.AWS_MANAGED,
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [
                {
                    expiration: cdk.Duration.days(90),
                },
            ],
        });
        // Bucket containing the output data uploaded by the crawler
        let dataBucket;
        if (!props.existingOutputBucketObj) {
            let tmpBucket;
            if (!props.bucketOutputProps) {
                tmpBucket = new s3.Bucket(this, 'webCrawlerDataBucket' + this.stage, {
                    accessControl: s3.BucketAccessControl.PRIVATE,
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'outputbucket' + this.stage.toLowerCase() + '-' + cdk.Aws.REGION + '-' + cdk.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [
                        {
                            expiration: cdk.Duration.days(90),
                        },
                    ],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'webCrawlerDataBucket' + this.stage, props.bucketOutputProps);
            }
            dataBucket = tmpBucket;
            this.dataBucket = tmpBucket;
        }
        else {
            dataBucket = props.existingOutputBucketObj;
        }
        const snsTopic = new sns.Topic(this, 'webCrawlerTopic', {
            masterKey: kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns'),
        });
        // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
        snsTopic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            sid: 'TopicOwnerOnlyAccess',
            resources: [`${snsTopic.topicArn}`],
            actions: [
                'SNS:Publish',
                'SNS:RemovePermission',
                'SNS:SetTopicAttributes',
                'SNS:DeleteTopic',
                'SNS:ListSubscriptionsByTopic',
                'SNS:GetTopicAttributes',
                'SNS:Receive',
                'SNS:AddPermission',
                'SNS:Subscribe',
            ],
            principals: [new aws_iam_1.AccountPrincipal(cdk.Stack.of(snsTopic).account)],
            effect: aws_iam_1.Effect.ALLOW,
            conditions: {
                StringEquals: {
                    'AWS:SourceOwner': cdk.Stack.of(snsTopic).account,
                },
            },
        }));
        // Apply Topic policy to enforce encryption of data in transit
        snsTopic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            sid: 'HttpsOnly',
            resources: [`${snsTopic.topicArn}`],
            actions: [
                'SNS:Publish',
                'SNS:RemovePermission',
                'SNS:SetTopicAttributes',
                'SNS:DeleteTopic',
                'SNS:ListSubscriptionsByTopic',
                'SNS:GetTopicAttributes',
                'SNS:Receive',
                'SNS:AddPermission',
                'SNS:Subscribe',
            ],
            principals: [new aws_iam_1.AnyPrincipal()],
            effect: aws_iam_1.Effect.DENY,
            conditions: {
                Bool: {
                    'aws:SecureTransport': 'false',
                },
            },
        }));
        const computeEnvironment = new batch.FargateComputeEnvironment(this, 'webCrawlerEnvironment', {
            vpc: this.vpc,
            maxvCpus: 8,
            replaceComputeEnvironment: true,
            updateTimeout: cdk.Duration.minutes(30),
            updateToLatestImageVersion: true,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
        });
        const jobQueue = new batch.JobQueue(this, 'webCrawlerJobQueue', {
            computeEnvironments: [
                {
                    computeEnvironment,
                    order: 1,
                },
            ],
            priority: 1,
        });
        const webCrawlerJobRole = new iam.Role(this, 'webCrawlerJobRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            inlinePolicies: {
                FargateContainerServiceRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage'],
                            effect: iam.Effect.ALLOW,
                            resources: ['arn:' + cdk.Aws.PARTITION + ':ecr:' + cdk.Aws.REGION + ':' + cdk.Aws.ACCOUNT_ID + ':repository/*'],
                        }),
                        new iam.PolicyStatement({
                            actions: ['ecr:GetAuthorizationToken'],
                            effect: iam.Effect.ALLOW,
                            resources: ['*'],
                        }),
                        new iam.PolicyStatement({
                            actions: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                            effect: iam.Effect.ALLOW,
                            resources: ['arn:' + cdk.Aws.PARTITION + ':logs:' + cdk.Aws.REGION + ':' + cdk.Aws.ACCOUNT_ID + ':log-group:*'],
                        }),
                    ],
                }),
            },
        });
        const webCrawlerContainer = new batch.EcsFargateContainerDefinition(this, 'webCrawlerContainer', {
            cpu: 2,
            memory: cdk.Size.mebibytes(6144),
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../../../../resources/gen-ai/aws-web-crawler-container'), {
                platform: aws_ecr_assets.Platform.LINUX_AMD64,
            }),
            jobRole: webCrawlerJobRole,
            environment: {
                AWS_DEFAULT_REGION: cdk.Stack.of(this).region,
                TARGETS_TABLE_NAME: targetsTable.tableName,
                JOBS_TABLE_NAME: jobsTable.tableName,
                DATA_BUCKET_NAME: dataBucket.bucketName,
                SNS_TOPIC_ARN: snsTopic.topicArn,
            },
        });
        targetsTable.grantReadWriteData(webCrawlerJobRole);
        jobsTable.grantReadWriteData(webCrawlerJobRole);
        dataBucket.grantReadWrite(webCrawlerJobRole);
        snsTopic.grantPublish(webCrawlerJobRole);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(webCrawlerContainer.executionRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'CDK CustomResource Provider has a wildcard to invoke any version of the specific Custom Resource function.',
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(webCrawlerContainer.executionRole, [
            {
                id: 'AwsSolutions-IAM4',
                reason: 'The AWSLambdaBasicExecutionRole managed policy is required for ' + 'the Lambda function to write logs to CloudWatch.',
            },
        ]);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(webCrawlerJobRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'CDK CustomResource Provider has a wildcard to invoke any version of the specific Custom Resource function.',
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(webCrawlerJobRole, [
            {
                id: 'AwsSolutions-IAM4',
                reason: 'The AWSLambdaBasicExecutionRole managed policy is required for ' + 'the Lambda function to write logs to CloudWatch.',
            },
        ]);
        const webCrawlerJobDefinition = new batch.EcsJobDefinition(this, 'webCrawlerJob', {
            container: webCrawlerContainer,
            retryAttempts: 1,
            timeout: cdk.Duration.hours(48),
            retryStrategies: [
                batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER),
                batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.custom({
                    onExitCode: '137',
                })),
            ],
        });
        for (const target of props.targets ?? []) {
            let targetUrl = target.url.trim();
            if (!/^https?:\/\//.test(targetUrl.toLowerCase())) {
                targetUrl = `https://${targetUrl}`;
            }
            let target_s3_key = encodeURIComponent(url.parse(targetUrl).hostname?.replace(/\./g, '-').substring(0, 30) ?? targetUrl);
            const hash = crypto
                .createHash('sha256')
                .update(targetUrl)
                .digest('base64')
                .replace(/\+/g, '-')
                .replace(/\//g, '_')
                .substring(0, 12);
            target_s3_key = `${target_s3_key}-${hash}`;
            new cr.AwsCustomResource(this, `target-${target_s3_key}`, {
                onCreate: {
                    service: 'DynamoDB',
                    action: 'putItem',
                    parameters: {
                        TableName: targetsTable.tableArn,
                        Item: {
                            target_url: { S: targetUrl },
                            target_s3_key: { S: target_s3_key },
                            target_type: { S: target.targetType },
                            sitemaps: { L: [] },
                            max_requests: { N: `${target.maxRequests ?? 0}` },
                            max_files: { N: `${target.maxFiles ?? 0}` },
                            download_files: { BOOL: target.downloadFiles ?? true },
                            file_types: { L: target.fileTypes ?? [] },
                            crawl_interval_hours: { N: `${target.crawlIntervalHours ?? 0}` },
                            last_finished_job_id: { S: '' },
                            last_finished_job_date: { S: '' },
                            created_at: { N: `${Date.now()}` },
                            updated_at: { N: `${Date.now()}` },
                        },
                    },
                    physicalResourceId: cr.PhysicalResourceId.of(targetUrl),
                },
                onUpdate: {
                    service: 'DynamoDB',
                    action: 'updateItem',
                    parameters: {
                        TableName: targetsTable.tableArn,
                        Key: {
                            target_url: { S: targetUrl },
                        },
                        UpdateExpression: 'SET target_type = :target_type, max_requests = :max_requests, max_files = :max_files, download_files = :download_files, file_types = :file_types, crawl_interval_hours = :crawl_interval_hours',
                        ExpressionAttributeValues: {
                            ':target_type': { S: target.targetType },
                            ':max_requests': { N: `${target.maxRequests ?? 0}` },
                            ':max_files': { N: `${target.maxFiles ?? 0}` },
                            ':download_files': { BOOL: target.downloadFiles ?? true },
                            ':file_types': { L: target.fileTypes ?? [] },
                            ':crawl_interval_hours': { N: `${target.crawlIntervalHours ?? 0}` },
                        },
                        ReturnValues: 'UPDATED_NEW',
                    },
                },
                onDelete: {
                    service: 'DynamoDB',
                    action: 'deleteItem',
                    parameters: {
                        TableName: targetsTable.tableArn,
                        Key: {
                            target_url: { S: targetUrl },
                        },
                    },
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['dynamodb:PutItem', 'dynamodb:GetItem', 'dynamodb:DeleteItem', 'dynamodb:UpdateItem'],
                        resources: [targetsTable.tableArn],
                    }),
                ]),
            });
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(cdk.Stack.of(this), `${cdk.Stack.of(this).stackName}/AWS${cr.AwsCustomResource.PROVIDER_FUNCTION_UUID.replace(/-/g, '')}/ServiceRole/Resource`, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'CDK CustomResource Lambda uses the AWSLambdaBasicExecutionRole AWS Managed Policy.',
                },
            ]);
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(cdk.Stack.of(this), `${cdk.Stack.of(this).stackName}/AWS${cr.AwsCustomResource.PROVIDER_FUNCTION_UUID.replace(/-/g, '')}/Resource`, [
                {
                    id: 'AwsSolutions-L1',
                    reason: 'Lambda runtime version is managed upstream by CDK.',
                },
            ]);
        }
        const schedulerFunction = new lambda.Function(this, 'webCrawlerSchedulerFunction', {
            runtime: lambda.Runtime.PYTHON_3_12,
            handler: 'lambda.handler',
            functionName: 'lambda_crawler_scheduler' + this.stage,
            tracing: this.lambdaTracing,
            timeout: cdk.Duration.minutes(15),
            memorySize: 256,
            code: lambda.Code.fromAsset(path.join(__dirname, '../../../../lambda/aws-web-crawler-scheduler')),
            logGroup: new logs.LogGroup(this, 'webCrawlerSchedulerLogGroup', { retention: logs.RetentionDays.ONE_WEEK }),
            environment: {
                TARGETS_TABLE_NAME: targetsTable.tableName,
                JOBS_TABLE_NAME: jobsTable.tableName,
                JOB_QUEUE_ARN: jobQueue.jobQueueArn,
                JOB_DEFINITION_ARN: webCrawlerJobDefinition.jobDefinitionArn,
            },
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(schedulerFunction.role, [
            {
                id: 'AwsSolutions-IAM4',
                reason: 'Lambda uses the AWSLambdaBasicExecutionRole AWS Managed Policy.',
            },
        ]);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(schedulerFunction.role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        targetsTable.grantReadWriteData(schedulerFunction);
        jobsTable.grantReadWriteData(schedulerFunction);
        schedulerFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['batch:SubmitJob'],
            resources: [webCrawlerJobDefinition.jobDefinitionArn, jobQueue.jobQueueArn],
        }));
        const rule = new events.Rule(this, 'webCrawlerSchedulerRule', {
            schedule: events.Schedule.expression('cron(0/15 * * * ? *)'),
        });
        rule.addTarget(new targets.LambdaFunction(schedulerFunction));
        if (props.enableLambdaCrawler) {
            const lambdaCrawler = new lambda.Function(this, 'lambdaCrawler', {
                architecture: lambda.Architecture.X86_64,
                functionName: 'lambda_crawler_function' + this.stage,
                tracing: this.lambdaTracing,
                timeout: cdk.Duration.minutes(15),
                memorySize: 2048,
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_20_X,
                code: lambda.Code.fromDockerBuild(path.join(__dirname, '../../../../lambda/aws-web-crawler-lambda')),
            });
            this.lambdaCrawler = lambdaCrawler;
            cdk_nag_1.NagSuppressions.addResourceSuppressions(lambdaCrawler.role, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'AWSLambdaBasicExecutionRole is used.',
                },
            ], true);
            cdk_nag_1.NagSuppressions.addResourceSuppressions(lambdaCrawler.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'Lambda uses the AWSLambdaBasicExecutionRole AWS Managed Policy.',
                },
            ]);
        }
        this.dataBucket = dataBucket;
        this.snsTopic = snsTopic;
        this.targetsTable = targetsTable;
        this.jobsTable = jobsTable;
        this.jobQueue = jobQueue;
        this.webCrawlerJobDefinition = webCrawlerJobDefinition;
        this.lambdaCrawlerApiSchemaPath = path.join(__dirname, '../../../../lambda/aws-web-crawler-lambda/action-group.yaml');
    }
}
exports.WebCrawler = WebCrawler;
_a = JSII_RTTI_SYMBOL_1;
WebCrawler[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.WebCrawler", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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