import * as batch from 'aws-cdk-lib/aws-batch';
import * as dynamodb from 'aws-cdk-lib/aws-dynamodb';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as sns from 'aws-cdk-lib/aws-sns';
import { Construct } from 'constructs';
import { BaseClass } from '../../../common/base-class/base-class';
export interface CrawlerTarget {
    /**
     * Target URL to be crawled.
     */
    readonly url: string;
    /**
     * Type of URL to be crawled.
     */
    readonly targetType: CrawlerTargetType;
    /**
     * Maximum number of requests to be made by crawler.
     *
     * @default - crawler limit
     */
    readonly maxRequests?: number;
    /**
     * Download files from the web site.
     *
     * @default - true
     */
    readonly downloadFiles?: boolean;
    /**
     * Maximum number of files to be downloaded.
     *
     * @default - crawler limit
     */
    readonly maxFiles?: number;
    /**
     * File types (extensions) to be downloaded.
     *
     * @default - all file types
     */
    readonly fileTypes?: string[];
    /**
     * Schedule the crawler to run every N hours following the completion of the previous job.
     *
     * @default - not scheduled
     */
    readonly crawlIntervalHours?: number;
}
export interface WebCrawlerProps {
    /**
     * Value will be appended to resources name.
     *
     * @default - _dev
     */
    readonly stage?: string;
    /**
     * Enable observability. Warning: associated cost with the services
     * used. Best practice to enable by default.
     *
     * @default - true
     */
    readonly observability?: boolean;
    /**
     * Existing instance of S3 Bucket object, providing both this and `bucketOutputProps` will cause an error.
     *
     * @default - None
     */
    readonly existingOutputBucketObj?: s3.IBucket;
    /**
     * Optional user provided props to override the default props for the S3 Bucket.
     * Providing both this and `existingOutputBucketObj` will cause an error.
     *
     * @default - Default props are used
     */
    readonly bucketOutputProps?: s3.BucketProps;
    /**
     * Optional An existing VPC in which to deploy the construct. Providing both this and
     * vpcProps is an error.
     *
     * @default - none
     */
    readonly existingVpc?: ec2.IVpc;
    /**
     * Optional custom properties for a VPC the construct will create. This VPC will
     * be used by the compute resources the construct creates. Providing
     * both this and existingVpc is an error.
     *
     * @default - none
     */
    readonly vpcProps?: ec2.VpcProps;
    /**
     *  Targets to be crawled.
     *
     * @default - none
     */
    readonly targets?: CrawlerTarget[];
    /**
     *  Deploy Lambda crawler.
     *
     * @default - false
     */
    readonly enableLambdaCrawler?: boolean;
}
export declare enum CrawlerTargetType {
    WEBSITE = "website",
    RSS_FEED = "rss_feed"
}
export declare class WebCrawler extends BaseClass {
    /**
     * Returns the instance of ec2.IVpc used by the construct
     */
    readonly vpc: ec2.IVpc;
    /**
     * Returns the instance of S3 bucket used by the construct
     */
    readonly dataBucket: s3.IBucket;
    /**
     * Returns the instance of SNS Topic used by the construct
     */
    readonly snsTopic: sns.ITopic;
    /**
     * Returns the instance of Targets DynamoDB table
     */
    readonly targetsTable: dynamodb.ITable;
    /**
     * Returns the instance of Jobs DynamoDB table
     */
    readonly jobsTable: dynamodb.ITable;
    /**
     * Returns the instance of JobQueue used by the construct
     */
    readonly jobQueue: batch.IJobQueue;
    /**
     * Returns the instance of JobDefinition used by the construct
     */
    readonly webCrawlerJobDefinition: batch.IJobDefinition;
    /**
     * Lambda crawler
     */
    readonly lambdaCrawler: lambda.IFunction | undefined;
    /**
     * Lambda crawler API schema path
     */
    readonly lambdaCrawlerApiSchemaPath: string;
    /**
     * @summary Constructs a new instance of the WebCrawler class.
     * @param {Construct} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {WebCrawlerProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: WebCrawlerProps);
}
