"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SummarizationAppsyncStepfn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfnTask = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class/base-class");
const construct_name_enum_1 = require("../../../common/base-class/construct-name-enum");
const eventBridge = require("../../../common/helpers/eventbridge-helper");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const s3BucketHelper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
class SummarizationAppsyncStepfn extends base_class_1.BaseClass {
    /**
       * @summary Constructs a new instance of the SummarizationAppsyncStepfn class.
       * @param {Construct} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {SummarizationAppsyncStepfnProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            constructName: construct_name_enum_1.ConstructName.AWSSUMMARIZATIONAPPSYNCSTEPFN,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        // vpc
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'summarizationConstructLogGroup');
        const vpcFlowLogrole = new iam.Role(this, 'summarizationConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, vpcFlowLogrole),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // bucket for input document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingInputAssetsBucketObj) {
            this.inputAssetBucket = props.existingInputAssetsBucketObj;
        }
        else if (props?.bucketInputsAssetsProps) {
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + this.stage, props.bucketInputsAssetsProps);
        }
        else {
            const bucketName = (0, utils_1.generatePhysicalNameV2)(this, 'input-assets-bucket' + this.stage, { maxLength: 63, lower: true });
            this.inputAssetBucket = new s3.Bucket(this, bucketName, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // bucket for transformed document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingProcessedAssetsBucketObj) {
            this.processedAssetBucket = props.existingProcessedAssetsBucketObj;
        }
        else if (props?.bucketProcessedAssetsProps) {
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + this.stage, props.bucketProcessedAssetsProps);
        }
        else {
            const bucketName = (0, utils_1.generatePhysicalNameV2)(this, 'processed-assets-bucket' + this.stage, { maxLength: 63, lower: true });
            this.processedAssetBucket = new s3.Bucket(this, bucketName, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        eventBridge.CheckEventBridgeProps(props);
        // Create event bridge
        this.eventBridgeBus = eventBridge.buildEventBus(this, {
            existingEventBusInterface: props.existingBusInterface,
            eventBusProps: props.eventBusProps,
        });
        // make it generic for other auth config as well
        const authorizationConfig = {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.USER_POOL,
                userPoolConfig: { userPool: props.cognitoUserPool },
            },
            additionalAuthorizationModes: [
                {
                    authorizationType: appsync.AuthorizationType.IAM,
                },
            ],
        };
        const apiName = props.summaryApiName || (0, utils_1.generatePhysicalNameV2)(this, 'summaryApi' + this.stage, { maxLength: 32, lower: true });
        // graphql api for summary. client invoke this api with given schema and cognito user pool auth.
        const summarizationGraphqlApi = new appsync.GraphqlApi(this, apiName, {
            name: apiName + this.stage,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-summarization-appsync-stepfn/schema.graphql')),
            authorizationConfig: authorizationConfig,
            xrayEnabled: this.enablexray,
        });
        this.graphqlApi = summarizationGraphqlApi;
        this.graphqlApiId = summarizationGraphqlApi.apiId;
        this.graphqlUrl = summarizationGraphqlApi.graphqlUrl;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? summarizationGraphqlApi.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? summarizationGraphqlApi.apiId : props.existingMergedApi.attrApiId;
        // bucket
        const transformedAssetBucketName = this.processedAssetBucket.bucketName;
        const inputAssetBucketName = this.inputAssetBucket.bucketName;
        const isFileTransformationRequired = props?.isFileTransformationRequired || 'false';
        // role
        const inputvalidatorLambdaRole = new iam.Role(this, 'inputvalidatorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                'appsync:GraphQL'],
            resources: ['arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + inputAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + transformedAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(inputvalidatorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_input_validation_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/input_validator')),
            functionName: (0, utils_1.generatePhysicalNameV2)(this, 'summary_input_validator' + this.stage, { maxLength: 63, lower: true }),
            description: 'Lambda function to validate input for summary api',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 1),
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: inputvalidatorLambdaRole,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        // Lambda function used to validate inputs in the step function
        const inputValidatorLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'inputValidatorLambda' + this.stage, construct_input_validation_lambda_props, props.customInputValidationDockerLambdaProps);
        // role
        const documentReaderLambdaRole = new iam.Role(this, 'documentReaderLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL'],
            resources: ['arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + inputAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + transformedAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'rekognition:DetectModerationLabels',
            ],
            resources: ['*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(documentReaderLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used',
            },
        ], true);
        const construct_document_reader_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/document_reader')),
            functionName: 'summary_document_reader' + this.stage,
            description: 'Lambda function to read the input transformed document',
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 1),
            tracing: this.lambdaTracing,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: documentReaderLambdaRole,
            environment: {
                TRANSFORMED_ASSET_BUCKET: transformedAssetBucketName,
                INPUT_ASSET_BUCKET: inputAssetBucketName,
                IS_FILE_TRANSFORMED: isFileTransformationRequired,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        // Lambda function used to read documents in the step function
        const documentReaderLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'documentReaderLambda' + this.stage, construct_document_reader_lambda_props, props.customDocumentReaderDockerLambdaProps);
        const summaryChainType = props?.summaryChainType || 'stuff';
        // role
        const summaryGeneratorLambdaRole = new iam.Role(this, 'summaryGeneratorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL',
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream'],
            resources: ['arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + inputAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + transformedAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(summaryGeneratorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const functionName = (0, utils_1.generatePhysicalNameV2)(this, 'summary_generator' + this.stage, { maxLength: 32, lower: true });
        const construct_generate_summary_lambda_props = {
            functionName: functionName,
            description: 'Lambda function to generate the summary',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/summary_generator')),
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            tracing: this.lambdaTracing,
            role: summaryGeneratorLambdaRole,
            environment: {
                ASSET_BUCKET_NAME: transformedAssetBucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                SUMMARY_LLM_CHAIN_TYPE: summaryChainType,
            },
        };
        // Lambda function used to generate the summary in the step function
        const generateSummarylambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, functionName, construct_generate_summary_lambda_props, props.customSummaryGeneratorDockerLambdaProps);
        this.inputAssetBucket?.grantRead(generateSummarylambda);
        this.processedAssetBucket?.grantReadWrite(generateSummarylambda);
        this.inputAssetBucket?.grantRead(documentReaderLambda);
        this.processedAssetBucket?.grantReadWrite(documentReaderLambda);
        const lambdaFunctions = [documentReaderLambda, generateSummarylambda, inputValidatorLambda];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // create datasource at appsync
        const SummaryStatusDataSource = new appsync.NoneDataSource(this, 'noneDataSource' + this.stage, {
            api: summarizationGraphqlApi,
            name: 'SummaryStatusDataSource',
        });
        SummaryStatusDataSource.createResolver('summaryResponseresolver' + this.stage, {
            typeName: 'Mutation',
            fieldName: 'updateSummaryJobStatus',
            requestMappingTemplate: appsync.MappingTemplate.fromString('{ "version": "2017-02-28", "payload": $util.toJson($context.args) }'),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        // step functions
        const inputValidationTask = new sfnTask.LambdaInvoke(this, 'Validate Input ', {
            lambdaFunction: inputValidatorLambda,
            resultPath: '$.validation_result',
        });
        const documentReaderTask = new sfnTask.LambdaInvoke(this, 'Read document.', {
            lambdaFunction: documentReaderLambda,
            resultPath: '$.document_result',
        });
        const generateSummaryTask = new sfnTask.LambdaInvoke(this, 'Generate Summary with llm', {
            lambdaFunction: generateSummarylambda,
            resultPath: '$.summary_result',
        });
        const dlq = new sqs.Queue(this, 'dlq', {
            queueName: (0, utils_1.generatePhysicalNameV2)(this, 'summarydlq' + this.stage, { maxLength: 32, lower: true }),
            retentionPeriod: aws_cdk_lib_1.Duration.days(7),
            enforceSSL: true,
        });
        const jobFailed = new sfn.Fail(this, 'Failed', {
            comment: 'AWS summary Job failed',
        });
        const jobSuccess = new sfn.Succeed(this, 'succeeded', {
            comment: 'AWS summary Job succeeded',
        });
        // step function choice steps
        const validateInputChoice = new sfn.Choice(this, 'is Valid Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const fileStatusForSummarization = new sfn.Choice(this, 'is file status ready for summarization?', {
            outputPath: '$.document_result.Payload',
        });
        // step function, run files in parallel
        const runFilesInparallel = new sfn.Map(this, 'Run Files in Parallel', {
            maxConcurrency: 100,
        }).itemProcessor(documentReaderTask.next(fileStatusForSummarization.when(sfn.Condition.stringMatches('$.document_result.Payload.status', 'Error'), jobSuccess).otherwise(generateSummaryTask.next(jobSuccess))));
        const maxLogGroupNameLength = 255;
        const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
        const maxGeneratedNameLength = maxLogGroupNameLength - logGroupPrefix.length;
        const logGroupName = (0, utils_1.generatePhysicalNameV2)(this, logGroupPrefix, { maxLength: maxGeneratedNameLength, lower: true });
        const summarizationLogGroup = new logs.LogGroup(this, 'summarizationLogGroup', {
            logGroupName: logGroupName,
        });
        // step function definition
        const definition = inputValidationTask.next(validateInputChoice
            .when(sfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed)
            .otherwise(runFilesInparallel));
        // step function
        const summarizationStepFunction = new sfn.StateMachine(this, 'summarizationStepFunction', {
            definitionBody: sfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            logs: {
                destination: summarizationLogGroup,
                level: sfn.LogLevel.ALL,
            },
            tracingEnabled: this.enablexray,
        });
        this.stateMachine = summarizationStepFunction;
        // event bridge datasource for summarization api
        const eventBridgeDataSource = summarizationGraphqlApi.addEventBridgeDataSource('eventBridgeDataSource', this.eventBridgeBus);
        this.eventBridgeBus.grantPutEventsTo(eventBridgeDataSource.grantPrincipal);
        // add resolver on summary graphql schema
        eventBridgeDataSource.createResolver('generateSummary', {
            typeName: 'Mutation',
            fieldName: 'generateSummary',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
         {
          "version": "2018-05-29",
          "operation": "PutEvents",
          "events": [{
              "source": "summary",
              "detail": $util.toJson($context.arguments),
              "detailType": "genAIdemo"
          }
          ]
  } 
  `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('#if($ctx.error)$utilerror($ctx.error.message, $ctx.error.type, $ctx.result) #end $util.toJson($ctx.result)'),
        });
        new events.Rule(this, 'SummaryMutationRule', {
            description: 'Summary Mutation Rule',
            eventBus: this.eventBridgeBus,
            eventPattern: {
                source: ['summary'],
            },
            targets: [
                new targets.SfnStateMachine(summarizationStepFunction, {
                    deadLetterQueue: dlq,
                    retryAttempts: 1,
                }),
            ],
        });
        this.documentReaderLambdaFunction = documentReaderLambda;
        this.inputValidationLambdaFunction = inputValidatorLambda;
        this.summaryGeneratorLambdaFunction = generateSummarylambda;
    }
}
exports.SummarizationAppsyncStepfn = SummarizationAppsyncStepfn;
_a = JSII_RTTI_SYMBOL_1;
SummarizationAppsyncStepfn[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.SummarizationAppsyncStepfn", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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