"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RagAppsyncStepfnOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const stepfn = require("aws-cdk-lib/aws-stepfunctions");
const stepfn_task = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const construct_name_enum_1 = require("../../../common/base-class/construct-name-enum");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const opensearch_helper = require("../../../common/helpers/opensearch-helper");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
   * @summary The RagAppsyncStepfnOpensearch class.
   */
class RagAppsyncStepfnOpensearch extends base_class_1.BaseClass {
    /**
       * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
       * @param {cdk.App} scope - represents the scope for all the resources.
       * @param {string} id - this is a scope-unique id.
       * @param {RagAppsyncStepfnOpensearchProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            constructName: construct_name_enum_1.ConstructName.AWSRAGAPPSYNCSTEPFNOPENSEARCH,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        vpc_helper.CheckVpcProps(props);
        opensearch_helper.CheckOpenSearchProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + this.stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'ingestionConstructLogGroup');
        const role = new iam.Role(this, 'ingestionConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // Bucket containing the inputs assets (documents - multiple modalities) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsBucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-assets-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsBucket' + this.stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // Bucket containing the processed assets (documents - text format) uploaded by the user
        let processedAssetsBucket;
        if (!props.existingProcessedAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'processed-assets-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + this.stage, props.bucketProcessedAssetsProps);
            }
            processedAssetsBucket = tmpBucket;
            this.s3ProcessedAssetsBucket = tmpBucket;
        }
        else {
            processedAssetsBucket = props.existingProcessedAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3ProcessedAssetsBucketInterface = processedAssetsBucket;
        // GraphQL API
        const ingestion_graphql_api = new appsync.GraphqlApi(this, 'ingestionGraphqlApi', {
            name: 'ingestionGraphqlApi' + this.stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-rag-appsync-stepfn-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: this.enablexray,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
        });
        this.graphqlApi = ingestion_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? ingestion_graphql_api.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? ingestion_graphql_api.apiId : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceIngestion', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateIngestionJobStatusResolver', {
            fieldName: 'updateIngestionJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2017-02-28",
                        "payload": $util.toJson($context.args)
                    }
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.ingestionBus = new events.EventBus(this, 'ingestionEventBus' + this.stage, {
                eventBusName: 'ingestionEventBus' + this.stage,
            });
        }
        else {
            this.ingestionBus = props.existingBusInterface;
        }
        // create httpdatasource with ingestion_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('ingestionEventBridgeDataSource' + this.stage, this.ingestionBus, {
            name: 'ingestionEventBridgeDataSource' + this.stage,
        });
        const construct_input_validation_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/input_validation/src')),
            functionName: 'ingestion_input_validation_docker' + this.stage,
            description: 'Lambda function for validating input files formats',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        // Lambda function used to validate inputs in the step function
        const validate_input_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_function_validation_input' + this.stage, construct_input_validation_lambda_props, props.customInputValidationDockerLambdaProps);
        // Add GraphQl permissions to the IAM role for the Lambda function
        validate_input_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // The lambda will pull documents from the input bucket, transform them, and upload
        // the artifacts to the processed bucket
        // we don't use grant read here since it has no effect in case of existing buckets provided by the user
        const s3_transformer_job_function_role = new iam.Role(this, 's3_transformer_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'rekognition:DetectModerationLabels',
            ],
            resources: ['*'],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['bedrock:*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3InputAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:PutObjectRetention',
                's3:List*',
                's3:GetBucket*',
                's3:Abort*',
                's3:DeleteObject*',
                's3:PutObjectLegalHold',
                's3:PutObjectTagging',
                's3:PutObjectVersionTagging',
                's3:PutObject',
                's3:GetObject*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        // Add GraphQl permissions to the IAM role for the Lambda function
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(s3_transformer_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct__file_transformer_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/s3_file_transformer/src')),
            functionName: 's3_file_transformer_docker' + this.stage,
            description: 'Lambda function for converting files from their input format to text',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: s3_transformer_job_function_role,
            environment: {
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        const s3_transformer_job_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_function_s3_file_transformer' + this.stage, construct__file_transformer_lambda_props, props.customFileTransformerDockerLambdaProps);
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        const embeddings_job_function_role = new iam.Role(this, 'embeddings_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        if (props.existingOpensearchDomain) {
            embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['es:*'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName + '/*',
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName,
                ],
            }));
        }
        if (props.existingOpensearchServerlessCollection) {
            embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['aoss:APIAccessAll'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':aoss:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':collection/' + props.existingOpensearchServerlessCollection.attrId,
                ],
            }));
        }
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['bedrock:*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(embeddings_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(embeddings_job_function_role);
        }
        const construct_embeddings_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/embeddings_job/src')),
            functionName: 'embeddings_job_docker' + this.stage,
            description: 'Lambda function for creating documents chunks, embeddings and storing them in Amazon Opensearch',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: embeddings_job_function_role,
            environment: {
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_API_NAME: opensearch_helper.getOpenSearchApiName(props),
                OPENSEARCH_DOMAIN_ENDPOINT: opensearch_helper.getOpenSearchEndpoint(props),
                OPENSEARCH_SECRET_ID: SecretId,
            },
        };
        // Lambda function performing the embedding job
        const embeddings_job_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_function_embeddings_job' + this.stage, construct_embeddings_lambda_props, props.customEmbeddingsDockerLambdaProps);
        const lambdaFunctions = [embeddings_job_function, s3_transformer_job_function, validate_input_function];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // Add GraphQl permissions to the IAM role for the Lambda function
        embeddings_job_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // Step function definition
        const inputValidationTask = new stepfn_task.LambdaInvoke(this, 'Validate Ingestion Input', {
            lambdaFunction: validate_input_function,
            resultPath: '$.validation_result',
        });
        const fileTransformationTask = new stepfn_task.LambdaInvoke(this, 'Download and transform document to raw text', {
            lambdaFunction: s3_transformer_job_function,
            resultPath: '$.s3_transformer_result',
        });
        const embeddingsTask = new stepfn_task.LambdaInvoke(this, 'Generate embeddings from processed documents and store them', {
            lambdaFunction: embeddings_job_function,
            resultPath: '$',
        });
        const validate_input_choice = new stepfn.Choice(this, 'Is Valid Ingestion Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const run_files_in_parallel = new stepfn.Map(this, 'Map State', {
            maxConcurrency: 100,
        }).itemProcessor(fileTransformationTask);
        const jobFailed = new stepfn.Fail(this, 'Job Failed', {
            cause: 'Validation job failed',
            error: 'DescribeJob returned FAILED',
        });
        const definition = inputValidationTask.next(validate_input_choice.when(stepfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed).otherwise(run_files_in_parallel.next(embeddingsTask)));
        const maxLogGroupNameLength = 255;
        const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
        const maxGeneratedNameLength = maxLogGroupNameLength - logGroupPrefix.length;
        const nameParts = [
            aws_cdk_lib_1.Stack.of(scope).stackName, // Name of the stack
            scope.node.id, // Construct ID
            'StateMachineLogRag', // Literal string for log group name portion
        ];
        const logGroupName = (0, utils_1.generatePhysicalName)(logGroupPrefix, nameParts, maxGeneratedNameLength);
        const ragLogGroup = new logs.LogGroup(this, 'ingestionStepFunctionLogGroup', {
            logGroupName: logGroupName,
        });
        const ingestion_step_function = new stepfn.StateMachine(this, 'IngestionStateMachine', {
            stateMachineName: 'IngestionStateMachine' + this.stage,
            definitionBody: stepfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(30),
            logs: {
                destination: ragLogGroup,
                level: stepfn.LogLevel.ALL,
            },
            tracingEnabled: this.enablexray,
        });
        this.stateMachine = ingestion_step_function;
        this.ingestionBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('ingestDocumentResolver', {
            fieldName: 'ingestDocuments',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2018-05-29",
                        "operation": "PutEvents",
                        "events": [{
                            "source": "ingestion",
                            "detail": $util.toJson($context.arguments),
                            "detailType": "genAIdemo"
                        }
                        ]
                    } 
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                    #if($ctx.error)
                        $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                    #end
                        $util.toJson($ctx.result)
                    `),
        });
        const rule = new events.Rule(this, 'ingestionRule' + this.stage, {
            description: 'Rule to trigger ingestion function',
            eventBus: this.ingestionBus,
            eventPattern: {
                source: ['ingestion'],
            },
        });
        rule.addTarget(new targets.SfnStateMachine(this.stateMachine));
        this.embeddingsLambdaFunction = embeddings_job_function;
        this.fileTransformerLambdaFunction = s3_transformer_job_function;
        this.inputValidationLambdaFunction = validate_input_function;
    }
}
exports.RagAppsyncStepfnOpensearch = RagAppsyncStepfnOpensearch;
_a = JSII_RTTI_SYMBOL_1;
RagAppsyncStepfnOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.RagAppsyncStepfnOpensearch", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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