"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RagAppsyncStepfnKendra = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const base_class_1 = require("../../../common/base-class");
const dynamodb_helper_1 = require("../../../common/helpers/dynamodb-helper");
const kendra_construct_1 = require("../../../common/helpers/kendra-construct");
const kendra_helper_1 = require("../../../common/helpers/kendra-helper");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper_1 = require("../../../common/helpers/vpc-helper");
var ServiceEndpointTypeEnum;
(function (ServiceEndpointTypeEnum) {
    ServiceEndpointTypeEnum["DYNAMODB"] = "DDB";
    ServiceEndpointTypeEnum["ECR_API"] = "ECR_API";
    ServiceEndpointTypeEnum["ECR_DKR"] = "ECR_DKR";
    ServiceEndpointTypeEnum["EVENTS"] = "CLOUDWATCH_EVENTS";
    ServiceEndpointTypeEnum["KENDRA"] = "KENDRA";
    ServiceEndpointTypeEnum["KINESIS_FIREHOSE"] = "KINESIS_FIREHOSE";
    ServiceEndpointTypeEnum["KINESIS_STREAMS"] = "KINESIS_STREAMS";
    ServiceEndpointTypeEnum["S3"] = "S3";
    ServiceEndpointTypeEnum["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypeEnum["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypeEnum["SNS"] = "SNS";
    ServiceEndpointTypeEnum["SQS"] = "SQS";
    ServiceEndpointTypeEnum["SSM"] = "SSM";
    ServiceEndpointTypeEnum["STEP_FUNCTIONS"] = "STEP_FUNCTIONS";
})(ServiceEndpointTypeEnum || (ServiceEndpointTypeEnum = {}));
/**
 * @summary The RagAppsyncStepfnKendra class.
 */
class RagAppsyncStepfnKendra extends base_class_1.BaseClass {
    /**
       * @summary Constructs a new instance of the RagAppsyncStepfnKendra class.
       * @param {cdk.App} scope - represents the scope for all the resources.
       * @param {string} id - this is a scope-unique id.
       * @param {RagAppsyncStepfnKendraProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaTracing = lambda.Tracing.ACTIVE;
        const baseProps = {
            stage: props.stage,
            constructName: base_class_1.ConstructName.AWSRAGAPPSYNCSTEPFNKENDRA,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        this.cognito = props.cognitoUserPool;
        this.awsAccountId = cdk.Stack.of(this).account;
        this.awsRegion = cdk.Stack.of(this).region;
        this.stack = cdk.Stack.of(this);
        this.removalPolicy = cdk.RemovalPolicy.DESTROY;
        // Staging environment
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + stage,
            });
        }
        let enableXRay = true;
        let apiLogConfig = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability === false) {
            enableXRay = false;
            apiLogConfig = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        if (props.kendraIndexProps && props.existingKendraIndexObj) {
            throw new Error('You may not provide both kendraIndexProps and existingKendraIndexObj');
        }
        if (props.kendraIndexProps && props.kendraDataSourcesProps) {
            throw new Error('You may not provide both kendraDataSourcesProps and existingKendraIndexObj');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = (0, vpc_helper_1.buildVpc)(scope, {
                defaultVpcProps: (0, vpc_helper_1.createDefaultIsolatedVpcProps)(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            (0, vpc_helper_1.AddAwsServiceEndpoint)(scope, this.vpc, ServiceEndpointTypeEnum.KENDRA);
        }
        this.kendraInputBucket = new aws_s3_1.Bucket(this, 'kendraInputBucket', {
            accessControl: aws_s3_1.BucketAccessControl.PRIVATE,
            enforceSSL: true,
            eventBridgeEnabled: true,
        });
        this.cognito = props.cognitoUserPool;
        this.kendraIndex = new kendra_construct_1.KendraConstruct(this, 'llmDemoIndex', {
            IndexName: 'llmDemoKendraIndex',
            Edition: 'DEVELOPER_EDITION',
            kendraDataSyncInputBucketName: this.kendraInputBucket.bucketName,
            CognitoUserPoolId: this.cognito.userPoolId,
        });
        this.kendraIndexId = this.kendraIndex.KendraIndexId;
        this.kendraDataSourceIndexId = this.kendraIndex.KendraDataSourceIndexId;
        this.kendraInputBucketArn = this.kendraInputBucket.bucketArn;
        this.kendraInputBucket.addCorsRule({
            allowedOrigins: ['*'],
            allowedMethods: [aws_s3_1.HttpMethods.GET, aws_s3_1.HttpMethods.PUT, aws_s3_1.HttpMethods.POST, aws_s3_1.HttpMethods.DELETE, aws_s3_1.HttpMethods.HEAD],
            allowedHeaders: ['*'],
            exposedHeaders: ['ETag'], // For Storage Put with Object size > 5MB
            maxAge: 3000,
        });
        const kendraACL = [
            {
                keyPrefix: `s3://${this.kendraInputBucket.bucketName}/public/BusinessTeam1/`,
                aclEntries: [
                    {
                        Name: 'BusinessTeam1',
                        Type: 'GROUP',
                        Access: 'ALLOW',
                    },
                    {
                        Name: 'Admin',
                        Type: 'GROUP',
                        Access: 'ALLOW',
                    },
                ],
            },
            {
                keyPrefix: `s3://${this.kendraInputBucket.bucketName}/public/BusinessTeam2/`,
                aclEntries: [
                    {
                        Name: 'BusinessTeam2',
                        Type: 'GROUP',
                        Access: 'ALLOW',
                    },
                    {
                        Name: 'Admin',
                        Type: 'GROUP',
                        Access: 'ALLOW',
                    },
                ],
            },
            {
                keyPrefix: `s3://${this.kendraInputBucket.bucketName}/public/AdminsOnly/`,
                aclEntries: [
                    {
                        Name: 'Admin',
                        Type: 'GROUP',
                        Access: 'ALLOW',
                    },
                ],
            },
        ];
        const kendraAclUploader = new custom_resources_1.AwsCustomResource(this, 'kendraAclUploader', {
            onUpdate: {
                action: 'putObject',
                parameters: {
                    Body: JSON.stringify(kendraACL),
                    Bucket: this.kendraInputBucket.bucketName,
                    CacheControl: 'max-age=0, no-cache, no-store, must-revalidate',
                    ContentType: 'application/json',
                    Key: 'kendra_acl.json',
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of('kendra_acl'),
                service: 'S3',
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromStatements([
                new aws_iam_1.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: [this.kendraInputBucket.arnForObjects('kendra_acl.json')],
                }),
            ]),
        });
        this.kendraIndex.node.addDependency(kendraAclUploader);
        new cdk.CfnOutput(this, 'KendraAccessControlListUrl', {
            value: `s3://${this.kendraInputBucket.bucketName}/kendra_acl.json`,
        });
        // TODO(miketran): Fix this schema.
        const ingestionGraphqlApi = new appsync.GraphqlApi(this, 'ingestionGraphqlApi', {
            name: 'ingestionGraphqlApi' + stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-rag-appsync-stepfn-kendra/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: this.cognito },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: enableXRay,
            logConfig: apiLogConfig,
        });
        this.graphqlApi = ingestionGraphqlApi;
        this.syncRunTable = (0, dynamodb_helper_1.createKendraSyncRunTable)(this);
        const lambdaPropsEnv = {
            KENDRA_INDEX_ID: this.kendraIndexId,
            KENDRA_DATA_SOURCE_INDEX_ID: this.kendraDataSourceIndexId,
            DOCUMENTS_TABLE: this.syncRunTable.tableName,
            S3_BUCKET_NAME: this.kendraInputBucket.bucketName,
        };
        const generatePresignedUrlRole = (0, kendra_helper_1.getGeneratePresignedUrlLambdaRole)(this, this.kendraInputBucket);
        const constructGeneratePresignedUrlLambdaProps = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-kendra/generate_presigned_url/src')),
            functionName: 's3_pre_signed_links_generator_docker' + stage,
            description: 'Lambda function for pre-signed links generation',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: generatePresignedUrlRole,
            environment: lambdaPropsEnv,
        };
        const generatePresignedUrlLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'generatePresignedUrlFN', constructGeneratePresignedUrlLambdaProps, props.generatePresignedUrlLambdaProps);
        const startDataSyncRole = (0, kendra_helper_1.getKendraStartDataSyncLambdaRole)(this, this.syncRunTable, this.awsRegion, this.awsAccountId, this.kendraIndexId, this.kendraDataSourceIndexId);
        const constructStartDataSyncJobLambdaProps = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-kendra/kendra_sync/src')),
            functionName: 'kendra_start_sync_job_docker' + stage,
            description: 'Lambda function for Kendra  sync job starting',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: startDataSyncRole,
            environment: lambdaPropsEnv,
        };
        const kendraSyncLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'kendraStartDataSync', constructStartDataSyncJobLambdaProps, props.kendraStartDataSyncLambdaProps);
        const checkJobStatusLambdaRole = (0, kendra_helper_1.getCheckJobStatusLambdaRole)(this, this.awsRegion, this.awsAccountId, this.kendraIndexId, this.kendraDataSourceIndexId);
        const constructCheckJobStatusLambdaProps = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-kendra/kendra_sync_status/src')),
            functionName: 'kendra_check_sync_job_status_docker' + stage,
            description: 'Lambda function for getting kendra sync status',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            environment: lambdaPropsEnv,
            role: checkJobStatusLambdaRole,
        };
        const createCheckJobsStatusLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'checkJobStatusFN', constructCheckJobStatusLambdaProps, props.checkJobsStatusLambdaProps);
        const updateKendraJobStatusLambdaRole = (0, kendra_helper_1.getUpdateKendraJobStatusLambdaRole)(this, this.syncRunTable);
        const constructKendraJobStatusLambdaProps = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-kendra/kendra_job_manager/src')),
            functionName: 'kendra_job_manager_docker' + stage,
            description: 'Lambda function for Kendra job status updates',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: updateKendraJobStatusLambdaRole,
            environment: lambdaPropsEnv,
        };
        const updateKendraJobStatusLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'updateKendraJobStatusFn', constructKendraJobStatusLambdaProps, props.updateKendraJobStatusLambdaProps);
        this.docProcessingStateMachine = (0, kendra_helper_1.createKendraWorkflowStepFunction)(this.stack, updateKendraJobStatusLambda, kendraSyncLambda, createCheckJobsStatusLambda);
        this.stepFunctionsExecutionHandlerRole = (0, kendra_helper_1.createStepFunctionsExecutionHandlerRole)(this.stack, this.docProcessingStateMachine);
        const startKendraSyncStepFnRole = (0, kendra_helper_1.getStartKendraSyncStepFnRole)(this, this.docProcessingStateMachine);
        const constructStartKendraSyncStepFnLambdaProps = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-kendra/start_kendra_sync_stepfn/src')),
            functionName: 'start_kndra_sync_step_fn' + stage,
            description: 'Lambda for starting execution',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: startKendraSyncStepFnRole,
            environment: {
                ...lambdaPropsEnv,
                STEP_FUNCTION_ARN: this.docProcessingStateMachine.stateMachineArn,
            },
        };
        const startKendraStepFnLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'startKendraStepFnLambda', constructStartKendraSyncStepFnLambdaProps, undefined);
        const presignedUrlDataSource = ingestionGraphqlApi.addLambdaDataSource('presignedUrlDataSource', generatePresignedUrlLambda);
        presignedUrlDataSource.createResolver('presignedUrlResolver', {
            typeName: 'Mutation',
            fieldName: 'generatePresignedUrl',
        });
        const kendraSyncLambdaDataSource = ingestionGraphqlApi.addLambdaDataSource('kendraSyncLambdaDataSource', startKendraStepFnLambda);
        kendraSyncLambdaDataSource.createResolver('kendraSyncLambdaResolver', {
            typeName: 'Mutation',
            fieldName: 'startKendraSyncJob',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
    {
      "version": "2017-02-28",
      "operation": "Invoke",
      "payload": {}
    }
  `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
    $util.toJson($context.result)
  `),
        });
    }
}
exports.RagAppsyncStepfnKendra = RagAppsyncStepfnKendra;
//# sourceMappingURL=data:application/json;base64,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