"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QaAppsyncOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const opensearch_helper = require("../../../common/helpers/opensearch-helper");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
 * @summary The QaAppsyncOpensearch class.
 */
class QaAppsyncOpensearch extends base_class_1.BaseClass {
    /**
     * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {QaAppsyncOpensearchProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            constructName: base_class_1.ConstructName.AWSQAAPPSYNCOPENSEARCH,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + this.stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'qaConstructLogGroup');
        const role = new iam.Role(this, 'qaConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(90),
                },
            ],
        });
        // Bucket containing the inputs assets (documents - text format) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsQABucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-asset-qa-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [
                        {
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        },
                    ],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsQABucket' + this.stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // GraphQL API
        const question_answering_graphql_api = new appsync.GraphqlApi(this, 'questionAnsweringGraphqlApi', {
            name: 'questionAnsweringGraphqlApi' + this.stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-qa-appsync-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: this.enablexray,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
        });
        this.graphqlApi = question_answering_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi
            ? question_answering_graphql_api.graphqlUrl
            : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi
            ? question_answering_graphql_api.apiId
            : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceQuestionAnswering', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateQAJobStatusResolver', {
            fieldName: 'updateQAJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                          {
                              "version": "2017-02-28",
                              "payload": $util.toJson($context.args)
                          }
                          `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.qaBus = new events.EventBus(this, 'questionAnsweringEventBus' + this.stage, {
                eventBusName: 'questionAnsweringEventBus' + this.stage,
            });
        }
        else {
            this.qaBus = props.existingBusInterface;
        }
        // create httpdatasource with question_answering_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('questionAnsweringEventBridgeDataSource' + this.stage, this.qaBus, {
            name: 'questionAnsweringEventBridgeDataSource' + this.stage,
        });
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        // Lambda function used to validate inputs in the step function
        const question_answering_function_role = new iam.Role(this, 'question_answering_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [
                                `arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        // Minimum permissions for a Lambda functienvon to execute while accessing a resource within a VPC
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ec2:DescribeNetworkInterfaces'],
            resources: ['*'],
        }));
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(question_answering_function_role);
        }
        // The lambda will pull processed files and create embeddings
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:GetObject*', 's3:GetBucket*', 's3:List*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' +
                    this.s3InputAssetsBucketInterface?.bucketName +
                    '/*',
            ],
        }));
        if (props.existingOpensearchDomain) {
            question_answering_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['es:*'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' +
                        aws_cdk_lib_1.Aws.REGION +
                        ':' +
                        aws_cdk_lib_1.Aws.ACCOUNT_ID +
                        ':domain/' +
                        props.existingOpensearchDomain.domainName +
                        '/*',
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' +
                        aws_cdk_lib_1.Aws.REGION +
                        ':' +
                        aws_cdk_lib_1.Aws.ACCOUNT_ID +
                        ':domain/' +
                        props.existingOpensearchDomain.domainName,
                ],
            }));
        }
        if (props.existingOpensearchServerlessCollection) {
            question_answering_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['aoss:APIAccessAll'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':aoss:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':collection/' + props.existingOpensearchServerlessCollection.attrId,
                ],
            }));
        }
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream',
                'bedrock:ListFoundationModels',
            ],
            // ListFoundationModels has no specific resource type
            resources: [
                '*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(question_answering_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_docker_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-qa-appsync-opensearch/question_answering/src')),
            functionName: 'lambda_question_answering' + this.stage,
            description: 'Lambda function for question answering',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: question_answering_function_role,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OPENSEARCH_API_NAME: opensearch_helper.getOpenSearchApiName(props),
                OPENSEARCH_DOMAIN_ENDPOINT: opensearch_helper.getOpenSearchEndpoint(props),
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_SECRET_ID: SecretId,
            },
            ...(props.lambdaProvisionedConcurrency && {
                currentVersionOptions: {
                    provisionedConcurrentExecutions: props.lambdaProvisionedConcurrency,
                },
            }),
        };
        const question_answering_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_question_answering' + this.stage, construct_docker_lambda_props, props.customDockerLambdaProps);
        question_answering_function.currentVersion;
        const lambdaFunctions = [question_answering_function];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // Add GraphQl permissions to the IAM role for the Lambda function
        question_answering_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['appsync:GraphQL'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':apis/' +
                    updateGraphQlApiId +
                    '/*',
            ],
        }));
        this.qaBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('QuestionAnsweringResolver', {
            fieldName: 'postQuestion',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                        {
                            "version": "2018-05-29",
                            "operation": "PutEvents",
                            "events": [{
                                "source": "questionanswering",
                                "detail": $util.toJson($context.arguments),
                                "detailType": "Question answering"
                            }
                            ]
                        } 
                        `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                        #if($ctx.error)
                            $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                        #end
                            $util.toJson($ctx.result)
                        `),
        });
        const rule = new events.Rule(this, 'QuestionAnsweringRule' + this.stage, {
            description: 'Rule to trigger question answering function',
            eventBus: this.qaBus,
            eventPattern: {
                source: ['questionanswering'],
            },
        });
        rule.addTarget(new targets.LambdaFunction(question_answering_function));
        this.qaLambdaFunction = question_answering_function;
    }
}
exports.QaAppsyncOpensearch = QaAppsyncOpensearch;
_a = JSII_RTTI_SYMBOL_1;
QaAppsyncOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.QaAppsyncOpensearch", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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