"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JumpStartSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const core_1 = require("aws-cdk-lib/core");
const jumpstart_constants_1 = require("./private/jumpstart-constants");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
const construct_name_enum_1 = require("../../../common/base-class/construct-name-enum");
/**
 * @summary The JumpStartSageMakerEndpoint class.
 */
class JumpStartSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            constructName: construct_name_enum_1.ConstructName.JUMPSTARTSAGEMAKERENDPOINT,
            constructId: id,
        };
        // No lambda function to use AWS SDK for service metric
        const lambdaFunctions = [];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        this.model = props.model;
        this.instanceType = props.instanceType;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.acceptEula = props.acceptEula ?? false;
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.spec = this.model.bind();
        if (!this.acceptEula && this.spec.requiresEula) {
            throw new Error('The AcceptEula value must be explicitly defined as True in order to accept the EULA for the model ' + this.spec.modelId + '. You are responsible for reviewing and complying with any applicable license terms and making sure they are acceptable for your use case before downloading or using a model.');
        }
        this.region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(this.region)) {
            throw new Error('Region is unresolved. You should explicitly specify the region in the environment.');
        }
        const instanceType = this.verifyInstanceType();
        const instanseBaseType = instanceType.split('.')[1];
        let model;
        if (this.spec.modelPackageArns) {
            if (this.environment) {
                throw new Error('Environment variables are not supported for model packages.');
            }
            model = this.getModelFromPackage(scope, id, props.vpcConfig);
        }
        else {
            const environment = this.buildEnvironment(instanceType);
            model = this.getModelFromArtifact(scope, id, instanceType, instanseBaseType, environment, props.vpcConfig);
        }
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType,
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${this.spec.modelId}-endpoint-${id}`, {
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            endpointName: 'jumpstart-' + props.endpointName,
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
    verifyInstanceType() {
        let instanceType = this.spec.defaultInstanceType;
        if (this.instanceType) {
            instanceType = this.instanceType.toString();
        }
        const supportedInstanceTypes = this.spec.instanceTypes;
        if (!supportedInstanceTypes.includes(instanceType)) {
            throw new Error(`The instance type ${instanceType} is not supported. Default instance type: ${this.spec.defaultInstanceType}. Supported instance types: ${supportedInstanceTypes.join(', ')}.`);
        }
        return instanceType;
    }
    buildEnvironment(instanceType) {
        const configEnvironment = this.spec.instanceVariants?.find((v) => v.instanceType === instanceType)?.environment;
        const environment = {
            ...(this.spec.environment ?? {}),
            ...configEnvironment,
            ...this.environment,
        };
        return environment;
    }
    getModelFromArtifact(scope, id, instanceType, instanceBaseType, environment, vpcConfig) {
        const key = this.spec.prepackedArtifactKey ?? this.spec.artifactKey;
        const bucket = jumpstart_constants_1.JumpStartConstants.JUMPSTART_LAUNCHED_REGIONS[this.region]?.contentBucket;
        if (!bucket) {
            throw new Error(`JumpStart is not available in the region ${this.region}.`);
        }
        const modelArtifactUrl = `s3://${bucket}/${key}`;
        const isArtifactCompressed = modelArtifactUrl.endsWith('.tar.gz');
        const imageUriKey = this.spec.instanceVariants
            ?.find((v) => v.instanceType === instanceBaseType)
            ?.imageUri?.replace('$', '');
        if (!imageUriKey) {
            throw new Error(`The image uri is not available for instance type ${instanceType}.`);
        }
        const image = this.spec.instanceAliases?.find((v) => v.region === this.region)?.aliases[imageUriKey];
        if (!image) {
            throw new Error(`The image uri is not available for instance type ${instanceType} in region ${this.region}.`);
        }
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: isArtifactCompressed ? {
                // True: Artifact is a tarball
                image,
                modelDataUrl: modelArtifactUrl,
                environment,
            } : {
                // False: Model is uncompressed
                image,
                modelDataSource: {
                    s3DataSource: {
                        compressionType: 'None',
                        s3DataType: 'S3Prefix',
                        s3Uri: modelArtifactUrl,
                        modelAccessConfig: {
                            acceptEula: this.acceptEula,
                        },
                    },
                },
                environment,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
            vpcConfig: vpcConfig,
        });
        return model;
    }
    getModelFromPackage(scope, id, vpcConfig) {
        const modelPackageArns = this.spec.modelPackageArns || {};
        const supportedRegions = Object.keys(modelPackageArns);
        if (!supportedRegions.includes(this.region)) {
            throw new Error(`The model package is not available in the region ${this.region}. Supported regions: ${supportedRegions.join(', ')}.`);
        }
        const modelPackageArn = modelPackageArns[this.region];
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: {
                modelPackageName: modelPackageArn,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
            vpcConfig: vpcConfig,
        });
        return model;
    }
}
exports.JumpStartSageMakerEndpoint = JumpStartSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
JumpStartSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.JumpStartSageMakerEndpoint", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoianVtcHN0YXJ0LXNhZ2VtYWtlci1lbmRwb2ludC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9wYXR0ZXJucy9nZW4tYWkvYXdzLW1vZGVsLWRlcGxveW1lbnQtc2FnZW1ha2VyL2p1bXBzdGFydC1zYWdlbWFrZXItZW5kcG9pbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFhQSwyQ0FBMkM7QUFDM0MsdURBQXVEO0FBQ3ZELDJDQUFnRDtBQUdoRCx1RUFBbUU7QUFDbkUsdUVBQWtFO0FBR2xFLHdGQUErRTtBQWUvRTs7R0FFRztBQUNILE1BQWEsMEJBQTJCLFNBQVEsK0NBQXFCO0lBa0JuRSxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNDO1FBQzlFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxTQUFTLEdBQWlCO1lBQzlCLGFBQWEsRUFBRSxtQ0FBYSxDQUFDLDBCQUEwQjtZQUN2RCxXQUFXLEVBQUUsRUFBRTtTQUNoQixDQUFDO1FBRUYsdURBQXVEO1FBQ3ZELE1BQU0sZUFBZSxHQUF1QyxFQUFFLENBQUM7UUFDL0QsSUFBSSxDQUFDLDhCQUE4QixDQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFHeEUsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxhQUFhLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDM0QsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUU1QyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFDckQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBRWhDLElBQUksQ0FBQyxrQ0FBa0MsR0FBRyxLQUFLLENBQUMsa0NBQWtDLElBQUksR0FBRyxDQUFDO1FBQzFGLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFOUIsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUMvQyxNQUFNLElBQUksS0FBSyxDQUNiLG9HQUFvRyxHQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxHQUFDLGdMQUFnTCxDQUN4UyxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxNQUFNLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFFcEMsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO1lBQ3BDLE1BQU0sSUFBSSxLQUFLLENBQ2Isb0ZBQW9GLENBQ3JGLENBQUM7UUFDSixDQUFDO1FBRUQsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUM7UUFDL0MsTUFBTSxnQkFBZ0IsR0FBRyxZQUFZLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXBELElBQUksS0FBeUIsQ0FBQztRQUM5QixJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztZQUMvQixJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztnQkFDckIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO1lBQ2pGLENBQUM7WUFFRCxLQUFLLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQy9ELENBQUM7YUFBTSxDQUFDO1lBQ04sTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ3hELEtBQUssR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxZQUFZLEVBQUUsZ0JBQWdCLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUM3RyxDQUFDO1FBRUQsTUFBTSxjQUFjLEdBQUcsSUFBSSxTQUFTLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFLEVBQUUsRUFBRTtZQUNwRixrQkFBa0IsRUFBRTtnQkFDbEI7b0JBQ0UsWUFBWTtvQkFDWixvQkFBb0IsRUFBRSxDQUFDO29CQUN2QixvQkFBb0IsRUFBRSxJQUFJLENBQUMsYUFBYTtvQkFDeEMsV0FBVyxFQUFFLFlBQVk7b0JBQ3pCLFNBQVMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFFBQVEsRUFBRTtvQkFDL0MsMkNBQTJDLEVBQUUsSUFBSSxDQUFDLGtDQUFrQztpQkFDckY7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILGNBQWMsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFcEMsTUFBTSxRQUFRLEdBQUcsSUFBSSxTQUFTLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxhQUFhLEVBQUUsRUFBRSxFQUFFO1lBQ3ZGLGtCQUFrQixFQUFFLGNBQWMsQ0FBQyxNQUFNLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxRQUFRLEVBQUU7WUFDMUUsWUFBWSxFQUFFLFlBQVksR0FBQyxLQUFLLENBQUMsWUFBWTtTQUM5QyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXZDLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO1FBQ3RCLElBQUksQ0FBQyxXQUFXLEdBQUcsUUFBUSxDQUFDO1FBQzVCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxjQUFjLENBQUM7UUFDeEMsSUFBSSxDQUFDLFdBQVcsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQ2xDLENBQUM7SUFFTSxlQUFlLENBQUMsU0FBOEI7UUFDbkQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNmLE9BQU87UUFDVCxDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVNLFdBQVcsQ0FBQyxPQUF1QjtRQUN4QyxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxPQUFPLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQztZQUNyQyxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO1NBQ2pDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxrQkFBa0I7UUFDeEIsSUFBSSxZQUFZLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztRQUNqRCxJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUN0QixZQUFZLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUM5QyxDQUFDO1FBRUQsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQztRQUN2RCxJQUFJLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUM7WUFDbkQsTUFBTSxJQUFJLEtBQUssQ0FDYixxQkFBcUIsWUFBWSw2Q0FDL0IsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFDWiwrQkFBK0Isc0JBQXNCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQ3BFLENBQUM7UUFDSixDQUFDO1FBRUQsT0FBTyxZQUFZLENBQUM7SUFDdEIsQ0FBQztJQUVPLGdCQUFnQixDQUFDLFlBQW9CO1FBQzNDLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxJQUFJLENBQ3hELENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBWSxLQUFLLFlBQVksQ0FDdkMsRUFBRSxXQUFXLENBQUM7UUFFZixNQUFNLFdBQVcsR0FBRztZQUNsQixHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLElBQUksRUFBRSxDQUFDO1lBQ2hDLEdBQUcsaUJBQWlCO1lBQ3BCLEdBQUcsSUFBSSxDQUFDLFdBQVc7U0FDcEIsQ0FBQztRQUVGLE9BQU8sV0FBVyxDQUFDO0lBQ3JCLENBQUM7SUFFTyxvQkFBb0IsQ0FDMUIsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLFlBQW9CLEVBQ3BCLGdCQUF3QixFQUN4QixXQUF5RCxFQUN6RCxTQUEyRDtRQUUzRCxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO1FBQ3BFLE1BQU0sTUFBTSxHQUFHLHdDQUFrQixDQUFDLDBCQUEwQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxhQUFhLENBQUM7UUFDekYsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFDOUUsQ0FBQztRQUVELE1BQU0sZ0JBQWdCLEdBQUcsUUFBUSxNQUFNLElBQUksR0FBRyxFQUFFLENBQUM7UUFDakQsTUFBTSxvQkFBb0IsR0FBRyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFbEUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0I7WUFDNUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxZQUFZLEtBQUssZ0JBQWdCLENBQUM7WUFDbEQsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUUvQixJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7WUFDakIsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsWUFBWSxHQUFHLENBQUMsQ0FBQztRQUN2RixDQUFDO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxLQUFLLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLENBQ3JGLFdBQVcsQ0FDWixDQUFDO1FBQ0YsSUFBSSxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQ1gsTUFBTSxJQUFJLEtBQUssQ0FDYixvREFBb0QsWUFBWSxjQUFjLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FDN0YsQ0FBQztRQUNKLENBQUM7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLFVBQVUsRUFBRSxFQUFFLEVBQUU7WUFDOUUsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ25DLHNCQUFzQixFQUFFLElBQUk7WUFDNUIsZ0JBQWdCLEVBQUUsb0JBQW9CLENBQUMsQ0FBQyxDQUFDO2dCQUN6Qyw4QkFBOEI7Z0JBQzVCLEtBQUs7Z0JBQ0wsWUFBWSxFQUFFLGdCQUFnQjtnQkFDOUIsV0FBVzthQUNaLENBQUMsQ0FBQyxDQUFDO2dCQUNGLCtCQUErQjtnQkFDL0IsS0FBSztnQkFDTCxlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLGVBQWUsRUFBRSxNQUFNO3dCQUN2QixVQUFVLEVBQUUsVUFBVTt3QkFDdEIsS0FBSyxFQUFFLGdCQUFnQjt3QkFDdkIsaUJBQWlCLEVBQUU7NEJBQ2pCLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTt5QkFDNUI7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsV0FBVzthQUNaO1lBQ0QsSUFBSSxFQUFFO2dCQUNKO29CQUNFLEdBQUcsRUFBRSxTQUFTO29CQUNkLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87aUJBQ3pCO2dCQUNEO29CQUNFLEdBQUcsRUFBRSxjQUFjO29CQUNuQixLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO2lCQUN6QjthQUNGO1lBQ0QsU0FBUyxFQUFFLFNBQVM7U0FDckIsQ0FBQyxDQUFDO1FBRUgsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRU8sbUJBQW1CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsU0FBMkQ7UUFDbkgsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixJQUFJLEVBQUUsQ0FBQztRQUMxRCxNQUFNLGdCQUFnQixHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUN2RCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO1lBQzVDLE1BQU0sSUFBSSxLQUFLLENBQ2Isb0RBQ0UsSUFBSSxDQUFDLE1BQ1Asd0JBQXdCLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUN2RCxDQUFDO1FBQ0osQ0FBQztRQUVELE1BQU0sZUFBZSxHQUFHLGdCQUFnQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV0RCxNQUFNLEtBQUssR0FBRyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLFVBQVUsRUFBRSxFQUFFLEVBQUU7WUFDOUUsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ25DLHNCQUFzQixFQUFFLElBQUk7WUFDNUIsZ0JBQWdCLEVBQUU7Z0JBQ2hCLGdCQUFnQixFQUFFLGVBQWU7YUFDbEM7WUFDRCxJQUFJLEVBQUU7Z0JBQ0o7b0JBQ0UsR0FBRyxFQUFFLFNBQVM7b0JBQ2QsS0FBSyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTztpQkFDekI7Z0JBQ0Q7b0JBQ0UsR0FBRyxFQUFFLGNBQWM7b0JBQ25CLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87aUJBQ3pCO2FBQ0Y7WUFDRCxTQUFTLEVBQUUsU0FBUztTQUNyQixDQUFDLENBQUM7UUFFSCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7O0FBOVBILGdFQStQQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc2FnZW1ha2VyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zYWdlbWFrZXInO1xuaW1wb3J0IHsgU3RhY2ssIFRva2VuIH0gZnJvbSAnYXdzLWNkay1saWIvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEp1bXBTdGFydE1vZGVsLCBJSnVtcFN0YXJ0TW9kZWxTcGVjIH0gZnJvbSAnLi9qdW1wc3RhcnQtbW9kZWwnO1xuaW1wb3J0IHsgSnVtcFN0YXJ0Q29uc3RhbnRzIH0gZnJvbSAnLi9wcml2YXRlL2p1bXBzdGFydC1jb25zdGFudHMnO1xuaW1wb3J0IHsgU2FnZU1ha2VyRW5kcG9pbnRCYXNlIH0gZnJvbSAnLi9zYWdlbWFrZXItZW5kcG9pbnQtYmFzZSc7XG5pbXBvcnQgeyBTYWdlTWFrZXJJbnN0YW5jZVR5cGUgfSBmcm9tICcuL3NhZ2VtYWtlci1pbnN0YW5jZS10eXBlJztcbmltcG9ydCB7IEJhc2VDbGFzc1Byb3BzIH0gZnJvbSAnLi4vLi4vLi4vY29tbW9uL2Jhc2UtY2xhc3MvYmFzZS1jbGFzcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3ROYW1lIH0gZnJvbSAnLi4vLi4vLi4vY29tbW9uL2Jhc2UtY2xhc3MvY29uc3RydWN0LW5hbWUtZW51bSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSnVtcFN0YXJ0U2FnZU1ha2VyRW5kcG9pbnRQcm9wcyB7XG4gIHJlYWRvbmx5IG1vZGVsOiBKdW1wU3RhcnRNb2RlbDtcbiAgcmVhZG9ubHkgZW5kcG9pbnROYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IFNhZ2VNYWtlckluc3RhbmNlVHlwZTtcbiAgcmVhZG9ubHkgaW5zdGFuY2VDb3VudD86IG51bWJlcjtcbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5Sb2xlO1xuICByZWFkb25seSB2cGNDb25maWc/OiBzYWdlbWFrZXIuQ2ZuTW9kZWwuVnBjQ29uZmlnUHJvcGVydHkgfCB1bmRlZmluZWQ7XG4gIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcmVhZG9ubHkgc3RhcnR1cEhlYWx0aENoZWNrVGltZW91dEluU2Vjb25kcz86IG51bWJlcjtcbiAgcmVhZG9ubHkgYWNjZXB0RXVsYT86IGJvb2xlYW47XG5cbn1cblxuLyoqXG4gKiBAc3VtbWFyeSBUaGUgSnVtcFN0YXJ0U2FnZU1ha2VyRW5kcG9pbnQgY2xhc3MuXG4gKi9cbmV4cG9ydCBjbGFzcyBKdW1wU3RhcnRTYWdlTWFrZXJFbmRwb2ludCBleHRlbmRzIFNhZ2VNYWtlckVuZHBvaW50QmFzZSB7XG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gIHB1YmxpYyByZWFkb25seSBlbmRwb2ludEFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY2ZuTW9kZWw6IHNhZ2VtYWtlci5DZm5Nb2RlbDtcbiAgcHVibGljIHJlYWRvbmx5IGNmbkVuZHBvaW50OiBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnQ7XG4gIHB1YmxpYyByZWFkb25seSBjZm5FbmRwb2ludENvbmZpZzogc2FnZW1ha2VyLkNmbkVuZHBvaW50Q29uZmlnO1xuXG4gIHB1YmxpYyByZWFkb25seSBtb2RlbDogSnVtcFN0YXJ0TW9kZWw7XG4gIHB1YmxpYyByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBTYWdlTWFrZXJJbnN0YW5jZVR5cGU7XG4gIHB1YmxpYyByZWFkb25seSBpbnN0YW5jZUNvdW50OiBudW1iZXI7XG4gIHB1YmxpYyByZWFkb25seSByb2xlOiBpYW0uUm9sZTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGFjY2VwdEV1bGE6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVnaW9uOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgc3BlYzogSUp1bXBTdGFydE1vZGVsU3BlYztcbiAgcHJpdmF0ZSByZWFkb25seSBzdGFydHVwSGVhbHRoQ2hlY2tUaW1lb3V0SW5TZWNvbmRzOiBudW1iZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBKdW1wU3RhcnRTYWdlTWFrZXJFbmRwb2ludFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IGJhc2VQcm9wczogQmFzZUNsYXNzUHJvcHM9e1xuICAgICAgY29uc3RydWN0TmFtZTogQ29uc3RydWN0TmFtZS5KVU1QU1RBUlRTQUdFTUFLRVJFTkRQT0lOVCxcbiAgICAgIGNvbnN0cnVjdElkOiBpZCxcbiAgICB9O1xuXG4gICAgLy8gTm8gbGFtYmRhIGZ1bmN0aW9uIHRvIHVzZSBBV1MgU0RLIGZvciBzZXJ2aWNlIG1ldHJpY1xuICAgIGNvbnN0IGxhbWJkYUZ1bmN0aW9uczogY2RrLmF3c19sYW1iZGEuRG9ja2VySW1hZ2VGdW5jdGlvbltdPVtdO1xuICAgIHRoaXMudXBkYXRlQ29uc3RydWN0VXNhZ2VNZXRyaWNDb2RlKCBiYXNlUHJvcHMsIHNjb3BlLCBsYW1iZGFGdW5jdGlvbnMpO1xuXG5cbiAgICB0aGlzLm1vZGVsID0gcHJvcHMubW9kZWw7XG4gICAgdGhpcy5pbnN0YW5jZVR5cGUgPSBwcm9wcy5pbnN0YW5jZVR5cGU7XG4gICAgdGhpcy5pbnN0YW5jZUNvdW50ID0gTWF0aC5tYXgoMSwgcHJvcHMuaW5zdGFuY2VDb3VudCA/PyAxKTtcbiAgICB0aGlzLmFjY2VwdEV1bGEgPSBwcm9wcy5hY2NlcHRFdWxhID8/IGZhbHNlO1xuXG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSA/PyB0aGlzLmNyZWF0ZVNhZ2VNYWtlclJvbGUoKTtcbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGhpcy5yb2xlO1xuXG4gICAgdGhpcy5zdGFydHVwSGVhbHRoQ2hlY2tUaW1lb3V0SW5TZWNvbmRzID0gcHJvcHMuc3RhcnR1cEhlYWx0aENoZWNrVGltZW91dEluU2Vjb25kcyA/PyA2MDA7XG4gICAgdGhpcy5lbnZpcm9ubWVudCA9IHByb3BzLmVudmlyb25tZW50O1xuICAgIHRoaXMuc3BlYyA9IHRoaXMubW9kZWwuYmluZCgpO1xuXG4gICAgaWYgKCF0aGlzLmFjY2VwdEV1bGEgJiYgdGhpcy5zcGVjLnJlcXVpcmVzRXVsYSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAnVGhlIEFjY2VwdEV1bGEgdmFsdWUgbXVzdCBiZSBleHBsaWNpdGx5IGRlZmluZWQgYXMgVHJ1ZSBpbiBvcmRlciB0byBhY2NlcHQgdGhlIEVVTEEgZm9yIHRoZSBtb2RlbCAnK3RoaXMuc3BlYy5tb2RlbElkKycuIFlvdSBhcmUgcmVzcG9uc2libGUgZm9yIHJldmlld2luZyBhbmQgY29tcGx5aW5nIHdpdGggYW55IGFwcGxpY2FibGUgbGljZW5zZSB0ZXJtcyBhbmQgbWFraW5nIHN1cmUgdGhleSBhcmUgYWNjZXB0YWJsZSBmb3IgeW91ciB1c2UgY2FzZSBiZWZvcmUgZG93bmxvYWRpbmcgb3IgdXNpbmcgYSBtb2RlbC4nLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICB0aGlzLnJlZ2lvbiA9IFN0YWNrLm9mKHRoaXMpLnJlZ2lvbjtcblxuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQodGhpcy5yZWdpb24pKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICdSZWdpb24gaXMgdW5yZXNvbHZlZC4gWW91IHNob3VsZCBleHBsaWNpdGx5IHNwZWNpZnkgdGhlIHJlZ2lvbiBpbiB0aGUgZW52aXJvbm1lbnQuJyxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgaW5zdGFuY2VUeXBlID0gdGhpcy52ZXJpZnlJbnN0YW5jZVR5cGUoKTtcbiAgICBjb25zdCBpbnN0YW5zZUJhc2VUeXBlID0gaW5zdGFuY2VUeXBlLnNwbGl0KCcuJylbMV07XG5cbiAgICBsZXQgbW9kZWw6IHNhZ2VtYWtlci5DZm5Nb2RlbDtcbiAgICBpZiAodGhpcy5zcGVjLm1vZGVsUGFja2FnZUFybnMpIHtcbiAgICAgIGlmICh0aGlzLmVudmlyb25tZW50KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignRW52aXJvbm1lbnQgdmFyaWFibGVzIGFyZSBub3Qgc3VwcG9ydGVkIGZvciBtb2RlbCBwYWNrYWdlcy4nKTtcbiAgICAgIH1cblxuICAgICAgbW9kZWwgPSB0aGlzLmdldE1vZGVsRnJvbVBhY2thZ2Uoc2NvcGUsIGlkLCBwcm9wcy52cGNDb25maWcpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBlbnZpcm9ubWVudCA9IHRoaXMuYnVpbGRFbnZpcm9ubWVudChpbnN0YW5jZVR5cGUpO1xuICAgICAgbW9kZWwgPSB0aGlzLmdldE1vZGVsRnJvbUFydGlmYWN0KHNjb3BlLCBpZCwgaW5zdGFuY2VUeXBlLCBpbnN0YW5zZUJhc2VUeXBlLCBlbnZpcm9ubWVudCwgcHJvcHMudnBjQ29uZmlnKTtcbiAgICB9XG5cbiAgICBjb25zdCBlbmRwb2ludENvbmZpZyA9IG5ldyBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnRDb25maWcoc2NvcGUsIGBFbmRwb2ludENvbmZpZy0ke2lkfWAsIHtcbiAgICAgIHByb2R1Y3Rpb25WYXJpYW50czogW1xuICAgICAgICB7XG4gICAgICAgICAgaW5zdGFuY2VUeXBlLFxuICAgICAgICAgIGluaXRpYWxWYXJpYW50V2VpZ2h0OiAxLFxuICAgICAgICAgIGluaXRpYWxJbnN0YW5jZUNvdW50OiB0aGlzLmluc3RhbmNlQ291bnQsXG4gICAgICAgICAgdmFyaWFudE5hbWU6ICdBbGxUcmFmZmljJyxcbiAgICAgICAgICBtb2RlbE5hbWU6IG1vZGVsLmdldEF0dCgnTW9kZWxOYW1lJykudG9TdHJpbmcoKSxcbiAgICAgICAgICBjb250YWluZXJTdGFydHVwSGVhbHRoQ2hlY2tUaW1lb3V0SW5TZWNvbmRzOiB0aGlzLnN0YXJ0dXBIZWFsdGhDaGVja1RpbWVvdXRJblNlY29uZHMsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgZW5kcG9pbnRDb25maWcuYWRkRGVwZW5kZW5jeShtb2RlbCk7XG5cbiAgICBjb25zdCBlbmRwb2ludCA9IG5ldyBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnQoc2NvcGUsIGAke3RoaXMuc3BlYy5tb2RlbElkfS1lbmRwb2ludC0ke2lkfWAsIHtcbiAgICAgIGVuZHBvaW50Q29uZmlnTmFtZTogZW5kcG9pbnRDb25maWcuZ2V0QXR0KCdFbmRwb2ludENvbmZpZ05hbWUnKS50b1N0cmluZygpLFxuICAgICAgZW5kcG9pbnROYW1lOiAnanVtcHN0YXJ0LScrcHJvcHMuZW5kcG9pbnROYW1lLFxuICAgIH0pO1xuXG4gICAgZW5kcG9pbnQuYWRkRGVwZW5kZW5jeShlbmRwb2ludENvbmZpZyk7XG5cbiAgICB0aGlzLmNmbk1vZGVsID0gbW9kZWw7XG4gICAgdGhpcy5jZm5FbmRwb2ludCA9IGVuZHBvaW50O1xuICAgIHRoaXMuY2ZuRW5kcG9pbnRDb25maWcgPSBlbmRwb2ludENvbmZpZztcbiAgICB0aGlzLmVuZHBvaW50QXJuID0gZW5kcG9pbnQucmVmO1xuICB9XG5cbiAgcHVibGljIGFkZFRvUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICBpZiAoIXRoaXMucm9sZSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMucm9sZS5hZGRUb1BvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgcHVibGljIGdyYW50SW52b2tlKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zOiBbJ3NhZ2VtYWtlcjpJbnZva2VFbmRwb2ludCddLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbdGhpcy5lbmRwb2ludEFybl0sXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIHZlcmlmeUluc3RhbmNlVHlwZSgpIHtcbiAgICBsZXQgaW5zdGFuY2VUeXBlID0gdGhpcy5zcGVjLmRlZmF1bHRJbnN0YW5jZVR5cGU7XG4gICAgaWYgKHRoaXMuaW5zdGFuY2VUeXBlKSB7XG4gICAgICBpbnN0YW5jZVR5cGUgPSB0aGlzLmluc3RhbmNlVHlwZS50b1N0cmluZygpO1xuICAgIH1cblxuICAgIGNvbnN0IHN1cHBvcnRlZEluc3RhbmNlVHlwZXMgPSB0aGlzLnNwZWMuaW5zdGFuY2VUeXBlcztcbiAgICBpZiAoIXN1cHBvcnRlZEluc3RhbmNlVHlwZXMuaW5jbHVkZXMoaW5zdGFuY2VUeXBlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgVGhlIGluc3RhbmNlIHR5cGUgJHtpbnN0YW5jZVR5cGV9IGlzIG5vdCBzdXBwb3J0ZWQuIERlZmF1bHQgaW5zdGFuY2UgdHlwZTogJHtcbiAgICAgICAgICB0aGlzLnNwZWMuZGVmYXVsdEluc3RhbmNlVHlwZVxuICAgICAgICB9LiBTdXBwb3J0ZWQgaW5zdGFuY2UgdHlwZXM6ICR7c3VwcG9ydGVkSW5zdGFuY2VUeXBlcy5qb2luKCcsICcpfS5gLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gaW5zdGFuY2VUeXBlO1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZEVudmlyb25tZW50KGluc3RhbmNlVHlwZTogc3RyaW5nKSB7XG4gICAgY29uc3QgY29uZmlnRW52aXJvbm1lbnQgPSB0aGlzLnNwZWMuaW5zdGFuY2VWYXJpYW50cz8uZmluZChcbiAgICAgICh2KSA9PiB2Lmluc3RhbmNlVHlwZSA9PT0gaW5zdGFuY2VUeXBlLFxuICAgICk/LmVudmlyb25tZW50O1xuXG4gICAgY29uc3QgZW52aXJvbm1lbnQgPSB7XG4gICAgICAuLi4odGhpcy5zcGVjLmVudmlyb25tZW50ID8/IHt9KSxcbiAgICAgIC4uLmNvbmZpZ0Vudmlyb25tZW50LFxuICAgICAgLi4udGhpcy5lbnZpcm9ubWVudCxcbiAgICB9O1xuXG4gICAgcmV0dXJuIGVudmlyb25tZW50O1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRNb2RlbEZyb21BcnRpZmFjdChcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgaW5zdGFuY2VUeXBlOiBzdHJpbmcsXG4gICAgaW5zdGFuY2VCYXNlVHlwZTogc3RyaW5nLFxuICAgIGVudmlyb25tZW50OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB8IG51bWJlciB8IGJvb2xlYW4gfSxcbiAgICB2cGNDb25maWc6IHNhZ2VtYWtlci5DZm5Nb2RlbC5WcGNDb25maWdQcm9wZXJ0eSB8IHVuZGVmaW5lZCxcbiAgKSB7XG4gICAgY29uc3Qga2V5ID0gdGhpcy5zcGVjLnByZXBhY2tlZEFydGlmYWN0S2V5ID8/IHRoaXMuc3BlYy5hcnRpZmFjdEtleTtcbiAgICBjb25zdCBidWNrZXQgPSBKdW1wU3RhcnRDb25zdGFudHMuSlVNUFNUQVJUX0xBVU5DSEVEX1JFR0lPTlNbdGhpcy5yZWdpb25dPy5jb250ZW50QnVja2V0O1xuICAgIGlmICghYnVja2V0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEp1bXBTdGFydCBpcyBub3QgYXZhaWxhYmxlIGluIHRoZSByZWdpb24gJHt0aGlzLnJlZ2lvbn0uYCk7XG4gICAgfVxuXG4gICAgY29uc3QgbW9kZWxBcnRpZmFjdFVybCA9IGBzMzovLyR7YnVja2V0fS8ke2tleX1gO1xuICAgIGNvbnN0IGlzQXJ0aWZhY3RDb21wcmVzc2VkID0gbW9kZWxBcnRpZmFjdFVybC5lbmRzV2l0aCgnLnRhci5neicpO1xuXG4gICAgY29uc3QgaW1hZ2VVcmlLZXkgPSB0aGlzLnNwZWMuaW5zdGFuY2VWYXJpYW50c1xuICAgICAgPy5maW5kKCh2KSA9PiB2Lmluc3RhbmNlVHlwZSA9PT0gaW5zdGFuY2VCYXNlVHlwZSlcbiAgICAgID8uaW1hZ2VVcmk/LnJlcGxhY2UoJyQnLCAnJyk7XG5cbiAgICBpZiAoIWltYWdlVXJpS2V5KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBpbWFnZSB1cmkgaXMgbm90IGF2YWlsYWJsZSBmb3IgaW5zdGFuY2UgdHlwZSAke2luc3RhbmNlVHlwZX0uYCk7XG4gICAgfVxuXG4gICAgY29uc3QgaW1hZ2UgPSB0aGlzLnNwZWMuaW5zdGFuY2VBbGlhc2VzPy5maW5kKCh2KSA9PiB2LnJlZ2lvbiA9PT0gdGhpcy5yZWdpb24pPy5hbGlhc2VzW1xuICAgICAgaW1hZ2VVcmlLZXlcbiAgICBdO1xuICAgIGlmICghaW1hZ2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYFRoZSBpbWFnZSB1cmkgaXMgbm90IGF2YWlsYWJsZSBmb3IgaW5zdGFuY2UgdHlwZSAke2luc3RhbmNlVHlwZX0gaW4gcmVnaW9uICR7dGhpcy5yZWdpb259LmAsXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IG1vZGVsID0gbmV3IHNhZ2VtYWtlci5DZm5Nb2RlbChzY29wZSwgYCR7dGhpcy5zcGVjLm1vZGVsSWR9LW1vZGVsLSR7aWR9YCwge1xuICAgICAgZXhlY3V0aW9uUm9sZUFybjogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICBlbmFibGVOZXR3b3JrSXNvbGF0aW9uOiB0cnVlLFxuICAgICAgcHJpbWFyeUNvbnRhaW5lcjogaXNBcnRpZmFjdENvbXByZXNzZWQgPyB7XG4gICAgICAvLyBUcnVlOiBBcnRpZmFjdCBpcyBhIHRhcmJhbGxcbiAgICAgICAgaW1hZ2UsXG4gICAgICAgIG1vZGVsRGF0YVVybDogbW9kZWxBcnRpZmFjdFVybCxcbiAgICAgICAgZW52aXJvbm1lbnQsXG4gICAgICB9IDoge1xuICAgICAgICAvLyBGYWxzZTogTW9kZWwgaXMgdW5jb21wcmVzc2VkXG4gICAgICAgIGltYWdlLFxuICAgICAgICBtb2RlbERhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIGNvbXByZXNzaW9uVHlwZTogJ05vbmUnLFxuICAgICAgICAgICAgczNEYXRhVHlwZTogJ1MzUHJlZml4JyxcbiAgICAgICAgICAgIHMzVXJpOiBtb2RlbEFydGlmYWN0VXJsLFxuICAgICAgICAgICAgbW9kZWxBY2Nlc3NDb25maWc6IHtcbiAgICAgICAgICAgICAgYWNjZXB0RXVsYTogdGhpcy5hY2NlcHRFdWxhLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBlbnZpcm9ubWVudCxcbiAgICAgIH0sXG4gICAgICB0YWdzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBrZXk6ICdtb2RlbElkJyxcbiAgICAgICAgICB2YWx1ZTogdGhpcy5zcGVjLm1vZGVsSWQsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBrZXk6ICdtb2RlbFZlcnNpb24nLFxuICAgICAgICAgIHZhbHVlOiB0aGlzLnNwZWMudmVyc2lvbixcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB2cGNDb25maWc6IHZwY0NvbmZpZyxcbiAgICB9KTtcblxuICAgIHJldHVybiBtb2RlbDtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0TW9kZWxGcm9tUGFja2FnZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCB2cGNDb25maWc6IHNhZ2VtYWtlci5DZm5Nb2RlbC5WcGNDb25maWdQcm9wZXJ0eSB8IHVuZGVmaW5lZCkge1xuICAgIGNvbnN0IG1vZGVsUGFja2FnZUFybnMgPSB0aGlzLnNwZWMubW9kZWxQYWNrYWdlQXJucyB8fCB7fTtcbiAgICBjb25zdCBzdXBwb3J0ZWRSZWdpb25zID0gT2JqZWN0LmtleXMobW9kZWxQYWNrYWdlQXJucyk7XG4gICAgaWYgKCFzdXBwb3J0ZWRSZWdpb25zLmluY2x1ZGVzKHRoaXMucmVnaW9uKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgVGhlIG1vZGVsIHBhY2thZ2UgaXMgbm90IGF2YWlsYWJsZSBpbiB0aGUgcmVnaW9uICR7XG4gICAgICAgICAgdGhpcy5yZWdpb25cbiAgICAgICAgfS4gU3VwcG9ydGVkIHJlZ2lvbnM6ICR7c3VwcG9ydGVkUmVnaW9ucy5qb2luKCcsICcpfS5gLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBtb2RlbFBhY2thZ2VBcm4gPSBtb2RlbFBhY2thZ2VBcm5zW3RoaXMucmVnaW9uXTtcblxuICAgIGNvbnN0IG1vZGVsID0gbmV3IHNhZ2VtYWtlci5DZm5Nb2RlbChzY29wZSwgYCR7dGhpcy5zcGVjLm1vZGVsSWR9LW1vZGVsLSR7aWR9YCwge1xuICAgICAgZXhlY3V0aW9uUm9sZUFybjogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICBlbmFibGVOZXR3b3JrSXNvbGF0aW9uOiB0cnVlLFxuICAgICAgcHJpbWFyeUNvbnRhaW5lcjoge1xuICAgICAgICBtb2RlbFBhY2thZ2VOYW1lOiBtb2RlbFBhY2thZ2VBcm4sXG4gICAgICB9LFxuICAgICAgdGFnczogW1xuICAgICAgICB7XG4gICAgICAgICAga2V5OiAnbW9kZWxJZCcsXG4gICAgICAgICAgdmFsdWU6IHRoaXMuc3BlYy5tb2RlbElkLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAga2V5OiAnbW9kZWxWZXJzaW9uJyxcbiAgICAgICAgICB2YWx1ZTogdGhpcy5zcGVjLnZlcnNpb24sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdnBjQ29uZmlnOiB2cGNDb25maWcsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gbW9kZWw7XG4gIH1cbn1cbiJdfQ==