"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HuggingFaceSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
const base_class_1 = require("../../../common/base-class");
/**
 * @summary The HuggingFaceSageMakerEndpoint class.
 */
class HuggingFaceSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            constructName: base_class_1.ConstructName.HUGGINGFACESAGEMAKERENDPOINT,
            constructId: id,
        };
        // No lambda function to use AWS SDK for service metric
        const lambdaFunctions = [];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        this.modelId = props.modelId;
        this.instanceType = props.instanceType;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        const image = props.container.bind(this, this.grantPrincipal).imageName;
        const modelIdStr = this.modelId.split('/').join('-').split('.').join('-');
        const model = new sagemaker.CfnModel(scope, `${modelIdStr}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            primaryContainer: {
                image,
                mode: 'SingleModel',
                environment: {
                    SAGEMAKER_CONTAINER_LOG_LEVEL: '20',
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    HF_MODEL_ID: this.modelId,
                    ...this.environment,
                },
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
            vpcConfig: props.vpcConfig,
        });
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType: this.instanceType.toString(),
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${modelIdStr}-endpoint-${id}`, {
            endpointName: props.endpointName,
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
}
exports.HuggingFaceSageMakerEndpoint = HuggingFaceSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
HuggingFaceSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.HuggingFaceSageMakerEndpoint", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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