"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
const base_class_1 = require("../../../common/base-class");
class CustomSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            constructName: base_class_1.ConstructName.CUSTOMSAGEMAKERENDPOINT,
            constructId: id,
        };
        // No lambda function to use AWS SDK for service metric
        const lambdaFunctions = [];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        this.instanceType = props.instanceType;
        this.modelId = props.modelId;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.modelDataUrl = props.modelDataUrl;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.modelDataDownloadTimeoutInSeconds = props.modelDataDownloadTimeoutInSeconds ?? 600;
        const image = props.container.bind(this, this.grantPrincipal).imageName;
        const modelIdStr = this.modelId.split('/').join('-').split('.').join('-');
        const isArtifactCompressed = this.modelDataUrl.endsWith('.tar.gz');
        const model = new sagemaker.CfnModel(scope, `${modelIdStr}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            primaryContainer: isArtifactCompressed ? {
                image,
                mode: 'SingleModel',
                modelDataUrl: this.modelDataUrl,
                environment: {
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    ...this.environment,
                },
            } : {
                image,
                mode: 'SingleModel',
                modelDataSource: {
                    s3DataSource: {
                        compressionType: 'None',
                        s3DataType: 'S3Prefix',
                        s3Uri: this.modelDataUrl,
                    },
                },
                environment: {
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    ...this.environment,
                },
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
            vpcConfig: props.vpcConfig,
        });
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType: this.instanceType.toString(),
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    volumeSizeInGb: props.volumeSizeInGb,
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                    modelDataDownloadTimeoutInSeconds: this.modelDataDownloadTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${modelIdStr}-endpoint-${id}`, {
            endpointName: props.endpointName,
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
}
exports.CustomSageMakerEndpoint = CustomSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
CustomSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.CustomSageMakerEndpoint", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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