"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.download_data = exports.generateJumpStartModels = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const fs = require("fs");
const path = require("path");
const zlib = require("zlib");
const generate_utils_1 = require("./generate-utils");
const jumpstart_constants_1 = require("../private/jumpstart-constants");
const JUMPSTART_CACHE_PATH = path.join(__dirname, './.cache/jumpstart-models-cache.json');
const JUMPSTART_MODEL_PATH = path.join(__dirname, '../jumpstart-model.ts');
const JUMPSTART_MODELS_PATH = path.join(__dirname, '../jumpstart-models.json');
const ALLOWED_FRAMEWORKS = [
    'huggingface',
    'huggingface-llm',
    'djl-deepspeed',
    'djl-fastertransformer',
    'stabilityai',
];
async function generateJumpStartModels() {
    console.log('Getting JumpStart models data');
    await download_data();
    generateCode();
}
exports.generateJumpStartModels = generateJumpStartModels;
async function download_data() {
    console.log('Downloading JumpStart models data');
    const regions = jumpstart_constants_1.JumpStartConstants.JUMPSTART_LAUNCHED_REGIONS;
    const regionNames = Object.keys(regions).filter((c) => c === 'us-east-1');
    const models = {};
    const frameworks = new Set();
    for (const regionName of regionNames) {
        console.log(`Processing region ${regionName}`);
        const regionData = regions[regionName];
        const manifestS3Key = jumpstart_constants_1.JumpStartConstants.JUMPSTART_DEFAULT_MANIFEST_FILE_S3_KEY;
        const url = `https://${regionData.contentBucket}.s3.${regionName}.amazonaws.com/${manifestS3Key}`;
        const [manifest] = await generate_utils_1.GenerateUtils.downloadJSON(url);
        for (const model of manifest) {
            const specUrl = `https://${regionData.contentBucket}.s3.${regionName}.amazonaws.com/${model.spec_key}`;
            const [modelSpec] = await generate_utils_1.GenerateUtils.downloadJSON(specUrl);
            const { deprecated, hosting_ecr_specs, default_inference_instance_type, supported_inference_instance_types, hosting_model_package_arns, hosting_use_script_uri, hosting_script_key, hosting_artifact_key, hosting_prepacked_artifact_key, inference_environment_variables, hosting_instance_type_variants, hosting_eula_key, } = modelSpec;
            const allowedFramework = ALLOWED_FRAMEWORKS.includes(hosting_ecr_specs.framework);
            console.log(`${deprecated ? '[DEPRECATED] ' : ''}${!allowedFramework ? '[SKIP:' + hosting_ecr_specs.framework + '] ' : ''}${model.model_id}/${model.version}`);
            frameworks.add(hosting_ecr_specs.framework);
            if (deprecated)
                continue;
            if (!ALLOWED_FRAMEWORKS.includes(hosting_ecr_specs.framework))
                continue;
            if (hosting_use_script_uri ||
                (!hosting_prepacked_artifact_key && !hosting_model_package_arns && !hosting_artifact_key)) {
                throw new Error('No model data');
            }
            models[model.model_id] = models[model.model_id] ?? {};
            models[model.model_id][model.version] = models[model.model_id][model.version] ?? {};
            models[model.model_id][model.version] = {
                deprecated,
                hosting_ecr_specs,
                default_inference_instance_type,
                supported_inference_instance_types,
                hosting_model_package_arns,
                hosting_use_script_uri,
                hosting_artifact_key,
                hosting_script_key,
                hosting_prepacked_artifact_key,
                inference_environment_variables,
                hosting_instance_type_variants,
                hosting_eula_key,
            };
        }
    }
    generate_utils_1.GenerateUtils.writeFileSyncWithDirs(JUMPSTART_CACHE_PATH, JSON.stringify(models));
    console.log('Frameworks', Array.from(frameworks));
}
exports.download_data = download_data;
function generateCode() {
    console.log('Generating JumpStart models data');
    const data = JSON.parse(fs.readFileSync(JUMPSTART_CACHE_PATH, 'utf8'));
    let modelsStr = '';
    let specs = {};
    for (const modelId of Object.keys(data)) {
        for (const version of Object.keys(data[modelId])) {
            const modelName = `${generate_utils_1.GenerateUtils.replaceAll(modelId, '-', '_')}_${generate_utils_1.GenerateUtils.replaceAll(version, '\\.', '_')}`.toUpperCase();
            const specSource = data[modelId][version];
            const environment = {};
            for (const env of specSource.inference_environment_variables) {
                environment[env.name] = env.default;
            }
            const hosting_eula_key = specSource.hosting_eula_key;
            const instanceVariants = specSource.hosting_instance_type_variants?.variants;
            const instanceAliases = specSource.hosting_instance_type_variants?.regional_aliases;
            let instanceVariantsArr;
            let instanceAliasesArr;
            if (instanceVariants) {
                instanceVariantsArr = [];
                for (const instanceType of Object.keys(instanceVariants)) {
                    const current = instanceVariants[instanceType];
                    instanceVariantsArr.push({
                        instanceType,
                        imageUri: current.regional_properties?.image_uri,
                        environment: current.properties?.environment_variables,
                    });
                }
            }
            if (instanceAliases) {
                instanceAliasesArr = [];
                for (const region of Object.keys(instanceAliases)) {
                    const current = instanceAliases[region];
                    instanceAliasesArr.push({
                        region,
                        aliases: current,
                    });
                }
            }
            const spec = {
                modelId,
                version,
                defaultInstanceType: specSource.default_inference_instance_type,
                instanceTypes: specSource.supported_inference_instance_types,
                modelPackageArns: specSource.hosting_model_package_arns,
                prepackedArtifactKey: specSource.hosting_prepacked_artifact_key,
                artifactKey: specSource.hosting_artifact_key,
                environment,
                instanceAliases: instanceAliasesArr,
                instanceVariants: instanceVariantsArr,
                requiresEula: hosting_eula_key,
            };
            if (spec.modelPackageArns) {
                delete spec.artifactKey;
                delete spec.prepackedArtifactKey;
            }
            specs[modelName] = spec;
            modelsStr += '  ' + `public static readonly ${modelName} = this.of('${modelName}');\n`;
        }
    }
    const fileStr = `/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as zlib from 'zlib';
import * as data from './jumpstart-models.json';

export interface IInstanceAliase {
  region: string;
  aliases: { [key: string]: string };
}

export interface IInstanceValiant {
  instanceType: string;
  imageUri?: string;
  environment?: { [key: string]: string };
}

export interface IJumpStartModelSpec {
  modelId: string;
  version: string;
  defaultInstanceType: string;
  instanceTypes: string[];
  modelPackageArns?: { [region: string]: string };
  prepackedArtifactKey?: string;
  artifactKey?: string;
  environment: { [key: string]: string | number | boolean };
  instanceAliases?: IInstanceAliase[];
  instanceVariants?: IInstanceValiant[];
  requiresEula: boolean;
}

export class JumpStartModel {
${modelsStr}

  public static of(name: string): JumpStartModel {
    return new JumpStartModel(name);
  }

  constructor(private readonly name: string) {}

  public bind(): IJumpStartModelSpec {
    const bufferSource = (data as { data: number[] }).data;
    const buffer = Buffer.from(bufferSource);
    const bufferStr = zlib.inflateRawSync(buffer);
    const json = JSON.parse(bufferStr.toString());

    return json[this.name];
  }
}`;
    generate_utils_1.GenerateUtils.writeFileSyncWithDirs(JUMPSTART_MODELS_PATH, JSON.stringify(zlib.deflateRawSync(JSON.stringify(specs)).toJSON()));
    generate_utils_1.GenerateUtils.writeFileSyncWithDirs(JUMPSTART_MODEL_PATH, fileStr);
}
//# sourceMappingURL=data:application/json;base64,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