"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getRepositories = exports.generateDLCContainerImages = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const node_child_process_1 = require("node:child_process");
const path = require("path");
const generate_utils_1 = require("./generate-utils");
const regionName = 'us-west-2';
const startVersionRegex = /^\d+\.\d+\.\d+-/;
const versionRegex = /-v\d+(\.\d+)*(-\d{4}-\d{2}-\d{2}-\d{2}-\d{2}-\d{2})?$/;
const repositories = [
    'huggingface-pytorch-tgi-inference',
    'huggingface-pytorch-inference',
    'huggingface-tensorflow-inference',
    'huggingface-pytorch-inference-neuronx',
    'djl-inference',
];
const DEEP_LEARNING_CONTAINER_IMAGE_PATH = path.join(__dirname, '../deep-learning-container-image.ts');
async function generateDLCContainerImages() {
    console.log('Getting DLC container image data');
    const output = (0, node_child_process_1.execSync)(`aws ecr get-authorization-token --output text --query 'authorizationData[].authorizationToken' --region ${regionName}`);
    const token = output.toString();
    const repositoryTagData = {};
    for (const repositoryName of repositories) {
        console.log(repositoryName);
        const pageSize = 1000;
        const hostname = '763104351884.dkr.ecr.us-west-2.amazonaws.com';
        let link = `/v2/${repositoryName}/tags/list?n=${pageSize}`;
        let tags = [];
        while (link) {
            const [data, response] = await generate_utils_1.GenerateUtils.downloadJSON({
                hostname,
                port: 443,
                method: 'GET',
                path: link,
                headers: {
                    Authorization: `Basic ${token.trim()}`,
                },
            });
            tags.push(...data.tags);
            link = response.headers.link;
            if (link) {
                console.log(link);
                link = link.substring(1, link.indexOf('>')).split(hostname)[1];
                console.log('Link:', link);
            }
        }
        tags = tags.filter((tag) => startVersionRegex.test(tag));
        if (repositoryName == 'huggingface-tensorflow-inference' ||
            repositoryName == 'huggingface-pytorch-inference' ||
            repositoryName == 'huggingface-pytorch-inference-neuronx') {
            tags = tags.filter((tag) => tag.includes('-transformers'));
        }
        else if (repositoryName == 'huggingface-pytorch-tgi-inference') {
            tags = tags.filter((tag) => tag.includes('-tgi'));
        }
        else if (repositoryName == 'djl-inference') {
            tags = tags.filter((tag) => tag.includes('-deepspeed') ||
                tag.includes('-fastertransformer') ||
                tag.includes('-neuronx'));
        }
        tags = tags.map((tag) => tag.replace(versionRegex, ''));
        tags = Array.from(new Set(tags)).sort();
        repositoryTagData[repositoryName] = tags;
        console.log(tags);
    }
    generateCode(repositoryTagData);
}
exports.generateDLCContainerImages = generateDLCContainerImages;
async function getRepositories() {
    const repositoryNames = new Set();
    const GITHUB_URL = 'https://raw.githubusercontent.com/aws/sagemaker-python-sdk/master/src/sagemaker/image_uri_config';
    const fileNames = [
        `${GITHUB_URL}/huggingface-llm.json`,
        `${GITHUB_URL}/huggingface.json`,
        `${GITHUB_URL}/huggingface-neuronx.json`,
        `${GITHUB_URL}/djl-deepspeed.json`,
        `${GITHUB_URL}/djl-fastertransformer.json`,
        `${GITHUB_URL}/djl-neuronx.json`,
    ];
    for (const fileName of fileNames) {
        const [data] = await generate_utils_1.GenerateUtils.downloadJSON(fileName);
        console.log('Processing file:', fileName);
        const versions = data.versions || data.inference.versions;
        for (const version of Object.keys(versions)) {
            const versionData = versions[version];
            const versionAliases = versionData.version_aliases;
            const items = [];
            if (versionAliases) {
                for (const versionAlias of Object.keys(versionAliases)) {
                    items.push(versionData[versionAliases[versionAlias]]);
                }
            }
            else {
                items.push(versionData);
            }
            for (const item of items) {
                const repositoryName = item.repository;
                repositoryNames.add(repositoryName);
            }
        }
    }
    console.log('Repositories:', new Array(...repositories));
    return repositoryNames;
}
exports.getRepositories = getRepositories;
function generateCode(repositoryTagData) {
    let imagesStr = '';
    for (const repositoryName of Object.keys(repositoryTagData).sort()) {
        const tags = repositoryTagData[repositoryName].sort();
        const repositoryNameStr = generate_utils_1.GenerateUtils.replaceAll(repositoryName, '-', '_').toUpperCase();
        for (const tagName of tags) {
            const tagNameStr = generate_utils_1.GenerateUtils.replaceAllBatch(tagName, ['\\.', '-'], '_').toUpperCase();
            const name = `${repositoryNameStr}_${tagNameStr}`;
            imagesStr +=
                '  ' +
                    `public static readonly ${name} = this.fromDeepLearningContainerImage('${repositoryName}', '${tagName}');\n`;
        }
    }
    const fileStr = `/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { Stack } from 'aws-cdk-lib';
import * as ecr from 'aws-cdk-lib/aws-ecr';
import * as iam from 'aws-cdk-lib/aws-iam';
import { FactName } from 'aws-cdk-lib/region-info';
import { Construct } from 'constructs';
import { ContainerImage, ContainerImageConfig } from './container-image';

export class DeepLearningContainerImage extends ContainerImage {
${imagesStr}

  public static fromDeepLearningContainerImage(
    repositoryName: string,
    tag: string,
    accountId?: string,
  ): ContainerImage {
    return new DeepLearningContainerImage(repositoryName, tag, accountId);
  }

  constructor(
    private readonly repositoryName: string,
    private readonly tag: string,
    private readonly accountId?: string,
  ) {
    super();
  }

  public bind(
    scope: Construct,
    grantable: iam.IGrantable,
  ): ContainerImageConfig {
    const accountId =
      this.accountId ??
      Stack.of(scope).regionalFact(FactName.DLC_REPOSITORY_ACCOUNT);

    const repository = ecr.Repository.fromRepositoryAttributes(
      scope,
      'DeepLearningContainerRepository',
      {
        repositoryName: this.repositoryName,
        repositoryArn: ecr.Repository.arnForLocalRepository(
          this.repositoryName,
          scope,
          accountId,
        ),
      },
    );

    repository.grantPull(grantable);

    return { imageName: repository.repositoryUri + ':' + this.tag };
  }
}
`;
    generate_utils_1.GenerateUtils.writeFileSyncWithDirs(DEEP_LEARNING_CONTAINER_IMAGE_PATH, fileStr);
}
//# sourceMappingURL=data:application/json;base64,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