"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContentGenerationAppSyncLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
 * @summary The ContentGenerationAppSyncLambda class.
 */
class ContentGenerationAppSyncLambda extends base_class_1.BaseClass {
    /**
     * @summary Constructs a new instance of the ContentGenerationAppSyncLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ContentGenerationAppSyncLambdaProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            constructName: base_class_1.ConstructName.AWSCONTENTGENAPPSYNCLAMBDA,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingGeneratedAssetsBucketObj,
            bucketProps: props.generatedAssetsBucketProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + this.stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'generateImageConstructLogGroup');
        const role = new iam.Role(this, 'generateImageConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(90),
                },
            ],
        });
        // Bucket containing the inputs assets (documents - text format) uploaded by the user
        let generatedAssetsBucket;
        if (!props.existingGeneratedAssetsBucketObj) {
            let tmpBucket;
            if (!props.generatedAssetsBucketProps) {
                tmpBucket = new s3.Bucket(this, 'generatedAssetsBucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'generated-asset-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [
                        {
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        },
                    ],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'generatedAssetsBucket' + this.stage, props.generatedAssetsBucketProps);
            }
            generatedAssetsBucket = tmpBucket;
            this.s3GenerateAssetsBucket = tmpBucket;
        }
        else {
            generatedAssetsBucket = props.existingGeneratedAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3GenerateAssetsBucketInterface = generatedAssetsBucket;
        // GraphQL API
        const generate_image_graphql_api = new appsync.GraphqlApi(this, 'generateImageGraphqlApi', {
            name: 'generateImageGraphqlApi' + this.stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-contentgen-appsync-lambda/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: this.enablexray,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
        });
        this.graphqlApi = generate_image_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi
            ? generate_image_graphql_api.graphqlUrl
            : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi
            ? generate_image_graphql_api.apiId
            : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceQuestionAnswering', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateGenImageJobStatusResolver', {
            fieldName: 'updateGenerateImageStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                          {
                              "version": "2017-02-28",
                              "payload": $util.toJson($context.args)
                          }
                          `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.generatedImageBus = new events.EventBus(this, 'generateImageEventBus' + this.stage, {
                eventBusName: 'generateImageEventBus' + this.stage,
            });
        }
        else {
            this.generatedImageBus = props.existingBusInterface;
        }
        // create httpdatasource with generate_image_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('generateImageEventBridgeDataSource' + this.stage, this.generatedImageBus, {
            name: 'generateImageEventBridgeDataSource' + this.stage,
        });
        // Lambda function used to validate inputs in the step function
        const generate_image_function_role = new iam.Role(this, 'generate_image_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [
                                `arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ec2:DescribeNetworkInterfaces'],
            resources: ['*'],
        }));
        // The lambda will genearte image and upload to s3
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:GetBucket', 's3:ListBucket', 's3:PutObject'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3GenerateAssetsBucketInterface.bucketName + '/*',
            ],
        }));
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'comprehend:DetectToxicContent',
            ],
            resources: ['*'],
        }));
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'rekognition:DetectModerationLabels',
                'rekognition:DetectText',
                'rekognition:DetectLabels',
                'rekognition:DetectFaces',
            ],
            resources: ['*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(generate_image_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_docker_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-contentgen-appsync-lambda/src')),
            functionName: 'lambda_generate_image' + this.stage,
            description: 'Lambda function for generating image',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: generate_image_function_role,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                OUTPUT_BUCKET: this.s3GenerateAssetsBucketInterface.bucketName,
            },
            ...(props.lambdaProvisionedConcurrency && {
                currentVersionOptions: {
                    provisionedConcurrentExecutions: props.lambdaProvisionedConcurrency,
                },
            }),
        };
        const generate_image_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_content_generation' + this.stage, construct_docker_lambda_props, props.customDockerLambdaProps);
        generate_image_function.currentVersion;
        const lambdaFunctions = [generate_image_function];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // Add GraphQl permissions to the IAM role for the Lambda function
        generate_image_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['appsync:GraphQL'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':apis/' +
                    updateGraphQlApiId +
                    '/*',
            ],
        }));
        this.generatedImageBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('GenerateImageResolver', {
            fieldName: 'generateImage',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                {
                  "version": "2018-05-29",
                  "operation": "PutEvents",
                  "events": [{
                      "source": "textToImage",
                      "detail": $util.toJson($context.arguments),
                      "detailType": "genAIdemo"
                  }]
                }

 
                        `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                        #if($ctx.error)
                            $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                        #end
                            $util.toJson($ctx.result)
                        `),
        });
        const rule = new events.Rule(this, 'textToImageRule' + this.stage, {
            description: 'Rule to trigger textToImage function',
            eventBus: this.generatedImageBus,
            eventPattern: {
                source: ['textToImage'],
            },
        });
        rule.addTarget(new targets.LambdaFunction(generate_image_function));
        this.cgLambdaFunction = generate_image_function;
    }
}
exports.ContentGenerationAppSyncLambda = ContentGenerationAppSyncLambda;
_a = JSII_RTTI_SYMBOL_1;
ContentGenerationAppSyncLambda[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.ContentGenerationAppSyncLambda", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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