"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.createDefaultIsolatedVpcProps = exports.suppressEncryptedLogWarnings = exports.suppressMapPublicIpWarnings = exports.DefaultVpcProps = exports.getlambdaSecuritygroup = exports.getPrivateSubnetIDs = exports.buildVpc = exports.CheckVpcProps = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const kendra_helper_1 = require("./kendra-helper");
const utils_1 = require("./utils");
const types_1 = require("../../patterns/gen-ai/aws-rag-appsync-stepfn-kendra/types");
function CheckVpcProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if ((propsObject.deployVpc || propsObject.vpcProps) && propsObject.existingVpc) {
        errorMessages += 'Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckVpcProps = CheckVpcProps;
function buildVpc(scope, props) {
    if (props?.existingVpc) {
        return props?.existingVpc;
    }
    let defaultVpcProps = DefaultVpcProps();
    let cumulativeProps = defaultVpcProps;
    // Merge props provided by construct builder and by the end user
    // If user provided props are empty, the vpc will use only the builder provided props
    //cumulativeProps = consolidateProps(cumulativeProps, props?.userVpcProps, props?.constructVpcProps);
    const vpc = new aws_ec2_1.Vpc(scope, 'Vpc', cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog('FlowLog');
    suppressMapPublicIpWarnings(vpc);
    suppressEncryptedLogWarnings(flowLog);
    return vpc;
}
exports.buildVpc = buildVpc;
// get subnet id for passed vpc.
function getPrivateSubnetIDs(vpc) {
    return vpc.privateSubnets.map(subnet => subnet.subnetId);
}
exports.getPrivateSubnetIDs = getPrivateSubnetIDs;
// get lambda security group for passed VPC
function getlambdaSecuritygroup(scope, vpc) {
    let lambdaSecurityGroup = new aws_ec2_1.SecurityGroup(scope, 'lambdaSecurityGroup', {
        vpc: vpc,
        allowAllOutbound: true,
        description: 'security group for lambda',
        securityGroupName: 'lambdaSecurityGroup',
    });
    return lambdaSecurityGroup;
}
exports.getlambdaSecuritygroup = getlambdaSecuritygroup;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Creates the default vpc props with public , private_with_egress and private_isolated subnet configuration.
 */
function DefaultVpcProps() {
    return {
        subnetConfiguration: [
            {
                name: 'public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
                cidrMask: 24,
            },
            {
                name: 'private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
                cidrMask: 24,
            },
            {
                name: 'isolated',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                cidrMask: 24,
            },
        ],
        ipAddresses: aws_ec2_1.IpAddresses.cidr('10.0.0.0/16'),
    };
}
exports.DefaultVpcProps = DefaultVpcProps;
function suppressMapPublicIpWarnings(vpc) {
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        (0, utils_1.addCfnSuppressRules)(cfnSubnet, [
            {
                id: 'W33',
                reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true',
            },
        ]);
    });
}
exports.suppressMapPublicIpWarnings = suppressMapPublicIpWarnings;
function suppressEncryptedLogWarnings(flowLog) {
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = flowLog.logGroup?.node.defaultChild;
    (0, utils_1.addCfnSuppressRules)(cfnLogGroup, [
        {
            id: 'W84',
            reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)',
        },
    ]);
}
exports.suppressEncryptedLogWarnings = suppressEncryptedLogWarnings;
function AddInterfaceEndpoint(scope, vpc, service, interfaceTag) {
    const endpointDefaultSecurityGroup = (0, kendra_helper_1.buildSecurityGroup)(scope, `${scope.node.id}-${service.endpointName}`, {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: aws_ec2_1.Peer.ipv4(vpc.vpcCidrBlock), connection: aws_ec2_1.Port.tcp(443) }], []);
    vpc.addInterfaceEndpoint(interfaceTag, {
        service: service.endpointInterfaceService,
        securityGroups: [endpointDefaultSecurityGroup],
    });
}
function createDefaultIsolatedVpcProps() {
    return {
        natGateways: 0,
        subnetConfiguration: [
            {
                cidrMask: 18,
                name: 'isolated',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
            },
        ],
    };
}
exports.createDefaultIsolatedVpcProps = createDefaultIsolatedVpcProps;
function AddGatewayEndpoint(vpc, service, interfaceTag) {
    vpc.addGatewayEndpoint(interfaceTag, {
        service: service.endpointGatewayService,
    });
}
function CheckIfEndpointAlreadyExists(vpc, interfaceTag) {
    return vpc.node.children.some((child) => child.node.id === interfaceTag);
}
const endpointSettings = [
    {
        endpointName: types_1.ServiceEndpointTypeEnum.DYNAMODB,
        endpointType: types_1.EndpointTypes.GATEWAY,
        endpointGatewayService: aws_ec2_1.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.S3,
        endpointType: types_1.EndpointTypes.GATEWAY,
        endpointGatewayService: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.STEP_FUNCTIONS,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.STEP_FUNCTIONS,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.SNS,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.SQS,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.SAGEMAKER_RUNTIME,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.SECRETS_MANAGER,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.SSM,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.SSM,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.ECR_API,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.ECR,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.ECR_DKR,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.ECR_DOCKER,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.EVENTS,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.KINESIS_FIREHOSE,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.KINESIS_FIREHOSE,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.KINESIS_STREAMS,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.KINESIS_STREAMS,
    },
    {
        endpointName: types_1.ServiceEndpointTypeEnum.KENDRA,
        endpointType: types_1.EndpointTypes.INTERFACE,
        endpointInterfaceService: aws_ec2_1.InterfaceVpcEndpointAwsService.KENDRA,
    },
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (CheckIfEndpointAlreadyExists(vpc, interfaceTag)) {
        return;
    }
    const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
    if (!service) {
        throw new Error('Unsupported Service sent to AddServiceEndpoint');
    }
    if (service.endpointType === types_1.EndpointTypes.GATEWAY) {
        AddGatewayEndpoint(vpc, service, interfaceTag);
    }
    if (service.endpointType === types_1.EndpointTypes.INTERFACE) {
        AddInterfaceEndpoint(scope, vpc, service, interfaceTag);
    }
    // ESLint requires this return statement, so disabling SonarQube warning
    return; // NOSONAR
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
//# sourceMappingURL=data:application/json;base64,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