"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isPlainObject = exports.addCfnSuppressRules = exports.lambdaMemorySizeLimiter = exports.recommendedMaximumLambdaMemorySize = exports.maximumLambdaMemorySizeContextItem = exports.generatePhysicalNameV2 = exports.generatePhysicalName = exports.version = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
/**
 * The version of this package
 */
// eslint-disable-next-line @typescript-eslint/no-require-imports
exports.version = require('../../../package.json').version;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @summary Creates a physical resource name in the style of the CDK (string+hash) - this value incorporates Stack ID,
 * so it will remain static in multiple updates of a single stack, but will be different in a separate stack instance
 * @param {string} prefix - the prefix for the name
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if necessary) + a hash of the full concatenated parts
 *
 * @deprecated This function is deprecated and will be removed in a future major version.
 * Please use the new function generatePhysicalNameV2 instead.
 */
function generatePhysicalName(prefix, parts, maxLength) {
    // The result will consist of:
    //    -The prefix - unaltered
    //    -The parts concatenated, but reduced in size to meet the maxLength limit for the overall name
    //    -A hyphen delimiter
    //    -The GUID portion of the stack arn
    const stackIdGuidLength = 36;
    const prefixLength = prefix.length;
    const maxPartsLength = maxLength - prefixLength - 1 - stackIdGuidLength; // 1 is the hyphen
    // Extract the Stack ID Guid
    const uniqueStackIdPart = cdk.Fn.select(2, cdk.Fn.split('/', `${cdk.Aws.STACK_ID}`));
    let allParts = '';
    parts.forEach((part) => {
        allParts += part;
    });
    if (allParts.length > maxPartsLength) {
        const subStringLength = maxPartsLength / 2;
        allParts = allParts.substring(0, subStringLength) + allParts.substring(allParts.length - subStringLength);
    }
    if (prefix.length + allParts.length + stackIdGuidLength + 1 /* hyphen */ > maxLength) {
        throw new Error(`The generated name is longer than the maximum length of ${maxLength}`);
    }
    return prefix.toLowerCase() + allParts + '-' + uniqueStackIdPart;
}
exports.generatePhysicalName = generatePhysicalName;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @summary Creates a physical resource name in the style of the CDK (string+hash) - this value incorporates
 * the Stack Name and node ID, so it will remain static in multiple updates of a single stack, but will be
 * different in a separate stack instance.
 *
 * This new version allows for names shorter than 36 characters with control over casing.
 *
 * The minimum length is the length of the prefix and separator plus 10.
 */
function generatePhysicalNameV2(
/**
 * The CDK scope of the resource.
 */
scope, 
/**
 * The prefix for the name.
 */
prefix, 
/**
 * Options for generating the name.
 */
options) {
    const { maxLength = 256, lower = false, separator = '', allowedSpecialCharacters = undefined, } = options ?? {};
    if (maxLength < (prefix + separator).length) {
        throw new Error('The prefix is longer than the maximum length.');
    }
    const uniqueName = cdk.Names.uniqueResourceName(scope, { maxLength: maxLength - (prefix + separator).length, separator, allowedSpecialCharacters });
    const name = `${prefix}${separator}${uniqueName}`;
    if (name.length > maxLength) {
        throw new Error(`The generated name is longer than the maximum length of ${maxLength}`);
    }
    return lower ? name.toLowerCase() : name;
}
exports.generatePhysicalNameV2 = generatePhysicalNameV2;
exports.maximumLambdaMemorySizeContextItem = 'maximumLambdaMemorySize';
exports.recommendedMaximumLambdaMemorySize = 7076;
function lambdaMemorySizeLimiter(construct, requestedMemorySizeInMegabytes) {
    const maximumLambaMemorySize = construct.node.tryGetContext(exports.maximumLambdaMemorySizeContextItem) === undefined ?
        exports.recommendedMaximumLambdaMemorySize :
        parseInt(construct.node.tryGetContext(exports.maximumLambdaMemorySizeContextItem));
    if (maximumLambaMemorySize < exports.recommendedMaximumLambdaMemorySize) {
        console.warn(`Maximum Lambda memorySize, ${maximumLambaMemorySize}, is less than the recommended ${exports.recommendedMaximumLambdaMemorySize}.`);
    }
    if (requestedMemorySizeInMegabytes > maximumLambaMemorySize) {
        console.warn(`Reducing Lambda memorySize, ${requestedMemorySizeInMegabytes} to ${maximumLambaMemorySize} for ${construct.constructor.name}`);
        return maximumLambaMemorySize;
    }
    else {
        return requestedMemorySizeInMegabytes;
    }
}
exports.lambdaMemorySizeLimiter = lambdaMemorySizeLimiter;
/**
 * Adds CFN NAG suppress rules to the CDK resource.
 * @param resource The CDK resource
 * @param rules The CFN NAG suppress rules
 */
function addCfnSuppressRules(resource, rules) {
    if (resource instanceof cdk.Resource) {
        resource = resource.node.defaultChild;
    }
    if (resource.cfnOptions.metadata?.cfn_nag?.rules_to_suppress) {
        resource.cfnOptions.metadata?.cfn_nag.rules_to_suppress.push(...rules);
    }
    else {
        resource.addMetadata('cfn_nag', {
            rules_to_suppress: rules,
        });
    }
}
exports.addCfnSuppressRules = addCfnSuppressRules;
function isObject(val) {
    return val !== null && typeof val === 'object' && !Array.isArray(val);
}
function isPlainObject(o) {
    if (!isObject(o))
        return false;
    if (Object.getPrototypeOf(o) === null)
        return true;
    let proto = o;
    while (Object.getPrototypeOf(proto) !== null) {
        proto = Object.getPrototypeOf(proto);
    }
    return Object.getPrototypeOf(o) === proto;
}
exports.isPlainObject = isPlainObject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvY29tbW9uL2hlbHBlcnMvdXRpbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7QUFDSCxtQ0FBbUM7QUFJbkM7O0dBRUc7QUFDSCxpRUFBaUU7QUFDcEQsUUFBQSxPQUFPLEdBQUcsT0FBTyxDQUFDLHVCQUF1QixDQUFDLENBQUMsT0FBTyxDQUFDO0FBQ2hFOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILFNBQWdCLG9CQUFvQixDQUNsQyxNQUFjLEVBQ2QsS0FBZSxFQUNmLFNBQWlCO0lBRWpCLDhCQUE4QjtJQUM5Qiw2QkFBNkI7SUFDN0IsbUdBQW1HO0lBQ25HLHlCQUF5QjtJQUN6Qix3Q0FBd0M7SUFFeEMsTUFBTSxpQkFBaUIsR0FBRyxFQUFFLENBQUM7SUFDN0IsTUFBTSxZQUFZLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQztJQUNuQyxNQUFNLGNBQWMsR0FBRyxTQUFTLEdBQUcsWUFBWSxHQUFHLENBQUMsR0FBRyxpQkFBaUIsQ0FBQyxDQUFDLGtCQUFrQjtJQUUzRiw0QkFBNEI7SUFDNUIsTUFBTSxpQkFBaUIsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFckYsSUFBSSxRQUFRLEdBQVcsRUFBRSxDQUFDO0lBRTFCLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtRQUNyQixRQUFRLElBQUksSUFBSSxDQUFDO0lBQ25CLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLGNBQWMsRUFBRSxDQUFDO1FBQ3JDLE1BQU0sZUFBZSxHQUFHLGNBQWMsR0FBRyxDQUFDLENBQUM7UUFDM0MsUUFBUSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLGVBQWUsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxlQUFlLENBQUMsQ0FBQztJQUM1RyxDQUFDO0lBRUQsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLFFBQVEsQ0FBQyxNQUFNLEdBQUcsaUJBQWlCLEdBQUcsQ0FBQyxDQUFDLFlBQVksR0FBRyxTQUFTLEVBQUUsQ0FBQztRQUNyRixNQUFNLElBQUksS0FBSyxDQUFDLDJEQUEyRCxTQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQzFGLENBQUM7SUFFRCxPQUFPLE1BQU0sQ0FBQyxXQUFXLEVBQUUsR0FBRyxRQUFRLEdBQUcsR0FBRyxHQUFHLGlCQUFpQixDQUFDO0FBQ25FLENBQUM7QUFsQ0Qsb0RBa0NDO0FBV0Q7Ozs7Ozs7Ozs7R0FVRztBQUNILFNBQWdCLHNCQUFzQjtBQUNwQzs7R0FFRztBQUNILEtBQWlCO0FBQ2pCOztHQUVHO0FBQ0gsTUFBYztBQUNkOztHQUVHO0FBQ0gsT0FBdUM7SUFFdkMsTUFBTSxFQUNKLFNBQVMsR0FBRyxHQUFHLEVBQ2YsS0FBSyxHQUFFLEtBQUssRUFDWixTQUFTLEdBQUUsRUFBRSxFQUNiLHdCQUF3QixHQUFDLFNBQVMsR0FDbkMsR0FBRyxPQUFPLElBQUksRUFBRSxDQUFDO0lBQ2xCLElBQUksU0FBUyxHQUFHLENBQUMsTUFBTSxHQUFHLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztJQUNuRSxDQUFDO0lBQ0QsTUFBTSxVQUFVLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FDN0MsS0FBSyxFQUNMLEVBQUUsU0FBUyxFQUFFLFNBQVMsR0FBRyxDQUFDLE1BQU0sR0FBRyxTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsU0FBUyxFQUFFLHdCQUF3QixFQUFFLENBQzVGLENBQUM7SUFDRixNQUFNLElBQUksR0FBRyxHQUFHLE1BQU0sR0FBRyxTQUFTLEdBQUcsVUFBVSxFQUFFLENBQUM7SUFDbEQsSUFBSSxJQUFJLENBQUMsTUFBTSxHQUFHLFNBQVMsRUFBRSxDQUFDO1FBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELFNBQVMsRUFBRSxDQUFDLENBQUM7SUFDMUYsQ0FBQztJQUNELE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztBQUMzQyxDQUFDO0FBaENELHdEQWdDQztBQUVZLFFBQUEsa0NBQWtDLEdBQUcseUJBQXlCLENBQUM7QUFDL0QsUUFBQSxrQ0FBa0MsR0FBRyxJQUFJLENBQUM7QUFDdkQsU0FBZ0IsdUJBQXVCLENBQUMsU0FBcUIsRUFBRSw4QkFBc0M7SUFDbkcsTUFBTSxzQkFBc0IsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQywwQ0FBa0MsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBQzdHLDBDQUFrQyxDQUFDLENBQUM7UUFDcEMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDBDQUFrQyxDQUFDLENBQUMsQ0FBQztJQUM3RSxJQUFJLHNCQUFzQixHQUFHLDBDQUFrQyxFQUFFLENBQUM7UUFDaEUsT0FBTyxDQUFDLElBQUksQ0FBQyw4QkFBOEIsc0JBQXNCLGtDQUFrQywwQ0FBa0MsR0FBRyxDQUFDLENBQUM7SUFDNUksQ0FBQztJQUNELElBQUksOEJBQThCLEdBQUcsc0JBQXNCLEVBQUUsQ0FBQztRQUM1RCxPQUFPLENBQUMsSUFBSSxDQUFDLCtCQUErQiw4QkFBOEIsT0FBTyxzQkFBc0IsUUFBUSxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7UUFDN0ksT0FBTyxzQkFBc0IsQ0FBQztJQUNoQyxDQUFDO1NBQU0sQ0FBQztRQUNOLE9BQU8sOEJBQThCLENBQUM7SUFDeEMsQ0FBQztBQUNILENBQUM7QUFiRCwwREFhQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixtQkFBbUIsQ0FBQyxRQUF3QyxFQUFFLEtBQTJCO0lBQ3ZHLElBQUksUUFBUSxZQUFZLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUNyQyxRQUFRLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxZQUErQixDQUFDO0lBQzNELENBQUM7SUFFRCxJQUFJLFFBQVEsQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFLE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxDQUFDO1FBQzdELFFBQVEsQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQztJQUN6RSxDQUFDO1NBQU0sQ0FBQztRQUNOLFFBQVEsQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFO1lBQzlCLGlCQUFpQixFQUFFLEtBQUs7U0FDekIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztBQUNILENBQUM7QUFaRCxrREFZQztBQUVELFNBQVMsUUFBUSxDQUFDLEdBQVc7SUFDM0IsT0FBTyxHQUFHLEtBQUssSUFBSSxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDeEUsQ0FBQztBQUVELFNBQWdCLGFBQWEsQ0FBQyxDQUFTO0lBQ3JDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBQUUsT0FBTyxLQUFLLENBQUM7SUFFL0IsSUFBSSxNQUFNLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxLQUFLLElBQUk7UUFBRSxPQUFPLElBQUksQ0FBQztJQUVuRCxJQUFJLEtBQUssR0FBRyxDQUFDLENBQUM7SUFDZCxPQUFPLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDN0MsS0FBSyxHQUFHLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUNELE9BQU8sTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLENBQUM7QUFDNUMsQ0FBQztBQVZELHNDQVVDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5OYWdTdXBwcmVzc1J1bGUgfSBmcm9tICcuLi8uLi9wYXR0ZXJucy9nZW4tYWkvYXdzLXJhZy1hcHBzeW5jLXN0ZXBmbi1rZW5kcmEvdHlwZXMnO1xuXG4vKipcbiAqIFRoZSB2ZXJzaW9uIG9mIHRoaXMgcGFja2FnZVxuICovXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuZXhwb3J0IGNvbnN0IHZlcnNpb24gPSByZXF1aXJlKCcuLi8uLi8uLi9wYWNrYWdlLmpzb24nKS52ZXJzaW9uO1xuLyoqXG4gKiBAaW50ZXJuYWwgVGhpcyBpcyBhbiBpbnRlcm5hbCBjb3JlIGZ1bmN0aW9uIGFuZCBzaG91bGQgbm90IGJlIGNhbGxlZCBkaXJlY3RseSBieSBTb2x1dGlvbnMgQ29uc3RydWN0cyBjbGllbnRzLlxuICpcbiAqIEBzdW1tYXJ5IENyZWF0ZXMgYSBwaHlzaWNhbCByZXNvdXJjZSBuYW1lIGluIHRoZSBzdHlsZSBvZiB0aGUgQ0RLIChzdHJpbmcraGFzaCkgLSB0aGlzIHZhbHVlIGluY29ycG9yYXRlcyBTdGFjayBJRCxcbiAqIHNvIGl0IHdpbGwgcmVtYWluIHN0YXRpYyBpbiBtdWx0aXBsZSB1cGRhdGVzIG9mIGEgc2luZ2xlIHN0YWNrLCBidXQgd2lsbCBiZSBkaWZmZXJlbnQgaW4gYSBzZXBhcmF0ZSBzdGFjayBpbnN0YW5jZVxuICogQHBhcmFtIHtzdHJpbmd9IHByZWZpeCAtIHRoZSBwcmVmaXggZm9yIHRoZSBuYW1lXG4gKiBAcGFyYW0ge3N0cmluZ1tdfSBwYXJ0cyAtIHRoZSB2YXJpb3VzIHN0cmluZyBjb21wb25lbnRzIG9mIHRoZSBuYW1lIChlZyAtIHN0YWNrTmFtZSwgc29sdXRpb25zIGNvbnN0cnVjdCBJRCwgTDIgY29uc3RydWN0IElEKVxuICogQHBhcmFtIHtudW1iZXJ9IG1heExlbmd0aCAtIHRoZSBsb25nZXN0IHN0cmluZyB0aGF0IGNhbiBiZSByZXR1cm5lZFxuICogQHJldHVybnMge3N0cmluZ30gLSBhIHN0cmluZyB3aXRoIGNvbmNhdGVuYXRlZCBwYXJ0cyAodHJ1bmNhdGVkIGlmIG5lY2Vzc2FyeSkgKyBhIGhhc2ggb2YgdGhlIGZ1bGwgY29uY2F0ZW5hdGVkIHBhcnRzXG4gKlxuICogQGRlcHJlY2F0ZWQgVGhpcyBmdW5jdGlvbiBpcyBkZXByZWNhdGVkIGFuZCB3aWxsIGJlIHJlbW92ZWQgaW4gYSBmdXR1cmUgbWFqb3IgdmVyc2lvbi5cbiAqIFBsZWFzZSB1c2UgdGhlIG5ldyBmdW5jdGlvbiBnZW5lcmF0ZVBoeXNpY2FsTmFtZVYyIGluc3RlYWQuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZW5lcmF0ZVBoeXNpY2FsTmFtZShcbiAgcHJlZml4OiBzdHJpbmcsXG4gIHBhcnRzOiBzdHJpbmdbXSxcbiAgbWF4TGVuZ3RoOiBudW1iZXIsXG4pOiBzdHJpbmcge1xuICAvLyBUaGUgcmVzdWx0IHdpbGwgY29uc2lzdCBvZjpcbiAgLy8gICAgLVRoZSBwcmVmaXggLSB1bmFsdGVyZWRcbiAgLy8gICAgLVRoZSBwYXJ0cyBjb25jYXRlbmF0ZWQsIGJ1dCByZWR1Y2VkIGluIHNpemUgdG8gbWVldCB0aGUgbWF4TGVuZ3RoIGxpbWl0IGZvciB0aGUgb3ZlcmFsbCBuYW1lXG4gIC8vICAgIC1BIGh5cGhlbiBkZWxpbWl0ZXJcbiAgLy8gICAgLVRoZSBHVUlEIHBvcnRpb24gb2YgdGhlIHN0YWNrIGFyblxuXG4gIGNvbnN0IHN0YWNrSWRHdWlkTGVuZ3RoID0gMzY7XG4gIGNvbnN0IHByZWZpeExlbmd0aCA9IHByZWZpeC5sZW5ndGg7XG4gIGNvbnN0IG1heFBhcnRzTGVuZ3RoID0gbWF4TGVuZ3RoIC0gcHJlZml4TGVuZ3RoIC0gMSAtIHN0YWNrSWRHdWlkTGVuZ3RoOyAvLyAxIGlzIHRoZSBoeXBoZW5cblxuICAvLyBFeHRyYWN0IHRoZSBTdGFjayBJRCBHdWlkXG4gIGNvbnN0IHVuaXF1ZVN0YWNrSWRQYXJ0ID0gY2RrLkZuLnNlbGVjdCgyLCBjZGsuRm4uc3BsaXQoJy8nLCBgJHtjZGsuQXdzLlNUQUNLX0lEfWApKTtcblxuICBsZXQgYWxsUGFydHM6IHN0cmluZyA9ICcnO1xuXG4gIHBhcnRzLmZvckVhY2goKHBhcnQpID0+IHtcbiAgICBhbGxQYXJ0cyArPSBwYXJ0O1xuICB9KTtcblxuICBpZiAoYWxsUGFydHMubGVuZ3RoID4gbWF4UGFydHNMZW5ndGgpIHtcbiAgICBjb25zdCBzdWJTdHJpbmdMZW5ndGggPSBtYXhQYXJ0c0xlbmd0aCAvIDI7XG4gICAgYWxsUGFydHMgPSBhbGxQYXJ0cy5zdWJzdHJpbmcoMCwgc3ViU3RyaW5nTGVuZ3RoKSArIGFsbFBhcnRzLnN1YnN0cmluZyhhbGxQYXJ0cy5sZW5ndGggLSBzdWJTdHJpbmdMZW5ndGgpO1xuICB9XG5cbiAgaWYgKHByZWZpeC5sZW5ndGggKyBhbGxQYXJ0cy5sZW5ndGggKyBzdGFja0lkR3VpZExlbmd0aCArIDEgLyogaHlwaGVuICovID4gbWF4TGVuZ3RoKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgZ2VuZXJhdGVkIG5hbWUgaXMgbG9uZ2VyIHRoYW4gdGhlIG1heGltdW0gbGVuZ3RoIG9mICR7bWF4TGVuZ3RofWApO1xuICB9XG5cbiAgcmV0dXJuIHByZWZpeC50b0xvd2VyQ2FzZSgpICsgYWxsUGFydHMgKyAnLScgKyB1bmlxdWVTdGFja0lkUGFydDtcbn1cblxuXG5leHBvcnQgaW50ZXJmYWNlIEdlbmVyYXRlUGh5c2ljYWxOYW1lVjJPcHRpb25zIGV4dGVuZHMgY2RrLlVuaXF1ZVJlc291cmNlTmFtZU9wdGlvbnMge1xuICAvKipcbiAgICogV2hldGhlciB0byBjb252ZXJ0IHRoZSBuYW1lIHRvIGxvd2VyIGNhc2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICBsb3dlcj86IGJvb2xlYW47XG59XG4vKipcbiAqIEBpbnRlcm5hbCBUaGlzIGlzIGFuIGludGVybmFsIGNvcmUgZnVuY3Rpb24gYW5kIHNob3VsZCBub3QgYmUgY2FsbGVkIGRpcmVjdGx5IGJ5IFNvbHV0aW9ucyBDb25zdHJ1Y3RzIGNsaWVudHMuXG4gKlxuICogQHN1bW1hcnkgQ3JlYXRlcyBhIHBoeXNpY2FsIHJlc291cmNlIG5hbWUgaW4gdGhlIHN0eWxlIG9mIHRoZSBDREsgKHN0cmluZytoYXNoKSAtIHRoaXMgdmFsdWUgaW5jb3Jwb3JhdGVzXG4gKiB0aGUgU3RhY2sgTmFtZSBhbmQgbm9kZSBJRCwgc28gaXQgd2lsbCByZW1haW4gc3RhdGljIGluIG11bHRpcGxlIHVwZGF0ZXMgb2YgYSBzaW5nbGUgc3RhY2ssIGJ1dCB3aWxsIGJlXG4gKiBkaWZmZXJlbnQgaW4gYSBzZXBhcmF0ZSBzdGFjayBpbnN0YW5jZS5cbiAqXG4gKiBUaGlzIG5ldyB2ZXJzaW9uIGFsbG93cyBmb3IgbmFtZXMgc2hvcnRlciB0aGFuIDM2IGNoYXJhY3RlcnMgd2l0aCBjb250cm9sIG92ZXIgY2FzaW5nLlxuICpcbiAqIFRoZSBtaW5pbXVtIGxlbmd0aCBpcyB0aGUgbGVuZ3RoIG9mIHRoZSBwcmVmaXggYW5kIHNlcGFyYXRvciBwbHVzIDEwLlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2VuZXJhdGVQaHlzaWNhbE5hbWVWMihcbiAgLyoqXG4gICAqIFRoZSBDREsgc2NvcGUgb2YgdGhlIHJlc291cmNlLlxuICAgKi9cbiAgc2NvcGU6IElDb25zdHJ1Y3QsXG4gIC8qKlxuICAgKiBUaGUgcHJlZml4IGZvciB0aGUgbmFtZS5cbiAgICovXG4gIHByZWZpeDogc3RyaW5nLFxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgZ2VuZXJhdGluZyB0aGUgbmFtZS5cbiAgICovXG4gIG9wdGlvbnM/OiBHZW5lcmF0ZVBoeXNpY2FsTmFtZVYyT3B0aW9ucyxcbik6IHN0cmluZyB7XG4gIGNvbnN0IHtcbiAgICBtYXhMZW5ndGggPSAyNTYsXG4gICAgbG93ZXI9IGZhbHNlLFxuICAgIHNlcGFyYXRvcj0gJycsXG4gICAgYWxsb3dlZFNwZWNpYWxDaGFyYWN0ZXJzPXVuZGVmaW5lZCxcbiAgfSA9IG9wdGlvbnMgPz8ge307XG4gIGlmIChtYXhMZW5ndGggPCAocHJlZml4ICsgc2VwYXJhdG9yKS5sZW5ndGgpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSBwcmVmaXggaXMgbG9uZ2VyIHRoYW4gdGhlIG1heGltdW0gbGVuZ3RoLicpO1xuICB9XG4gIGNvbnN0IHVuaXF1ZU5hbWUgPSBjZGsuTmFtZXMudW5pcXVlUmVzb3VyY2VOYW1lKFxuICAgIHNjb3BlLFxuICAgIHsgbWF4TGVuZ3RoOiBtYXhMZW5ndGggLSAocHJlZml4ICsgc2VwYXJhdG9yKS5sZW5ndGgsIHNlcGFyYXRvciwgYWxsb3dlZFNwZWNpYWxDaGFyYWN0ZXJzIH0sXG4gICk7XG4gIGNvbnN0IG5hbWUgPSBgJHtwcmVmaXh9JHtzZXBhcmF0b3J9JHt1bmlxdWVOYW1lfWA7XG4gIGlmIChuYW1lLmxlbmd0aCA+IG1heExlbmd0aCkge1xuICAgIHRocm93IG5ldyBFcnJvcihgVGhlIGdlbmVyYXRlZCBuYW1lIGlzIGxvbmdlciB0aGFuIHRoZSBtYXhpbXVtIGxlbmd0aCBvZiAke21heExlbmd0aH1gKTtcbiAgfVxuICByZXR1cm4gbG93ZXIgPyBuYW1lLnRvTG93ZXJDYXNlKCkgOiBuYW1lO1xufVxuXG5leHBvcnQgY29uc3QgbWF4aW11bUxhbWJkYU1lbW9yeVNpemVDb250ZXh0SXRlbSA9ICdtYXhpbXVtTGFtYmRhTWVtb3J5U2l6ZSc7XG5leHBvcnQgY29uc3QgcmVjb21tZW5kZWRNYXhpbXVtTGFtYmRhTWVtb3J5U2l6ZSA9IDcwNzY7XG5leHBvcnQgZnVuY3Rpb24gbGFtYmRhTWVtb3J5U2l6ZUxpbWl0ZXIoY29uc3RydWN0OiBJQ29uc3RydWN0LCByZXF1ZXN0ZWRNZW1vcnlTaXplSW5NZWdhYnl0ZXM6IG51bWJlcikge1xuICBjb25zdCBtYXhpbXVtTGFtYmFNZW1vcnlTaXplID0gY29uc3RydWN0Lm5vZGUudHJ5R2V0Q29udGV4dChtYXhpbXVtTGFtYmRhTWVtb3J5U2l6ZUNvbnRleHRJdGVtKSA9PT0gdW5kZWZpbmVkID9cbiAgICByZWNvbW1lbmRlZE1heGltdW1MYW1iZGFNZW1vcnlTaXplIDpcbiAgICBwYXJzZUludChjb25zdHJ1Y3Qubm9kZS50cnlHZXRDb250ZXh0KG1heGltdW1MYW1iZGFNZW1vcnlTaXplQ29udGV4dEl0ZW0pKTtcbiAgaWYgKG1heGltdW1MYW1iYU1lbW9yeVNpemUgPCByZWNvbW1lbmRlZE1heGltdW1MYW1iZGFNZW1vcnlTaXplKSB7XG4gICAgY29uc29sZS53YXJuKGBNYXhpbXVtIExhbWJkYSBtZW1vcnlTaXplLCAke21heGltdW1MYW1iYU1lbW9yeVNpemV9LCBpcyBsZXNzIHRoYW4gdGhlIHJlY29tbWVuZGVkICR7cmVjb21tZW5kZWRNYXhpbXVtTGFtYmRhTWVtb3J5U2l6ZX0uYCk7XG4gIH1cbiAgaWYgKHJlcXVlc3RlZE1lbW9yeVNpemVJbk1lZ2FieXRlcyA+IG1heGltdW1MYW1iYU1lbW9yeVNpemUpIHtcbiAgICBjb25zb2xlLndhcm4oYFJlZHVjaW5nIExhbWJkYSBtZW1vcnlTaXplLCAke3JlcXVlc3RlZE1lbW9yeVNpemVJbk1lZ2FieXRlc30gdG8gJHttYXhpbXVtTGFtYmFNZW1vcnlTaXplfSBmb3IgJHtjb25zdHJ1Y3QuY29uc3RydWN0b3IubmFtZX1gKTtcbiAgICByZXR1cm4gbWF4aW11bUxhbWJhTWVtb3J5U2l6ZTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gcmVxdWVzdGVkTWVtb3J5U2l6ZUluTWVnYWJ5dGVzO1xuICB9XG59XG5cbi8qKlxuICogQWRkcyBDRk4gTkFHIHN1cHByZXNzIHJ1bGVzIHRvIHRoZSBDREsgcmVzb3VyY2UuXG4gKiBAcGFyYW0gcmVzb3VyY2UgVGhlIENESyByZXNvdXJjZVxuICogQHBhcmFtIHJ1bGVzIFRoZSBDRk4gTkFHIHN1cHByZXNzIHJ1bGVzXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBhZGRDZm5TdXBwcmVzc1J1bGVzKHJlc291cmNlOiBjZGsuUmVzb3VyY2UgfCBjZGsuQ2ZuUmVzb3VyY2UsIHJ1bGVzOiBDZm5OYWdTdXBwcmVzc1J1bGVbXSkge1xuICBpZiAocmVzb3VyY2UgaW5zdGFuY2VvZiBjZGsuUmVzb3VyY2UpIHtcbiAgICByZXNvdXJjZSA9IHJlc291cmNlLm5vZGUuZGVmYXVsdENoaWxkIGFzIGNkay5DZm5SZXNvdXJjZTtcbiAgfVxuXG4gIGlmIChyZXNvdXJjZS5jZm5PcHRpb25zLm1ldGFkYXRhPy5jZm5fbmFnPy5ydWxlc190b19zdXBwcmVzcykge1xuICAgIHJlc291cmNlLmNmbk9wdGlvbnMubWV0YWRhdGE/LmNmbl9uYWcucnVsZXNfdG9fc3VwcHJlc3MucHVzaCguLi5ydWxlcyk7XG4gIH0gZWxzZSB7XG4gICAgcmVzb3VyY2UuYWRkTWV0YWRhdGEoJ2Nmbl9uYWcnLCB7XG4gICAgICBydWxlc190b19zdXBwcmVzczogcnVsZXMsXG4gICAgfSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gaXNPYmplY3QodmFsOiBvYmplY3QpIHtcbiAgcmV0dXJuIHZhbCAhPT0gbnVsbCAmJiB0eXBlb2YgdmFsID09PSAnb2JqZWN0JyAmJiAhQXJyYXkuaXNBcnJheSh2YWwpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNQbGFpbk9iamVjdChvOiBvYmplY3QpIHtcbiAgaWYgKCFpc09iamVjdChvKSkgcmV0dXJuIGZhbHNlO1xuXG4gIGlmIChPYmplY3QuZ2V0UHJvdG90eXBlT2YobykgPT09IG51bGwpIHJldHVybiB0cnVlO1xuXG4gIGxldCBwcm90byA9IG87XG4gIHdoaWxlIChPYmplY3QuZ2V0UHJvdG90eXBlT2YocHJvdG8pICE9PSBudWxsKSB7XG4gICAgcHJvdG8gPSBPYmplY3QuZ2V0UHJvdG90eXBlT2YocHJvdG8pO1xuICB9XG4gIHJldHVybiBPYmplY3QuZ2V0UHJvdG90eXBlT2YobykgPT09IHByb3RvO1xufVxuIl19