"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckRedisClusterProps = exports.setInboundRules = exports.getRedisSecurityGroup = exports.buildRedisCluster = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const ec2 = require("aws-cdk-lib/aws-ec2");
const elasticache = require("aws-cdk-lib/aws-elasticache");
function buildRedisCluster(scope, props) {
    const cacheNodeType = props.cfnCacheClusterProps?.cacheNodeType || 'cache.r6g.xlarge';
    const engine = props.cfnCacheClusterProps?.engine || 'redis';
    const numCacheNodes = props.cfnCacheClusterProps?.numCacheNodes || 1;
    let redisCulster = new elasticache.CfnCacheCluster(scope, 'redisCluster', {
        cacheNodeType: cacheNodeType,
        engine: engine,
        numCacheNodes: numCacheNodes,
        cacheSubnetGroupName: getRedisSubnetGroup(scope, props).ref,
        vpcSecurityGroupIds: [props.redisSecurityGroup.securityGroupId],
        port: props.redisPort,
    });
    return redisCulster;
}
exports.buildRedisCluster = buildRedisCluster;
// get redis subnet group from existing vpc
function getRedisSubnetGroup(scope, props) {
    let redisSubnetGroup = new elasticache.CfnSubnetGroup(scope, 'redisSubnetGroup', {
        description: 'Redis subnet group',
        subnetIds: props.subnetIds,
    });
    return redisSubnetGroup;
}
// get redis security group from existing vpc
function getRedisSecurityGroup(scope, props) {
    let redisSecurityGroup = new ec2.SecurityGroup(scope, 'redisSecurityGroup', {
        vpc: props.existingVpc,
        allowAllOutbound: true,
        description: 'security group for elasticache',
    });
    return redisSecurityGroup;
}
exports.getRedisSecurityGroup = getRedisSecurityGroup;
function setInboundRules(redisSecurityGroup, sourceSecuritygroup, redisPort) {
    redisSecurityGroup.connections.allowFrom(sourceSecuritygroup, ec2.Port.tcp(redisPort));
}
exports.setInboundRules = setInboundRules;
function CheckRedisClusterProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if (propsObject.existingRedisCulster && propsObject.cfnCacheClusterProps) {
        errorMessages += 'Error - Either provide existingRedisCulster or cfnCacheClusterProps, but not both.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckRedisClusterProps = CheckRedisClusterProps;
//# sourceMappingURL=data:application/json;base64,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