"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGeneratePresignedUrlFn = exports.createS3FileUploader = exports.buildSecurityGroup = exports.createStepFunctionsExecutionHandlerRole = exports.getUpdateKendraJobStatusLambdaRole = exports.getCheckJobStatusLambdaRole = exports.getKendraStartDataSyncLambdaRole = exports.getStartKendraSyncStepFnRole = exports.getGeneratePresignedUrlLambdaRole = exports.createKendraWorkflowStepFunction = exports.createS3DataSource = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("node:path");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kendra = require("aws-cdk-lib/aws-kendra");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const iam_roles_helper_1 = require("./iam-roles-helper");
const kendra_utils_1 = require("./kendra-utils");
const utils_1 = require("./utils");
function createS3DataSource(scope, targetIndex, id, clientProps) {
    // We go through some hoops here to extract the various inputs, because we need to narrow
    // the type to remove the union with IResolvable
    const dataSourceConfig = clientProps.dataSourceConfiguration;
    if (!dataSourceConfig) {
        throw new Error('Error - an S3 Kendra DataSource requires an DataSourceConfiguration prop');
    }
    const s3DataSourceConfig = dataSourceConfig.s3Configuration;
    if (!s3DataSourceConfig) {
        throw new Error('Error - an S3 Kendra DataSource requires an DataSourceConfiguration.S3Configuration prop');
    }
    // No Bucket name is an error
    if (!s3DataSourceConfig.bucketName) {
        throw new Error('Error - an S3 Kendra DataSource requires the DataSourceConfiguration.S3Configuration.bucketName prop');
    }
    // If there's no role, make a role and put it into defaultProps
    // Put bucket name in default props
    let defaultProps = {
        indexId: targetIndex.ref,
        name: (0, utils_1.generatePhysicalName)('', ['s3-datasource', id], 1000),
        type: 'S3',
    };
    // Return consolidated default and user props
    if (!clientProps.roleArn) {
        const s3CrawlPolicy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [
                        `arn:aws:s3:::${s3DataSourceConfig.bucketName}/*`,
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    actions: [
                        's3:ListBucket',
                    ],
                    resources: [
                        `arn:aws:s3:::${s3DataSourceConfig.bucketName}`,
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'kendra:BatchPutDocument',
                        'kendra:BatchDeleteDocument',
                    ],
                    resources: [
                        targetIndex.attrArn,
                    ],
                }),
            ],
        });
        const dataSourceRole = new iam.Role(scope, `data-source-role-${id}`, {
            assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
            description: 'Policy for Kendra S3 Data Source',
            inlinePolicies: {
                s3CrawlPolicy,
            },
        });
        defaultProps = (0, kendra_utils_1.overrideProps)(defaultProps, { roleArn: dataSourceRole.roleArn });
    }
    const consolidatedProps = (0, kendra_utils_1.consolidateProps)(defaultProps, clientProps);
    return new kendra.CfnDataSource(scope, `data-source-${id}`, consolidatedProps);
}
exports.createS3DataSource = createS3DataSource;
function createKendraWorkflowStepFunction(cdkStack, updateKendraJobStatusFn, kendraSyncLambda, createCheckJobsStatusLambda) {
    const docProcessingLogGroup = new cdk.aws_logs.LogGroup(cdkStack, 'DocProcessingStateMachineLog', {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
    });
    // TODO(miketran): Eventually make this event driven
    const waitFor30Secs = new cdk.aws_stepfunctions.Wait(cdkStack, 'Wait 30 Seconds', {
        time: cdk.aws_stepfunctions.WaitTime.duration(cdk.Duration.seconds(30)),
    });
    const getKendraJobStatus = (0, kendra_utils_1.getStepFnLambdaInvoke)(cdkStack, 'Get Textract Job Status', createCheckJobsStatusLambda);
    // Step function Def
    const docProcessingDefinition = (0, kendra_utils_1.getStepFnLambdaInvoke)(cdkStack, 'Starts a new Kendra Data Sync Job', kendraSyncLambda)
        .next(getKendraJobStatus)
        .next(new cdk.aws_stepfunctions.Choice(cdkStack, 'Kendra DataSync Job Complete?')
        .when(cdk.aws_stepfunctions.Condition.stringEquals('$.KendraJobStatus', 'FAILED'), (0, kendra_utils_1.getStepFnLambdaInvoke)(cdkStack, 'Update Document Status as Failure', updateKendraJobStatusFn))
        .when(cdk.aws_stepfunctions.Condition.stringEquals('$.KendraJobStatus', 'ABORTED'), (0, kendra_utils_1.getStepFnLambdaInvoke)(cdkStack, 'Update Document Status as Aborted', updateKendraJobStatusFn))
        .when(cdk.aws_stepfunctions.Condition.stringEquals('$.KendraJobStatus', 'INCOMPLETE'), (0, kendra_utils_1.getStepFnLambdaInvoke)(cdkStack, 'Update Document Status as Incomplete', updateKendraJobStatusFn))
        .when(cdk.aws_stepfunctions.Condition.stringEquals('$.KendraJobStatus', 'SUCCEEDED'), (0, kendra_utils_1.getStepFnLambdaInvoke)(cdkStack, 'Update Document Status as Completed', updateKendraJobStatusFn))
        .otherwise(waitFor30Secs.next(getKendraJobStatus)));
    const definitionDocProcessingBody = aws_stepfunctions_1.DefinitionBody.fromChainable(docProcessingDefinition);
    return new cdk.aws_stepfunctions.StateMachine(cdkStack, 'DocProcessingStateMachine', {
        definitionBody: definitionDocProcessingBody,
        tracingEnabled: true,
        logs: {
            destination: docProcessingLogGroup,
            level: cdk.aws_stepfunctions.LogLevel.ALL,
        },
    });
}
exports.createKendraWorkflowStepFunction = createKendraWorkflowStepFunction;
function getGeneratePresignedUrlLambdaRole(scope, bucket) {
    const role = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(scope, 'generatePresignedUrlRole', 'Role used by the Generate Pre-signed URL Lambda functio');
    (0, iam_roles_helper_1.addRolePolicies)(role, [
        {
            actions: ['s3:PutObject'],
            resources: [bucket.bucketArn],
        },
        {
            actions: [
                'ec2:DescribeInstances',
                'ec2:CreateNetworkInterface',
                'ec2:AttachNetworkInterface',
                'ec2:DescribeNetworkInterfaces',
                'autoscaling:CompleteLifecycleAction',
                'ec2:DeleteNetworkInterface',
            ],
            resources: ['*'],
        },
    ]);
    return role;
}
exports.getGeneratePresignedUrlLambdaRole = getGeneratePresignedUrlLambdaRole;
function getStartKendraSyncStepFnRole(scope, stateMachine) {
    const role = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(scope, 'startKendraSyncStepFnRole', 'Role used for starting Kendra sync state machine');
    (0, iam_roles_helper_1.addRolePolicies)(role, [
        {
            actions: ['states:StartExecution'],
            resources: [stateMachine.stateMachineArn],
        },
        {
            actions: [
                'ec2:DescribeInstances',
                'ec2:CreateNetworkInterface',
                'ec2:AttachNetworkInterface',
                'ec2:DescribeNetworkInterfaces',
                'autoscaling:CompleteLifecycleAction',
                'ec2:DeleteNetworkInterface',
            ],
            resources: ['*'],
        },
    ]);
    return role;
}
exports.getStartKendraSyncStepFnRole = getStartKendraSyncStepFnRole;
function getKendraStartDataSyncLambdaRole(scope, table, awsRegion, awsAccountId, kendraIndexId, kendraDataSourceIndexId) {
    const role = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(scope, 'startDataSyncRole', 'Role used by the Document Status Update Lambda function');
    (0, iam_roles_helper_1.addRolePolicies)(role, [
        {
            actions: ['dynamodb:PutItem', 'dynamodb:Query', 'dynamodb:GetItem', 'dynamodb:UpdateItem'],
            resources: [table.tableArn],
        },
        {
            actions: ['kendra:StartDataSourceSyncJob'],
            resources: [
                `arn:aws:kendra:${awsRegion}:${awsAccountId}:index/${kendraIndexId}`,
                `arn:aws:kendra:${awsRegion}:${awsAccountId}:index/${kendraIndexId}/data-source/${kendraDataSourceIndexId}`,
            ],
        },
        {
            actions: [
                'ec2:DescribeInstances',
                'ec2:CreateNetworkInterface',
                'ec2:AttachNetworkInterface',
                'ec2:DescribeNetworkInterfaces',
                'autoscaling:CompleteLifecycleAction',
                'ec2:DeleteNetworkInterface',
            ],
            resources: ['*'],
        },
    ]);
    return role;
}
exports.getKendraStartDataSyncLambdaRole = getKendraStartDataSyncLambdaRole;
function getCheckJobStatusLambdaRole(scope, awsRegion, awsAccountId, kendraIndexId, kendraDataSourceIndexId) {
    const role = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(scope, 'textTractLambdaRole', 'Role used by the Text Extract Lambda function');
    (0, iam_roles_helper_1.addRolePolicies)(role, [
        {
            actions: ['kendra:ListDataSourceSyncJobs'],
            resources: [`arn:aws:kendra:${awsRegion}:${awsAccountId}:index/${kendraIndexId}`],
        },
        {
            actions: ['kendra:ListDataSourceSyncJobs'],
            resources: [`arn:aws:kendra:${awsRegion}:${awsAccountId}:index/${kendraIndexId}/data-source/${kendraDataSourceIndexId}`],
        },
    ]);
    return role;
}
exports.getCheckJobStatusLambdaRole = getCheckJobStatusLambdaRole;
function getUpdateKendraJobStatusLambdaRole(scope, table) {
    const role = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(scope, 'updateKendraJobStatus', 'Role used by the Document Status Update Lambda function');
    (0, iam_roles_helper_1.addRolePolicies)(role, [
        {
            actions: ['dynamodb:PutItem', 'dynamodb:Query', 'dynamodb:GetItem', 'dynamodb:UpdateItem'],
            resources: [table.tableArn],
        },
        {
            actions: [
                'ec2:DescribeInstances',
                'ec2:CreateNetworkInterface',
                'ec2:AttachNetworkInterface',
                'ec2:DescribeNetworkInterfaces',
                'autoscaling:CompleteLifecycleAction',
                'ec2:DeleteNetworkInterface',
            ],
            resources: ['*'],
        },
    ]);
    return role;
}
exports.getUpdateKendraJobStatusLambdaRole = getUpdateKendraJobStatusLambdaRole;
function createStepFunctionsExecutionHandlerRole(cdkStack, docProcessingStateMachine) {
    const stepFunctionsExecutionHandlerRole = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(cdkStack, 'stepFunctionsExecutionHandlerRole', 'Role used by the stepFunctionsExecutionHandlerFn Lambda function');
    stepFunctionsExecutionHandlerRole.node.addDependency(docProcessingStateMachine);
    (0, iam_roles_helper_1.addRolePolicies)(stepFunctionsExecutionHandlerRole, [{
            actions: ['states:StartExecution'],
            resources: [
                docProcessingStateMachine.stateMachineArn,
            ],
        }]);
    return stepFunctionsExecutionHandlerRole;
}
exports.createStepFunctionsExecutionHandlerRole = createStepFunctionsExecutionHandlerRole;
function buildSecurityGroup(scope, name, props, ingressRules, egressRules) {
    const newSecurityGroup = new aws_ec2_1.SecurityGroup(scope, `${name}-security-group`, props);
    ingressRules.forEach(rule => {
        newSecurityGroup.addIngressRule(rule.peer, rule.connection, rule.description, rule.remoteRule);
    });
    egressRules.forEach(rule => {
        newSecurityGroup.addEgressRule(rule.peer, rule.connection, rule.description, rule.remoteRule);
    });
    (0, utils_1.addCfnSuppressRules)(newSecurityGroup, [
        {
            id: 'W5',
            reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
        },
        {
            id: 'W40',
            reason: 'Egress IPProtocol of -1 is default and generally considered OK',
        },
    ]);
    return newSecurityGroup;
}
exports.buildSecurityGroup = buildSecurityGroup;
function createS3FileUploader(cdkStack, s3_bucket, props) {
    const createS3FileUploaderRole = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(cdkStack, 's3FileUploader', 'Role used by the S3 file uploader Lambda function');
    (0, iam_roles_helper_1.addRolePolicies)(createS3FileUploaderRole, [{
            actions: ['s3:PutObject', 's3:PutObjectAcl', 's3:GetObject'],
            resources: [s3_bucket.bucketArn],
        }]);
    return new aws_lambda_1.DockerImageFunction(cdkStack, 's3FileUploaderFn', {
        ...props,
        role: createS3FileUploaderRole,
    });
}
exports.createS3FileUploader = createS3FileUploader;
function createGeneratePresignedUrlFn(cdkStack, bucket, environment) {
    const generatePresignedUrlRole = (0, iam_roles_helper_1.createIAMRoleWithBasicExecutionPolicy)(cdkStack, 'generatePresignedUrlRole', 'Role used by the Generate Pre-signed URL Lambda functio');
    (0, iam_roles_helper_1.addRolePolicies)(generatePresignedUrlRole, [
        {
            actions: ['s3:PutObject'],
            resources: [bucket.bucketArn],
        },
    ]);
    const lambdaGenerateUrlFn = new cdk.aws_lambda.Function(cdkStack, 'generatePresignedUrlFN', {
        runtime: cdk.aws_lambda.Runtime.PYTHON_3_10,
        handler: 'generate_presigned_url.lambda_handler',
        code: cdk.aws_lambda.Code.fromAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-kendra/generate_presigned_url/')),
        timeout: cdk.Duration.seconds(60),
        memorySize: 256,
        role: generatePresignedUrlRole,
        environment,
    });
    return lambdaGenerateUrlFn;
}
exports.createGeneratePresignedUrlFn = createGeneratePresignedUrlFn;
//# sourceMappingURL=data:application/json;base64,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