"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KendraConstruct = void 0;
const cdk = require("aws-cdk-lib");
const kendra = require("aws-cdk-lib/aws-kendra");
const constructs_1 = require("constructs");
const utils_1 = require("./utils");
class KendraConstruct extends constructs_1.Construct {
    constructor(parent, name, props) {
        super(parent, name);
        this.props = props;
        const awsAccountId = cdk.Stack.of(this).account;
        const awsRegion = cdk.Stack.of(this).region;
        const indexRole = new cdk.aws_iam.Role(this, 'kendraIndexRole', {
            description: 'Role that Kendra uses to push logging and metrics to Amazon Cloudwatch',
            assumedBy: new cdk.aws_iam.ServicePrincipal('kendra.amazonaws.com'),
        });
        indexRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
            conditions: {
                StringEquals: {
                    'cloudwatch:namespace': 'Kendra',
                },
            },
        }));
        indexRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: ['*'],
        }));
        indexRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: ['logs:CreateLogGroup'],
            resources: [`arn:aws:logs:${awsRegion}:${awsAccountId}:log-group:/aws/kendra/*`],
        }));
        indexRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: [
                'logs:DescribeLogStreams',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [`arn:aws:logs:${awsRegion}:${awsAccountId}:log-group:/aws/kendra/*:log-stream:*`],
        }));
        const cfnIndex = new kendra.CfnIndex(this, 'llmdemoIndex', {
            edition: this.props.Edition,
            name: this.props.IndexName,
            roleArn: indexRole.roleArn,
            userContextPolicy: 'USER_TOKEN',
            userTokenConfigurations: [{
                    jwtTokenTypeConfiguration: {
                        keyLocation: 'URL',
                        url: `https://cognito-idp.${awsRegion}.amazonaws.com/${this.props.CognitoUserPoolId}/.well-known/jwks.json`,
                        groupAttributeField: 'cognito:groups',
                        userNameAttributeField: 'cognito:username',
                    },
                }],
        });
        this.kendraIndex = cfnIndex;
        ////////////////// Kendra //////////////////////////////////////
        const kendraS3AccessRole = new cdk.aws_iam.Role(this, 'kendraS3AccessRole', {
            description: 'Role that Kendra uses to access documents in S3 bucket',
            assumedBy: new cdk.aws_iam.ServicePrincipal('kendra.amazonaws.com'),
        });
        kendraS3AccessRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: ['s3:GetObject'],
            resources: [`arn:aws:s3:::${this.props.kendraDataSyncInputBucketName}/*`],
        }));
        kendraS3AccessRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: ['s3:ListBucket'],
            resources: [`arn:aws:s3:::${this.props.kendraDataSyncInputBucketName}`],
        }));
        kendraS3AccessRole.addToPolicy(new cdk.aws_iam.PolicyStatement({
            actions: [
                'kendra:BatchPutDocument',
                'kendra:BatchDeleteDocument',
            ],
            resources: [`arn:aws:kendra:${awsRegion}:${awsAccountId}:index/${cfnIndex.attrId}`],
        }));
        this.kendraDataSource = new kendra.CfnDataSource(this, 'llmdemoIndexDataSource', {
            indexId: cfnIndex.attrId,
            name: 'llmdemoIndexDataSource',
            type: 'S3',
            roleArn: kendraS3AccessRole.roleArn,
            dataSourceConfiguration: {
                s3Configuration: {
                    bucketName: this.props.kendraDataSyncInputBucketName,
                    accessControlListConfiguration: {
                        keyPath: `s3://${this.props.kendraDataSyncInputBucketName}/kendra_acl.json`,
                    },
                },
            },
        });
        // https://github.com/awslabs/aws-solutions-constructs/blob/1598893b57f99aa6c5a3a8c1ea18e734ff619f0a/source/patterns/%40aws-solutions-constructs/core/lib/kendra-helper.ts#L60
        (0, utils_1.addCfnSuppressRules)(this.kendraIndex, [{
                id: 'W80',
                reason: 'We consulted the Kendra TFC and they confirmed the default encryption is sufficient for general use cases',
            }]);
    }
    get KendraIndexId() {
        return this.kendraIndex.attrId;
    }
    get KendraDataSourceIndexId() {
        return this.kendraDataSource.attrId;
    }
}
exports.KendraConstruct = KendraConstruct;
//# sourceMappingURL=data:application/json;base64,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