"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildCustomResourceProvider = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const cr = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
function buildCustomResourceProvider(props) {
    const { providerName, codePath, handler, runtime, layers, vpc, securityGroup } = props;
    class CRProvider extends constructs_1.Construct {
        static getProvider(scope) {
            const stack = cdk.Stack.of(scope);
            const existing = stack.node.tryFindChild(providerName);
            if (existing) {
                return existing;
            }
            return new CRProvider(cdk.Stack.of(scope), providerName);
        }
        constructor(scope, id) {
            super(scope, id);
            this.role = new iam.Role(this, 'CRRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                ],
            });
            const customResourceFunction = new lambda.Function(this, 'CustomResourcesFunction', {
                code: lambda.Code.fromDockerBuild(codePath),
                handler,
                runtime,
                layers,
                role: this.role,
                timeout: cdk.Duration.minutes(15),
                memorySize: 128,
                vpc,
                vpcSubnets: vpc ? { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS } : undefined,
                securityGroups: vpc && securityGroup ? [securityGroup] : undefined,
                logRetention: logs.RetentionDays.ONE_WEEK,
                description: 'Custom Resource Provider',
            });
            const providerRole = new iam.Role(this, 'ProviderRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                ],
            });
            this.provider = new cr.Provider(this, 'Provider', {
                onEventHandler: customResourceFunction,
                logRetention: logs.RetentionDays.ONE_WEEK,
                role: providerRole,
            });
            this.serviceToken = this.provider.serviceToken;
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(cdk.Stack.of(this), `${this.provider.node.path}/framework-onEvent/Resource`, [
                {
                    id: 'AwsSolutions-L1',
                    reason: 'Lambda runtime version is managed upstream by CDK.',
                },
            ]);
            for (let role of [this.role, providerRole]) {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(role, [
                    {
                        id: 'AwsSolutions-IAM4',
                        reason: 'CDK CustomResource Lambda uses the AWSLambdaBasicExecutionRole AWS Managed Policy.',
                    },
                ]);
            }
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'CDK CustomResource Provider has a wildcard to invoke any version of the specific Custom Resource function.',
                    appliesTo: [{ regex: `/^Resource::<${id}${customResourceFunction.node.id}[A-Z0-9]+\\.Arn>:\\*$/g` }],
                },
            ], true);
        }
    }
    return CRProvider;
}
exports.buildCustomResourceProvider = buildCustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,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