"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setMergedApiRole = exports.checkAppsyncMergedApiProps = exports.buildMergedAPI = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const iam = require("aws-cdk-lib/aws-iam");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * build app sync merge api with source api associations
 *
 * @param AppsyncMergedApiProps The  props to be used by the construct
 * @param  apiType - MERGED, GRAPHQL
 * @returns App sync merge api
 */
function buildMergedAPI(scope, id, props) {
    if (props.existingMergeApi) {
        return props.existingMergeApi;
    }
    else {
        const mergeAPIname = props.cfnGraphQLApiProps?.name || 'appsyncmergeAPI';
        const apiType = props.cfnGraphQLApiProps?.apiType || 'MERGED';
        const fieldLogLevel = props?.fieldLogLevel || appsync.FieldLogLevel.NONE;
        const excludeVerboseContent = props?.excludeVerboseContent || false;
        const xRayEnabled = props?.xRayEnabled || false;
        let mergedApi = new appsync.CfnGraphQLApi(scope, id, {
            apiType: apiType,
            name: mergeAPIname,
            authenticationType: props.cfnGraphQLApiProps.authenticationType,
            userPoolConfig: props.cfnGraphQLApiProps?.userPoolConfig,
            additionalAuthenticationProviders: [{
                    authenticationType: 'AWS_IAM',
                }],
            logConfig: {
                cloudWatchLogsRoleArn: setAppsyncCloudWatchlogsRole(scope, props).roleArn,
                fieldLogLevel: fieldLogLevel,
                excludeVerboseContent: excludeVerboseContent,
            },
            xrayEnabled: xRayEnabled,
            mergedApiExecutionRoleArn: props.mergedApiRole.roleArn,
            ownerContact: props?.cfnGraphQLApiProps.ownerContact,
        });
        return mergedApi;
    }
}
exports.buildMergedAPI = buildMergedAPI;
function checkAppsyncMergedApiProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if (propsObject.existingMergeApi && propsObject.cfnGraphQLApiProps) {
        errorMessages += 'Error - Either provide existingMergeApi or cfnGraphQLApiProps, but not both.\n';
        errorFound = true;
    }
    if (!propsObject.existingMergeApi && !propsObject.cfnGraphQLApiProps) {
        errorMessages += 'Error - Atleast one is required either existingMergeApi or cfnGraphQLApiProps.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.checkAppsyncMergedApiProps = checkAppsyncMergedApiProps;
/**
 * @internal This is an internal core function and should not be called directly
 * by Solutions Constructs clients.
 * set the merge api role to access source api associations
 *
 * @param AppsyncMergedApiProps The  props to be used by the construct
 * @param mergedAPI app sync graphql api
 * @param mergedApiRole iam role
 * @returns App sync merge api role
 */
function setMergedApiRole(mergedApiID, sourceApiId, mergedApiRole) {
    mergedApiRole.addToPolicy(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ['appsync:SourceGraphQL',
            'appsync:StartSchemaMerge'],
        resources: [
            'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID
                + ':apis/' + sourceApiId + '/*',
            'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + mergedApiID + '/sourceApiAssociations/*',
            'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + sourceApiId + '/sourceApiAssociations/*',
        ],
    }));
    return mergedApiRole;
}
exports.setMergedApiRole = setMergedApiRole;
function setAppsyncCloudWatchlogsRole(scope, props) {
    const appsyncServicePrincipleRole = props.appsyncServicePrincipleRole || 'appsync.amazonaws.com';
    let appsynccloudWatchlogsRole = new iam.Role(scope, 'appsynccloudWatchlogsRole', {
        assumedBy: new iam.ServicePrincipal(appsyncServicePrincipleRole),
    });
    appsynccloudWatchlogsRole.addToPolicy(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        resources: ['*'],
    }));
    return appsynccloudWatchlogsRole;
}
//# sourceMappingURL=data:application/json;base64,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