"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VectorCollection = exports.VectorCollectionStandbyReplicas = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const oss = require("aws-cdk-lib/aws-opensearchserverless");
const constructs_1 = require("constructs");
const utils_1 = require("../../common/helpers/utils");
var VectorCollectionStandbyReplicas;
(function (VectorCollectionStandbyReplicas) {
    VectorCollectionStandbyReplicas["ENABLED"] = "ENABLED";
    VectorCollectionStandbyReplicas["DISABLED"] = "DISABLED";
})(VectorCollectionStandbyReplicas || (exports.VectorCollectionStandbyReplicas = VectorCollectionStandbyReplicas = {}));
/**
 * Deploys an OpenSearch Serverless Collection to be used as a vector store.
 *
 * It includes all policies.
 */
class VectorCollection extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An OpenSearch Access Policy document that will become `dataAccessPolicy`.
         * @private
         */
        this.dataAccessPolicyDocument = [];
        this.collectionName = props?.collectionName ?? (0, utils_1.generatePhysicalNameV2)(this, 'VectorStore', { maxLength: 32, lower: true });
        this.standbyReplicas = props?.standbyReplicas ?? VectorCollectionStandbyReplicas.ENABLED;
        const encryptionPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'EncryptionPolicy', { maxLength: 32, lower: true });
        const encryptionPolicy = new oss.CfnSecurityPolicy(this, 'EncryptionPolicy', {
            name: encryptionPolicyName,
            type: 'encryption',
            policy: JSON.stringify({
                Rules: [
                    {
                        ResourceType: 'collection',
                        Resource: [`collection/${this.collectionName}`],
                    },
                ],
                AWSOwnedKey: true,
            }),
        });
        const networkPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'NetworkPolicy', { maxLength: 32, lower: true });
        const networkPolicy = new oss.CfnSecurityPolicy(this, 'NetworkPolicy', {
            name: networkPolicyName,
            type: 'network',
            policy: JSON.stringify([
                {
                    Rules: [
                        {
                            ResourceType: 'collection',
                            Resource: [`collection/${this.collectionName}`],
                        },
                        {
                            ResourceType: 'dashboard',
                            Resource: [`collection/${this.collectionName}`],
                        },
                    ],
                    AllowFromPublic: true,
                },
            ]),
        });
        const collection = new oss.CfnCollection(this, 'VectorCollection', {
            name: this.collectionName,
            type: 'VECTORSEARCH',
            standbyReplicas: this.standbyReplicas,
        });
        this.collectionArn = collection.attrArn;
        this.collectionId = collection.attrId;
        this.aossPolicy = new iam.ManagedPolicy(this, 'AOSSApiAccessAll', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'aoss:APIAccessAll',
                    ],
                    resources: [collection.attrArn],
                }),
            ],
        });
        collection.addDependency(encryptionPolicy);
        collection.addDependency(networkPolicy);
        const isDataAccessPolicyNotEmpty = new cdk.CfnCondition(this, 'IsDataAccessPolicyNotEmpty', {
            expression: cdk.Fn.conditionNot(cdk.Fn.conditionEquals(0, cdk.Lazy.number({
                produce: () => this.dataAccessPolicyDocument.length,
            }))),
        });
        const dataAccessPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'DataAccessPolicy', { maxLength: 32, lower: true });
        this.dataAccessPolicy = new oss.CfnAccessPolicy(this, 'DataAccessPolicy', {
            name: dataAccessPolicyName,
            type: 'data',
            policy: cdk.Lazy.string({
                produce: () => JSON.stringify(this.dataAccessPolicyDocument),
            }),
        });
        this.dataAccessPolicy.cfnOptions.condition = isDataAccessPolicyNotEmpty;
    }
    /**
     * Grants the specified role access to data in the collection.
     * @param grantee The role to grant access to.
     */
    grantDataAccess(grantee) {
        this.dataAccessPolicyDocument.push({
            Rules: [
                {
                    Resource: [`collection/${this.collectionName}`],
                    Permission: [
                        'aoss:DescribeCollectionItems',
                        'aoss:CreateCollectionItems',
                        'aoss:UpdateCollectionItems',
                    ],
                    ResourceType: 'collection',
                },
                {
                    Resource: [`index/${this.collectionName}/*`],
                    Permission: [
                        'aoss:UpdateIndex',
                        'aoss:DescribeIndex',
                        'aoss:ReadDocument',
                        'aoss:WriteDocument',
                        'aoss:CreateIndex',
                    ],
                    ResourceType: 'index',
                },
            ],
            Principal: [
                grantee.roleArn,
            ],
            Description: '',
        });
        grantee.addManagedPolicy(this.aossPolicy);
    }
}
exports.VectorCollection = VectorCollection;
_a = JSII_RTTI_SYMBOL_1;
VectorCollection[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.opensearchserverless.VectorCollection", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmVjdG9yLWNvbGxlY3Rpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvY2RrLWxpYi9vcGVuc2VhcmNoc2VydmVybGVzcy92ZWN0b3ItY29sbGVjdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBQ0gsbUNBQW1DO0FBQ25DLDJDQUEyQztBQUMzQyw0REFBNEQ7QUFDNUQsMkNBQXVDO0FBQ3ZDLHNEQUFvRTtBQUVwRSxJQUFZLCtCQUdYO0FBSEQsV0FBWSwrQkFBK0I7SUFDekMsc0RBQW1CLENBQUE7SUFDbkIsd0RBQXFCLENBQUE7QUFDdkIsQ0FBQyxFQUhXLCtCQUErQiwrQ0FBL0IsK0JBQStCLFFBRzFDO0FBZ0JEOzs7O0dBSUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLHNCQUFTO0lBb0M3QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTZCO1FBQ3JFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFQbkI7OztXQUdHO1FBQ0ssNkJBQXdCLEdBQVUsRUFBRSxDQUFDO1FBSzNDLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxFQUFFLGNBQWMsSUFBSSxJQUFBLDhCQUFzQixFQUNuRSxJQUFJLEVBQ0osYUFBYSxFQUNiLEVBQUUsU0FBUyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUVsQyxJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssRUFBRSxlQUFlLElBQUksK0JBQStCLENBQUMsT0FBTyxDQUFDO1FBRXpGLE1BQU0sb0JBQW9CLEdBQUcsSUFBQSw4QkFBc0IsRUFBQyxJQUFJLEVBQ3RELGtCQUFrQixFQUNsQixFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbEMsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDM0UsSUFBSSxFQUFFLG9CQUFvQjtZQUMxQixJQUFJLEVBQUUsWUFBWTtZQUNsQixNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQztnQkFDckIsS0FBSyxFQUFFO29CQUNMO3dCQUNFLFlBQVksRUFBRSxZQUFZO3dCQUMxQixRQUFRLEVBQUUsQ0FBQyxjQUFjLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztxQkFDaEQ7aUJBQ0Y7Z0JBQ0QsV0FBVyxFQUFFLElBQUk7YUFDbEIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0saUJBQWlCLEdBQUcsSUFBQSw4QkFBc0IsRUFBQyxJQUFJLEVBQ25ELGVBQWUsRUFDZixFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbEMsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtZQUNyRSxJQUFJLEVBQUUsaUJBQWlCO1lBQ3ZCLElBQUksRUFBRSxTQUFTO1lBQ2YsTUFBTSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7Z0JBQ3JCO29CQUNFLEtBQUssRUFBRTt3QkFDTDs0QkFDRSxZQUFZLEVBQUUsWUFBWTs0QkFDMUIsUUFBUSxFQUFFLENBQUMsY0FBYyxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7eUJBQ2hEO3dCQUNEOzRCQUNFLFlBQVksRUFBRSxXQUFXOzRCQUN6QixRQUFRLEVBQUUsQ0FBQyxjQUFjLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQzt5QkFDaEQ7cUJBQ0Y7b0JBQ0QsZUFBZSxFQUFFLElBQUk7aUJBQ3RCO2FBQ0YsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDakUsSUFBSSxFQUFFLElBQUksQ0FBQyxjQUFjO1lBQ3pCLElBQUksRUFBRSxjQUFjO1lBQ3BCLGVBQWUsRUFBRSxJQUFJLENBQUMsZUFBZTtTQUN0QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUM7UUFDeEMsSUFBSSxDQUFDLFlBQVksR0FBRyxVQUFVLENBQUMsTUFBTSxDQUFDO1FBRXRDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUNyQyxJQUFJLEVBQ0osa0JBQWtCLEVBQUU7WUFDbEIsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztvQkFDeEIsT0FBTyxFQUFFO3dCQUNQLG1CQUFtQjtxQkFDcEI7b0JBQ0QsU0FBUyxFQUFFLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQztpQkFDaEMsQ0FBQzthQUNIO1NBQ0YsQ0FDRixDQUFDO1FBRUYsVUFBVSxDQUFDLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNDLFVBQVUsQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLENBQUM7UUFHeEMsTUFBTSwwQkFBMEIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLDRCQUE0QixFQUFFO1lBQzFGLFVBQVUsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7Z0JBQ3hFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsd0JBQXdCLENBQUMsTUFBTTthQUNwRCxDQUFDLENBQUMsQ0FBQztTQUNMLENBQUMsQ0FBQztRQUVILE1BQU0sb0JBQW9CLEdBQUcsSUFBQSw4QkFBc0IsRUFBQyxJQUFJLEVBQ3RELGtCQUFrQixFQUNsQixFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbEMsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDeEUsSUFBSSxFQUFFLG9CQUFvQjtZQUMxQixJQUFJLEVBQUUsTUFBTTtZQUNaLE1BQU0sRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztnQkFDdEIsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLHdCQUF3QixDQUFDO2FBQzdELENBQUM7U0FDSCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLFNBQVMsR0FBRywwQkFBMEIsQ0FBQztJQUMxRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsZUFBZSxDQUFDLE9BQWtCO1FBQ2hDLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUM7WUFDakMsS0FBSyxFQUFFO2dCQUNMO29CQUNFLFFBQVEsRUFBRSxDQUFDLGNBQWMsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO29CQUMvQyxVQUFVLEVBQUU7d0JBQ1YsOEJBQThCO3dCQUM5Qiw0QkFBNEI7d0JBQzVCLDRCQUE0QjtxQkFDN0I7b0JBQ0QsWUFBWSxFQUFFLFlBQVk7aUJBQzNCO2dCQUNEO29CQUNFLFFBQVEsRUFBRSxDQUFDLFNBQVMsSUFBSSxDQUFDLGNBQWMsSUFBSSxDQUFDO29CQUM1QyxVQUFVLEVBQUU7d0JBQ1Ysa0JBQWtCO3dCQUNsQixvQkFBb0I7d0JBQ3BCLG1CQUFtQjt3QkFDbkIsb0JBQW9CO3dCQUNwQixrQkFBa0I7cUJBQ25CO29CQUNELFlBQVksRUFBRSxPQUFPO2lCQUN0QjthQUNGO1lBQ0QsU0FBUyxFQUFFO2dCQUNULE9BQU8sQ0FBQyxPQUFPO2FBQ2hCO1lBQ0QsV0FBVyxFQUFFLEVBQUU7U0FDaEIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUM1QyxDQUFDOztBQXZLSCw0Q0F3S0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIG9zcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtb3BlbnNlYXJjaHNlcnZlcmxlc3MnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBnZW5lcmF0ZVBoeXNpY2FsTmFtZVYyIH0gZnJvbSAnLi4vLi4vY29tbW9uL2hlbHBlcnMvdXRpbHMnO1xuXG5leHBvcnQgZW51bSBWZWN0b3JDb2xsZWN0aW9uU3RhbmRieVJlcGxpY2FzIHtcbiAgRU5BQkxFRCA9ICdFTkFCTEVEJyxcbiAgRElTQUJMRUQgPSAnRElTQUJMRUQnLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFZlY3RvckNvbGxlY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29sbGVjdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGNvbGxlY3Rpb25OYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRvIHVzZSBzdGFuZGJ5IHJlcGxpY2FzIGZvciB0aGUgY29sbGVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgRU5BQkxFRFxuICAgKi9cbiAgcmVhZG9ubHkgc3RhbmRieVJlcGxpY2FzPzogVmVjdG9yQ29sbGVjdGlvblN0YW5kYnlSZXBsaWNhcztcbn1cblxuLyoqXG4gKiBEZXBsb3lzIGFuIE9wZW5TZWFyY2ggU2VydmVybGVzcyBDb2xsZWN0aW9uIHRvIGJlIHVzZWQgYXMgYSB2ZWN0b3Igc3RvcmUuXG4gKlxuICogSXQgaW5jbHVkZXMgYWxsIHBvbGljaWVzLlxuICovXG5leHBvcnQgY2xhc3MgVmVjdG9yQ29sbGVjdGlvbiBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29sbGVjdGlvbi5cbiAgICovXG4gIHB1YmxpYyBjb2xsZWN0aW9uTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0byB1c2Ugc3RhbmRieSByZXBsaWNhcyBmb3IgdGhlIGNvbGxlY3Rpb24uXG4gICAqL1xuICBwdWJsaWMgc3RhbmRieVJlcGxpY2FzOiBWZWN0b3JDb2xsZWN0aW9uU3RhbmRieVJlcGxpY2FzO1xuXG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIGNvbGxlY3Rpb24uXG4gICAqL1xuICBwdWJsaWMgY29sbGVjdGlvbklkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBjb2xsZWN0aW9uLlxuICAgKi9cbiAgcHVibGljIGNvbGxlY3Rpb25Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogQW4gSUFNIHBvbGljeSB0aGF0IGFsbG93cyBBUEkgYWNjZXNzIHRvIHRoZSBjb2xsZWN0aW9uLlxuICAgKi9cbiAgcHVibGljIGFvc3NQb2xpY3k6IGlhbS5NYW5hZ2VkUG9saWN5O1xuXG4gIC8qKlxuICAgKiBBbiBPcGVuU2VhcmNoIEFjY2VzcyBQb2xpY3kgdGhhdCBhbGxvd3MgYWNjZXNzIHRvIHRoZSBpbmRleC5cbiAgICovXG4gIHB1YmxpYyBkYXRhQWNjZXNzUG9saWN5OiBvc3MuQ2ZuQWNjZXNzUG9saWN5O1xuXG4gIC8qKlxuICAgKiBBbiBPcGVuU2VhcmNoIEFjY2VzcyBQb2xpY3kgZG9jdW1lbnQgdGhhdCB3aWxsIGJlY29tZSBgZGF0YUFjY2Vzc1BvbGljeWAuXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIGRhdGFBY2Nlc3NQb2xpY3lEb2N1bWVudDogYW55W10gPSBbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wcz86IFZlY3RvckNvbGxlY3Rpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmNvbGxlY3Rpb25OYW1lID0gcHJvcHM/LmNvbGxlY3Rpb25OYW1lID8/IGdlbmVyYXRlUGh5c2ljYWxOYW1lVjIoXG4gICAgICB0aGlzLFxuICAgICAgJ1ZlY3RvclN0b3JlJyxcbiAgICAgIHsgbWF4TGVuZ3RoOiAzMiwgbG93ZXI6IHRydWUgfSk7XG5cbiAgICB0aGlzLnN0YW5kYnlSZXBsaWNhcyA9IHByb3BzPy5zdGFuZGJ5UmVwbGljYXMgPz8gVmVjdG9yQ29sbGVjdGlvblN0YW5kYnlSZXBsaWNhcy5FTkFCTEVEO1xuXG4gICAgY29uc3QgZW5jcnlwdGlvblBvbGljeU5hbWUgPSBnZW5lcmF0ZVBoeXNpY2FsTmFtZVYyKHRoaXMsXG4gICAgICAnRW5jcnlwdGlvblBvbGljeScsXG4gICAgICB7IG1heExlbmd0aDogMzIsIGxvd2VyOiB0cnVlIH0pO1xuICAgIGNvbnN0IGVuY3J5cHRpb25Qb2xpY3kgPSBuZXcgb3NzLkNmblNlY3VyaXR5UG9saWN5KHRoaXMsICdFbmNyeXB0aW9uUG9saWN5Jywge1xuICAgICAgbmFtZTogZW5jcnlwdGlvblBvbGljeU5hbWUsXG4gICAgICB0eXBlOiAnZW5jcnlwdGlvbicsXG4gICAgICBwb2xpY3k6IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgICAgUnVsZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZXNvdXJjZVR5cGU6ICdjb2xsZWN0aW9uJyxcbiAgICAgICAgICAgIFJlc291cmNlOiBbYGNvbGxlY3Rpb24vJHt0aGlzLmNvbGxlY3Rpb25OYW1lfWBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIEFXU093bmVkS2V5OiB0cnVlLFxuICAgICAgfSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBuZXR3b3JrUG9saWN5TmFtZSA9IGdlbmVyYXRlUGh5c2ljYWxOYW1lVjIodGhpcyxcbiAgICAgICdOZXR3b3JrUG9saWN5JyxcbiAgICAgIHsgbWF4TGVuZ3RoOiAzMiwgbG93ZXI6IHRydWUgfSk7XG4gICAgY29uc3QgbmV0d29ya1BvbGljeSA9IG5ldyBvc3MuQ2ZuU2VjdXJpdHlQb2xpY3kodGhpcywgJ05ldHdvcmtQb2xpY3knLCB7XG4gICAgICBuYW1lOiBuZXR3b3JrUG9saWN5TmFtZSxcbiAgICAgIHR5cGU6ICduZXR3b3JrJyxcbiAgICAgIHBvbGljeTogSlNPTi5zdHJpbmdpZnkoW1xuICAgICAgICB7XG4gICAgICAgICAgUnVsZXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVzb3VyY2VUeXBlOiAnY29sbGVjdGlvbicsXG4gICAgICAgICAgICAgIFJlc291cmNlOiBbYGNvbGxlY3Rpb24vJHt0aGlzLmNvbGxlY3Rpb25OYW1lfWBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVzb3VyY2VUeXBlOiAnZGFzaGJvYXJkJyxcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IFtgY29sbGVjdGlvbi8ke3RoaXMuY29sbGVjdGlvbk5hbWV9YF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgQWxsb3dGcm9tUHVibGljOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgXSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBjb2xsZWN0aW9uID0gbmV3IG9zcy5DZm5Db2xsZWN0aW9uKHRoaXMsICdWZWN0b3JDb2xsZWN0aW9uJywge1xuICAgICAgbmFtZTogdGhpcy5jb2xsZWN0aW9uTmFtZSxcbiAgICAgIHR5cGU6ICdWRUNUT1JTRUFSQ0gnLFxuICAgICAgc3RhbmRieVJlcGxpY2FzOiB0aGlzLnN0YW5kYnlSZXBsaWNhcyxcbiAgICB9KTtcblxuICAgIHRoaXMuY29sbGVjdGlvbkFybiA9IGNvbGxlY3Rpb24uYXR0ckFybjtcbiAgICB0aGlzLmNvbGxlY3Rpb25JZCA9IGNvbGxlY3Rpb24uYXR0cklkO1xuXG4gICAgdGhpcy5hb3NzUG9saWN5ID0gbmV3IGlhbS5NYW5hZ2VkUG9saWN5KFxuICAgICAgdGhpcyxcbiAgICAgICdBT1NTQXBpQWNjZXNzQWxsJywge1xuICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAnYW9zczpBUElBY2Nlc3NBbGwnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW2NvbGxlY3Rpb24uYXR0ckFybl0sXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICk7XG5cbiAgICBjb2xsZWN0aW9uLmFkZERlcGVuZGVuY3koZW5jcnlwdGlvblBvbGljeSk7XG4gICAgY29sbGVjdGlvbi5hZGREZXBlbmRlbmN5KG5ldHdvcmtQb2xpY3kpO1xuXG5cbiAgICBjb25zdCBpc0RhdGFBY2Nlc3NQb2xpY3lOb3RFbXB0eSA9IG5ldyBjZGsuQ2ZuQ29uZGl0aW9uKHRoaXMsICdJc0RhdGFBY2Nlc3NQb2xpY3lOb3RFbXB0eScsIHtcbiAgICAgIGV4cHJlc3Npb246IGNkay5Gbi5jb25kaXRpb25Ob3QoY2RrLkZuLmNvbmRpdGlvbkVxdWFscygwLCBjZGsuTGF6eS5udW1iZXIoe1xuICAgICAgICBwcm9kdWNlOiAoKSA9PiB0aGlzLmRhdGFBY2Nlc3NQb2xpY3lEb2N1bWVudC5sZW5ndGgsXG4gICAgICB9KSkpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgZGF0YUFjY2Vzc1BvbGljeU5hbWUgPSBnZW5lcmF0ZVBoeXNpY2FsTmFtZVYyKHRoaXMsXG4gICAgICAnRGF0YUFjY2Vzc1BvbGljeScsXG4gICAgICB7IG1heExlbmd0aDogMzIsIGxvd2VyOiB0cnVlIH0pO1xuICAgIHRoaXMuZGF0YUFjY2Vzc1BvbGljeSA9IG5ldyBvc3MuQ2ZuQWNjZXNzUG9saWN5KHRoaXMsICdEYXRhQWNjZXNzUG9saWN5Jywge1xuICAgICAgbmFtZTogZGF0YUFjY2Vzc1BvbGljeU5hbWUsXG4gICAgICB0eXBlOiAnZGF0YScsXG4gICAgICBwb2xpY3k6IGNkay5MYXp5LnN0cmluZyh7XG4gICAgICAgIHByb2R1Y2U6ICgpID0+IEpTT04uc3RyaW5naWZ5KHRoaXMuZGF0YUFjY2Vzc1BvbGljeURvY3VtZW50KSxcbiAgICAgIH0pLFxuICAgIH0pO1xuICAgIHRoaXMuZGF0YUFjY2Vzc1BvbGljeS5jZm5PcHRpb25zLmNvbmRpdGlvbiA9IGlzRGF0YUFjY2Vzc1BvbGljeU5vdEVtcHR5O1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyB0aGUgc3BlY2lmaWVkIHJvbGUgYWNjZXNzIHRvIGRhdGEgaW4gdGhlIGNvbGxlY3Rpb24uXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSByb2xlIHRvIGdyYW50IGFjY2VzcyB0by5cbiAgICovXG4gIGdyYW50RGF0YUFjY2VzcyhncmFudGVlOiBpYW0uSVJvbGUpIHtcbiAgICB0aGlzLmRhdGFBY2Nlc3NQb2xpY3lEb2N1bWVudC5wdXNoKHtcbiAgICAgIFJ1bGVzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBSZXNvdXJjZTogW2Bjb2xsZWN0aW9uLyR7dGhpcy5jb2xsZWN0aW9uTmFtZX1gXSxcbiAgICAgICAgICBQZXJtaXNzaW9uOiBbXG4gICAgICAgICAgICAnYW9zczpEZXNjcmliZUNvbGxlY3Rpb25JdGVtcycsXG4gICAgICAgICAgICAnYW9zczpDcmVhdGVDb2xsZWN0aW9uSXRlbXMnLFxuICAgICAgICAgICAgJ2Fvc3M6VXBkYXRlQ29sbGVjdGlvbkl0ZW1zJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIFJlc291cmNlVHlwZTogJ2NvbGxlY3Rpb24nLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgUmVzb3VyY2U6IFtgaW5kZXgvJHt0aGlzLmNvbGxlY3Rpb25OYW1lfS8qYF0sXG4gICAgICAgICAgUGVybWlzc2lvbjogW1xuICAgICAgICAgICAgJ2Fvc3M6VXBkYXRlSW5kZXgnLFxuICAgICAgICAgICAgJ2Fvc3M6RGVzY3JpYmVJbmRleCcsXG4gICAgICAgICAgICAnYW9zczpSZWFkRG9jdW1lbnQnLFxuICAgICAgICAgICAgJ2Fvc3M6V3JpdGVEb2N1bWVudCcsXG4gICAgICAgICAgICAnYW9zczpDcmVhdGVJbmRleCcsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBSZXNvdXJjZVR5cGU6ICdpbmRleCcsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgUHJpbmNpcGFsOiBbXG4gICAgICAgIGdyYW50ZWUucm9sZUFybixcbiAgICAgIF0sXG4gICAgICBEZXNjcmlwdGlvbjogJycsXG4gICAgfSk7XG4gICAgZ3JhbnRlZS5hZGRNYW5hZ2VkUG9saWN5KHRoaXMuYW9zc1BvbGljeSk7XG4gIH1cbn1cblxuIl19