"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenSearchIndexCRProvider = exports.VectorIndex = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const oss = require("aws-cdk-lib/aws-opensearchserverless");
const custom_resource_provider_helper_1 = require("../../common/helpers/custom-resource-provider-helper");
const utils_1 = require("../../common/helpers/utils");
/**
 * Deploy a vector index on the collection.
 */
class VectorIndex extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.indexName = props.indexName;
        this.vectorField = props.vectorField;
        this.vectorDimensions = props.vectorDimensions;
        const crProvider = exports.OpenSearchIndexCRProvider.getProvider(this);
        crProvider.role.addManagedPolicy(props.collection.aossPolicy);
        const manageIndexPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'ManageIndexPolicy', { maxLength: 32, lower: true });
        const manageIndexPolicy = new oss.CfnAccessPolicy(this, 'ManageIndexPolicy', {
            name: manageIndexPolicyName,
            type: 'data',
            policy: JSON.stringify([
                {
                    Rules: [
                        {
                            Resource: [`index/${props.collection.collectionName}/*`],
                            Permission: [
                                'aoss:DescribeIndex',
                                'aoss:CreateIndex',
                                'aoss:DeleteIndex',
                                'aoss:UpdateIndex',
                            ],
                            ResourceType: 'index',
                        },
                        {
                            Resource: [`collection/${props.collection.collectionName}`],
                            Permission: [
                                'aoss:DescribeCollectionItems',
                            ],
                            ResourceType: 'collection',
                        },
                    ],
                    Principal: [
                        crProvider.role.roleArn,
                    ],
                    Description: '',
                },
            ]),
        });
        const vectorIndex = new cdk.CustomResource(this, 'VectorIndex', {
            serviceToken: crProvider.serviceToken,
            properties: {
                Endpoint: `${props.collection.collectionId}.${cdk.Stack.of(this).region}.aoss.amazonaws.com`,
                IndexName: props.indexName,
                VectorField: props.vectorField,
                Dimensions: props.vectorDimensions,
                MetadataManagement: props.mappings.map((m) => {
                    return {
                        MappingField: m.mappingField,
                        DataType: m.dataType,
                        Filterable: m.filterable,
                    };
                }),
            },
            resourceType: 'Custom::OpenSearchIndex',
        });
        vectorIndex.node.addDependency(manageIndexPolicy);
        vectorIndex.node.addDependency(props.collection);
        vectorIndex.node.addDependency(props.collection.dataAccessPolicy);
    }
}
exports.VectorIndex = VectorIndex;
_a = JSII_RTTI_SYMBOL_1;
VectorIndex[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.opensearch_vectorindex.VectorIndex", version: "0.1.198" };
/**
 * Custom Resource provider for OpenSearch Index operations.
 *
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
exports.OpenSearchIndexCRProvider = (0, custom_resource_provider_helper_1.buildCustomResourceProvider)({
    providerName: 'OpenSearchIndexCRProvider',
    codePath: path.join(__dirname, '../../../lambda/opensearch-serverless-custom-resources'),
    handler: 'custom_resources.on_event',
    runtime: lambda.Runtime.PYTHON_3_12,
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmVjdG9yLWluZGV4LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2Nkay1saWIvb3BlbnNlYXJjaC12ZWN0b3JpbmRleC92ZWN0b3ItaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7Ozs7Ozs7Ozs7R0FXRztBQUNILDZCQUE2QjtBQUM3QixtQ0FBbUM7QUFDbkMsaURBQWlEO0FBQ2pELDREQUE0RDtBQUU1RCwwR0FBbUc7QUFDbkcsc0RBQW9FO0FBZ0dwRTs7R0FFRztBQUNILE1BQWEsV0FBWSxTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBYzNDLFlBQ0UsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLEtBQXVCO1FBRXZCLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1FBQy9DLE1BQU0sVUFBVSxHQUFHLGlDQUF5QixDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvRCxVQUFVLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFOUQsTUFBTSxxQkFBcUIsR0FBRyxJQUFBLDhCQUFzQixFQUFDLElBQUksRUFDdkQsbUJBQW1CLEVBQ25CLEVBQUUsU0FBUyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNsQyxNQUFNLGlCQUFpQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7WUFDM0UsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixJQUFJLEVBQUUsTUFBTTtZQUNaLE1BQU0sRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDO2dCQUNyQjtvQkFDRSxLQUFLLEVBQUU7d0JBQ0w7NEJBQ0UsUUFBUSxFQUFFLENBQUMsU0FBUyxLQUFLLENBQUMsVUFBVSxDQUFDLGNBQWMsSUFBSSxDQUFDOzRCQUN4RCxVQUFVLEVBQUU7Z0NBQ1Ysb0JBQW9CO2dDQUNwQixrQkFBa0I7Z0NBQ2xCLGtCQUFrQjtnQ0FDbEIsa0JBQWtCOzZCQUNuQjs0QkFDRCxZQUFZLEVBQUUsT0FBTzt5QkFDdEI7d0JBQ0Q7NEJBQ0UsUUFBUSxFQUFFLENBQUMsY0FBYyxLQUFLLENBQUMsVUFBVSxDQUFDLGNBQWMsRUFBRSxDQUFDOzRCQUMzRCxVQUFVLEVBQUU7Z0NBQ1YsOEJBQThCOzZCQUMvQjs0QkFDRCxZQUFZLEVBQUUsWUFBWTt5QkFDM0I7cUJBQ0Y7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTztxQkFDeEI7b0JBQ0QsV0FBVyxFQUFFLEVBQUU7aUJBQ2hCO2FBQ0YsQ0FBQztTQUNILENBQUMsQ0FBQztRQUdILE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzlELFlBQVksRUFBRSxVQUFVLENBQUMsWUFBWTtZQUNyQyxVQUFVLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxZQUFZLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxxQkFBcUI7Z0JBQzVGLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDMUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO2dCQUM5QixVQUFVLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtnQkFDbEMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRTtvQkFDM0MsT0FBTzt3QkFDTCxZQUFZLEVBQUUsQ0FBQyxDQUFDLFlBQVk7d0JBQzVCLFFBQVEsRUFBRSxDQUFDLENBQUMsUUFBUTt3QkFDcEIsVUFBVSxFQUFFLENBQUMsQ0FBQyxVQUFVO3FCQUN6QixDQUFDO2dCQUNKLENBQUMsQ0FBQzthQUN5QjtZQUM3QixZQUFZLEVBQUUseUJBQXlCO1NBQ3hDLENBQUMsQ0FBQztRQUVILFdBQVcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDbEQsV0FBVyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2pELFdBQVcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNwRSxDQUFDOztBQXBGSCxrQ0FzRkM7OztBQUVEOzs7O0dBSUc7QUFDVSxRQUFBLHlCQUF5QixHQUFHLElBQUEsNkRBQTJCLEVBQUM7SUFDbkUsWUFBWSxFQUFFLDJCQUEyQjtJQUN6QyxRQUFRLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FDakIsU0FBUyxFQUFFLHdEQUF3RCxDQUFDO0lBQ3RFLE9BQU8sRUFBRSwyQkFBMkI7SUFDcEMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztDQUNwQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBvc3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLW9wZW5zZWFyY2hzZXJ2ZXJsZXNzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgYnVpbGRDdXN0b21SZXNvdXJjZVByb3ZpZGVyIH0gZnJvbSAnLi4vLi4vY29tbW9uL2hlbHBlcnMvY3VzdG9tLXJlc291cmNlLXByb3ZpZGVyLWhlbHBlcic7XG5pbXBvcnQgeyBnZW5lcmF0ZVBoeXNpY2FsTmFtZVYyIH0gZnJvbSAnLi4vLi4vY29tbW9uL2hlbHBlcnMvdXRpbHMnO1xuaW1wb3J0IHsgVmVjdG9yQ29sbGVjdGlvbiB9IGZyb20gJy4uL29wZW5zZWFyY2hzZXJ2ZXJsZXNzJztcblxuLyoqXG4gKiBNZXRhZGF0YSBmaWVsZCBkZWZpbml0aW9ucy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNZXRhZGF0YU1hbmFnZW1lbnRGaWVsZFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBmaWVsZC5cbiAgICovXG4gIHJlYWRvbmx5IG1hcHBpbmdGaWVsZDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGRhdGEgdHlwZSBvZiB0aGUgZmllbGQuXG4gICAqL1xuICByZWFkb25seSBkYXRhVHlwZTogc3RyaW5nO1xuICAvKipcbiAgICogV2hldGhlciB0aGUgZmllbGQgaXMgZmlsdGVyYWJsZS5cbiAgICovXG4gIHJlYWRvbmx5IGZpbHRlcmFibGU6IGJvb2xlYW47XG59XG5cbi8qKlxuICogTWV0YWRhdGEgZmllbGQgZGVmaW5pdGlvbnMgYXMgdGhlIEFQSSBleHBlY3RzIHRoZW0uXG4gKlxuICogQGludGVybmFsIC0gSlNJSSByZXF1aXJlcyB0aGUgZXhwb3J0ZWQgaW50ZXJmYWNlIHRvIGhhdmUgY2FtZWwgY2FtZWxDYXNlIHByb3BlcnRpZXMsXG4gKiBidXQgdGhlIEFQSSBleHBlY3QgUGFzY2FsQ2FzZSBwcm9wZXJ0aWVzXG4gKi9cbnR5cGUgTWV0YWRhdGFNYW5hZ2VtZW50RmllbGQgPSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZmllbGQuXG4gICAqL1xuICByZWFkb25seSBNYXBwaW5nRmllbGQ6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBkYXRhIHR5cGUgb2YgdGhlIGZpZWxkLlxuICAgKi9cbiAgcmVhZG9ubHkgRGF0YVR5cGU6IHN0cmluZztcbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIGZpZWxkIGlzIGZpbHRlcmFibGUuXG4gICAqL1xuICByZWFkb25seSBGaWx0ZXJhYmxlOiBib29sZWFuO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHRoZSBDdXN0b206Ok9wZW5TZWFyY2hJbmRleCBjdXN0b20gcmVzb3VyY2UuXG4gKlxuICogQGludGVybmFsXG4gKi9cbmludGVyZmFjZSBWZWN0b3JJbmRleFJlc291cmNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIE9wZW5TZWFyY2ggRW5kcG9pbnQuXG4gICAqL1xuICByZWFkb25seSBFbmRwb2ludDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGluZGV4LlxuICAgKi9cbiAgcmVhZG9ubHkgSW5kZXhOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgdmVjdG9yIGZpZWxkLlxuICAgKi9cbiAgcmVhZG9ubHkgVmVjdG9yRmllbGQ6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGltZW5zaW9ucyBpbiB0aGUgdmVjdG9yLlxuICAgKi9cbiAgcmVhZG9ubHkgRGltZW5zaW9uczogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIG1ldGFkYXRhIG1hbmFnZW1lbnQgZmllbGRzLlxuICAgKi9cbiAgcmVhZG9ubHkgTWV0YWRhdGFNYW5hZ2VtZW50OiBNZXRhZGF0YU1hbmFnZW1lbnRGaWVsZFtdO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHRoZSBWZWN0b3JJbmRleC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWZWN0b3JJbmRleFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBPcGVuU2VhcmNoIFZlY3RvciBDb2xsZWN0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgY29sbGVjdGlvbjogVmVjdG9yQ29sbGVjdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBpbmRleC5cbiAgICovXG4gIHJlYWRvbmx5IGluZGV4TmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHZlY3RvciBmaWVsZC5cbiAgICovXG4gIHJlYWRvbmx5IHZlY3RvckZpZWxkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGRpbWVuc2lvbnMgaW4gdGhlIHZlY3Rvci5cbiAgICovXG4gIHJlYWRvbmx5IHZlY3RvckRpbWVuc2lvbnM6IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBtZXRhZGF0YSBtYW5hZ2VtZW50IGZpZWxkcy5cbiAgICovXG4gIHJlYWRvbmx5IG1hcHBpbmdzOiBNZXRhZGF0YU1hbmFnZW1lbnRGaWVsZFByb3BzW107XG59XG5cbi8qKlxuICogRGVwbG95IGEgdmVjdG9yIGluZGV4IG9uIHRoZSBjb2xsZWN0aW9uLlxuICovXG5leHBvcnQgY2xhc3MgVmVjdG9ySW5kZXggZXh0ZW5kcyBjZGsuUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGluZGV4LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGluZGV4TmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHZlY3RvciBmaWVsZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2ZWN0b3JGaWVsZDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBkaW1lbnNpb25zIGluIHRoZSB2ZWN0b3IuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdmVjdG9yRGltZW5zaW9uczogbnVtYmVyO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBwcm9wczogVmVjdG9ySW5kZXhQcm9wcyxcbiAgKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuaW5kZXhOYW1lID0gcHJvcHMuaW5kZXhOYW1lO1xuICAgIHRoaXMudmVjdG9yRmllbGQgPSBwcm9wcy52ZWN0b3JGaWVsZDtcbiAgICB0aGlzLnZlY3RvckRpbWVuc2lvbnMgPSBwcm9wcy52ZWN0b3JEaW1lbnNpb25zO1xuICAgIGNvbnN0IGNyUHJvdmlkZXIgPSBPcGVuU2VhcmNoSW5kZXhDUlByb3ZpZGVyLmdldFByb3ZpZGVyKHRoaXMpO1xuICAgIGNyUHJvdmlkZXIucm9sZS5hZGRNYW5hZ2VkUG9saWN5KHByb3BzLmNvbGxlY3Rpb24uYW9zc1BvbGljeSk7XG5cbiAgICBjb25zdCBtYW5hZ2VJbmRleFBvbGljeU5hbWUgPSBnZW5lcmF0ZVBoeXNpY2FsTmFtZVYyKHRoaXMsXG4gICAgICAnTWFuYWdlSW5kZXhQb2xpY3knLFxuICAgICAgeyBtYXhMZW5ndGg6IDMyLCBsb3dlcjogdHJ1ZSB9KTtcbiAgICBjb25zdCBtYW5hZ2VJbmRleFBvbGljeSA9IG5ldyBvc3MuQ2ZuQWNjZXNzUG9saWN5KHRoaXMsICdNYW5hZ2VJbmRleFBvbGljeScsIHtcbiAgICAgIG5hbWU6IG1hbmFnZUluZGV4UG9saWN5TmFtZSxcbiAgICAgIHR5cGU6ICdkYXRhJyxcbiAgICAgIHBvbGljeTogSlNPTi5zdHJpbmdpZnkoW1xuICAgICAgICB7XG4gICAgICAgICAgUnVsZXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IFtgaW5kZXgvJHtwcm9wcy5jb2xsZWN0aW9uLmNvbGxlY3Rpb25OYW1lfS8qYF0sXG4gICAgICAgICAgICAgIFBlcm1pc3Npb246IFtcbiAgICAgICAgICAgICAgICAnYW9zczpEZXNjcmliZUluZGV4JyxcbiAgICAgICAgICAgICAgICAnYW9zczpDcmVhdGVJbmRleCcsXG4gICAgICAgICAgICAgICAgJ2Fvc3M6RGVsZXRlSW5kZXgnLFxuICAgICAgICAgICAgICAgICdhb3NzOlVwZGF0ZUluZGV4JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgUmVzb3VyY2VUeXBlOiAnaW5kZXgnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IFtgY29sbGVjdGlvbi8ke3Byb3BzLmNvbGxlY3Rpb24uY29sbGVjdGlvbk5hbWV9YF0sXG4gICAgICAgICAgICAgIFBlcm1pc3Npb246IFtcbiAgICAgICAgICAgICAgICAnYW9zczpEZXNjcmliZUNvbGxlY3Rpb25JdGVtcycsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIFJlc291cmNlVHlwZTogJ2NvbGxlY3Rpb24nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIFByaW5jaXBhbDogW1xuICAgICAgICAgICAgY3JQcm92aWRlci5yb2xlLnJvbGVBcm4sXG4gICAgICAgICAgXSxcbiAgICAgICAgICBEZXNjcmlwdGlvbjogJycsXG4gICAgICAgIH0sXG4gICAgICBdKSxcbiAgICB9KTtcblxuXG4gICAgY29uc3QgdmVjdG9ySW5kZXggPSBuZXcgY2RrLkN1c3RvbVJlc291cmNlKHRoaXMsICdWZWN0b3JJbmRleCcsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogY3JQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIEVuZHBvaW50OiBgJHtwcm9wcy5jb2xsZWN0aW9uLmNvbGxlY3Rpb25JZH0uJHtjZGsuU3RhY2sub2YodGhpcykucmVnaW9ufS5hb3NzLmFtYXpvbmF3cy5jb21gLFxuICAgICAgICBJbmRleE5hbWU6IHByb3BzLmluZGV4TmFtZSxcbiAgICAgICAgVmVjdG9yRmllbGQ6IHByb3BzLnZlY3RvckZpZWxkLFxuICAgICAgICBEaW1lbnNpb25zOiBwcm9wcy52ZWN0b3JEaW1lbnNpb25zLFxuICAgICAgICBNZXRhZGF0YU1hbmFnZW1lbnQ6IHByb3BzLm1hcHBpbmdzLm1hcCgobSkgPT4ge1xuICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBNYXBwaW5nRmllbGQ6IG0ubWFwcGluZ0ZpZWxkLFxuICAgICAgICAgICAgRGF0YVR5cGU6IG0uZGF0YVR5cGUsXG4gICAgICAgICAgICBGaWx0ZXJhYmxlOiBtLmZpbHRlcmFibGUsXG4gICAgICAgICAgfTtcbiAgICAgICAgfSksXG4gICAgICB9IGFzIFZlY3RvckluZGV4UmVzb3VyY2VQcm9wcyxcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6T3BlblNlYXJjaEluZGV4JyxcbiAgICB9KTtcblxuICAgIHZlY3RvckluZGV4Lm5vZGUuYWRkRGVwZW5kZW5jeShtYW5hZ2VJbmRleFBvbGljeSk7XG4gICAgdmVjdG9ySW5kZXgubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLmNvbGxlY3Rpb24pO1xuICAgIHZlY3RvckluZGV4Lm5vZGUuYWRkRGVwZW5kZW5jeShwcm9wcy5jb2xsZWN0aW9uLmRhdGFBY2Nlc3NQb2xpY3kpO1xuICB9XG5cbn1cblxuLyoqXG4gKiBDdXN0b20gUmVzb3VyY2UgcHJvdmlkZXIgZm9yIE9wZW5TZWFyY2ggSW5kZXggb3BlcmF0aW9ucy5cbiAqXG4gKiBAaW50ZXJuYWwgVGhpcyBpcyBhbiBpbnRlcm5hbCBjb3JlIGZ1bmN0aW9uIGFuZCBzaG91bGQgbm90IGJlIGNhbGxlZCBkaXJlY3RseSBieSBTb2x1dGlvbnMgQ29uc3RydWN0cyBjbGllbnRzLlxuICovXG5leHBvcnQgY29uc3QgT3BlblNlYXJjaEluZGV4Q1JQcm92aWRlciA9IGJ1aWxkQ3VzdG9tUmVzb3VyY2VQcm92aWRlcih7XG4gIHByb3ZpZGVyTmFtZTogJ09wZW5TZWFyY2hJbmRleENSUHJvdmlkZXInLFxuICBjb2RlUGF0aDogcGF0aC5qb2luKFxuICAgIF9fZGlybmFtZSwgJy4uLy4uLy4uL2xhbWJkYS9vcGVuc2VhcmNoLXNlcnZlcmxlc3MtY3VzdG9tLXJlc291cmNlcycpLFxuICBoYW5kbGVyOiAnY3VzdG9tX3Jlc291cmNlcy5vbl9ldmVudCcsXG4gIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzEyLFxufSk7Il19