"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3DataSource = exports.ChunkingStrategy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const CHUNKING_OVERLAP = 20;
const CHUNKING_MAX_TOKENS = 300;
/**
 * Knowledge base can split your source data into chunks. A chunk refers to an
 * excerpt from a data source that is returned when the knowledge base that it
 * belongs to is queried. You have the following options for chunking your
 * data. If you opt for NONE, then you may want to pre-process your files by
 * splitting them up such that each file corresponds to a chunk.
 */
var ChunkingStrategy;
(function (ChunkingStrategy) {
    /**
     * Amazon Bedrock splits your source data into chunks of the approximate size
     * that you set in the `fixedSizeChunkingConfiguration`.
     */
    ChunkingStrategy["FIXED_SIZE"] = "FIXED_SIZE";
    /**
     * `FIXED_SIZE` with the default chunk size of 300 tokens and 20% overlap.
     * If default is selected, chunk size and overlap set by the user will be
     * ignored.
     */
    ChunkingStrategy["DEFAULT"] = "DEFAULT";
    /**
     * Amazon Bedrock treats each file as one chunk. If you choose this option,
     * you may want to pre-process your documents by splitting them into separate
     * files.
     */
    ChunkingStrategy["NONE"] = "NONE";
})(ChunkingStrategy || (exports.ChunkingStrategy = ChunkingStrategy = {}));
/**
 * Sets up a data source to be added to a knowledge base.
 */
class S3DataSource extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const knowledgeBase = props.knowledgeBase;
        const dataSourceName = props.dataSourceName;
        const bucket = props.bucket;
        const inclusionPrefixes = props.inclusionPrefixes;
        const chunkingStrategy = props.chunkingStrategy ?? ChunkingStrategy.DEFAULT;
        const maxTokens = props.maxTokens ?? CHUNKING_MAX_TOKENS;
        const overlapPercentage = props.overlapPercentage ?? CHUNKING_OVERLAP;
        const kmsKey = props.kmsKey;
        bucket.grantRead(knowledgeBase.role);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(knowledgeBase.role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'The KB role needs read only access to all objects in the data source bucket.',
            },
        ], true);
        this.dataSource = new aws_cdk_lib_1.aws_bedrock.CfnDataSource(this, 'DataSource', {
            knowledgeBaseId: knowledgeBase.knowledgeBaseId,
            name: dataSourceName,
            dataSourceConfiguration: {
                type: 'S3',
                s3Configuration: {
                    bucketArn: bucket.bucketArn,
                    inclusionPrefixes: inclusionPrefixes,
                },
            },
            vectorIngestionConfiguration: vectorIngestionConfiguration(chunkingStrategy, maxTokens, overlapPercentage),
            serverSideEncryptionConfiguration: kmsKey ? {
                kmsKeyArn: kmsKey.keyArn,
            } : undefined,
        });
        this.dataSourceId = this.dataSource.attrDataSourceId;
    }
}
exports.S3DataSource = S3DataSource;
_a = JSII_RTTI_SYMBOL_1;
S3DataSource[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.S3DataSource", version: "0.1.198" };
function vectorIngestionConfiguration(chunkingStrategy, maxTokens = CHUNKING_MAX_TOKENS, overlapPercentage = CHUNKING_OVERLAP) {
    if (chunkingStrategy === ChunkingStrategy.FIXED_SIZE) {
        if (maxTokens <= 20 || maxTokens >= 8000) {
            throw new Error(`maxTokens must be between 20 and 8000, got ${maxTokens}`);
        }
        if (overlapPercentage < 0 || overlapPercentage > 100) {
            throw new Error(`overlapPercentage must be between 0 and 100, got ${overlapPercentage}`);
        }
        return {
            chunkingConfiguration: {
                chunkingStrategy: chunkingStrategy,
                fixedSizeChunkingConfiguration: {
                    maxTokens,
                    overlapPercentage,
                },
            },
        };
    }
    else if (chunkingStrategy === ChunkingStrategy.NONE) {
        return {
            chunkingConfiguration: {
                chunkingStrategy,
            },
        };
    }
    else { // DEFAULT
        return {
            chunkingConfiguration: {
                chunkingStrategy: ChunkingStrategy.FIXED_SIZE,
                fixedSizeChunkingConfiguration: {
                    maxTokens: CHUNKING_MAX_TOKENS,
                    overlapPercentage: CHUNKING_OVERLAP,
                },
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtZGF0YS1zb3VyY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvY2RrLWxpYi9iZWRyb2NrL3MzLWRhdGEtc291cmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7QUFFSCw2Q0FBcUQ7QUFHckQscUNBQTBDO0FBQzFDLDJDQUF1QztBQUl2QyxNQUFNLGdCQUFnQixHQUFHLEVBQUUsQ0FBQztBQUM1QixNQUFNLG1CQUFtQixHQUFHLEdBQUcsQ0FBQztBQUVoQzs7Ozs7O0dBTUc7QUFDSCxJQUFZLGdCQWtCWDtBQWxCRCxXQUFZLGdCQUFnQjtJQUMxQjs7O09BR0c7SUFDSCw2Q0FBeUIsQ0FBQTtJQUN6Qjs7OztPQUlHO0lBQ0gsdUNBQW1CLENBQUE7SUFDbkI7Ozs7T0FJRztJQUNILGlDQUFhLENBQUE7QUFDZixDQUFDLEVBbEJXLGdCQUFnQixnQ0FBaEIsZ0JBQWdCLFFBa0IzQjtBQWtERDs7R0FFRztBQUNILE1BQWEsWUFBYSxTQUFRLHNCQUFTO0lBVXpDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQzFDLE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFDNUMsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUM1QixNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNsRCxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxnQkFBZ0IsQ0FBQyxPQUFPLENBQUM7UUFDNUUsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSSxtQkFBbUIsQ0FBQztRQUN6RCxNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxnQkFBZ0IsQ0FBQztRQUN0RSxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBRzVCLE1BQU0sQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3JDLHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLGFBQWEsQ0FBQyxJQUFJLEVBQ2xCO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUFFLDhFQUE4RTthQUN2RjtTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFFRixJQUFJLENBQUMsVUFBVSxHQUFHLElBQUkseUJBQU8sQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUM5RCxlQUFlLEVBQUUsYUFBYSxDQUFDLGVBQWU7WUFDOUMsSUFBSSxFQUFFLGNBQWM7WUFDcEIsdUJBQXVCLEVBQUU7Z0JBQ3ZCLElBQUksRUFBRSxJQUFJO2dCQUNWLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUUsTUFBTSxDQUFDLFNBQVM7b0JBQzNCLGlCQUFpQixFQUFFLGlCQUFpQjtpQkFDckM7YUFDRjtZQUNELDRCQUE0QixFQUFFLDRCQUE0QixDQUN4RCxnQkFBZ0IsRUFBRSxTQUFTLEVBQUUsaUJBQWlCLENBQy9DO1lBQ0QsaUNBQWlDLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQztnQkFDMUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxNQUFNO2FBQ3pCLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FFZCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUM7SUFDdkQsQ0FBQzs7QUF0REgsb0NBdURDOzs7QUFnQkQsU0FBUyw0QkFBNEIsQ0FDbkMsZ0JBQWtDLEVBQ2xDLFlBQW9CLG1CQUFtQixFQUN2QyxvQkFBNEIsZ0JBQWdCO0lBRzVDLElBQUksZ0JBQWdCLEtBQUssZ0JBQWdCLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDckQsSUFBSSxTQUFTLElBQUksRUFBRSxJQUFJLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUN6QyxNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1FBQzdFLENBQUM7UUFFRCxJQUFJLGlCQUFpQixHQUFHLENBQUMsSUFBSSxpQkFBaUIsR0FBRyxHQUFHLEVBQUUsQ0FBQztZQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDM0YsQ0FBQztRQUVELE9BQU87WUFDTCxxQkFBcUIsRUFBRTtnQkFDckIsZ0JBQWdCLEVBQUUsZ0JBQWdCO2dCQUNsQyw4QkFBOEIsRUFBRTtvQkFDOUIsU0FBUztvQkFDVCxpQkFBaUI7aUJBQ2xCO2FBQ0Y7U0FDRixDQUFDO0lBRUosQ0FBQztTQUFNLElBQUksZ0JBQWdCLEtBQUssZ0JBQWdCLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFdEQsT0FBTztZQUNMLHFCQUFxQixFQUFFO2dCQUNyQixnQkFBZ0I7YUFDakI7U0FDRixDQUFDO0lBRUosQ0FBQztTQUFNLENBQUMsQ0FBQyxVQUFVO1FBQ2pCLE9BQU87WUFDTCxxQkFBcUIsRUFBRTtnQkFDckIsZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUMsVUFBVTtnQkFDN0MsOEJBQThCLEVBQUU7b0JBQzlCLFNBQVMsRUFBRSxtQkFBbUI7b0JBQzlCLGlCQUFpQixFQUFFLGdCQUFnQjtpQkFDcEM7YUFDRjtTQUNGLENBQUM7SUFDSixDQUFDO0FBRUgsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IGF3c19iZWRyb2NrIGFzIGJlZHJvY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0IHsgTmFnU3VwcHJlc3Npb25zIH0gZnJvbSAnY2RrLW5hZyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuaW1wb3J0IHsgS25vd2xlZGdlQmFzZSB9IGZyb20gJy4va25vd2xlZGdlLWJhc2UnO1xuXG5jb25zdCBDSFVOS0lOR19PVkVSTEFQID0gMjA7XG5jb25zdCBDSFVOS0lOR19NQVhfVE9LRU5TID0gMzAwO1xuXG4vKipcbiAqIEtub3dsZWRnZSBiYXNlIGNhbiBzcGxpdCB5b3VyIHNvdXJjZSBkYXRhIGludG8gY2h1bmtzLiBBIGNodW5rIHJlZmVycyB0byBhblxuICogZXhjZXJwdCBmcm9tIGEgZGF0YSBzb3VyY2UgdGhhdCBpcyByZXR1cm5lZCB3aGVuIHRoZSBrbm93bGVkZ2UgYmFzZSB0aGF0IGl0XG4gKiBiZWxvbmdzIHRvIGlzIHF1ZXJpZWQuIFlvdSBoYXZlIHRoZSBmb2xsb3dpbmcgb3B0aW9ucyBmb3IgY2h1bmtpbmcgeW91clxuICogZGF0YS4gSWYgeW91IG9wdCBmb3IgTk9ORSwgdGhlbiB5b3UgbWF5IHdhbnQgdG8gcHJlLXByb2Nlc3MgeW91ciBmaWxlcyBieVxuICogc3BsaXR0aW5nIHRoZW0gdXAgc3VjaCB0aGF0IGVhY2ggZmlsZSBjb3JyZXNwb25kcyB0byBhIGNodW5rLlxuICovXG5leHBvcnQgZW51bSBDaHVua2luZ1N0cmF0ZWd5IHtcbiAgLyoqXG4gICAqIEFtYXpvbiBCZWRyb2NrIHNwbGl0cyB5b3VyIHNvdXJjZSBkYXRhIGludG8gY2h1bmtzIG9mIHRoZSBhcHByb3hpbWF0ZSBzaXplXG4gICAqIHRoYXQgeW91IHNldCBpbiB0aGUgYGZpeGVkU2l6ZUNodW5raW5nQ29uZmlndXJhdGlvbmAuXG4gICAqL1xuICBGSVhFRF9TSVpFID0gJ0ZJWEVEX1NJWkUnLFxuICAvKipcbiAgICogYEZJWEVEX1NJWkVgIHdpdGggdGhlIGRlZmF1bHQgY2h1bmsgc2l6ZSBvZiAzMDAgdG9rZW5zIGFuZCAyMCUgb3ZlcmxhcC5cbiAgICogSWYgZGVmYXVsdCBpcyBzZWxlY3RlZCwgY2h1bmsgc2l6ZSBhbmQgb3ZlcmxhcCBzZXQgYnkgdGhlIHVzZXIgd2lsbCBiZVxuICAgKiBpZ25vcmVkLlxuICAgKi9cbiAgREVGQVVMVCA9ICdERUZBVUxUJyxcbiAgLyoqXG4gICAqIEFtYXpvbiBCZWRyb2NrIHRyZWF0cyBlYWNoIGZpbGUgYXMgb25lIGNodW5rLiBJZiB5b3UgY2hvb3NlIHRoaXMgb3B0aW9uLFxuICAgKiB5b3UgbWF5IHdhbnQgdG8gcHJlLXByb2Nlc3MgeW91ciBkb2N1bWVudHMgYnkgc3BsaXR0aW5nIHRoZW0gaW50byBzZXBhcmF0ZVxuICAgKiBmaWxlcy5cbiAgICovXG4gIE5PTkUgPSAnTk9ORScsXG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gUzMgRGF0YSBTb3VyY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNEYXRhU291cmNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIGtub3dsZWRnZSBiYXNlIHRoYXQgdGhpcyBkYXRhIHNvdXJjZSBiZWxvbmdzIHRvLlxuICAgKi9cbiAgcmVhZG9ubHkga25vd2xlZGdlQmFzZTogS25vd2xlZGdlQmFzZTtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBkYXRhIHNvdXJjZS5cbiAgICovXG4gIHJlYWRvbmx5IGRhdGFTb3VyY2VOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgYnVja2V0IHRoYXQgY29udGFpbnMgdGhlIGRhdGEgc291cmNlLlxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0O1xuICAvKipcbiAgICogVGhlIHByZWZpeGVzIG9mIHRoZSBvYmplY3RzIGluIHRoZSBidWNrZXQgdGhhdCBzaG91bGQgYmUgaW5jbHVkZWQgaW4gdGhlIGRhdGEgc291cmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFsbCBvYmplY3RzIGluIHRoZSBidWNrZXQuXG4gICAqL1xuICByZWFkb25seSBpbmNsdXNpb25QcmVmaXhlcz86IHN0cmluZ1tdO1xuICAvKipcbiAgICogVGhlIGNodW5raW5nIHN0cmF0ZWd5IHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgQ2h1bmtpbmdTdHJhdGVneS5ERUZBVUxUXG4gICAqL1xuICByZWFkb25seSBjaHVua2luZ1N0cmF0ZWd5PzogQ2h1bmtpbmdTdHJhdGVneTtcbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiB0b2tlbnMgdG8gdXNlIGluIGEgY2h1bmsuXG4gICAqXG4gICAqIEBkZWZhdWx0IDMwMFxuICAgKi9cbiAgcmVhZG9ubHkgbWF4VG9rZW5zPzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIHBlcmNlbnRhZ2Ugb2Ygb3ZlcmxhcCB0byB1c2UgaW4gYSBjaHVuay5cbiAgICpcbiAgICogQGRlZmF1bHQgMjBcbiAgICovXG4gIHJlYWRvbmx5IG92ZXJsYXBQZXJjZW50YWdlPzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIEtNUyBrZXkgdG8gdXNlIHRvIGVuY3J5cHQgdGhlIGRhdGEgc291cmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBbWF6b24gQmVkcm9jayBlbmNyeXB0cyB5b3VyIGRhdGEgd2l0aCBhIGtleSB0aGF0IEFXUyBvd25zIGFuZCBtYW5hZ2VzXG4gICAqL1xuICByZWFkb25seSBrbXNLZXk/OiBrbXMuSUtleTtcbn1cblxuLyoqXG4gKiBTZXRzIHVwIGEgZGF0YSBzb3VyY2UgdG8gYmUgYWRkZWQgdG8gYSBrbm93bGVkZ2UgYmFzZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFMzRGF0YVNvdXJjZSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgRGF0YSBTb3VyY2UgY2ZuIHJlc291cmNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRhdGFTb3VyY2U6IGJlZHJvY2suQ2ZuRGF0YVNvdXJjZTtcbiAgLyoqXG4gICAqIFRoZSB1bmlxdWUgaWRlbnRpZmllciBvZiB0aGUgZGF0YSBzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGF0YVNvdXJjZUlkOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFMzRGF0YVNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICBjb25zdCBrbm93bGVkZ2VCYXNlID0gcHJvcHMua25vd2xlZGdlQmFzZTtcbiAgICBjb25zdCBkYXRhU291cmNlTmFtZSA9IHByb3BzLmRhdGFTb3VyY2VOYW1lO1xuICAgIGNvbnN0IGJ1Y2tldCA9IHByb3BzLmJ1Y2tldDtcbiAgICBjb25zdCBpbmNsdXNpb25QcmVmaXhlcyA9IHByb3BzLmluY2x1c2lvblByZWZpeGVzO1xuICAgIGNvbnN0IGNodW5raW5nU3RyYXRlZ3kgPSBwcm9wcy5jaHVua2luZ1N0cmF0ZWd5ID8/IENodW5raW5nU3RyYXRlZ3kuREVGQVVMVDtcbiAgICBjb25zdCBtYXhUb2tlbnMgPSBwcm9wcy5tYXhUb2tlbnMgPz8gQ0hVTktJTkdfTUFYX1RPS0VOUztcbiAgICBjb25zdCBvdmVybGFwUGVyY2VudGFnZSA9IHByb3BzLm92ZXJsYXBQZXJjZW50YWdlID8/IENIVU5LSU5HX09WRVJMQVA7XG4gICAgY29uc3Qga21zS2V5ID0gcHJvcHMua21zS2V5O1xuXG5cbiAgICBidWNrZXQuZ3JhbnRSZWFkKGtub3dsZWRnZUJhc2Uucm9sZSk7XG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAga25vd2xlZGdlQmFzZS5yb2xlLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNScsXG4gICAgICAgICAgcmVhc29uOiAnVGhlIEtCIHJvbGUgbmVlZHMgcmVhZCBvbmx5IGFjY2VzcyB0byBhbGwgb2JqZWN0cyBpbiB0aGUgZGF0YSBzb3VyY2UgYnVja2V0LicsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdHJ1ZSxcbiAgICApO1xuXG4gICAgdGhpcy5kYXRhU291cmNlID0gbmV3IGJlZHJvY2suQ2ZuRGF0YVNvdXJjZSh0aGlzLCAnRGF0YVNvdXJjZScsIHtcbiAgICAgIGtub3dsZWRnZUJhc2VJZDoga25vd2xlZGdlQmFzZS5rbm93bGVkZ2VCYXNlSWQsXG4gICAgICBuYW1lOiBkYXRhU291cmNlTmFtZSxcbiAgICAgIGRhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIHR5cGU6ICdTMycsXG4gICAgICAgIHMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldEFybjogYnVja2V0LmJ1Y2tldEFybixcbiAgICAgICAgICBpbmNsdXNpb25QcmVmaXhlczogaW5jbHVzaW9uUHJlZml4ZXMsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgdmVjdG9ySW5nZXN0aW9uQ29uZmlndXJhdGlvbjogdmVjdG9ySW5nZXN0aW9uQ29uZmlndXJhdGlvbihcbiAgICAgICAgY2h1bmtpbmdTdHJhdGVneSwgbWF4VG9rZW5zLCBvdmVybGFwUGVyY2VudGFnZSxcbiAgICAgICksXG4gICAgICBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkNvbmZpZ3VyYXRpb246IGttc0tleSA/IHtcbiAgICAgICAga21zS2V5QXJuOiBrbXNLZXkua2V5QXJuLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcblxuICAgIH0pO1xuXG4gICAgdGhpcy5kYXRhU291cmNlSWQgPSB0aGlzLmRhdGFTb3VyY2UuYXR0ckRhdGFTb3VyY2VJZDtcbiAgfVxufVxuXG5pbnRlcmZhY2UgRml4ZWRTaXplQ2h1bmtpbmdDb25maWcge1xuICBtYXhUb2tlbnM6IG51bWJlcjtcbiAgb3ZlcmxhcFBlcmNlbnRhZ2U6IG51bWJlcjtcbn1cblxuaW50ZXJmYWNlIENodW5raW5nQ29uZmlnIHtcbiAgY2h1bmtpbmdTdHJhdGVneTogQ2h1bmtpbmdTdHJhdGVneTtcbiAgZml4ZWRTaXplQ2h1bmtpbmdDb25maWd1cmF0aW9uPzogRml4ZWRTaXplQ2h1bmtpbmdDb25maWc7XG59XG5cbmludGVyZmFjZSBWZWN0b3JJbmdlc3Rpb25Db25maWcge1xuICBjaHVua2luZ0NvbmZpZ3VyYXRpb24/OiBDaHVua2luZ0NvbmZpZztcbn1cblxuZnVuY3Rpb24gdmVjdG9ySW5nZXN0aW9uQ29uZmlndXJhdGlvbihcbiAgY2h1bmtpbmdTdHJhdGVneTogQ2h1bmtpbmdTdHJhdGVneSxcbiAgbWF4VG9rZW5zOiBudW1iZXIgPSBDSFVOS0lOR19NQVhfVE9LRU5TLFxuICBvdmVybGFwUGVyY2VudGFnZTogbnVtYmVyID0gQ0hVTktJTkdfT1ZFUkxBUCxcbik6IFZlY3RvckluZ2VzdGlvbkNvbmZpZyB7XG5cbiAgaWYgKGNodW5raW5nU3RyYXRlZ3kgPT09IENodW5raW5nU3RyYXRlZ3kuRklYRURfU0laRSkge1xuICAgIGlmIChtYXhUb2tlbnMgPD0gMjAgfHwgbWF4VG9rZW5zID49IDgwMDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgbWF4VG9rZW5zIG11c3QgYmUgYmV0d2VlbiAyMCBhbmQgODAwMCwgZ290ICR7bWF4VG9rZW5zfWApO1xuICAgIH1cblxuICAgIGlmIChvdmVybGFwUGVyY2VudGFnZSA8IDAgfHwgb3ZlcmxhcFBlcmNlbnRhZ2UgPiAxMDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgb3ZlcmxhcFBlcmNlbnRhZ2UgbXVzdCBiZSBiZXR3ZWVuIDAgYW5kIDEwMCwgZ290ICR7b3ZlcmxhcFBlcmNlbnRhZ2V9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNodW5raW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBjaHVua2luZ1N0cmF0ZWd5OiBjaHVua2luZ1N0cmF0ZWd5LFxuICAgICAgICBmaXhlZFNpemVDaHVua2luZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBtYXhUb2tlbnMsXG4gICAgICAgICAgb3ZlcmxhcFBlcmNlbnRhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH07XG5cbiAgfSBlbHNlIGlmIChjaHVua2luZ1N0cmF0ZWd5ID09PSBDaHVua2luZ1N0cmF0ZWd5Lk5PTkUpIHtcblxuICAgIHJldHVybiB7XG4gICAgICBjaHVua2luZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgY2h1bmtpbmdTdHJhdGVneSxcbiAgICAgIH0sXG4gICAgfTtcblxuICB9IGVsc2UgeyAvLyBERUZBVUxUXG4gICAgcmV0dXJuIHtcbiAgICAgIGNodW5raW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBjaHVua2luZ1N0cmF0ZWd5OiBDaHVua2luZ1N0cmF0ZWd5LkZJWEVEX1NJWkUsXG4gICAgICAgIGZpeGVkU2l6ZUNodW5raW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIG1heFRva2VuczogQ0hVTktJTkdfTUFYX1RPS0VOUyxcbiAgICAgICAgICBvdmVybGFwUGVyY2VudGFnZTogQ0hVTktJTkdfT1ZFUkxBUCxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG59XG4iXX0=