"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KnowledgeBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const nag_suppressions_1 = require("cdk-nag/lib/nag-suppressions");
const constructs_1 = require("constructs");
const utils_1 = require("../../common/helpers/utils");
const amazonaurora_1 = require("../amazonaurora");
const opensearch_vectorindex_1 = require("../opensearch-vectorindex");
const opensearchserverless_1 = require("../opensearchserverless");
const pinecone_1 = require("../pinecone");
/**
 * Knowledge base can be backed by different vector databases.
 * This enum represents the different vector databases that can be used.
 *
 * `OPENSEARCH_SERVERLESS` is the default vector database.
 * `REDIS_ENTERPRISE_CLOUD` is the vector database for Redis Enterprise Cloud.
 * `PINECONE` is the vector database for Pinecone.
 * `AMAZON_AURORA` is the vector database for Amazon Aurora PostgreSQL.
 */
var VectorStoreType;
(function (VectorStoreType) {
    /**
     * `OPENSEARCH_SERVERLESS` is the vector store for OpenSearch Serverless.
     */
    VectorStoreType["OPENSEARCH_SERVERLESS"] = "OPENSEARCH_SERVERLESS";
    /**
     * `PINECONE` is the vector store for Pinecone.
     */
    VectorStoreType["PINECONE"] = "PINECONE";
    /**
     * `RDS` is the vector store for Amazon Aurora.
     */
    VectorStoreType["AMAZON_AURORA"] = "RDS";
})(VectorStoreType || (VectorStoreType = {}));
/**
 * Deploys a Bedrock Knowledge Base and configures a backend by OpenSearch Serverless,
 * Pinecone, Redis Enterprise Cloud or Amazon Aurora PostgreSQL.
 *
 */
class KnowledgeBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.instruction = props.instruction;
        const embeddingsModel = props.embeddingsModel;
        const indexName = props.indexName ?? 'bedrock-knowledge-base-default-index';
        const vectorField = props.vectorField ?? 'bedrock-knowledge-base-default-vector';
        const textField = 'AMAZON_BEDROCK_TEXT_CHUNK';
        const metadataField = 'AMAZON_BEDROCK_METADATA';
        this.description = props.description ?? 'CDK deployed Knowledge base'; // even though this prop is optional, if no value is provided it will fail to deploy
        this.knowledgeBaseState = props.knowledgeBaseState ?? 'ENABLED';
        validateModel(embeddingsModel);
        validateVectorIndex(props.vectorStore, props.vectorIndex, props.vectorField, props.indexName);
        if (props.vectorIndex) {
            validateIndexParameters(props.vectorIndex, indexName, vectorField);
        }
        this.name = props.name ?? (0, utils_1.generatePhysicalNameV2)(this, 'KB', { maxLength: 32 });
        if (props.existingRole) {
            this.role = props.existingRole;
        }
        else {
            const roleName = (0, utils_1.generatePhysicalNameV2)(this, 'AmazonBedrockExecutionRoleForKnowledgeBase', { maxLength: 64 });
            this.role = new iam.Role(this, 'Role', {
                roleName: roleName,
                assumedBy: new iam.ServicePrincipal('bedrock.amazonaws.com'),
            });
            this.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                principals: [new iam.ServicePrincipal('bedrock.amazonaws.com')],
                conditions: {
                    StringEquals: {
                        'aws:SourceAccount': cdk.Stack.of(this).account,
                    },
                    ArnLike: {
                        'aws:SourceArn': cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'knowledge-base',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    },
                },
            }));
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['bedrock:InvokeModel'],
                resources: [embeddingsModel.asArn(this)],
            }));
        }
        /**
         * Create the vector store if the vector store was provided by the user.
         * Otherwise check againts all possible vector datastores.
         * If none was provided create default OpenSearch Serverless Collection.
         */
        if (props.vectorStore instanceof opensearchserverless_1.VectorCollection) {
            ({
                vectorStore: this.vectorStore,
                vectorStoreType: this.vectorStoreType,
            } = this.handleOpenSearchCollection(props));
        }
        else if (props.vectorStore instanceof pinecone_1.PineconeVectorStore) {
            ({
                vectorStore: this.vectorStore,
                vectorStoreType: this.vectorStoreType,
            } = this.handlePineconeVectorStore(props));
        }
        else if (props.vectorStore instanceof amazonaurora_1.AmazonAuroraVectorStore) {
            ({
                vectorStore: this.vectorStore,
                vectorStoreType: this.vectorStoreType,
            } = this.handleAmazonAuroraVectorStore(props));
        }
        else if (props.vectorStore instanceof amazonaurora_1.AmazonAuroraDefaultVectorStore) {
            ({
                vectorStore: this.vectorStore,
                vectorStoreType: this.vectorStoreType,
            } = this.handleAmazonAuroraDefaultVectorStore(props));
        }
        else {
            ({
                vectorStore: this.vectorStore,
                vectorStoreType: this.vectorStoreType,
            } = this.handleOpenSearchDefaultVectorCollection());
        }
        /**
         * We need to add `secretsmanager:GetSecretValue` to the role
         * of the knowledge base if we use data sources
         * other than OpenSearch Serverless.
         */
        if (!(this.vectorStore instanceof opensearchserverless_1.VectorCollection)) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['secretsmanager:GetSecretValue'],
                resources: [
                    this.vectorStore.credentialsSecretArn,
                ],
            }));
        }
        /**
         * We need to add `rds-data:ExecuteStatement`,
         * `rds-data:BatchExecuteStatement` and
         * `rds:DescribeDBClusters` to the role
         * of the knowledge base if we use Amazon Aurora as
         * a data source.
         */
        if (this.vectorStore instanceof amazonaurora_1.AmazonAuroraDefaultVectorStore ||
            this.vectorStore instanceof amazonaurora_1.AmazonAuroraVectorStore) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'rds-data:ExecuteStatement',
                    'rds-data:BatchExecuteStatement',
                    'rds:DescribeDBClusters',
                ],
                resources: [
                    this.vectorStore.resourceArn,
                ],
            }));
        }
        /**
         * Create the vector index if the vector store is OpenSearch Serverless
         * and it was not provided. Otherwise use the provided vector index.
         */
        if (this.vectorStoreType === VectorStoreType.OPENSEARCH_SERVERLESS) {
            if (!props.vectorIndex) {
                this.vectorIndex = new opensearch_vectorindex_1.VectorIndex(this, 'KBIndex', {
                    collection: this.vectorStore,
                    indexName,
                    vectorField,
                    vectorDimensions: embeddingsModel.vectorDimensions,
                    mappings: [
                        {
                            mappingField: 'AMAZON_BEDROCK_TEXT_CHUNK',
                            dataType: 'text',
                            filterable: true,
                        },
                        {
                            mappingField: 'AMAZON_BEDROCK_METADATA',
                            dataType: 'text',
                            filterable: false,
                        },
                    ],
                });
                this.vectorIndex.node.addDependency(this.vectorStore);
            }
            else {
                this.vectorIndex = props.vectorIndex;
            }
        }
        /**
         * Create storage configuraion. If it is of type of
         * `AmazonAuroraVectorStore` get textField, metadataField,
         * vectorField from the arguments. Otherwise use default values.
         */
        const storageConfiguration = {
            indexName: indexName,
            vectorStore: this.vectorStore,
            vectorStoreType: this.vectorStoreType,
            vectorField: (this.vectorStore instanceof amazonaurora_1.AmazonAuroraVectorStore) ?
                this.vectorStore.vectorField : vectorField,
            textField: (this.vectorStore instanceof amazonaurora_1.AmazonAuroraVectorStore || this.vectorStore instanceof pinecone_1.PineconeVectorStore) ?
                this.vectorStore.textField : textField,
            metadataField: (this.vectorStore instanceof amazonaurora_1.AmazonAuroraVectorStore || this.vectorStore instanceof pinecone_1.PineconeVectorStore) ?
                this.vectorStore.metadataField : metadataField,
        };
        const knowledgeBase = new aws_cdk_lib_1.aws_bedrock.CfnKnowledgeBase(this, 'MyCfnKnowledgeBase', {
            knowledgeBaseConfiguration: {
                type: 'VECTOR',
                vectorKnowledgeBaseConfiguration: {
                    embeddingModelArn: embeddingsModel.asArn(this),
                },
            },
            name: this.name,
            roleArn: this.role.roleArn,
            storageConfiguration: getStorageConfiguration(storageConfiguration),
            description: props.description,
            tags: props.tags,
        });
        this.knowledgeBaseInstance = knowledgeBase;
        const kbCRPolicy = new iam.Policy(this, 'KBCRPolicy', {
            // roles: [crProvider.role],
            roles: [this.role],
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:CreateKnowledgeBase',
                        /**
                         * We need to add `bedrock:AssociateThirdPartyKnowledgeBase` if
                         * we are deploying Redis or Pinecone data sources
                         */
                        //...(this.vectorStoreType === VectorStoreType.REDIS_ENTERPRISE_CLOUD ||
                        ...(this.vectorStoreType === VectorStoreType.PINECONE ?
                            ['bedrock:AssociateThirdPartyKnowledgeBase'] : []),
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:UpdateKnowledgeBase',
                        'bedrock:DeleteKnowledgeBase',
                        'bedrock:TagResource',
                    ],
                    resources: [
                        cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'knowledge-base',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    ],
                }),
                new iam.PolicyStatement({
                    actions: ['iam:PassRole'],
                    resources: [this.role.roleArn],
                }),
            ],
        });
        knowledgeBase.node.addDependency(this.role);
        knowledgeBase.node.addDependency(kbCRPolicy);
        if (this.vectorStoreType === VectorStoreType.OPENSEARCH_SERVERLESS &&
            this.vectorIndex) {
            knowledgeBase.node.addDependency(this.vectorIndex);
        }
        if (this.vectorStoreType === VectorStoreType.AMAZON_AURORA &&
            this.vectorStore instanceof amazonaurora_1.AmazonAuroraDefaultVectorStore) {
            knowledgeBase.node.addDependency(this.vectorStore);
        }
        nag_suppressions_1.NagSuppressions.addResourceSuppressions(kbCRPolicy, [
            {
                id: 'AwsSolutions-IAM5',
                reason: "Bedrock CreateKnowledgeBase can't be restricted by resource.",
            },
        ], true);
        this.knowledgeBaseArn = knowledgeBase.attrKnowledgeBaseArn;
        this.knowledgeBaseId = knowledgeBase.attrKnowledgeBaseId;
    }
    /**
     * Handle VectorCollection type of VectorStore.
     *
     * @param props - The properties of the KnowledgeBase.
     * @returns The instance of VectorCollection, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    handleOpenSearchCollection(props) {
        const vectorStore = props.vectorStore;
        vectorStore.grantDataAccess(this.role);
        return {
            vectorStore: vectorStore,
            vectorStoreType: VectorStoreType.OPENSEARCH_SERVERLESS,
        };
    }
    /**
   * Handle PineconeVectorStore type of VectorStore.
   *
   * @param props - The properties of the KnowledgeBase.
   * @returns The instance of PineconeVectorStore, VectorStoreType.
   * @internal This is an internal core function and should not be called directly.
   */
    handlePineconeVectorStore(props) {
        const vectorStore = props.vectorStore;
        return {
            vectorStore: vectorStore,
            vectorStoreType: VectorStoreType.PINECONE,
        };
    }
    /**
     * Handle AmazonAuroraVectorStore type of VectorStore.
     *
     * @param props - The properties of the KnowledgeBase.
     * @returns The instance of AmazonAuroraVectorStore, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    handleAmazonAuroraVectorStore(props) {
        const vectorStore = props.vectorStore;
        return {
            vectorStore: vectorStore,
            vectorStoreType: VectorStoreType.AMAZON_AURORA,
        };
    }
    /**
     * Handle AmazonAuroraDefaultVectorStore type of VectorStore.
     *
     * @param props - The properties of the KnowledgeBase.
     * @returns The instance of AmazonAuroraDefaultVectorStore, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    handleAmazonAuroraDefaultVectorStore(props) {
        const vectorStore = props.vectorStore;
        return {
            vectorStore: vectorStore,
            vectorStoreType: VectorStoreType.AMAZON_AURORA,
        };
    }
    /**
     * Handle the default VectorStore type.
     *
     * @returns The instance of VectorCollection, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    handleOpenSearchDefaultVectorCollection() {
        const vectorStore = new opensearchserverless_1.VectorCollection(this, 'KBVectors');
        vectorStore.grantDataAccess(this.role);
        return {
            vectorStore: vectorStore,
            vectorStoreType: VectorStoreType.OPENSEARCH_SERVERLESS,
        };
    }
    /**
   * Associate knowledge base with an agent
   */
    associateToAgent(agent) {
        const agentKnowledgeBaseProperty = {
            description: this.description,
            knowledgeBaseId: this.knowledgeBaseId,
            knowledgeBaseState: this.knowledgeBaseState,
        };
        agent.knowledgeBases = [agentKnowledgeBaseProperty];
    }
}
exports.KnowledgeBase = KnowledgeBase;
_a = JSII_RTTI_SYMBOL_1;
KnowledgeBase[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.KnowledgeBase", version: "0.1.198" };
/**
 * Validate that Bedrock Knowledge Base can use the selected model.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateModel(foundationModel) {
    if (!foundationModel.supportsKnowledgeBase) {
        throw new Error(`The model ${foundationModel} is not supported by Bedrock Knowledge Base.`);
    }
}
/**
 * Validate if VectorIndex was provided for a VectorStore of type
 * other than `VectorCollection`.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateVectorIndex(vectorStore, vectorIndex, vectorField, indexName) {
    if (!(vectorStore instanceof opensearchserverless_1.VectorCollection) && vectorIndex) {
        throw new Error('If vectorStore is not of type VectorCollection, vectorIndex should not be provided ' +
            'in KnowledgeBase construct.');
    }
    if (!(vectorStore instanceof opensearchserverless_1.VectorCollection) && indexName) {
        throw new Error('If vectorStore is not of type VectorCollection, indexName should not be provided ' +
            'in KnowledgeBase construct.');
    }
    if (!(vectorStore instanceof opensearchserverless_1.VectorCollection) && vectorField) {
        throw new Error('If vectorStore is not of type VectorCollection, vectorField should not be provided ' +
            'in KnowledgeBase construct.');
    }
}
/**
 * Validate that indexName and vectorField parameters are identical
 * in KnowledgeBase construct if VectorIndex was created manually.
 *
 * By default we assign `vectorIndex` to `bedrock-knowledge-base-default-index`
 * value and if user provides `vectorIndex` manually, we need to make sure
 * they also provide it in KnowledgeBase construct if the value is not
 * `bedrock-knowledge-base-default-index`. Same for vectorField.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateIndexParameters(vectorIndex, indexName, vectorField) {
    if (vectorIndex.indexName !== 'bedrock-knowledge-base-default-index') {
        if (vectorIndex.indexName !== indexName) {
            throw new Error('Default value of indexName is `bedrock-knowledge-base-default-index`.' +
                ' If you create VectorIndex manually and assign vectorIndex to value other than' +
                ' `bedrock-knowledge-base-default-index` then you must provide the same value in KnowledgeBase construct.' +
                ' If you created VectorIndex manually and set it to `bedrock-knowledge-base-default-index`' +
                ' then do not assign indexName in KnowledgeBase construct.');
        }
    }
    if (vectorIndex.vectorField !== 'bedrock-knowledge-base-default-vector') {
        if (vectorIndex.vectorField !== vectorField) {
            throw new Error('Default value of vectorField is `bedrock-knowledge-base-default-vector`.' +
                ' If you create VectorIndex manually and assign vectorField to value other than' +
                ' `bedrock-knowledge-base-default-field` then you must provide the same value in KnowledgeBase construct.' +
                ' If you created VectorIndex manually and set it to `bedrock-knowledge-base-default-vector`' +
                ' then do not assign vectorField in KnowledgeBase construct.');
        }
    }
}
/**
 * Determine storage configuration based on vector store type.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function getStorageConfiguration(params) {
    switch (params.vectorStoreType) {
        case VectorStoreType.OPENSEARCH_SERVERLESS:
            params.vectorStore = params.vectorStore;
            return {
                type: VectorStoreType.OPENSEARCH_SERVERLESS,
                opensearchServerlessConfiguration: {
                    collectionArn: params.vectorStore.collectionArn,
                    fieldMapping: {
                        vectorField: params.vectorField,
                        textField: params.textField,
                        metadataField: params.metadataField,
                    },
                    vectorIndexName: params.indexName,
                },
            };
        case VectorStoreType.PINECONE:
            params.vectorStore = params.vectorStore;
            return {
                type: VectorStoreType.PINECONE,
                pineconeConfiguration: {
                    namespace: params.vectorStore.namespace || undefined,
                    connectionString: params.vectorStore.connectionString,
                    credentialsSecretArn: params.vectorStore.credentialsSecretArn,
                    fieldMapping: {
                        textField: params.textField,
                        metadataField: params.metadataField,
                    },
                },
            };
        case VectorStoreType.AMAZON_AURORA:
            params.vectorStore = params.vectorStore instanceof amazonaurora_1.AmazonAuroraVectorStore ?
                params.vectorStore :
                params.vectorStore;
            return {
                type: VectorStoreType.AMAZON_AURORA,
                rdsConfiguration: {
                    credentialsSecretArn: params.vectorStore.credentialsSecretArn,
                    databaseName: params.vectorStore.databaseName,
                    resourceArn: params.vectorStore.resourceArn,
                    tableName: params.vectorStore.tableName,
                    fieldMapping: {
                        vectorField: params.vectorField.replace(/-/g, '_'),
                        primaryKeyField: params.vectorStore.primaryKeyField,
                        textField: params.textField.toLowerCase(),
                        metadataField: params.metadataField.toLowerCase(),
                    },
                },
            };
        default:
            throw new Error(`Unsupported vector store type: ${params.vectorStoreType}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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