/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { aws_bedrock as bedrock } from 'aws-cdk-lib';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { Agent } from './../bedrock/agent';
import { BedrockFoundationModel } from './models';
import { AmazonAuroraDefaultVectorStore, AmazonAuroraVectorStore } from '../amazonaurora';
import { VectorIndex } from '../opensearch-vectorindex';
import { VectorCollection } from '../opensearchserverless';
import { PineconeVectorStore } from '../pinecone';
/**
 * Properties for a knowledge base
 */
export interface KnowledgeBaseProps {
    /**
     * The embeddings model for the knowledge base
     */
    readonly embeddingsModel: BedrockFoundationModel;
    /**
     * The name of the knowledge base.
     */
    readonly name?: string;
    /**
     * The description of the knowledge base.
     *
     * @default - No description provided.
     */
    readonly description?: string;
    /**
     * Existing IAM role with a policy statement
     * granting permission to invoke the specific embeddings model.
     * Any entity (e.g., an AWS service or application) that assumes
     * this role will be able to invoke or use the
     * specified embeddings model within the Bedrock service.
     */
    readonly existingRole?: iam.Role;
    /**
     * A narrative description of the knowledge base.
     *
     * A Bedrock Agent can use this instruction to determine if it should
     * query this Knowledge Base.
     *
     * @default - No description provided.
     */
    readonly instruction?: string;
    /**
     * The name of the vector index.
     * If vectorStore is not of type `VectorCollection`,
     * do not include this property as it will throw error.
     *
     * @default - 'bedrock-knowledge-base-default-index'
     */
    readonly indexName?: string;
    /**
     * The name of the field in the vector index.
     * If vectorStore is not of type `VectorCollection`,
     * do not include this property as it will throw error.
     *
     * @default - 'bedrock-knowledge-base-default-vector'
     */
    readonly vectorField?: string;
    /**
     * The vector store for the knowledge base. Must be either of
     * type `VectorCollection`, `RedisEnterpriseVectorStore`,
     * `PineconeVectorStore`, `AmazonAuroraVectorStore` or
     * `AmazonAuroraDefaultVectorStore`.
     *
     * @default - A new OpenSearch Serverless vector collection is created.
     */
    readonly vectorStore?: VectorCollection | PineconeVectorStore | AmazonAuroraVectorStore | AmazonAuroraDefaultVectorStore;
    /**
     * The vector index for the OpenSearch Serverless backed knowledge base.
     * If vectorStore is not of type `VectorCollection`, do not include
     * this property as it will throw error.
     *
     * @default - A new vector index is created on the Vector Collection
     * if vector store is of `VectorCollection` type.
     */
    readonly vectorIndex?: VectorIndex;
    /**
     * Specifies whether to use the knowledge base or not when sending an InvokeAgent request.
     */
    readonly knowledgeBaseState?: string;
    /**
     * OPTIONAL: Tag (KEY-VALUE) bedrock agent resource
     *
     * @default - false
     */
    readonly tags?: Record<string, string>;
}
/**
 * Deploys a Bedrock Knowledge Base and configures a backend by OpenSearch Serverless,
 * Pinecone, Redis Enterprise Cloud or Amazon Aurora PostgreSQL.
 *
 */
export declare class KnowledgeBase extends Construct {
    /**
     * The name of the knowledge base.
     */
    readonly name: string;
    /**
     * Instance of knowledge base.
     */
    readonly knowledgeBaseInstance: bedrock.CfnKnowledgeBase;
    /**
     * The role the Knowledge Base uses to access the vector store and data source.
     */
    readonly role: iam.Role;
    /**
     * The vector store for the knowledge base.
     */
    readonly vectorStore: VectorCollection | PineconeVectorStore | AmazonAuroraVectorStore | AmazonAuroraDefaultVectorStore;
    /**
     * A narrative instruction of the knowledge base.
     */
    readonly instruction?: string;
    /**
     * The ARN of the knowledge base.
     */
    readonly knowledgeBaseArn: string;
    /**
     * The ID of the knowledge base.
     */
    readonly knowledgeBaseId: string;
    /**
     * The OpenSearch vector index for the knowledge base.
     * @private
     */
    private vectorIndex?;
    /**
     * The description knowledge base.
     */
    readonly description: string;
    /**
     * Specifies whether to use the knowledge base or not when sending an InvokeAgent request.
     */
    readonly knowledgeBaseState: string;
    /**
     * The type of the knowledge base.
     * @private
     */
    private vectorStoreType;
    constructor(scope: Construct, id: string, props: KnowledgeBaseProps);
    /**
     * Handle VectorCollection type of VectorStore.
     *
     * @param props - The properties of the KnowledgeBase.
     * @returns The instance of VectorCollection, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    private handleOpenSearchCollection;
    /**
   * Handle PineconeVectorStore type of VectorStore.
   *
   * @param props - The properties of the KnowledgeBase.
   * @returns The instance of PineconeVectorStore, VectorStoreType.
   * @internal This is an internal core function and should not be called directly.
   */
    private handlePineconeVectorStore;
    /**
     * Handle AmazonAuroraVectorStore type of VectorStore.
     *
     * @param props - The properties of the KnowledgeBase.
     * @returns The instance of AmazonAuroraVectorStore, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    private handleAmazonAuroraVectorStore;
    /**
     * Handle AmazonAuroraDefaultVectorStore type of VectorStore.
     *
     * @param props - The properties of the KnowledgeBase.
     * @returns The instance of AmazonAuroraDefaultVectorStore, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    private handleAmazonAuroraDefaultVectorStore;
    /**
     * Handle the default VectorStore type.
     *
     * @returns The instance of VectorCollection, VectorStoreType.
     * @internal This is an internal core function and should not be called directly.
     */
    private handleOpenSearchDefaultVectorCollection;
    /**
   * Associate knowledge base with an agent
   */
    associateToAgent(agent: Agent): void;
}
