"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Guardrail = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const fs = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const kms = require("aws-cdk-lib/aws-kms");
const constructs_1 = require("constructs");
const content_policy_1 = require("./content-policy");
const guardrail_version_1 = require("./guardrail-version");
const pii_list_1 = require("./pii-list");
const utils_1 = require("../../common/helpers/utils");
/**
 * Deploy bedrock guardrail .
 */
class Guardrail extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.name = props.name ?? (0, utils_1.generatePhysicalNameV2)(this, 'bedrock-guardrail', { maxLength: 32, lower: true, separator: '-' });
        this.kmsKeyArn = props.kmsKeyArn ?? new kms.Key(scope, `'${id}Key'`, {
            enableKeyRotation: true,
        }).keyArn;
        const defaultBlockedInputMessaging = 'Sorry, your query voilates our usage policy.';
        const defaultBlockedOutputsMessaging = 'Sorry, I am unable to answer your question because of our usage policy.';
        this.guardrailInstance = new aws_cdk_lib_1.aws_bedrock.CfnGuardrail(this, 'MyGuardrail', {
            blockedInputMessaging: props.blockedInputMessaging ?? defaultBlockedInputMessaging,
            blockedOutputsMessaging: props.blockedOutputsMessaging ?? defaultBlockedOutputsMessaging,
            name: this.name,
            description: props.description,
            kmsKeyArn: this.kmsKeyArn,
            contentPolicyConfig: {
                filtersConfig: new content_policy_1.ContentPolicyConfig(this, `'CP${id}'`, props.filtersConfig).contentPolicyConfigList,
            },
        });
        this.guardrailVersion = this.guardrailInstance.attrVersion;
        this.guardrailId = this.guardrailInstance.attrGuardrailId;
    }
    addSensitiveInformationPolicyConfig(props, guardrailRegexesConfig) {
        if (props) {
            this.guardrailInstance.sensitiveInformationPolicyConfig =
                {
                    piiEntitiesConfig: new pii_list_1.SensitiveInformationPolicyConfig(this, 'PII', props).piiConfigList,
                    regexesConfig: [guardrailRegexesConfig],
                };
        }
        else {
            throw new Error('No guardrailPiiEntityConfig or guardrailRegexesConfig is set in GuardrailProps.');
        }
    }
    async uploadWordPolicyFromFile(filePath) {
        try {
            const wordsFilter = [];
            // Read the file line by line and extract the words
            const fileContents = await fs.promises.readFile(filePath, 'utf8');
            const lines = fileContents.trim().split(',');
            for (const line of lines) {
                const word = line.trim();
                if (word) {
                    wordsFilter.push({ text: word });
                }
            }
            // Add the word policy configuration
            this.addWordPolicyConfig(wordsFilter);
        }
        catch (error) {
            console.error('Error reading file and adding word policy config:', error);
            throw error;
        }
    }
    addWordPolicyConfig(wordsFilter) {
        if (wordsFilter && wordsFilter.length > 0) {
            this.guardrailInstance.wordPolicyConfig =
                {
                    managedWordListsConfig: [{
                            type: 'PROFANITY',
                        }],
                    wordsConfig: wordsFilter,
                };
        }
        else {
            throw new Error('props.wordsFilter is empty or undefined in GuardrailProps.');
        }
    }
    addTopicPolicyConfig(topic) {
        if (topic) {
            this.guardrailInstance.topicPolicyConfig =
                {
                    topicsConfig: topic.topicConfigPropertyList(),
                };
        }
        else {
            throw new Error('topic.topicConfigPropertylist is empty or undefined in GuardrailProps.');
        }
    }
    addTags(props) {
        if (props && props.tags) {
            this.guardrailInstance.tags = props.tags;
        }
    }
    /**
     * Creates a version of the guardrail.
     */
    addVersion(id, description) {
        const version = new guardrail_version_1.GuardrailVersion(this, id, {
            guardrailIdentifier: this.guardrailId,
            description,
        });
        return version;
    }
}
exports.Guardrail = Guardrail;
_a = JSII_RTTI_SYMBOL_1;
Guardrail[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.Guardrail", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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