import * as s3 from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
/**
 * Result of binding `ApiSchema` into an `ActionGroup`.
 */
export interface ApiSchemaConfig {
    /**
     * The JSON or YAML-formatted payload defining the OpenAPI schema for the action group.
     * (mutually exclusive with `s3`)
     */
    readonly payload?: string;
    /**
     * Contains details about the S3 object containing the OpenAPI schema for the action group.
     * (mutually exclusive with `payload`)
     */
    readonly s3?: S3Identifier;
}
/**
 * Result of the bind when `S3ApiSchema` is used.
 */
export interface S3Identifier {
    /**
     * The name of the S3 bucket.
     */
    readonly s3BucketName: string;
    /**
     * The S3 object key containing the resource.
     */
    readonly s3ObjectKey: string;
}
/**
 * Bedrock Agents Action Group API Schema definition.
 */
export declare abstract class ApiSchema {
    /**
     * Inline code for API Schema
     * @return `InlineApiSchema` with inline schema
     * @param schema The actual Open API schema
     */
    static fromInline(schema: string): InlineApiSchema;
    /**
     * Loads the API Schema from a local disk path.
     * @return `InlineApiSchema` with the contents of `path`
     * @param path Path to the Open API schema file in yaml or JSON
     */
    static fromAsset(path: string): InlineApiSchema;
    /**
     * API Schema as an S3 object.
     * @return `S3ApiSchema` with the S3 bucket and key.
     * @param bucket The S3 bucket
     * @param key The object key
     */
    static fromBucket(bucket: s3.IBucket, key: string): S3ApiSchema;
    /**
     * Called when the action group is initialized to allow this object to bind
     * to the stack, add resources and have fun.
     *
     * @param scope The binding scope. Don't be smart about trying to down-cast or
     * assume it's initialized. You may just use it as a construct scope.
     */
    abstract bind(scope: Construct): ApiSchemaConfig;
}
/**
 * API Schema from a string value.
 */
export declare class InlineApiSchema extends ApiSchema {
    private schema;
    constructor(schema: string);
    bind(_scope: Construct): ApiSchemaConfig;
}
/**
 * API Schema in an S3 object.
 */
export declare class S3ApiSchema extends ApiSchema {
    private bucket;
    private key;
    constructor(bucket: s3.IBucket, key: string);
    bind(_scope: Construct): ApiSchemaConfig;
}
