"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.validatePromptOverrideConfiguration = exports.validateInferenceConfiguration = exports.Agent = exports.PromptState = exports.PromptCreationMode = exports.ParserMode = exports.PromptType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const agent_action_group_1 = require("./agent-action-group");
const agent_alias_1 = require("./agent-alias");
const utils_1 = require("../../common/helpers/utils");
/**
 * The step in the agent sequence that this prompt configuration applies to.
 */
var PromptType;
(function (PromptType) {
    PromptType["PRE_PROCESSING"] = "PRE_PROCESSING";
    PromptType["ORCHESTRATION"] = "ORCHESTRATION";
    PromptType["POST_PROCESSING"] = "POST_PROCESSING";
    PromptType["KNOWLEDGE_BASE_RESPONSE_GENERATION"] = "KNOWLEDGE_BASE_RESPONSE_GENERATION";
})(PromptType || (exports.PromptType = PromptType = {}));
/**
 * Specifies whether to override the default parser Lambda function when
 * parsing the raw foundation model output in the part of the agent sequence
 * defined by the promptType. If you set the field as OVERRIDEN, the
 * overrideLambda field in the PromptOverrideConfiguration must be specified
 * with the ARN of a Lambda function.
 */
var ParserMode;
(function (ParserMode) {
    ParserMode["DEFAULT"] = "DEFAULT";
    ParserMode["OVERRIDDEN"] = "OVERRIDDEN";
})(ParserMode || (exports.ParserMode = ParserMode = {}));
/**
 * Specifies whether to override the default prompt template for this
 * promptType. Set this value to OVERRIDDEN to use the prompt that you
 * provide in the basePromptTemplate. If you leave it as DEFAULT, the agent
 * uses a default prompt template.
 */
var PromptCreationMode;
(function (PromptCreationMode) {
    PromptCreationMode["DEFAULT"] = "DEFAULT";
    PromptCreationMode["OVERRIDDEN"] = "OVERRIDDEN";
})(PromptCreationMode || (exports.PromptCreationMode = PromptCreationMode = {}));
/**
 * Specifies whether to allow the agent to carry out the step specified in the
 * promptType. If you set this value to DISABLED, the agent skips that step.
 * The default state for each promptType is as follows.
 *
 *     PRE_PROCESSING – ENABLED
 *     ORCHESTRATION – ENABLED
 *     KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
 *     POST_PROCESSING – DISABLED
 */
var PromptState;
(function (PromptState) {
    PromptState["ENABLED"] = "ENABLED";
    PromptState["DISABLED"] = "DISABLED";
})(PromptState || (exports.PromptState = PromptState = {}));
/**
 * Deploy a Bedrock Agent.
 */
class Agent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * A list of values to indicate if PrepareAgent or an Alias needs to be updated.
         * @private
         */
        this.resourceUpdates = [];
        /**
         * A list of action groups associated with the agent
         * @private
         */
        this.actionGroups = [];
        /**
         * A list of KnowledgeBases associated with the agent.
         *
         * @default - No knowledge base is used.
         */
        this.knowledgeBases = [];
        validatePromptOverrideConfiguration(props.promptOverrideConfiguration);
        validateModel(props.foundationModel);
        this.name = props.name ?? (0, utils_1.generatePhysicalNameV2)(this, 'bedrock-agent', { maxLength: 32, lower: true, separator: '-' });
        if (props.existingRole) {
            this.role = props.existingRole;
        }
        else {
            this.role = new iam.Role(this, 'Role', {
                assumedBy: new iam.ServicePrincipal('bedrock.amazonaws.com'),
                roleName: (0, utils_1.generatePhysicalNameV2)(this, 'AmazonBedrockExecutionRoleForAgents_', { maxLength: 64, lower: false }),
            });
            this.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                principals: [new iam.ServicePrincipal('bedrock.amazonaws.com')],
                conditions: {
                    StringEquals: {
                        'aws:SourceAccount': cdk.Stack.of(this).account,
                    },
                    ArnLike: {
                        'aws:SourceArn': cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'agent',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    },
                },
            }));
            new iam.Policy(this, 'AgentFMPolicy', {
                roles: [this.role],
                statements: [
                    new iam.PolicyStatement({
                        actions: ['bedrock:InvokeModel'],
                        resources: [props.foundationModel.asArn(this)],
                    }),
                ],
            });
        }
        const agent = new aws_cdk_lib_1.aws_bedrock.CfnAgent(this, 'Agent', {
            agentName: this.name,
            foundationModel: String(props.foundationModel),
            instruction: props.instruction,
            description: props.description,
            idleSessionTtlInSeconds: props.idleSessionTTL?.toSeconds(),
            agentResourceRoleArn: this.role.roleArn,
            customerEncryptionKeyArn: props.encryptionKey?.keyArn,
            tags: props.tags,
            promptOverrideConfiguration: props.promptOverrideConfiguration,
            autoPrepare: props.shouldPrepareAgent,
        });
        this.agentInstance = agent;
        this.agentId = agent.attrAgentId;
        this.agentArn = agent.attrAgentArn;
        this.agentversion = agent.attrAgentVersion;
        this._addAliasDependency(agent.attrUpdatedAt);
        if (props.aliasName) {
            const alias = this.addAlias({
                aliasName: props.aliasName,
            });
            this.aliasId = alias.aliasId;
            this.aliasArn = alias.aliasArn;
            this.aliasName = alias.aliasName;
        }
        if (props.knowledgeBases) {
            this.addKnowledgeBases(props.knowledgeBases);
        }
        if (props.actionGroups) {
            this.addActionGroups(props.actionGroups);
        }
        // To allow your agent to request the user for additional information
        // when trying to complete a task , add this action group
        if (props.enableUserInput) {
            this.addActionGroup(new agent_action_group_1.AgentActionGroup(this, 'userInputEnabledActionGroup', {
                actionGroupName: 'UserInputAction',
                parentActionGroupSignature: 'AMAZON.UserInput',
                actionGroupState: 'ENABLED',
            }));
        }
    }
    /**
     * Add an alias to the agent.
     */
    addAlias(props) {
        const alias = new agent_alias_1.AgentAlias(this, `AgentAlias-${props.aliasName}`, {
            agentId: this.agentId,
            agentVersion: props.agentVersion,
            resourceUpdates: cdk.Lazy.list({ produce: () => this.resourceUpdates }),
            aliasName: props.aliasName,
            description: props.description,
        });
        return alias;
    }
    /**
     * Add knowledge bases to the agent.
     */
    addKnowledgeBases(knowledgeBases) {
        for (const kb of knowledgeBases) {
            this.addKnowledgeBase(kb);
        }
    }
    /**
     * Add knowledge base to the agent.
     */
    addKnowledgeBase(knowledgeBase) {
        if (!knowledgeBase.instruction) {
            throw new Error('Agent Knowledge Bases require instructions.');
        }
        new iam.Policy(this, `AgentKBPolicy-${knowledgeBase.name}`, {
            roles: [this.role],
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:UpdateKnowledgeBase',
                        'bedrock:Retrieve',
                    ],
                    resources: [knowledgeBase.knowledgeBaseArn],
                }),
            ],
        });
        const agentKnowledgeBaseProperty = {
            description: knowledgeBase.instruction, // known issue: wrong parameter mapping in Cfn. Workaround: pass instruction through description
            knowledgeBaseId: knowledgeBase.knowledgeBaseId,
            knowledgeBaseState: knowledgeBase.knowledgeBaseState,
        };
        if (!this.agentInstance.knowledgeBases || !Array.isArray(this.agentInstance.knowledgeBases)) {
            this.agentInstance.knowledgeBases = [agentKnowledgeBaseProperty];
        }
        else {
            this.agentInstance.knowledgeBases.push(agentKnowledgeBaseProperty);
        }
    }
    /**
     * Add action group to the agent.
     */
    addActionGroup(actionGroup) {
        actionGroup.actionGroupExecutor?.lambda?.addPermission('AgentLambdaInvocationPolicy', {
            principal: new iam.ServicePrincipal('bedrock.amazonaws.com'),
            sourceArn: this.agentArn,
            sourceAccount: cdk.Stack.of(this).account,
        });
        if (!this.agentInstance.actionGroups || !Array.isArray(this.agentInstance.actionGroups)) {
            this.agentInstance.actionGroups = [actionGroup.actionGroupProperty];
        }
        else {
            this.agentInstance.actionGroups.push(actionGroup.actionGroupProperty);
        }
    }
    /**
     * Add action groups to the agent.
     */
    addActionGroups(actionGroups) {
        for (const actionGroup of actionGroups) {
            this.addActionGroup(actionGroup);
        }
    }
    /**
     * Register a dependency for aliases.
     *
     * @param updatedAt - The updatedAt of the resource that will be registered as a dependency.
     *
     * @internal This is an internal core function and should not be called directly.
     */
    _addAliasDependency(updatedAt) {
        if (updatedAt) {
            this.resourceUpdates.push(updatedAt);
        }
    }
}
exports.Agent = Agent;
_a = JSII_RTTI_SYMBOL_1;
Agent[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.Agent", version: "0.1.198" };
/**
 * Validate that Bedrock Agents can use the selected model.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateModel(foundationModel) {
    if (!foundationModel.supportsAgents) {
        throw new Error(`The model ${foundationModel} is not supported by Bedrock Agents.`);
    }
}
/**
 * Validate the inferenceConfiguration of a prompt override.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateInferenceConfiguration(inferenceConfiguration) {
    if (inferenceConfiguration.topK < 0 || inferenceConfiguration.topK > 500) {
        throw new Error('topK must be between 0 and 500');
    }
    if (!Number.isInteger(inferenceConfiguration.topK)) {
        throw new Error('topK must be an integer');
    }
    if (inferenceConfiguration.stopSequences.length > 4) {
        throw new Error('stopSequences cannot contain more than 4 elements');
    }
    if (inferenceConfiguration.maximumLength < 0 || inferenceConfiguration.maximumLength > 4096) {
        throw new Error('maximumLength must be between 0 and 4096');
    }
    if (!Number.isInteger(inferenceConfiguration.maximumLength)) {
        throw new Error('maximumLength must be an integer');
    }
    if (inferenceConfiguration.topP < 0 || inferenceConfiguration.topP > 1) {
        throw new Error('topP must be between 0 and 1');
    }
    if (inferenceConfiguration.temperature < 0 || inferenceConfiguration.temperature > 1) {
        throw new Error('temperature must be between 0 and 1');
    }
}
exports.validateInferenceConfiguration = validateInferenceConfiguration;
/**
 * Validate the promptOverrideConfiguration.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validatePromptOverrideConfiguration(promptOverrideConfiguration) {
    if (!promptOverrideConfiguration) {
        return;
    }
    if (promptOverrideConfiguration.overrideLambda &&
        promptOverrideConfiguration.promptConfigurations.some(pc => pc.parserMode !== ParserMode.OVERRIDDEN)) {
        throw new Error('overrideLambda can only be used if all promptConfigurations have a parserMode value of OVERRIDDEN');
    }
    if (!promptOverrideConfiguration.overrideLambda &&
        promptOverrideConfiguration.promptConfigurations.some(pc => pc.parserMode === ParserMode.OVERRIDDEN)) {
        throw new Error('At least one promptConfiguration has a parserMode value of OVERRIDDEN, but no overrideLambda is specified');
    }
    // check inferenceConfiguration number types
    Object.values(promptOverrideConfiguration.promptConfigurations).forEach(pc => {
        validateInferenceConfiguration(pc.inferenceConfiguration);
    });
    return;
}
exports.validatePromptOverrideConfiguration = validatePromptOverrideConfiguration;
//# sourceMappingURL=data:application/json;base64,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