/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { aws_bedrock as bedrock } from 'aws-cdk-lib';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import { Construct } from 'constructs';
import { ApiSchema, ApiSchemaConfig } from './api-schema';
export interface ActionGroupExecutor {
    /**
     * The Lambda function containing the business logic that is carried out upon invoking the action.
     */
    readonly lambda?: lambda.IFunction;
    /**
     * To return the action group invocation results directly in the InvokeAgent response, specify RETURN_CONTROL .
     */
    readonly customControl?: string;
}
export interface AgentActionGroupProps {
    readonly actionGroupExecutor?: ActionGroupExecutor;
    /**
     * The name of the action group.
     *
     * @default - a name is generated by CloudFormation.
     */
    readonly actionGroupName: string;
    /**
     * Specifies whether the action group is available for the agent to invoke or not when sending an InvokeAgent request.
     */
    readonly actionGroupState?: 'ENABLED' | 'DISABLED';
    /**
     * Contains details about the S3 object containing the OpenAPI schema for the action group. For more information, see
     * [Action group OpenAPI schemas](https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html).
     */
    readonly apiSchema?: ApiSchema;
    /**
     * A description of the action group.
     *
     * @note This object is a Union. Only one member of this object can be specified or returned.
     */
    readonly description?: string;
    /**
     * Defines functions that each define parameters that the agent needs to invoke from the user.
     *
     */
    readonly functionSchema?: bedrock.CfnAgent.FunctionSchemaProperty;
    /**
     * If you specify this value as AMAZON.UserInput, the agent will prompt additional information from the user when it
     * doesn't have enough information to respond to an utterance. Leave this field blank if you don't want the agent to
     * prompt additional information.
     */
    readonly parentActionGroupSignature?: 'AMAZON.UserInput';
    /**
     * Specifies whether the agent should skip the resource in use check on delete.
     */
    readonly skipResourceInUseCheckOnDelete?: boolean;
}
export declare class AgentActionGroup extends Construct {
    /**
     * The unique identifier of the action group.
     */
    readonly actionGroupName: string;
    /**
     * The Lambda function containing the business logic that is carried out upon invoking the action.
     */
    readonly actionGroupExecutor: ActionGroupExecutor | undefined;
    /**
     * The action group.
     */
    readonly actionGroupProperty: bedrock.CfnAgent.AgentActionGroupProperty;
    /**
     * The API schema.
     */
    readonly apiSchema: ApiSchemaConfig | undefined;
    /**
     * The description.
     */
    readonly description: string | undefined;
    /**
     * The parent action group signature.
     */
    readonly parentActionGroupSignature: string | undefined;
    /**
     * The action group state.
     */
    readonly actionGroupState: string | undefined;
    /**
     * The skip resource in use check on delete.
     *
     * @default - false
     */
    readonly skipResourceInUseCheckOnDelete: boolean | undefined;
    /**
     * A list of action groups associated with the agent
     */
    readonly functionSchema: bedrock.CfnAgent.FunctionSchemaProperty | undefined;
    constructor(scope: Construct, id: string, props: AgentActionGroupProps);
}
