"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmazonAuroraDefaultVectorStore = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const rds = require("aws-cdk-lib/aws-rds");
const cdk_nag_1 = require("cdk-nag");
const custom_resource_provider_helper_1 = require("../../common/helpers/custom-resource-provider-helper");
const utils_1 = require("../../common/helpers/utils");
const vpc_helper_1 = require("../../common/helpers/vpc-helper");
/**
  * Creates default AmazonAuroraVectorStore.
  *
  * It includes creation of a VPC with 3 subnets (public,
  * private with NAT Gateway, private without NAT Gateway),
  * with the Amazon Aurora Serverless V2 Cluster.
  * The cluster has 1 writer/reader of PostgreSQL version 15.5
  * instance (min capacity 0.5, max capacity 4). Lambda custom
  * resource executes required pgvector and Amazon Bedrock Knowledge
  * Base SQL queries against Aurora cluster
  * during deployment. The secret containing databases credentials is
  * being deployed and securely stored in AWS Secrets Manager.
  * You must specify the same embeddings model that you used in
  * KnowledgeBase construct.
  * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.VectorDB.html)
  */
class AmazonAuroraDefaultVectorStore extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.databaseName = 'bedrock_vector_db';
        this.tableName = 'bedrock_integration.bedrock_kb';
        this.primaryKeyField = 'id';
        this.clusterIdentifier = 'aurora-serverless-vector-cluster';
        this.embeddingsModelVectorDimension = props.embeddingsModelVectorDimension;
        const vpc = (0, vpc_helper_1.buildVpc)(this, {
            defaultVpcProps: (0, vpc_helper_1.DefaultVpcProps)(),
        });
        vpc.addFlowLog('VpcFlowLog', {
            destination: ec2.FlowLogDestination.toCloudWatchLogs(),
        });
        const auroraSecurityGroup = new ec2.SecurityGroup(this, 'AuroraSecurityGroup', {
            vpc: vpc,
            securityGroupName: 'aurora-security-group',
            description: 'Security group for access to Aurora from Lambda',
        });
        const lambdaSecurityGroup = new ec2.SecurityGroup(this, 'LambdaSecurityGroup', {
            vpc: vpc,
            securityGroupName: 'lambda-security-group',
            description: 'Security group for Lambda access to Aurora',
        });
        const auroraCluster = new rds.DatabaseCluster(this, 'AuroraCluster', {
            engine: rds.DatabaseClusterEngine.auroraPostgres({
                version: rds.AuroraPostgresEngineVersion.VER_15_5,
            }),
            credentials: rds.Credentials.fromGeneratedSecret('postgres'),
            clusterIdentifier: this.clusterIdentifier,
            defaultDatabaseName: this.databaseName,
            vpc: vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            securityGroups: [auroraSecurityGroup],
            iamAuthentication: true,
            storageEncrypted: true,
            serverlessV2MinCapacity: 0.5,
            serverlessV2MaxCapacity: 4,
            writer: rds.ClusterInstance.serverlessV2('AuroraServerlessWriter'),
            readers: [rds.ClusterInstance.serverlessV2('AuroraServerlessReader', { scaleWithWriter: true })],
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.credentialsSecretArn = auroraCluster.secret?.secretArn || '';
        this.resourceArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'cluster',
            resourceName: auroraCluster.clusterIdentifier,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        auroraCluster.addRotationSingleUser();
        auroraSecurityGroup.addIngressRule(lambdaSecurityGroup, ec2.Port.tcp(5432), 'Allow PostgreSQL access from Lambda security group');
        // Add Data API access to the Aurora cluster
        const cfnDbCluster = auroraCluster.node.defaultChild;
        cfnDbCluster.addOverride('Properties.EnableHttpEndpoint', true);
        this.auroraPgCRPolicy = new iam.ManagedPolicy(this, 'AuroraPgPolicy', {
            managedPolicyName: (0, utils_1.generatePhysicalNameV2)(this, 'AuroraPgPolicy', { maxLength: 32, lower: true }),
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ec2:DescribeInstances',
                        'ec2:CreateNetworkInterface',
                        'ec2:AttachNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'autoscaling:CompleteLifecycleAction',
                        'ec2:DeleteNetworkInterface',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'rds-data:BatchExecuteStatement',
                        'rds-data:BeginTransaction',
                        'rds-data:CommitTransaction',
                        'rds-data:ExecuteStatement',
                        'rds-data:RollbackTransaction',
                    ],
                    resources: [
                        cdk.Stack.of(this).formatArn({
                            service: 'rds',
                            resource: 'cluster',
                            resourceName: `${auroraCluster.clusterIdentifier}`,
                        }),
                    ],
                }),
            ],
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this.auroraPgCRPolicy, [
            {
                id: 'AwsSolutions-IAM4',
                reason: 'The AWSLambdaBasicExecutionRole managed policy is required for ' +
                    'the Lambda function to write logs to CloudWatch.',
            },
            {
                id: 'AwsSolutions-IAM5',
                reason: 'This policy is required to allow the custom resource to create a ' +
                    'network interface for the Aurora cluster and it has to be wildcard.',
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(auroraCluster, [
            {
                id: 'AwsSolutions-RDS10',
                reason: 'Deletion protection is disabled to make sure a customer can stop ' +
                    'incurring charges if they want to delete the construct.',
            },
        ], true);
        const customResource = (0, custom_resource_provider_helper_1.buildCustomResourceProvider)({
            providerName: 'AmazonAuroraPgVectorCRProvider',
            vpc: vpc,
            securityGroup: lambdaSecurityGroup,
            codePath: path.join(__dirname, '../../../lambda/amazon-aurora-pgvector-custom-resources'),
            handler: 'custom_resources.on_event',
            runtime: cdk.aws_lambda.Runtime.PYTHON_3_12,
        });
        const crProvider = customResource.getProvider(this);
        crProvider.role.addManagedPolicy(this.auroraPgCRPolicy);
        auroraCluster.secret?.grantRead(crProvider.role);
        const auroraPgVector = new cdk.CustomResource(this, 'AuroraPgVector', {
            resourceType: 'Custom::AmazonAuroraPgVector',
            serviceToken: crProvider.serviceToken,
            properties: {
                DatabaseName: this.databaseName,
                SecretName: auroraCluster.secret?.secretName || '',
                VectorDimensions: props.embeddingsModelVectorDimension,
            },
        });
        auroraPgVector.node.addDependency(this.auroraPgCRPolicy);
        this.auroraPgCRPolicy.node.addDependency(auroraCluster);
    }
}
exports.AmazonAuroraDefaultVectorStore = AmazonAuroraDefaultVectorStore;
_a = JSII_RTTI_SYMBOL_1;
AmazonAuroraDefaultVectorStore[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.amazonaurora.AmazonAuroraDefaultVectorStore", version: "0.1.198" };
//# sourceMappingURL=data:application/json;base64,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