/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
export interface AmazonAuroraDefaultVectorStoreProps {
    /**
     * The embeddings model vector dimension for the knowledge base.
     * Must be identical as in the KnowledgeBase construct.
     * This is due to the factor that the embeddings models
     * have different vector dimensions and this construct
     * needs to know the vector dimensions to create the vector
     * index of appropriate dimensions in the Aurora database.
     */
    readonly embeddingsModelVectorDimension: number;
}
/**
  * Creates default AmazonAuroraVectorStore.
  *
  * It includes creation of a VPC with 3 subnets (public,
  * private with NAT Gateway, private without NAT Gateway),
  * with the Amazon Aurora Serverless V2 Cluster.
  * The cluster has 1 writer/reader of PostgreSQL version 15.5
  * instance (min capacity 0.5, max capacity 4). Lambda custom
  * resource executes required pgvector and Amazon Bedrock Knowledge
  * Base SQL queries against Aurora cluster
  * during deployment. The secret containing databases credentials is
  * being deployed and securely stored in AWS Secrets Manager.
  * You must specify the same embeddings model that you used in
  * KnowledgeBase construct.
  * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.VectorDB.html)
  */
export declare class AmazonAuroraDefaultVectorStore extends cdk.Resource {
    /**
     * The ARN of your Amazon Aurora DB cluster.
     */
    readonly resourceArn: string;
    /**
     * The name of your Database.
     */
    readonly databaseName: string;
    /**
     * The Table Name of your Amazon Aurora DB cluster.
     */
    readonly tableName: string;
    /**
     * The Secret ARN of your Amazon Aurora DB cluster.
     */
    readonly credentialsSecretArn: string;
    /**
     * Primary key of your Amazon Aurora DB cluster.
     */
    readonly primaryKeyField: string;
    /**
     * Cluster identifier of your Amazon Aurora DB cluster.
     */
    readonly clusterIdentifier: string;
    /**
     * Model used for embeddings.
     */
    readonly embeddingsModelVectorDimension: number;
    /**
     * An IAM policy that allows Data API access to Aurora.
     * @private
     */
    private auroraPgCRPolicy;
    constructor(scope: Construct, id: string, props: AmazonAuroraDefaultVectorStoreProps);
}
