#
# Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
# with the License. A copy of the License is located at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
# OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
# and limitations under the License.
#
import base64
import json
import os
import boto3
from typing import Dict
from image_generator import image_generator
from aws_lambda_powertools import Logger, Tracer, Metrics
from aws_lambda_powertools.utilities.typing import LambdaContext
from aws_lambda_powertools.metrics import MetricUnit
from aws_lambda_powertools.utilities.validation import validate, SchemaValidationError



logger = Logger(service="CONTENT_GENERATION")
tracer = Tracer(service="CONTENT_GENERATION")
metrics = Metrics(namespace="content_generation", service="CONTENT_GENERATION")

aws_region = boto3.Session().region_name
bucket = os.environ['OUTPUT_BUCKET']

bedrock_client = boto3.client('bedrock-runtime')
rekognition_client=boto3.client('rekognition')
comprehend_client=boto3.client('comprehend', region_name=aws_region)


@logger.inject_lambda_context(log_event=True)
@tracer.capture_lambda_handler
@metrics.log_metrics(capture_cold_start_metric=True)
def handler(event,  context: LambdaContext) -> dict:

    """
        Invokes the Stability.ai Stable Diffusion XL model 
        to create an image using the input provided in the request body.

    """

    logger.info(f"event is {event}")
    
    input_params=event['detail']['imageInput']
    input_text=input_params['input_text']
    model_id=input_params['model_config']['modelId']
    
    # add timestamp to file name when uploading it to s3
    file_name="generatedimage_"

    response={
        "filename":file_name,
        "generatedImagePath":"",
        "status":"Generating",
        "image_path":bucket,
        "input_text":input_params['input_text'],
        "jobid":input_params["jobid"],
        "message":''
    }
    sample_string_bytes = base64.b64decode(input_text)
    decoded_input_text = sample_string_bytes.decode("utf-8")
    logger.info(f" entered text is :: {decoded_input_text}")
    
    if(len(decoded_input_text)==0):
        response["message"]="Input text is empty."
        return response
       
    img = image_generator(decoded_input_text, rekognition_client,comprehend_client,bedrock_client,bucket)
       
    text_moderation_response=img.text_moderation()
    if(text_moderation_response['isToxic']==True):
        response["message"]="In appropriate input prompt. Please change the prompt."
        response["status"]='Blocked'
    else:
        bedrock_response = img.generate_image(input_params)
        parsed_reponse = parse_response(bedrock_response,model_id)
        if(parsed_reponse['image_generated_status']=='Failed'):
            response["message"]="No image generated by bedrock API, Please check the prompt."
            response["status"]='Blocked'
        else:
            num_of_images=0 #if multiple image geneated iterate through all
            for image in parsed_reponse['image_generated']:
                logger.info(f'num_of_images {num_of_images}')
                if model_id=='stability.stable-diffusion-xl' :
                    imgbase64encoded= parsed_reponse['image_generated'][num_of_images]["base64"]
                if model_id=='amazon.titan-image-generator-v1' :
                    imgbase64encoded= parsed_reponse['image_generated'][num_of_images]
                imageGenerated=img.upload_file_to_s3(imgbase64encoded,file_name)
                num_of_images=+1
                uploaded_file_name=imageGenerated['file_name']
                image_moderation_response=img.image_moderation(uploaded_file_name)
                if(image_moderation_response['isToxic']==True):
                    response["message"]="In-appropriate image generated."
                    response["status"]='Blocked'
                else:
                    response={
                            "filename":uploaded_file_name,
                            "generatedImagePath":"",
                            "status":"Completed",
                            "image_path":bucket,
                            "input_text":input_params['input_text'],
                            "jobid":input_params["jobid"],
                            "message":'Image generated successfully'                 
                        }
                logger.info (f"response :: {response}")
                img.send_job_status(response)

    return response


def parse_response(query_response,model_id):
    """Parse response and return generated image and the prompt"""
    parsed_reponse={
        "image_generated":'',
        "image_generated_status":'Success'
    }
    if(not query_response):
         parsed_reponse['image_generated_status']='Failed'
         return parsed_reponse
    else:
        response_dict = json.loads(query_response["body"].read())

        if model_id=='stability.stable-diffusion-xl' :

            if(response_dict['artifacts'] is None):
                parsed_reponse['image_generated_status']='Failed'
            else:
                 parsed_reponse['image_generated']=response_dict['artifacts']

        if model_id=='amazon.titan-image-generator-v1' :
            if(response_dict['images'] is None):
                parsed_reponse['image_generated_status']='Failed'
            else:
                numiofimages=response_dict['images']
                parsed_reponse['image_generated']=response_dict['images']

        return parsed_reponse


