from eth_abi import decode_abi

from telliot_feeds.queries.evm_call import EVMCall


def test_evm_call_constructor():
    """Validate EVMCall query."""
    # Chain ID, address, and calldata for retrieving total supply of TRB on Ethereum mainnet
    q = EVMCall(
        chainId=1,
        contractAddress="0x88dF592F8eb5D7Bd38bFeF7dEb0fBc02cf3778a0",
        calldata=b"\x18\x16\x0d\xdd",
    )
    # expected query data generated with:
    # https://abi.hashex.org/
    expected_query_data = (
        "000000000000000000000000000000000000000000000000000000000000004"
        "0000000000000000000000000000000000000000000000000000000000000008"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "745564d43616c6c0000000000000000000000000000000000000000000000000"
        "000000000000000000000000000000000000000000000000000000000000000a"
        "00000000000000000000000000000000000000000000000000000000000000001"
        "00000000000000000000000088df592f8eb5d7bd38bfef7deb0fbc02cf3778a00"
        "00000000000000000000000000000000000000000000000000000000000006000"
        "0000000000000000000000000000000000000000000000000000000000000418160ddd"
        "00000000000000000000000000000000000000000000000000000000"
    )
    print(q.query_data.hex())
    print(q.query_id.hex())

    assert q.query_data.hex() == expected_query_data

    query_type, _ = decode_abi(["string", "bytes"], q.query_data)
    assert query_type == "EVMCall"

    exp_q_id = "d7472d51b2cd65a9c6b81da09854efdeeeff8afcda1a2934566f54b731a922f3"
    assert q.query_id.hex() == exp_q_id

    # Chain ID, address, and calldata for a fake read function with two arguments
    # (address,uint256)
    q2 = EVMCall(
        chainId=1,
        contractAddress="0x6b175474e89094c44da98b954eedeac495271d0f",
        calldata=bytes.fromhex(
            "a9059cbb0000000000000000000000003f5047bdb647dc39c88625e17bdbffee905a"
            "9f4400000000000000000000000000000000000000000000011c9a62d04ed0c80000"
        ),
    )
    q_data2 = (
        "000000000000000000000000000000000000000000000000000000000000004"
        "0000000000000000000000000000000000000000000000000000000000000008"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "745564d43616c6c0000000000000000000000000000000000000000000000000"
        "000000000000000000000000000000000000000000000000000000000000000e"
        "00000000000000000000000000000000000000000000000000000000000000001"
        "0000000000000000000000006b175474e89094c44da98b954eedeac495271d0f0"
        "00000000000000000000000000000000000000000000000000000000000006000"
        "00000000000000000000000000000000000000000000000000000000000044a90"
        "59cbb0000000000000000000000003f5047bdb647dc39c88625e17bdbffee905a"
        "9f4400000000000000000000000000000000000000000000011c9a62d04ed0c80"
        "00000000000000000000000000000000000000000000000000000000000"
    )
    q_id2 = "0be1ec9e2f8d903de70714f0097d8120ddb9821771d9725c74ae4a99b11714a9"

    assert q2.query_data.hex() == q_data2
    assert q2.query_id.hex() == q_id2


def test_evm_call_get_query_from_data():
    """Test get_query_from_data."""
    q = EVMCall(
        chainId=1,
        contractAddress="0x88dF592F8eb5D7Bd38bFeF7dEb0fBc02cf3778a0",
        calldata=b"\x18\x16\x0d\xdd",
    )
    q2 = EVMCall.get_query_from_data(q.query_data)
    assert q2.query_data == q.query_data
    assert isinstance(q2, EVMCall)
