# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['oeleo']

package_data = \
{'': ['*']}

install_requires = \
['Fabric>=2.7.0,<3.0.0',
 'SharePlum>=0.5.1,<0.6.0',
 'peewee>=3.15.0,<4.0.0',
 'python-dotenv>=0.20.0,<0.21.0',
 'rich>=12.4.4,<13.0.0']

setup_kwargs = {
    'name': 'oeleo',
    'version': '0.4.7',
    'description': 'A one-eyed tool to copy files with.',
    'long_description': '# oeleo\nPython package / app that can be used for transferring files from an instrument-PC to a data server.\n\n\n## Features (or limitations)\n- Transferring using an ssh connection should preferably be used with key-pairs. This might involve some\n  setting up on your server (ACL) to prevent security issues (the `oeleo` user should only have access to\n  the data folder on your server).\n- Accessing ssh can be done using password if you are not able to figure out how to set proper ownerships \n  on your server.\n- `oeleo` is one-eyed. Meaning that tracking of the "state of the duplicates" is only performed on the local side (where `oeleo` is running).\n- However, `oeleo` contains a `check` method that can help you figure out if starting copying is a  \n  good idea or not. And populate the database if you want.\n- The db that stores information about the "state of the duplicates" is stored relative to the folder \n  `oeleo` is running from. If you delete it (by accident?), `oeleo` will make a new empty one from scratch next time you run.\n- Configuration is done using environmental variables. \n\n## Usage\n\n### Install\n\n```bash\n$ pip install oeleo\n```\n### Run\n\n1. Create an `oeleo` worker instance.\n2. Connect the worker\'s `bookkeeper` to a `sqlite3` database.\n3. Filter local files.\n4. Run to copy files.\n5. Repeat from step 3.\n\n### Examples and descriptions\n\n#### Simple script for copying between local folders\n\n```python\nimport os\nfrom pathlib import Path\nimport time\n\nimport dotenv\n\nfrom oeleo.checkers import ChecksumChecker\nfrom oeleo.models import SimpleDbHandler\nfrom oeleo.connectors import LocalConnector\nfrom oeleo.workers import Worker\nfrom oeleo.utils import logger\n\n\ndef main():\n  log = logger()\n  # assuming you have made a .env file:\n  dotenv.load_dotenv()\n\n  db_name = os.environ["OELEO_DB_NAME"]\n  base_directory_from = Path(os.environ["OELEO_BASE_DIR_FROM"])\n  base_directory_to = Path(os.environ["OELEO_BASE_DIR_TO"])\n  filter_extension = os.environ["OELEO_FILTER_EXTENSION"]\n\n  # Making a worker using the Worker class.\n  # You can also use the `factory` functions in `oeleo.worker`\n  # (e.g. `ssh_worker` and `simple_worker`)\n  bookkeeper = SimpleDbHandler(db_name)\n  checker = ChecksumChecker()\n  local_connector = LocalConnector(directory=base_directory_from)\n  external_connector = LocalConnector(directory=base_directory_to)\n\n  worker = Worker(\n    checker=checker,\n    local_connector=local_connector,\n    external_connector=external_connector,\n    bookkeeper=bookkeeper,\n    extension=filter_extension\n  )\n\n  # Running the worker with 5 minutes intervals.\n  # You can also use an oeleo scheduler for this. The RichScheduler is kind-of cool.\n  worker.connect_to_db()\n  while True:\n    worker.filter_local()\n    worker.run()\n    time.sleep(300)\n\n\nif __name__ == "__main__":\n  main()\n```\n\n#### Environment `.env` file\n```.env\nOELEO_BASE_DIR_FROM=C:\\data\\local\nOELEO_BASE_DIR_TO=C:\\data\\pub\nOELEO_FILTER_EXTENSION=.csv\nOELEO_DB_NAME=local2pub.db\n\n## only needed for advanced connectors:\n# OELEO_DB_HOST=<db host>\n# OELEO_DB_PORT=<db port>\n# OELEO_DB_USER=<db user>\n# OELEO_DB_PASSWORD=<db user>\n# OELEO_EXTERNAL_HOST<ssh hostname>\n# OELEO_USERNAME=<ssh username>\n# OELEO_PASSWORD=<ssh password>\n# OELEO_KEY_FILENAME=<ssh key-pair filename>\n## only needed for SharePointConnector:\n# OELEO_SHAREPOINT_USERNAME=<sharepoint username (fallbacks to ssh username if missing)>\n# OELEO_SHAREPOINT_URL=<url to sharepoint>\n# OELEO_SHAREPOINT_SITENAME=<name of sharepoint site>\n# OELEO_SHAREPOINT_DOC_LIBRARY=<name of sharepoint library>\n```\n\n#### Database\n\nThe database contains one table called `filelist`:\n\n| id  | processed_date             | local_name         | external_name                         | checksum                         | code |\n|-----|:---------------------------|:-------------------|:--------------------------------------|:---------------------------------|-----:|\n| 1   | 2022-07-05 15:55:02.521154 | file_number_1.xyz\t | C:\\oeleo\\check\\to\\file_number_1.xyz   | c976e564825667d7c11ba200457af263 |    1 |\n| 2   | 2022-07-05 15:55:02.536152 | file_number_10.xyz | C:\\oeleo\\check\\to\\file_number_10.xyz\t | d502512c0d32d7503feb3fd3dd287376 |    1 |\n| 3   | 2022-07-05 15:55:02.553157 | file_number_2.xyz\t | C:\\oeleo\\check\\to\\file_number_2.xyz   | cb89d576f5bd57566c78247892baffa3 |    1 |\n\nThe `processed_date` is when the file was last updated (meaning last time `oeleo` found a new checksum for it).\n\nThe table below shows what the different values of `code` mean:\n\n| code | meaning                       |\n|:-----|:------------------------------|\n| 0    | `should-be-copied`            |\n| 1    | `should-be-copied-if-changed` |\n| 2    | `should-not-be-copied`        |\n\nHint! You can **lock** (chose to never copy) a file by editing the `code` manually to 2. \n\n\n#### Using an `oeleo` scheduler\n\nInstead of for example using a while loop to keep `oeleo` running continuously or at selected intervals, \nyou can use a scheduler (e.g. `rocketry`, `watchdog`, `schedule`, or more advanced options such as `AirFlow`).\n\n`oeleo` also includes its own schedulers. This is an example of how to use the `oeleo.RichScheduler`:\n\n\n```python\nimport dotenv\n\nfrom oeleo.schedulers import RichScheduler\nfrom oeleo.workers import simple_worker\n\n# assuming you have created an appropriate .env file\ndotenv.load_dotenv()\nworker = simple_worker()\ns = RichScheduler(\n        worker,\n        run_interval_time=4,  # seconds\n        max_run_intervals=4,\n    )\ns.start()\n```\n\n\n#### Copy files from a Windows PC to a Linux server through ssh\n\n```python\nimport logging\nimport os\nfrom pathlib import Path\n\nimport dotenv\n\nfrom oeleo.connectors import register_password\nfrom oeleo.utils import logger\nfrom oeleo.workers import ssh_worker\n\nlog = logger()\n\nprint(" ssh ".center(80, "-"))\nlog.setLevel(logging.DEBUG)\nlog.info(f"Starting oeleo!")\ndotenv.load_dotenv()\n\nexternal_dir = "/srv/data"\nfilter_extension = ".res"\n\nregister_password(os.environ["OELEO_PASSWORD"])\n\nworker = ssh_worker(\n    db_name="ssh_to_server.db",\n    base_directory_from=Path(r"data\\raw"),\n    base_directory_to=external_dir,\n    extension=filter_extension,\n)\nworker.connect_to_db()\ntry:\n    worker.check(update_db=True)\n    worker.filter_local()\n    worker.run()\nfinally:\n    worker.close()\n```\n\n## Future planned improvements\n\nJust plans, no promises given.\n\n- add `worker.add_local(something)` for updating the db (can be used instead of `worker.filter_local()`)\n- make even nicer printing and logging.\n- create CLI.\n- create an executable.\n- create a web-app.\n- create a GUI (not likely).\n\n## Status\n\n- [x] Works on my PC &rarr; PC\n- [x] Works on my PC &rarr; my server\n- [x] Works on my server &rarr; my server\n- [x] Works on my instrument PC &rarr; my instrument PC\n- [x] Works on my instrument PC &rarr; my server\n- [x] Works OK\n- [x] Deployable\n- [x] On testpypi\n- [x] On pypi\n- [x] Code understandable for others\n- [x] Looking good\n- [x] Fairly easy to use\n- [ ] Easy to use\n\n## Licence\nMIT\n\n## Development\n\n- Developed using `poetry` on `python 3.10`.\n- Must also run on `python 3.8` for Windows 7 support.\n\n### Some useful commands\n\n#### Update version\n\n```bash\n# update version e.g. from 0.3.1 to 0.3.2:\npoetry version patch\n```\nThen edit `__init__.py`:\n```python\n__version__ = "0.3.2"\n```\n#### Build\n\n```bash\npoetry build\n```\n\n#### Publish\n\n```bash\npoetry publish\n```\n\n### Next\n- Take some time off and enjoy the summer.\n\n### Development lead\n- Jan Petter Maehlen, IFE\n',
    'author': 'jepegit',
    'author_email': 'jepe@ife.no',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/ife-bat/oeleo',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
