import backoff
import requests
import singer
from singer import metrics

IS_AQUA = False
IS_REST = True
IS_PROD = False
IS_SAND = True
NOT_EURO = False
IS_EURO = True

URLS = {
    (IS_AQUA, IS_PROD, NOT_EURO): "https://www.zuora.com/",
    (IS_AQUA, IS_SAND, NOT_EURO): "https://apisandbox.zuora.com/",
    (IS_AQUA, IS_PROD, IS_EURO ): "https://rest.eu.zuora.com/",
    (IS_AQUA, IS_SAND, IS_EURO ): "https://rest.sandbox.eu.zuora.com/",
    (IS_REST, IS_PROD, NOT_EURO): "https://rest.zuora.com/",
    (IS_REST, IS_SAND, NOT_EURO): "https://rest.apisandbox.zuora.com/",
    (IS_REST, IS_PROD, IS_EURO ): "https://rest.eu.zuora.com/",
    (IS_REST, IS_SAND, IS_EURO ): "https://rest.sandbox.eu.zuora.com/",
}

LATEST_WSDL_VERSION = "91.0"

LOGGER = singer.get_logger()

class RateLimitException(Exception):
    def __init__(self, resp):
        self.resp = resp
        super(RateLimitException, self).__init__("Rate Limit Exceeded (429) - {}".format(self.resp.content))

class ApiException(Exception):
    def __init__(self, resp):
        self.resp = resp
        super(ApiException, self).__init__("{0.status_code}: {0.content}".format(self.resp))


class Client:
    def __init__(self, username, password, partner_id, sandbox=False, european=False):
        self.username = username
        self.password = password
        self.sandbox = sandbox
        self.european = european
        self.partner_id = partner_id
        self._session = requests.Session()

        adapter = requests.adapters.HTTPAdapter(max_retries=1) # Try again in the case the TCP socket closes
        self._session.mount('https://', adapter)

    @staticmethod
    def from_config(config):
        sandbox = config.get('sandbox', False) == 'true'
        european = config.get('european', False) == 'true'
        partner_id = config.get('partner_id', None)
        return Client(config['username'], config['password'], partner_id, sandbox, european)

    def get_url(self, url, rest=False):
        return URLS[(rest, self.sandbox, self.european)] + url

    @property
    def aqua_auth(self):
        return (self.username, self.password)

    @property
    def rest_headers(self):
        return {
            'apiAccessKeyId': self.username,
            'apiSecretAccessKey': self.password,
            'X-Zuora-WSDL-Version': LATEST_WSDL_VERSION,
            'Content-Type': 'application/json',
        }

    def _request(self, method, url, stream=False, **kwargs):
        req = requests.Request(method, url, **kwargs).prepare()
        LOGGER.info("%s: %s", method, req.url)
        resp = self._session.send(req, stream=stream)
        if resp.status_code == 429:
            raise RateLimitException(resp)
        if resp.status_code != 200:
            raise ApiException(resp)

        return resp

    # NB> Backoff as recommended by Zuora here:
    # https://community.zuora.com/t5/Release-Notifications/Upcoming-Change-for-AQuA-and-Data-Source-Export-January-2021/ba-p/35024
    @backoff.on_exception(backoff.expo,
                          RateLimitException,
                          max_time=5 * 60, # in seconds
                          factor=30,
                          jitter=None)
    def aqua_request(self, method, url, **kwargs):
        with metrics.http_request_timer(url):
            url = self.get_url(url, rest=False)
            return self._request(method, url, auth=self.aqua_auth, **kwargs)

    def rest_request(self, method, url, **kwargs):
        with metrics.http_request_timer(url):
            url = self.get_url(url, rest=True)
            return self._request(method, url, headers=self.rest_headers, **kwargs)
