"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheBehavior = void 0;
const iam = require("@aws-cdk/aws-iam");
const cache_policy_1 = require("../cache-policy");
const distribution_1 = require("../distribution");
/**
 * Allows configuring a variety of CloudFront functionality for a given URL path pattern.
 *
 * Note: This really should simply by called 'Behavior', but this name is already taken by the legacy
 * CloudFrontWebDistribution implementation.
 */
class CacheBehavior {
    constructor(originId, props) {
        this.props = props;
        this.originId = originId;
        this.validateEdgeLambdas(props.edgeLambdas);
        this.grantEdgeLambdaFunctionExecutionRole(props.edgeLambdas);
    }
    /**
     * Creates and returns the CloudFormation representation of this behavior.
     * This renders as a "CacheBehaviorProperty" regardless of if this is a default
     * cache behavior or not, as the two are identical except that the pathPattern
     * is omitted for the default cache behavior.
     *
     * @internal
     */
    _renderBehavior() {
        var _a, _b, _c, _d, _e, _f;
        return {
            pathPattern: this.props.pathPattern,
            targetOriginId: this.originId,
            allowedMethods: (_a = this.props.allowedMethods) === null || _a === void 0 ? void 0 : _a.methods,
            cachedMethods: (_b = this.props.cachedMethods) === null || _b === void 0 ? void 0 : _b.methods,
            cachePolicyId: ((_c = this.props.cachePolicy) !== null && _c !== void 0 ? _c : cache_policy_1.CachePolicy.CACHING_OPTIMIZED).cachePolicyId,
            compress: (_d = this.props.compress) !== null && _d !== void 0 ? _d : true,
            originRequestPolicyId: (_e = this.props.originRequestPolicy) === null || _e === void 0 ? void 0 : _e.originRequestPolicyId,
            smoothStreaming: this.props.smoothStreaming,
            viewerProtocolPolicy: (_f = this.props.viewerProtocolPolicy) !== null && _f !== void 0 ? _f : distribution_1.ViewerProtocolPolicy.ALLOW_ALL,
            lambdaFunctionAssociations: this.props.edgeLambdas
                ? this.props.edgeLambdas.map(edgeLambda => ({
                    lambdaFunctionArn: edgeLambda.functionVersion.edgeArn,
                    eventType: edgeLambda.eventType.toString(),
                    includeBody: edgeLambda.includeBody,
                }))
                : undefined,
        };
    }
    validateEdgeLambdas(edgeLambdas) {
        const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
        if (edgeLambdas && edgeLambdas.some(lambda => lambda.includeBody && !includeBodyEventTypes.includes(lambda.eventType))) {
            throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
        }
    }
    grantEdgeLambdaFunctionExecutionRole(edgeLambdas) {
        if (!edgeLambdas || edgeLambdas.length === 0) {
            return;
        }
        edgeLambdas.forEach((edgeLambda) => {
            const role = edgeLambda.functionVersion.role;
            if (role && role instanceof iam.Role && role.assumeRolePolicy) {
                role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                }));
            }
        });
    }
}
exports.CacheBehavior = CacheBehavior;
//# sourceMappingURL=data:application/json;base64,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