"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * (experimental) A Origin Request Policy configuration.
 *
 * @experimental
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        const originRequestPolicyName = (_a = props.originRequestPolicyName) !== null && _a !== void 0 ? _a : this.node.uniqueId;
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = (_b = props.cookieBehavior) !== null && _b !== void 0 ? _b : OriginRequestCookieBehavior.none();
        const headers = (_c = props.headerBehavior) !== null && _c !== void 0 ? _c : OriginRequestHeaderBehavior.none();
        const queryStrings = (_d = props.queryStringBehavior) !== null && _d !== void 0 ? _d : OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
    /**
     * (experimental) Imports a Origin Request Policy from its id.
     *
     * @experimental
     */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
}
exports.OriginRequestPolicy = OriginRequestPolicy;
/**
 * (experimental) This policy includes only the User-Agent and Referer headers.
 *
 * It doesn’t include any query strings or cookies.
 *
 * @experimental
 */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/**
 * (experimental) This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin.
 *
 * @experimental
 */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/**
 * (experimental) This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket.
 *
 * @experimental
 */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/**
 * (experimental) This policy includes all values (query strings, headers, and cookies) in the viewer request.
 *
 * @experimental
 */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/**
 * (experimental) This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint.
 *
 * @experimental
 */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/**
 * (experimental) Ddetermines whether any cookies in viewer requests (and if so, which cookies) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * (experimental) Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /**
     * (experimental) All cookies in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /**
     * (experimental) Only the provided `cookies` are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
}
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
/**
 * (experimental) Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * (experimental) HTTP headers are not included in requests that CloudFront sends to the origin.
     *
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * (experimental) All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     * @experimental
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /**
     * (experimental) Listed headers are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
}
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
/**
 * (experimental) Determines whether any URL query strings in viewer requests (and if so, which query strings) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * (experimental) Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /**
     * (experimental) All query strings in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /**
     * (experimental) Only the provided `queryStrings` are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
}
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,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