"""
SPDX-License-Identifier: MIT

  Copyright (c) 2021, SCANOSS

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
"""

import sys


class ScanossBase:
    """
    SCANOSS base class containing default properties and methods to be shared
    """

    def __init__(self, debug: bool = False, trace: bool = False, quiet: bool = False):
        """
        Initialise class
        :param debug:  enable debug
        :param trace:  enable tracing
        :param quiet:  enable quiet mode
        """
        self.debug = debug
        self.quiet = quiet
        self.trace = trace

    @staticmethod
    def print_stderr(*args, **kwargs):
        """
        Print the given message to STDERR
        """
        print(*args, file=sys.stderr, **kwargs)

    def print_msg(self, *args, **kwargs):
        """
        Print message if quite mode is not enabled
        """
        if not self.quiet:
            self.print_stderr(*args, **kwargs)

    def print_debug(self, *args, **kwargs):
        """
        Print debug message if enabled
        """
        if self.debug:
            self.print_stderr(*args, **kwargs)

    def print_trace(self, *args, **kwargs):
        """
        Print trace message if enabled
        """
        if self.trace:
            self.print_stderr(*args, **kwargs)

    @staticmethod
    def print_stdout(*args, **kwargs):
        """
        Print message to STDOUT
        """
        print(
            *args,
            file=sys.stdout,
            **kwargs,
        )

    def print_to_file_or_stdout(self, content: str, file: str = None):
        """
        Print message to file if provided or stdout
        """
        if not content:
            return

        if file:
            with open(file, 'w') as f:
                f.write(content)
        else:
            self.print_stdout(content)

    def print_to_file_or_stderr(self, msg: str, file: str = None):
        """
        Print message to file if provided or stderr
        """
        if not msg:
            return

        if file:
            with open(file, 'w') as f:
                f.write(msg)
        else:
            self.print_stderr(msg)
