# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['edelweiss_data']

package_data = \
{'': ['*']}

install_requires = \
['iso8601>=0.1.11',
 'pandas>=0.23.0',
 'pyjwt>=1.5.3,<2.0.0',
 'requests>=2.18.4,<3.0.0']

setup_kwargs = {
    'name': 'edelweiss-data',
    'version': '0.2.13',
    'description': 'Python client for EdelweissData',
    'long_description': 'This python client library allows easy access to [Edelweiss Data](https://www.saferworldbydesign.com/edelweissdata) servers.\n\n# Table of Contents\n\n- [Overview](#overview)\n- [Getting started](#getting-started)\n  - [Requirements](#requirements)\n  - [Installation](#installation)\n- [Common use cases](#common-use-cases)\n  - [Initialization](#initialization)\n  - [Authentication](#authentication)\n  - [Create a new dataset](#create-a-new-dataset)\n  - [Search for datasets](#search-for-datasets)\n  - [Filter and retrieve data](#filter-and-retrieve-data)\n  - [Delete a new dataset](#delete-a-dataset)\n- [API reference](#api-reference)\n\n# Overview\n\nThe core concept of Edelweiss Data is that of a **Dataset**. A Dataset is a single table of data (usually originating from a csv file) and carries the following additional pieces of information:\n* a **schema** describing the structure of the tabular data (data types, explanatory text for each column etc)\n* a human readable **description text** (markdown formatted - like the readme of a repository on github)\n* a **metadata json** structure (of arbitrary complexity - this can be used to store things like author information, instrument settings used to generate the data, ...).\n\nDatasets are **versioned** through a processes called publishing. Once a version of a dataset is published, it is "frozen" and becomes immutable. Any change to it has to be done by creating a new version. Users of Edelweiss Data will always see the version history of a dataset and be able to ask for the latest version or specific earlier version.\n\nDatasets can be public or **access restricted**. Public datasets can be accessed without any access restrictions. To access restricted datasets or to upload/edit your own dataset OpenIDConnect/OAuth is used - in the python client this process is done by calling the authenticate method on the Api instance that triggers a web based login at the end of which a token is confirmed.\n\nWhen retrieving the tabular data of a dataset, the data can be **filtered and ordered** and only specific columns requested - this makes request for subsets of data much faster than if all filtering happened only on the client. Conditions for filtering and ordering are created by constructing QueryExpression instances using classmethods on the QueryExpression class to create specific Expressions. You can access the data either in it\'s raw form (as json data) or, more conveniently, as a Pandas Dataframe.\n\nJust like the tabular data of one particular dataset can be retrieved as a **Pandas DataFrame**, you can also query for datasets using the same filtering and ordering capabilities - i.e. you can retrieve a DataFrame where each row represents a Dataset with it\'s name, description and optionally metadata and schema (not including the data though).\n\nWhen you are searching for Datasets, a lot of the interesting information that you may want to filter by is hidden in the **metadata** (e.g. maybe most of your datasets have a metadata field "Species" at the top level of the metadata json that indicates from what kind of animal cells the data in this dataset originate from). To make such filtering easy, our Datasets query function take an optional list of "column mappings" that allow you to specify a **JsonPath expression** to extract a field from the metadata and include it with a given name in the resulting DataFrame. In the Species example, you could pass the list of column mappings [("Species from Metadata", "$.Species")] and the resulting DataFrame would contain an additional column "Species from Metadata" and for every row the result of evaluating the JsonPath $.Species would be included in this column and you could filter on it using conditions to e.g. only retrieve Datasets where the Species is set to "Mouse".\n\nEdelweiss Data servers provide a rich **User Interface** as well that let\'s you visually browse and filter datasets and the data (and associated information) of each dataset. This UI is built to integrate nicely with the python client. The DataExplorer that is used to explore a dataset has a button in the upper right corner to generate the python code to get the exact same filtering and ordering you see in the UI into a Pandas DataFrame using the Edelweiss Data library for your convenience.\n\n# Getting started\n\n## Requirements\n\nPython 3.6+\n\n## Installation\n\n```bash\npip install edelweiss_data\n```\n\n# Common use cases\n\n## Initialization\n\nYou interact with the Edelweiss Data API mainly via the API class of the edelweiss_data python library. Import it, point it at the Edelweiss Data instance you want to interact with and instantiate it like this:\n\n```python\nfrom edelweiss_data import API, QueryExpression as Q\n\n# Set this to the url of the Edelweiss Data server you want to interact with\nedelweiss_api_url = \'https://api.develop.edelweiss.douglasconnect.com\'\n\napi = API(edelweiss_api_url)\n\n```\n\n## Authentication\n\nSome operations in Edelweiss Data are accessible without authentication (e.g. retrieving public datasets). For others (e.g. to create datasets), you need to be authenticated. Authentication is done with the authenticate call. Be aware that this call is currently built for interactive use like in a Jupyter environment - it will block execution for a up to a few minutes while it waits for you to log in to your account and confirm the access to the API on your behalf. Once accepted the python client will store the authentication token so that you will not have to enter it again for a few days (the token is stored in your home directory in the .edelweiss directory).\n\n```python\napi.authenticate()\n```\n\n## Create a new dataset\n\nCreating and publishing a new dataset form a csv file can be done in one quick operation like so:\n\n```python\nmetadata = {"metadata-dummy-string": "string value", "metadata-dummy-number": 42.0}\nwith open (\'FILENAME\') as f:\n    dataset = api.create_published_dataset_from_csv_file("DATASETNAME", f, metadata)\n```\n\nThis creates a new dataset form the file FILENAME with the name DATASETNAME. A trivial example metadata is used here as well.\n\nWhen creating and publishing datasets like this you don\'t have a lot of control over details of the schema or to set a more elaborate dataset description. If you need more control, you can create a dataset like so:\n\n```python\ndatafile = \'../../tests/Serialization/data/small1.csv\'\nname = \'My dataset\'\nschemafile = None # if none, schema will be inferred below\nmetadata = None # dict object that will be serialized to json or None\nmetadatafile = None # path to the metadata file or None\ndescription = "This is a *markdown* description that can use [hyperlinks](https://edelweissconnect.com)"\n\ndataset1 = api.create_in_progress_dataset(name)\nprint(\'DATASET:\', dataset1)\ntry:\n    with open(datafile) as f:\n        dataset1.upload_data(f)\n    if schemafile is not None:\n        print(\'uploading schema from file ...\')\n        with open(schemafile) as f:\n            dataset1.upload_schema_file(f)\n    else:\n        print(\'inferring schema from file ...\')\n        dataset1.infer_schema()\n    if metadata is not None:\n        print(\'uploading metadata ...\')\n        dataset1.upload_metadata(metadata)\n    elif metadatafile is not None:\n        print(\'uploading metadata from file ...\')\n        with open(metadatafile) as f:\n            dataset1.upload_metadata_file(f)\n\n    dataset1.set_description(description)\n\n    published_dataset = dataset1.publish(\'My first commit\')\n    print(\'DATASET published:\',published_dataset)\nexcept requests.HTTPError as err:\n    print(\'not published: \', err.response.text)\n```\n\n## Filter and retrieve data\n\nThe tabular data of an individual dataset can be retrieved into a pandas dataframe easily like this:\n\n```python\ndataframe = dataset.get_data()\n```\n\nYou can also filter and order data with QueryExpressions, often aliased to Q in the import statement. In the following example we assume the data to have a column "Species" which we want to filter to the value "Mouse" with fuzzy text matching and "Chemical name" which we want to order by ascending:\n\n```python\ndataframe = dataset.get_data(condition=Q.fuzzy_search(Q.column("Species"), "Mouse"), order_by=[Q.column("Chemical name")])\n```\n\nIn this example you can see how to do a chemical substructure search so that only molecules with the fragment "CC=O" are returned and the results are sorted descending by similarity to the molecule "C(C(CO)(CO)N)O". Chemical similarity for ordering is calculated using the rdkit library using tanimoto distance between rdkit fingerprints (other fingerprints or distance metrics could be supported in the future)\n\n```python\ndataframe = dataset.get_data(condition=Q.substructure_search("CC=O", Q.column("SMILES")), order_by=[Q.tanimoto_similarity("C(C(CO)(CO)N)O", Q.column("SMILES"))], ascending=False)\n```\n\n## Search for datasets\n\nTo just retrieve a pandas dataframe with all published datasets that you are allowed to see use get_published_datasets(). This will return a pandas dataframe with a multiindex of the dataset id and version and then a single object column for the dataset class instance. This class instance can be used to retrieve e.g. the name property of the dataset or it can be used to retrieve the data for this dataset by calling the .get_data() method or similar operations.\n\n```python\ndatasets = api.get_published_datasets()\ndataset = datasets.iloc[0].dataset\nprint("Found {} datasets. The name of the first is: ".format(len(datasets), dataset.name))\n```\n\nTo retrieve a single dataset by index value rather than based on numerical position use the pandas multiindex functionality passing in tuples:\n\n```python\ndatasetid = \'49fd99ee-ec6f-44df-a8cd-73f0f8bbcd76\' # example dataset ID\nversion = 1\ndataset = datasets.loc[(datasetid, version)]\n```\n\nJust like above with data you can use QueryExpressions to filter to only find datasets matching certain predicates. Below we filter on datasets that have the string "LTKB" somewhere in them (name etc)\n\n```python\ndatasets_filter = Q.search_anywhere("LTKB")\ndatasets = api.get_published_datasets(condition=datasets_filter)\n```\n\nSince very often the most interesting filter and sort critieria will be in the metadata (which is a Json of arbitrary structure), the Api gives you a way to add additional columns by extracting pieces from the metadata json with JsonPath expressions. Below we attempt to treat the metadata json of each dataset as an object with a key "Species" and if it is present we extract it and map it into the "Species from metadata json" column:\n\n```python\ncolumns = [("Species from metadata json", "$.Species")]\ndatasets = api.get_published_datasets(columns=columns)\n```\n\nThe result of such a query will always be a column containing lists of results as the jsonpath query could return not just a single primitive value or null or an object but also json arrays. Working with columns that contain lists in Pandas can be a bit awkward as some operations like filtering default to operating on series instead of individual cells. For cases like these, working with the Pandas apply method usually makes things easier - e.g. if you want to extract the first element if any from the above column this an easy way to do it that also extends well to more complex operations:\n\n```python\ndatasets[\'first_species_or_none\'] = datasets.apply(lambda row: row[\'Species from metadata json\'][0] if len(row[\'Species from metadata json\']) > 0 else None, axis=1)\n```\n\n## Delete a dataset\n\nTo delete a dataset and all versions call delete_all_versions:\n\n```python\ndataset.delete_all_versions()\n```\n\n# API reference\n\n## `API`\n\nThe central interaction point with EdelweissData. You instantiate this class with the base url of the\n\nEdelweissData server you want to interact with. The __init__ method will verify that the server is reachable\nand speaks a compatible API version, then you can use instance methods to interact with the API server.\n\n### `add_dataset_group_permission`\n\nAdd a group to a dataset\n\n**dataset_id**  : *the id of the dataset*\n\n**group**  : *the Group to add*\n\n### `add_dataset_user_permission`\n\nAdd a user to a dataset\n\n**dataset_id**  : *the id of the dataset*\n\n**user**  : *the User to add*\n\n### `authenticate`\n\n### `change_dataset_visibility`\n\nSet if the dataset should be public or access protected when published\n\n**dataset_id**  : *the id of the dataset*\n\n**is_public**  : *boolean to indicate if the dataset should be public*\n\n### `create_in_progress_dataset`\n\nCreates a new in-progress dataset on the server and returns it.\n\n**Returns:** The InProgressDataset that was created.\n\n**name**  : *the name of the dataset to create*\n\n### `create_in_progress_dataset_from_csv_file`\n\nCreates a new in-progress dataset from a CSV file on the server.\n\n**Returns:** the updated dataset\n\n**name**  : *the name of the dataset*\n\n**file**  : *opened text file to read the csv data from*\n\n**metadata**  : *dict of the metadata to store as json together with the dataset*\n\n### `create_published_dataset_from_csv_file`\n\nCreates a new published dataset from a CSV file on the server.\n\n**Returns:** the published dataset\n\n**name**  : *the name of the dataset*\n\n**file**  : *opened text file to read the csv data from*\n\n**metadata**  : *dict of the metadata to store as json together with the dataset*\n\n**changelog**  : *Publishing message to store for the first version*\n\n### `delete`\n\nSends a DELETE request to a server.\n\n**Returns:** dict with the JSON response.\n\n**route**  : *route to which the request will be sent*\n\n### `get`\n\nSends a GET request to a server.\n\n**Returns:** dict with the JSON response.\n\n**route**  : *route to which the request will be sent*\n\n**json**  : *dict with the JSON body to send*\n\n### `get_dataset_permissions`\n\nGet the permissions for the given dataset id\n\n**Returns:** the DatasetPermissions instance for this dataset\n\n**dataset_id**  : *the id of the dataset*\n\n### `get_in_progress_dataset`\n\nReturns an in-progress datasets with a given id.\n\n**Returns:** an InProgressDataset\n\n**id**  : *the id of the dataset to retrieve*\n\n### `get_in_progress_datasets`\n\nReturns a list of all in-progress datasets you are allowed to access (needs authentication).\n\n**Returns:** a list of InProgressDatasets\n\n**limit**  : *Number of datasets to retrieve - if None (default) it will retrieve all.*\n\n**offset**  : *Starting offset from which to retrieve the datasets*\n\n### `get_published_dataset`\n\nReturns a published dataset with a given id and version.\n\n**Returns:** the PublishedDataset\n\n**id**  : *id of the dataset to retrieve*\n\n**version**  : *version of the dataset to retrieve. Defaults to LATEST if none specified.*\n\n### `get_published_dataset_aggregations`\n\nReturns aggregation buckets and their sizes for each column.\n\n**Returns:** aggregations as a Series with an index of buckets and terms, for example\nbucket     term\norgan      liver          10\n           kidney         20\nspecies    mouse           5\n           elephant       30\n\n**columns**  : *same as in self.get_published_datasets*\n\n**condition**  : *same as in self.get_published_datasets*\n\n**aggregation_filters**  : *same as in self.get_published_datasets*\n\n### `get_published_dataset_versions`\n\nReturns all published versions of dataset with a given id.\n\n**Returns:** a list of dicts containing id, version and name for each version of the dataset\n\n**id**  : *id of the dataset*\n\n### `get_published_datasets`\n\nReturns a dataframe of all published datasets that match query.\n\n**Returns:** a dataframe indexed by the id and version, which in addition\nto user-specified columns, contains a column with a PublishedDataset\nobject for each dataset. Unless included explicitly, description, schema,\nand metadata are omitted from the datasets and the corresponding\nattributes are set to None. On the first access to any of the missing\nattributes of a given dataset, all three them are fetched from the server\nand set to the actual values, resulting in a single request for each dataset.\nIf there are many datasets for which the attributes are required, it makes\nsense to include the content in the bulk request.\n\n**columns**  : *a list of pairs (column_name, json_path) describing\nthe name of the new column to generate and which jsonpath to use to\nextract the values from the metadata to fill this column.*\n\n**condition**  : *a QueryExpression object limiting the fetched datasets.*\n\n**include_description**  : *a boolean specifying if the datasets in\nthe response should include the description*\n\n**include_schema**  : *a boolean specifying if the datasets in\nthe response should include the schema*\n\n**include_metadata**  : *a boolean specifying if the datasets in\nthe response should include the metadata*\n\n**aggregation_filters**  : *a dict limiting the fetched datasets to ones\nwhere column values fall into one of the selected aggregation buckets.\nFor example, using the dict\n  {\'organ\': [\'liver\', \'kidney\'], \'species\': [\'mouse\', \'elephant\']}\nwould return the datasets where both organ is either liver or kidney,\nAND species is either mouse or elephant.*\n\n**limit**  : *the number of rows to return.\nReturns all rows if set to None (default).*\n\n**offset**  : *the initial offset (default 0).*\n\n**order_by**  : *a list of QueryExpression objects by which to order\nthe resulting datasets.*\n\n**ascending**  : *a boolean or list of booleans to select the ordering.\nIf the single boolean is True (the default), the list is ascending\naccording to order_by, if False, it is descending. If given as a list,\nit must be of the same length as the order_by list, and the order is\nthe ascending/descending for each particular component.*\n\n**dataset_column_name**  : *the name of the dataframe column in which\nthe corresponding PublishedDataset objects are available.*\n\n**latest_only**  : *a boolean specifying whether to return only the latest\nversion of each dataset*\n\n### `get_raw_datasets`\n\nGet the published datasets. Unlike the more high-level get_published_datasets this method\n\ndoes not create a dataframe but returns the raw list of dicts representing the json response.\n     Unless explicity included the fields schema, metadata and description will not be included\n     in the response.\n\n**Returns:** The published datasets as a list of dicts (raw json response)\n\n**columns**  : *a list of pairs (column_name, json_path) describing\ncolumns in the dataframe.*\n\n**condition**  : *a QueryExpression object limiting the fetched datasets.*\n\n**include_description**  : *a boolean specifying if the datasets in\nthe response should include the description*\n\n**include_schema**  : *a boolean specifying if the datasets in\nthe response should include the schema*\n\n**include_metadata**  : *a boolean specifying if the datasets in\nthe response should include the metadata*\n\n**aggregation_filters**  : *a dict limiting the fetched datasets to ones\nwhere column values fall into one of the selected aggregation buckets.\nFor example, using the dict\n  {\'organ\': [\'liver\', \'kidney\'], \'species\': [\'mouse\', \'elephant\']}\nwould return the datasets where both organ is either liver or kidney,\nAND species is either mouse or elephant.*\n\n**limit**  : *the number of rows to return (default 100).\nReturns all rows if set to None.*\n\n**offset**  : *the initial offset (default 0).*\n\n**order_by**  : *a list of QueryExpression objects by which to order\nthe resulting datasets.*\n\n**ascending**  : *a boolean or list of booleans to select the ordering.\nIf the single boolean is True (the default), the list is ascending\naccording to order_by, if False, it is descending. If given as a list,\nit must be of the same length as the order_by list, and the order is\nthe ascending/descending for each particular component.*\n\n**dataset_column_name**  : *the name of the dataframe column in which\nthe corresponding PublishedDataset objects are available.*\n\n**latest_only**  : *a boolean specifying whether to return only the latest\nversion of each dataset*\n\n### `oidc_config`\n\nReturns the OpenID Connect configuration.\n\n### `openapi`\n\nReturns the OpenAPI definition of the entire EdelweissData REST API.\n\n**Returns:** The OpenAPI definition as a dict\n\n### `openapi_documents`\n\nReturns a list of all dataset specific openapi descriptions (i.e. one openapi document for each dataset with the\n\nprecise Json Schema of the particular datasets data endpoint).\n\n**Returns:** A list of url strings at which to retrieve the openapi.json documents for the documents\n\n### `post`\n\nSends a POST request to a server.\n\n**Returns:** dict with the JSON response.\n\n**route**  : *route to which the request will be sent*\n\n**json**  : *dict with the JSON body to send*\n\n### `post_raw`\n\nSends a POST request with a given body to a server.\n\n**Returns:** dict with the JSON response.\n\n**route**  : *route to which the request will be sent*\n\n**body**  : *raw body to send (a bytes object or a string that will be encoded as UTF-8)*\n\n### `remove_dataset_group_permission`\n\nRemove a group from a dataset\n\n**dataset_id**  : *the id of the dataset*\n\n**name**  : *the name of the group to remove*\n\n### `remove_dataset_user_permission`\n\nRemove a user from a dataset\n\n**dataset_id**  : *the id of the dataset*\n\n**user**  : *the email of the user to remove*\n\n### `upload`\n\nUploads a POST request that uploads files to a server.\n\n**Returns:** dict with the JSON response.\n\n**route**  : *route to which the request will be sent*\n\n**files**  : *a dictionary of files in which the keys are filenames\nand corresponding values are file objects*\n\n## `DatasetPermissions`\n\nThe permission information for a dataset. A list of users (email + flag if they can write), groups (name + flag if they can write) and\n\nan is_public field that indicates whether unauthenticated users can see this dataset when published.\n\n### `Group`\n\n#### `decode`\n\n#### `encode`\n\n### `User`\n\n#### `decode`\n\n#### `encode`\n\n### `decode`\n\n### `encode`\n\n## `InProgressDataset`\n\nInProgressDataset - datasets that are not yet published and for which data can be uploaded, the schema modified, metadata changed etc.\n\n### `copy_from`\n\nCopies all content from a PublishedDataset to this InProgressDataset. Useful to create new versions. See also set_data_source for\n\na more lightweight operation if you don\'t need to change the data or schema structure.\n\n### `decode`\n\n### `delete`\n\nDeletes the InProgressDataset\n\n### `encode`\n\n### `get_permissions`\n\n### `infer_schema`\n\nTriggers schema inference from uploaded data (this creates a schema on the server and sets it on the InProgressDataset)\n\n### `publish`\n\nAttempts to publish the dataset. This means that a new version of a PublishedDataset will be created (and returned by this call)\n\nand this InProgressDataset is no longer useable.\n\n### `sample`\n\nRetrieve a list of lists representing a sample of the tabular data of this dataset. This\n\nincludes only a sample (e.g. the first N rows) of the data so that they can be displayed to a\nuser as an example or similar.\n\n### `set_data_source`\n\nSet the data source for an in-progress dataset. This allows you to efficiently re-use the data of a PublishedDataset\n\nto create a new dataset without re-uploading the data. It is also useful if you want to create a new version of a\n    PublishedDataset to fix a mistake in the metadata or description.\n\n**dataset**  : *the PublishedDataset to copy data from when publishing*\n\n### `set_description`\n\nSet the description of the dataset. The description is assumed to be markdown formatted text, similar to a Github README.md\n\n### `upload_data`\n\nUpload tabular data (a CSV file)\n\n**data**  : *An open text file containing the csv data to upload*\n\n### `upload_metadata`\n\nUpload metadata (as a dict, not a file).\n\n**schema**  : *The metadata to upload*\n\n### `upload_metadata_file`\n\nUpload a metadata file (an open text file containing the metadata in Json form).\n\n**file**  : *The open text file to upload the metadata from*\n\n### `upload_schema`\n\nUpload a Schema (an instance of the class, not a file).\n\n**schema**  : *The schema to upload*\n\n### `upload_schema_file`\n\nUpload a schema file (an open text file containing the schema in Json form).\n\n**file**  : *The open text file to upload the schema from*\n\n## `PublishedDataset`\n\nRepresents a published dataset\n\n### `decode`\n\n### `delete_all_versions`\n\nDeletes all versions of a published dataset\n\n### `encode`\n\n### `get_data`\n\nGets the (tabular) data of a PublishedDataset as a pandas Dataframe. The data can be filtered so that only required columns or rows\n\nare retrieved.\n\n**Returns:** A pandas DataFrame with the tabular data\n\n**columns**  : *a list of column names that should appear in the result.\nIf None, all columns are included.*\n\n**condition**  : *a QueryExpression object limiting the fetched datasets.*\n\n**aggregation_filters**  : *a dict limiting the fetched datasets to ones\nwhere column values fall into one of the selected aggregation buckets.\nFor example, using the dict\n  {\'organ\': [\'liver\', \'kidney\'], \'species\': [\'mouse\', \'elephant\']}\nwould return the datasets where both organ is either liver or kidney,\nAND species is either mouse or elephant.*\n\n**limit**  : *the number of rows to return.\nReturns all rows if set to None (default).*\n\n**offset**  : *the initial offset (default 0).*\n\n**order_by**  : *a list of QueryExpression objects by which to order\nthe resulting datasets.*\n\n**ascending**  : *a boolean or list of booleans to select the ordering.\nIf the single boolean is True (the default), the list is ascending\naccording to order_by, if False, it is descending. If given as a list,\nit must be of the same length as the order_by list, and the order is\nthe ascending/descending for each particular component.*\n\n### `get_data_aggregations`\n\nReturns aggregation buckets and their sizes for each column.\n\n**Returns:** aggregations as a Series with an index of buckets and terms, for example\nbucket     term\norgan      liver          10\n           kidney         20\nspecies    mouse           5\n           elephant       30\n\n**columns**  : *a list of column names that should appear in the result.\nIf None, all columns are included.*\n\n**condition**  : *a QueryExpression object limiting the fetched datasets.*\n\n**aggregation_filters**  : *a dict limiting the fetched datasets to ones\nwhere column values fall into one of the selected aggregation buckets.\nFor example, using the dict\n  {\'organ\': [\'liver\', \'kidney\'], \'species\': [\'mouse\', \'elephant\']}\nwould return the datasets where both organ is either liver or kidney,\nAND species is either mouse or elephant.*\n\n### `get_permissions`\n\nReturns the Permissions object of this PublishedDataset\n\n### `get_raw_data`\n\nGets the raw tabular data JSON response for a PublishedDataset. The data can be filtered so that only required columns or rows\n\nare retrieved.\n\n**Returns:** A dict representing the JSON response\n\n**columns**  : *a list of column names that should appear in the result.\nIf None, all columns are included.*\n\n**condition**  : *a QueryExpression object limiting the fetched datasets.*\n\n**aggregation_filters**  : *a dict limiting the fetched datasets to ones\nwhere column values fall into one of the selected aggregation buckets.\nFor example, using the dict\n  {\'organ\': [\'liver\', \'kidney\'], \'species\': [\'mouse\', \'elephant\']}\nwould return the datasets where both organ is either liver or kidney,\nAND species is either mouse or elephant.*\n\n**limit**  : *the number of rows to return.\nReturns all rows if set to None (default).*\n\n**offset**  : *the initial offset (default 0).*\n\n**order_by**  : *a list of QueryExpression objects by which to order\nthe resulting datasets.*\n\n**ascending**  : *a boolean or list of booleans to select the ordering.\nIf the single boolean is True (the default), the list is ascending\naccording to order_by, if False, it is descending. If given as a list,\nit must be of the same length as the order_by list, and the order is\nthe ascending/descending for each particular component.*\n\n### `new_version`\n\nCreate a new version of this PublishedDataset. This will create and return a new InProgressDataset\n\nthat can be filled with content by uploading new files or copying data from a PublishedDataset\n\n**Returns:** The InProgressDataset\n\n### `openapi`\n\nReturns a OpenAPI descriptions for the data endpoint of this PublishedDataset, taking the schema\n\nand thus the precise JSON structure of the response into account.\n\n**Returns:** A dict respresenting the JSON decoded OpenAPI document\n\n## `QueryExpression`\n\nUsed to create filters or expressions to order records by. Use the classmethods on this\n\nclass to create instances, e.g. QueryExpression.fuzzySearch(QueryExpression.column("species"), "Monkey")\n\n### `cast`\n\nCreates a Cast expression. This attempts to convert one datatype into another.\n\n**expr**  : *The expression to cast*\n\n**data_type**  : *The datatype to cast to*\n\n### `column`\n\nConstructs a Column expression.\n\n**column_name**  : *the name of the column*\n\n### `contained_in`\n\nCreates a ContainedIn expression. Tests if an expression is contained in an element. Often used\n\nto check if columns of an Array datatype are contained in a value.\n\n**expr**  : *The expression to search for*\n\n**element**  : *The element to search in*\n\n### `contains`\n\nCreates a Contains expression. Tests if an expression contains an element. Often used\n\nto check if columns of an Array datatype contain a value.\n\n**expr**  : *The expression to search in*\n\n**element**  : *The element to search for*\n\n### `decode`\n\n### `encode`\n\n### `exact_search`\n\nConstructs an ExactSearch expression. Only rows where the expr expression exactly matches the term will be returned. This can be used\n\nto match exact substrings or exact numerical values\n\n**expr**  : *the search expression to evaluate (often a column QueryExpression)*\n\n**term**  : *the search term*\n\n### `fuzzy_search`\n\nConstructs a FuzzySearch expression. Only rows where the expr expression fuzzy-matches the term will be returned. Fuzzy-matching\n\nuses trigram indexing to match slightly different spellings.\n\n**expr**  : *the search expression to evaluate (often a column QueryExpression)*\n\n**term**  : *the search term*\n\n### `search_anywhere`\n\nConstructs a SearchAnywhere expression. Only rows will be returned that contain the search term in one of their text-like columns.\n\n**term**  : *The string to search for in all text-like columns.*\n\n### `substructure_search`\n\nConstructs a SubstructureSearch expression that uses chemical substructure testing. Only rows where the chemical substructure is contained in\n\nthe chemical superstructure are returned.\n\n**substructure**  : *the substructure to search (often a SMILES string constant value)*\n\n**superstructure**  : *the search term (often a Column of datatype SMILES)*\n\n### `system_column`\n\nConstructs a SystemColumn expression. SystemColumns are special columns maintained by EdelweissData.\n\nThe following SystemColumns are available:\n        name (text): the name of a dataset\n        created (text/datetime): the timestamp the dataset was created at\n        version: (int): the version number of the dataset\n\n**column_name**  : *the name of the column*\n\n### `tanimoto_similarity`\n\nCalculates the tanimoto distance between two molecular fingerprints.\n\n**left**  : *the left argument. Often a SMILES string constant value or Column of datatype SMILES.*\n\n**right**  : *the right argument. Often a SMILES string constant value or Column of datatype SMILES.*\n\n## `Schema`\n\nThe schema of the dataset describing the columns (name, description, datatype, rdf predicate, ...)\n\n### `Column`\n\nThe schema data of one column. This tells EdelweissData the name of the column, the datatype to use, how to handle missing values, ...\n\n#### `decode`\n\n#### `encode`\n\n### `decode`\n\n### `encode`\n\n',
    'author': 'Edelweiss Connect',
    'author_email': 'info@edelweissconnect.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/DouglasConnect/edelweiss-data-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
