"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const fc = require("fast-check");
const util_1 = require("./util");
/**
 * Synthesize the given step scaling setup to a template
 */
function setupStepScaling(intervals) {
    const stack = new cdk.Stack();
    const target = util_1.createScalableTarget(stack);
    target.scaleOnMetric('ScaleInterval', {
        metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Success' }),
        scalingSteps: intervals
    });
    return new ScalingStackTemplate(assert_1.SynthUtils.synthesize(stack).template);
}
class ScalingStackTemplate {
    constructor(template) {
        this.template = template;
        this.lowerPolicy = 'TargetScaleIntervalLowerPolicy6F26D597';
        this.lowerAlarm = 'TargetScaleIntervalLowerAlarm4B5CE869';
        this.upperPolicy = 'TargetScaleIntervalUpperPolicy7C751132';
        this.upperAlarm = 'TargetScaleIntervalUpperAlarm69FD1BBB';
    }
    get lowerThreshold() {
        return this.threshold(this.lowerAlarm);
    }
    get upperThreshold() {
        return this.threshold(this.upperAlarm);
    }
    get lowerSteps() {
        return this.steps(this.lowerPolicy);
    }
    get upperSteps() {
        return this.steps(this.upperPolicy);
    }
    allStepsAbsolute() {
        const ret = new Array();
        const lowerThreshold = this.lowerThreshold;
        if (lowerThreshold !== undefined) {
            ret.push(...this.lowerSteps.map(x => makeAbsolute(lowerThreshold, x)));
        }
        const upperThreshold = this.upperThreshold;
        if (upperThreshold !== undefined) {
            ret.push(...this.upperSteps.map(x => makeAbsolute(upperThreshold, x)));
        }
        return ret;
    }
    resource(id) {
        return this.template.Resources[id];
    }
    threshold(id) {
        return apply(this.resource(id), x => x.Properties.Threshold);
    }
    steps(id) {
        return apply(this.resource(id), x => x.Properties.StepScalingPolicyConfiguration.StepAdjustments);
    }
}
function makeAbsolute(threshold, step) {
    return concrete({
        MetricIntervalLowerBound: apply(step.MetricIntervalLowerBound, x => x + threshold),
        MetricIntervalUpperBound: apply(step.MetricIntervalUpperBound, x => x + threshold),
        ScalingAdjustment: step.ScalingAdjustment
    });
}
function overlaps(a, b) {
    return (a.MetricIntervalLowerBound < b.MetricIntervalUpperBound
        && a.MetricIntervalUpperBound > b.MetricIntervalLowerBound);
}
function concrete(step) {
    return {
        MetricIntervalLowerBound: ifUndefined(step.MetricIntervalLowerBound, -Infinity),
        MetricIntervalUpperBound: ifUndefined(step.MetricIntervalUpperBound, Infinity),
        ScalingAdjustment: step.ScalingAdjustment
    };
}
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
function apply(x, f) {
    if (x === undefined) {
        return undefined;
    }
    return f(x);
}
/**
 * Helper function to print variables in case of a failing property check
 */
function reportFalse(cond, ...repr) {
    if (!cond) {
        // tslint:disable-next-line:no-console
        console.error('PROPERTY FAILS ON:', ...repr);
    }
    return cond;
}
module.exports = {
    'alarm thresholds are valid numbers'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const lowerThreshold = template.lowerThreshold;
            const upperThreshold = template.upperThreshold;
            return reportFalse((lowerThreshold === undefined || (lowerThreshold > 0 && lowerThreshold !== Infinity))
                && (upperThreshold === undefined || (upperThreshold > 0 && upperThreshold !== Infinity)), lowerThreshold, upperThreshold);
        }));
        test.done();
    },
    'generated step intervals are valid intervals'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            return reportFalse(steps.every(step => {
                return step.MetricIntervalLowerBound < step.MetricIntervalUpperBound;
            }), steps, 'template', JSON.stringify(template, undefined, 2));
        }));
        test.done();
    },
    'generated step intervals are nonoverlapping'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            for (let i = 0; i < steps.length; i++) {
                const compareTo = steps.slice(i + 1);
                if (compareTo.some(x => overlaps(steps[i], x))) {
                    return reportFalse(false, steps);
                }
            }
            return true;
        }), { verbose: true });
        test.done();
    },
    'all template intervals occur in input array'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            return steps.every(step => {
                return reportFalse(intervals.find(interval => {
                    const acceptableLowerBounds = step.MetricIntervalLowerBound === -Infinity ? [undefined, 0] : [undefined, step.MetricIntervalLowerBound];
                    // tslint:disable-next-line:max-line-length
                    const acceptableUpperBounds = step.MetricIntervalUpperBound === Infinity ? [undefined, Infinity] : [undefined, step.MetricIntervalUpperBound];
                    return (acceptableLowerBounds.includes(interval.lower) && acceptableUpperBounds.includes(interval.upper));
                }) !== undefined, step, intervals);
            });
        }));
        test.done();
    },
    'lower alarm uses lower policy'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const alarm = template.resource(template.lowerAlarm);
            fc.pre(alarm !== undefined);
            return reportFalse(alarm.Properties.AlarmActions[0].Ref === template.lowerPolicy, alarm);
        }));
        test.done();
    },
    'upper alarm uses upper policy'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const alarm = template.resource(template.upperAlarm);
            fc.pre(alarm !== undefined);
            return reportFalse(alarm.Properties.AlarmActions[0].Ref === template.upperPolicy, alarm);
        }));
        test.done();
    },
    'test step scaling on metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const target = util_1.createScalableTarget(stack);
        // WHEN
        target.scaleOnMetric('Tracking', {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            scalingSteps: [
                { upper: 0, change: -1 },
                { lower: 100, change: +1 },
                { lower: 500, change: +5 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "StepScaling",
            ScalingTargetId: {
                Ref: "Target3191CF44"
            },
            StepScalingPolicyConfiguration: {
                AdjustmentType: "ChangeInCapacity",
                MetricAggregationType: "Average",
                StepAdjustments: [
                    {
                        MetricIntervalUpperBound: 0,
                        ScalingAdjustment: -1
                    }
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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