"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error(`Exactly one of 'customMetric' or 'predefinedMetric' must be specified.`);
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error(`Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.`);
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue
            }
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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