"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
    /**
     * Lambda
     */
    ServiceNamespace["LAMBDA"] = "lambda";
    /**
     * Comprehend
     */
    ServiceNamespace["COMPREHEND"] = "comprehend";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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