import logging
from typing import List, Union, Dict, Optional
from uuid import UUID

from slapp_py.helpers.dict_helper import first_key, deserialize_uuids_into_list, serialize_uuids


class TeamsHandler:
    _teams: Dict[UUID, List[UUID]]

    def __init__(self, teams: Optional[Dict[UUID, List[UUID]]] = None):
        self._teams = teams or {}

    @property
    def count(self):
        return len(self._teams)

    @property
    def current_team(self) -> UUID:
        """Get the current team by getting the first key. Ideally, this is checked against the source..."""
        from slapp_py.core_classes.builtins import UnknownTeam
        return first_key(self._teams, UnknownTeam.guid)

    def add(self, incoming: Union[UUID, List[UUID]], sources: Union[UUID, List[UUID]]):
        if not incoming or not sources:
            return
        if not isinstance(incoming, list):
            incoming = [incoming]

        uuids = deserialize_uuids_into_list(sources)

        for to_add in incoming:
            self._teams.setdefault(to_add, []).extend(uuids)

    def merge(self, handler: 'TeamsHandler'):
        for key, value in handler._teams.items():
            self.add(key, value)

    def filter_to_source(self, source_id: Union[str, UUID]) -> 'TeamsHandler':
        search_uuid = source_id if isinstance(source_id, UUID) else UUID(source_id)
        return TeamsHandler(
            teams={k: v for k, v in self._teams.items() if source_id in search_uuid}
        )

    def get_sources_for_team(self, team: UUID) -> List[UUID]:
        """Gets UUID sources for the team, or empty if not found."""
        return self._teams.get(team, [])

    def get_sources_flat(self) -> List[UUID]:
        """Gets UUIDs contained in this information."""
        return list(set([result for sublist in self._teams.values() for result in sublist]))

    def get_teams_unordered(self):
        """Get the teams without the sources."""
        return list(self._teams.keys())

    def get_teams_sourced(self) -> Dict[UUID, List[UUID]]:
        """Get the divisions with sources."""
        return {k: v for k, v in self._teams.items()}

    @staticmethod
    def from_dict(obj: dict) -> 'TeamsHandler':
        assert isinstance(obj, dict)
        try:
            val_dict = obj.get("T")
            result = TeamsHandler()
            for key, value in val_dict.items():
                result.add(UUID(key) if not isinstance(key, UUID) else key, value)
            return result
        except Exception as e:
            logging.exception(exc_info=e, msg=f"Exception occurred loading Divisions Handler: {e}, {e.args}")
            raise e

    def to_dict(self) -> dict:
        result = {}
        if len(self._teams) > 0:
            result["T"] = {k.__str__(): serialize_uuids(v) for k, v in self._teams.items()}
        return result

    def __str__(self):
        return f"{self.count} Teams{(f', current={first_key(self._teams)}' if self.count > 0 else '')}"
